package Cron::Methods::Dictionaries;

use qbit;

use base qw(QBit::Cron::Methods);

use Exception::DB;
use PiConstants qw($MYSQL_MIN_DATE);
use Utils::Logger qw(INFO INFOF ERROR ERRORF);
use Utils::MonitoringUtils;

__PACKAGE__->model_accessors(
    api_yql          => 'QBit::Application::Model::API::Yandex::YQL',
    api_yt           => 'QBit::Application::Model::API::Yandex::YT',
    partner_db       => 'Application::Model::PartnerDB',
    product_manager  => 'Application::Model::ProductManager',
    video_scenaries  => 'Application::Model::VideoScenaries',
    design_templates => 'Application::Model::DesignTemplates',
    bk_statistics    => 'Application::Model::BKStatistics',
);

my $LIMIT                    = 10_000;
my $LIMIT_UPDATE             = 1_000;
my $STAT_PERIOD              = 7;
my $TIMEOUT_RENAME_ALL_PAGES = 5;

sub model_path {'dictionaries'}

sub update_all_sources : CRON('*/10 * * * *') : LOCK : STAGE('TEST') : STAGE('PRODUCTION') {
    my ($self, %opts) = @_;

    my ($accessors) = @opts{qw( --accessors)};

    my $fields = [
        {
            dictionary_field => 'local_id',
            possible_options => [{field => 'id'},]
        },
        {
            dictionary_field => 'model',
            possible_options => [{field => 'model'},],
        },
        {
            dictionary_field => 'domain',
            possible_options => [{field => 'domain'},],
        },
        {
            dictionary_field => 'multistate',
            possible_options => [{field => 'multistate'},],
        },
        {
            dictionary_field => 'is_graysite',
            possible_options => [
                {
                    field => 'is_graysite',
                    check => sub {
                        my ($app, $block_model) = @_;

                        return $block_model eq 'site';
                    },
                },
            ],
        },
        {
            dictionary_field => 'type',
            possible_options => [
                {
                    field => 'type',
                    check => sub {
                        my ($app, $block_model) = @_;

                        return $block_model eq 'mobile_app' || $block_model eq 'internal_mobile_app';
                    },
                },
            ],
        },
    ];

    _get_data_from_model(
        $self->partner_db->all_sources,
        $fields,
        (
            $accessors
            ? [split /,/, $accessors]
            : $self->product_manager->get_source_model_accessors()
        ),
        ignore => TRUE
    );

    return TRUE;
}

sub update_all_pages : CRON('*/2 * * * *') : LOCK : STAGE('TEST') : STAGE('PRODUCTION') {
    my ($self) = @_;

    $self->partner_db->set_isolation_level_read_committed();

    $self->partner_db->update_table(
        'all_pages',
        sub {
            my ($new_table) = @_;

            my $db = $new_table->db;

            my @fields = sort $db->all_pages_view->field_names();

            $new_table->add_multi(
                [
                    $db->query->select(
                        table  => $db->all_pages_view,
                        fields => \@fields,
                    )
                ],
                fields => \@fields
            );
        },
        {
            timeout => $TIMEOUT_RENAME_ALL_PAGES,
            tries   => 3,
        }
    );

    return TRUE;
}

sub update_all_blocks : CRON('*/20 * * * *') : LOCK : STAGE('TEST') : STAGE('PRODUCTION') {
    my ($self) = @_;

    $self->partner_db->set_isolation_level_read_committed();

    $self->partner_db->update_table(
        'all_blocks',
        sub {
            my ($new_table) = @_;

            my $db = $new_table->db;

            my @fields = sort $db->all_blocks_view->field_names();

            $new_table->add_multi(
                [
                    $db->query->select(
                        table  => $db->all_blocks_view,
                        fields => \@fields
                    )
                ],
                fields => \@fields
            );
        }
    );

    return TRUE;
}

sub _get_data_from_model {
    my ($new_table, $fields, $source_models, %opts) = @_;

    my $app = $new_table->db->app;

    foreach my $source_model (sort @$source_models) {
        my $model_fields      = $app->$source_model->get_model_fields();
        my $orig_model_fields = clone($model_fields);

        $model_fields->{'model'} = {db => TRUE, db_expr => \$source_model};

        my $get_fields = {};
        foreach my $def (@$fields) {
            my ($dictionary_field, $possible_options) = @$def{qw(dictionary_field possible_options)};

            foreach my $option (@$possible_options) {
                if (exists($option->{'check'}) && !$option->{'check'}->($app, $source_model)) {
                    next;
                }

                if (exists($model_fields->{$option->{'field'}})) {
                    $get_fields->{$dictionary_field} = $option->{'field'};

                    if ($dictionary_field ne $option->{'field'}) {
                        if ($model_fields->{$option->{'field'}}{'db'}
                            && !exists($model_fields->{$option->{'field'}}{'get'}))
                        {
                            $model_fields->{$dictionary_field} = {
                                db      => TRUE,
                                db_expr => $option->{'field'},
                            };
                        } else {
                            $model_fields->{$dictionary_field} = {
                                depends_on => [$option->{'field'}],
                                get        => sub {$_[1]->{$option->{'field'}}}
                            };
                        }
                    }

                    last;
                }
            }
        }

        foreach (grep {!exists($get_fields->{$_})} map {$_->{'dictionary_field'}} @$fields) {
            $get_fields->{$_} = $_;

            $model_fields->{$_} = {db => TRUE, db_expr => \undef};
        }

        $app->$source_model->model_fields($model_fields);

        my $offset = 0;
        my $data   = [];

        while (
            @{
                $data = $app->$source_model->get_all(
                    fields   => [sort keys(%$get_fields)],
                    order_by => $app->$source_model->partner_db_table->primary_key,
                    offset   => $offset,
                    limit    => $LIMIT,
                )
            }
          )
        {
            $new_table->add_multi($data, %opts);

            $offset += $LIMIT;
        }

        $app->$source_model->model_fields($orig_model_fields);
    }
}

sub update_yt_vmap : CRON('*/5 * * * *') : LOCK : STAGE('TEST') : STAGE('PRODUCTION') {
    my ($self, $opts) = @_;

    my $data = $self->video_scenaries->get_all(
        fields   => [qw(page_id id caption)],
        order_by => [qw(id)],
    );
    $_ = {
        hash_transform(
            $_,
            [],
            {
                page_id => 'PageID',
                id      => 'VmapID',
                caption => 'VmapName',
            }
        )
    } foreach @$data;
    $self->api_yt->rewrite_table_on_cluster(
        path   => '//home/partner/dict/vmap',
        data   => $data,
        params => {
            timeout  => 20,
            attempts => 3,
            delay    => 0,
        }
    );

    Utils::MonitoringUtils::send_to_graphite(
        interval => 'five_min',
        path     => 'vmap.export_dict',
        value    => scalar @$data,
    );
}

sub update_design_templates : CRON('1 * * * *') : LOCK : FREQUENCY_LIMIT('1d') : STAGE('PRODUCTION') {
    my ($self) = @_;

    my $start_date = date_sub(curdate(), oformat => 'db', day => $STAT_PERIOD);
    my $finish_date = curdate(oformat => 'db');
    my $last = 0;

    while (
        my @list = @{
            $self->design_templates->get_all(
                distinct => TRUE,
                fields   => [qw(page_id)],
                filter   => [
                    'AND',
                    [
                        [    # Основной фильтр
                            'OR',
                            [
                                # Не удалённые (по ним всегда обновляем)
                                ['multistate', '<>', 'deleted'],

                                # Или которые ещё ни разу не обрабатывали (без учёта удалённости)
                                ['stat_last_date', 'IS', undef],
                            ]
                        ],

                        # Итерационный фильтр, работает быстрее, чем limit-offset
                        ['page_id', '>', $last],
                    ]
                ],
                limit    => $LIMIT,
                order_by => [qw(page_id)],
            )
        }
      )
    {
        # Обновление итератора
        $last = $list[-1]->{page_id};

        my %page_id_list = map {$_->{page_id} => 1} @list;
        my $pages = [sort keys %page_id_list];

        # Получение статы по выбранным пейджам за заданный период с разбивкой по дизайнам
        my $stat = $self->bk_statistics->get_statistics2(
            "entity_fields" => ["design_id"],
            "fields"        => ["shows"],
            "period"        => [$start_date, $finish_date],
            "levels"        => [
                {
                    "filter" => ["AND", [["design_id", "<>", 0], ["page_id", "IN", $pages],]],
                    "id" => "payment"
                }
            ],
            "order_by" => [{"field" => "design_id", "dir" => "asc"}],
        );
        my %design_dates;
        foreach my $point (@{$stat->{points}}) {
            my $design_id = $point->{dimensions}{design_id};
            $design_dates{$design_id} = $point->{measures}[0]{shows} ? $start_date : $MYSQL_MIN_DATE;
        }

        my $designs = $self->design_templates->get_all(
            fields   => [qw(id stat_last_date)],
            filter   => ['page_id', 'IN', $pages],
            order_by => [qw(id)],
        );

        my @designs_with_stat;
        my @designs_without_stat;
        foreach (@$designs) {
            if ($design_dates{$_->{id}}) {
                if ($design_dates{$_->{id}} eq $MYSQL_MIN_DATE) {
                    push @designs_without_stat, $_->{id}
                      if (!$_->{stat_last_date} || $design_dates{$_->{id}} gt $_->{stat_last_date});
                } else {
                    push @designs_with_stat, $_->{id}
                      if (!$_->{stat_last_date} || $design_dates{$_->{id}} gt $_->{stat_last_date});
                }
            } else {
                push @designs_without_stat, $_->{id} unless ($_->{stat_last_date});
            }
        }

        if (scalar @designs_with_stat) {
            INFO scalar @designs_with_stat . " designs WITH stat found, updating\n";
            while (my @chunk = splice @designs_with_stat, 0, $LIMIT_UPDATE) {
                $self->partner_db->design_templates->edit(\@chunk,
                    {opts => {json_set => ['opts', \'$.stat_last_date', \$start_date]}});
            }
        }

        if (scalar @designs_without_stat) {
            INFO scalar @designs_without_stat . " designs WITHOUT stat found, updating\n";
            while (my @chunk = splice @designs_without_stat, 0, $LIMIT_UPDATE) {
                $self->partner_db->design_templates->edit(\@chunk,
                    {opts => {json_set => ['opts', \'$.stat_last_date', \$MYSQL_MIN_DATE]}});
            }
        }
    }

    return TRUE;
}

sub cleanup_design_templates : CRON('31 * * * *') : LOCK : STAGE('PRODUCTION') {
    my ($self) = @_;

    my $date_today = curdate(oformat => 'db');
    my $date_2_years_ago = trdate('norm', 'db', date_sub(curdate(), day => 365 * 2 - 1));

    my $limit = 5000;
    while (
        my @list = @{
            $self->design_templates->get_all(
                filter => ['AND', [['multistate', '=', 'deleted'], ['stat_last_date', '<', $date_2_years_ago]]],
                fields => [qw(id)],
                limit  => $limit,
            )
        }
      )
    {
        my %design_id_list = map {$_->{id} => 1} @list;

        INFO scalar @list . " deleted designs without stat since $date_2_years_ago found\n";

        try {
            $self->app->design_templates_action_log->partner_db_table()
              ->delete($self->partner_db->filter(['elem_id' => 'IN' => \[map {$_->{id}} @list]]));
            my $deleted_count = $self->design_templates->partner_db_table()->delete([map {{id => $_->{id}}} @list]);
            INFO $deleted_count, " design_templates deleted\n";
        }
        catch {
            ERROR $_[0]->message;
        };
    }

    return TRUE;
}

TRUE;
