package Cron::Methods::DictionariesYT;

use qbit;
use base qw(QBit::Cron::Methods);

use PiConstants qw($VIDEO_BLOCK_TYPES);
use Utils::Logger qw(INFO ERROR);

__PACKAGE__->model_accessors(
    api_yt          => 'QBit::Application::Model::API::Yandex::YT',
    product_manager => 'Application::Model::ProductManager',
    partner_db      => 'Application::Model::PartnerDB',
);

my $LIMIT = 50_000;

sub model_path {'dictionaries_yt'}

sub export_dsp : CRON('0 * * * *') : LOCK : STAGE('TEST') : STAGE('PRODUCTION') {
    my ($self) = @_;

    my $query = {
        model  => 'dsp',
        fields => [
            qw(
              display_name
              id
              login
              multistate
              postmoderated
              short_caption
              show_probability
              skipnoud
              tag
              types
              unmoderated_rtb_auction
              url
              )
        ],
        order_by => [qw(id)],
    };

    my $callback = sub {
        my ($obj) = ($_);

        my $multistate = delete $obj->{multistate};
        $obj->{disabled} = !$self->app->dsp->check_multistate_flag($multistate, 'working');
        $obj->{use_pnocsy} = $self->app->dsp->_get_use_pnocsy_value($obj->{tag} // '');
        $obj->{$_} = ($obj->{$_} ? 1 : 0)
          foreach (qw(disabled postmoderated skipnoud unmoderated_rtb_auction use_pnocsy));
        if (exists($obj->{types})) {
            my $bk_type;
            $bk_type |= 2**$_ foreach @{$obj->{types}};
            $obj->{dsp_types} = $bk_type;
        } else {
            $obj->{dsp_types} = 0;
        }

        my %record = hash_transform(
            $obj,
            [],
            {
                id               => 'DSPID',
                login            => 'Login',
                tag              => 'DSPTag',
                dsp_types        => 'DSPType',
                show_probability => 'ShowProbability',
                short_caption    => 'Title',
                display_name     => 'TitleExternal',
                url              => 'Url',
            }
        );
        $record{Options} =
          join(',', grep {$obj->{$_}} qw(disabled postmoderated skipnoud unmoderated_rtb_auction use_pnocsy));
        $record{Options} =~ s/_/-/g;
        $record{$_} += 0 foreach (qw(DSPID DSPType ShowProbability));
        \%record;
    };

    return $self->_export_dictionary_table('//home/partner/dict/dsp', $query, $callback);
}

# DO NOT overlap with dictionaries->update_all_blocks
sub export_block : CRON('0 * * * *') : LOCK : STAGE('TEST') : STAGE('PRODUCTION') {
    my ($self) = @_;

    my $query = {
        table  => 'all_blocks',
        filter => ["id", "!=", \[100500]],
        fields => [
            qw(
              page_id
              id
              caption
              mobile_block_type
              video_block_type
              adfox_block
              multistate
              model
              site_version
              place_id
              )
        ],
        order_by => [qw(id_autoincrement)],
    };

    my $callback = sub {
        my ($obj, $show_video_block_hash) = @_;

        if (defined $obj->{mobile_block_type}) {
            $obj->{block_type} = 'app-' . $obj->{mobile_block_type};
        } elsif (defined $obj->{video_block_type} && exists $VIDEO_BLOCK_TYPES->{$obj->{video_block_type}}) {
            $obj->{block_type} = 'video-' . $VIDEO_BLOCK_TYPES->{$obj->{video_block_type}}{bk};
        } else {
            $obj->{block_type} = '';
        }

        $obj->{adfox_block} = $obj->{adfox_block} ? JSON::XS::true : JSON::XS::false;

        $obj->{show_video} =
          exists $show_video_block_hash->{$obj->{page_id}}{$obj->{id}} ? JSON::XS::true : JSON::XS::false;

        my %record = hash_transform(
            $obj,
            [],
            {
                page_id      => 'PageID',
                id           => 'ImpID',
                caption      => 'BlockCaption',
                block_type   => 'BlockType',
                adfox_block  => 'optionsAdfox',
                multistate   => 'MultiState',
                model        => 'BlockModel',
                site_version => 'SiteVersion',
                show_video   => 'ShowVideo',
                place_id     => 'PlaceID',
            }
        );

        foreach (qw(PageID ImpID MultiState PlaceID)) {
            $record{$_} = defined $record{$_} ? $record{$_} + 0 : 0;
        }

        foreach (qw(SiteVersion)) {
            $record{$_} //= '';
        }

        \%record;
    };

    my $block_with_show_video_hash = $self->_get_block_with_show_video_hash;

    return $self->_export_dictionary_table('//home/partner/dict/block', $query, $callback, $block_with_show_video_hash);
}

sub _get_block_with_show_video_hash {
    my ($self) = @_;

    my $result = {};

    my $query;
    for my $model_name (@{$self->product_manager->get_block_model_names}) {

        my $table = $self->app->$model_name->partner_db_table();

        next unless $table->have_fields('show_video');

        my $page_id_name = $self->app->$model_name->get_page_id_field_name;

        my $query_addition = $self->partner_db->query->select(
            table  => $table,
            fields => {page_id => $page_id_name, id => ''},
            filter => {show_video => 1, ($table->have_fields('model') ? (model => $model_name) : ())}
        );

        if ($query) {
            $query->union_all($query_addition);
        } else {
            $query = $query_addition;
        }
    }
    if ($query) {
        map {$result->{$_->{page_id}}{$_->{id}} = 1} @{$query->get_all};
    }

    return $result;
}

sub export_pages : CRON('36 * * * *') : LOCK : STAGE('TEST') : STAGE('PRODUCTION') {
    my ($self) = @_;

    # NOTE! пейжди Дистрибуции не должны попадать dict/pages, иначе менеджеры будут видеть
    #       пейджи Дистрибуции в МОЛ, к которым у них нет доступа, потому что
    #       эта таблица вляется справочником для МОЛ, для флага "isPi2"
    my $except_page_accessors = $self->app->product_manager->get_special_page_model_accessors();

    my $query = {
        table    => 'all_pages',
        fields   => [qw(model is_internal page_id caption login domain domain_id client_id send_time)],
        filter   => ['model', 'NOT IN', \$except_page_accessors],
        order_by => [qw(page_id)],
    };

    my $callback = sub {
        my ($obj, $extra_fields_by_page_hash) = @_;
        my %record = %{$obj};
        $record{page_model}  = delete $record{model};
        $record{source_id}   = delete $record{domain_id};
        $record{is_internal} = $record{is_internal} ? JSON::XS::true : JSON::XS::false;
        foreach my $f (qw(developer apple_store_id)) {
            $record{$f} =
              exists $extra_fields_by_page_hash->{$record{page_id}}
              ? $extra_fields_by_page_hash->{$record{page_id}}{$f} // ''
              : '';
        }
        $record{$_} += 0 foreach (qw(page_id source_id client_id));
        \%record;
    };

    my %extra_page_fields =
      map {$_->{page_id} => {developer => $_->{developer}, apple_store_id => $_->{apple_store_id},};}
      @{$self->app->mobile_app_settings->get_all(fields => [qw(page_id developer apple_store_id)])};

    return $self->_export_dictionary_table('//home/partner/dict/pages', $query, $callback, \%extra_page_fields);
}

sub export_partner_stat_id : CRON('37 * * * *') : LOCK : STAGE('TEST') : STAGE('PRODUCTION') {
    my ($self) = @_;

    my $query = {
        table    => 'block_tags',
        fields   => [qw( page_id tag_id caption )],
        order_by => [qw(page_id tag_id)],
    };

    my $callback = sub {
        my ($obj) = (@_);
        map {$obj->{$_} += 0} qw(page_id tag_id);
        $obj;
    };

    # NOTE! К сожалению таблица dict/block_tags уже занята (PI-20478)
    return $self->_export_dictionary_table('//home/partner/dict/partner_stat_id', $query, $callback);
}

sub export_users : CRON('47 * * * *') : LOCK : STAGE('TEST') : STAGE('PRODUCTION') {
    my ($self) = @_;

    my $query = {
        table  => 'users',
        fields => [
            qw( id login is_adfox_partner  is_efir_blogger  is_tutby  is_video_blogger  is_games  has_mobile_mediation  )
        ],
        order_by => [qw(id)],
    };

    my $callback = sub {
        my ($obj) = (@_);
        map {$obj->{$_} += 0} qw(id);
        map {$obj->{$_} = $obj->{$_} ? JSON::XS::true : JSON::XS::false}
          qw(is_adfox_partner  is_efir_blogger  is_tutby  is_video_blogger  is_games has_mobile_mediation);
        $obj;
    };

    return $self->_export_dictionary_table('//home/partner/dict/users', $query, $callback);
}

sub export_product_levels : CRON('57 * * * *') : LOCK : STAGE('TEST') : STAGE('PRODUCTION') {
    my ($self) = @_;

    # my $levels_dict = MenuHierarchy::get_product_levels_dict($self->app);
    my $levels_dict = _get_product_levels_dict($self->app);

    $self->_insert_rows_into_yt_table('//home/partner/dict/product_levels',
        [map {$levels_dict->{$_}} sort keys %$levels_dict]);

    return 1;
}

sub _get_product_levels_dict {
    my ($app) = @_;

    my $levels_dict = {
        (map {$_ => {model => $_, type => 'page'}} @{$app->product_manager->get_page_model_accessors()}),
        (map {$_ => {model => $_, type => 'block'}} @{$app->product_manager->get_block_model_accessors()})
    };

    my $locales = $app->get_locales();
    foreach my $lang (keys %$locales) {
        my $tmp_locale = $app->set_tmp_app_locale($lang);

        foreach my $accessor (keys %$levels_dict) {
            $levels_dict->{$accessor}->{'name_' . $lang} = $app->$accessor->get_product_name() // '';
        }
    }

    return $levels_dict;
}

sub export_design_templates : CRON('*/40 * * * *') : LOCK : STAGE('TEST') : STAGE('PRODUCTION') {
    my ($self) = @_;

    my $stat_last_date = date_sub(curdate(), month => 24, oformat => 'db');

    my $query = {
        table  => 'design_templates',
        filter => [{json_unquote => [{json_extract => ['opts', \'$.stat_last_date']}]}, '>=', \$stat_last_date],
        fields => {
            (map {$_ => ''} qw( id caption type)),
            'format_name' => {json_unquote => [{json_extract => ['opts', \'$.design_settings.name']}]},
        },
        order_by => [qw(id)],
    };

    my $callback = sub {
        my ($obj) = ($_);
        my %record = %{$obj};
        $record{$_} += 0 foreach (qw(id));
        $record{$_} //= '' foreach (qw(caption format_name type));
        \%record;
    };

    $self->_export_dictionary_table('//home/partner/dict/design_templates', $query, $callback);

    my $data = $self->api_yt->read_table_from_any_replica(
        replicas => $self->app->get_option('yt')->{'replicas'},
        path     => '//home/partner/dict/common_design_templates',
        headers  => {'X-YT-Output-Format' => 'json',},
        params   => {
            timeout  => 20,
            attempts => 3,
            delay    => 0,
        }
    );

    my @rows = map {from_json($_)} split(/\n/, $data);

    if (@rows) {
        $self->_insert_rows_into_yt_table('//home/partner/dict/design_templates', \@rows);
    }
}

sub _export_dictionary_table {
    my ($self, $yt_table_path, $query, $callback, @callback_parameters) = @_;

    unless ($yt_table_path && $query) {
        ERROR('yt_table_path, query parameters are required');
        return;
    }

    $callback //= sub {$_};

    my $model_name = delete $query->{model};
    my $table_name = delete $query->{table};

    my $model =
        $model_name
      ? $self->app->$model_name
      : $self->app->partner_db->$table_name;

    my ($offset, $limit) = (0, $LIMIT);
    while (
        my @list = map {&$callback($_, @callback_parameters)} @{
            $model->get_all(
                fields   => $query->{fields},
                order_by => $query->{order_by},
                $query->{filter} ? (filter => $query->{filter}) : (),
                limit  => $limit,
                offset => $offset,
            )
        }
      )
    {
        $offset += $limit;

        $self->_insert_rows_into_yt_table($yt_table_path, \@list);
    }
}

sub _insert_rows_into_yt_table {
    my ($self, $yt_table_path, $data) = @_;

    $self->api_yt->insert_rows(
        path                 => $yt_table_path,
        data                 => $data,
        require_sync_replica => FALSE,
        is_delete            => 0,
        params               => {
            ':timeout'  => 20,
            ':attempts' => 3,
            ':delay'    => 0,
        }
    );
}

TRUE;
