package Cron::Methods::Domain;

use qbit;

use base qw(QBit::Cron::Methods);

use Utils::MonitoringUtils;
use Exception::IncorrectParams;
use PiConstants qw($MANAGERS_MAILING_LIST_FOR_FOCUS_DOMAINS);
use Utils::Logger qw(WARN);

__PACKAGE__->model_accessors(
    domain     => 'Application::Model::Domain',
    kv_store   => 'QBit::Application::Model::KvStore',
    mailer     => 'Application::Model::SendMail',
    partner_db => 'Application::Model::PartnerDB',
);

sub model_path {'domain'}

my $LIMIT_OF_ROWS_PER_QUERY = 500;

#sub auto_confirm : CRON('*/5 * * * *') : LOCK {
#    my ($self) = @_;
#
#    $self->domain->auto_confirm();
#}

sub get_new_owners_kv_key {
    'focus_domains_for_managers_mailing_lists'
}

sub check_new_owners : CRON('*/5 * * * *') : LOCK {
    my ($self, $opts) = @_;

    my $from_date = $self->kv_store->get($self->get_new_owners_kv_key());
    throw Exception::IncorrectParams 'From-date is not specified',
      sentry => {fingerprint => ['Cron', 'Domain', 'check_new_owners', 'From-date is not specified']}
      unless $from_date;

    my $to_date = date_sub(curdate(), minute => 5, oformat => 'db_time');
    throw Exception::IncorrectParams "From-date is after to-date", sentry => {
        extra => {
            from_date => $from_date,
            to_date   => $to_date,
        },
        fingerprint => ['Cron', 'domain', 'check_new_owners', 'From-date is after to-date'],
      }
      if (compare_dates($from_date, $to_date, iformat1 => 'db_time', iformat2 => 'db_time',) > 0);

    my $site = $self->partner_db->query->select(
        table  => $self->partner_db->owner_site,
        fields => [qw(domain_id user_id create_date)],
        filter => [
            AND => [
                [create_date => '>=' => \$from_date],    # включительно
                [create_date => '<'  => \$to_date],      # не включительно
            ]
        ],
      )->join(
        table   => $self->partner_db->users,
        fields  => [qw(login)],
        join_on => [id => '=' => {user_id => $self->partner_db->owner_site}],
      )->join(
        table   => $self->partner_db->site,
        fields  => [qw(domain)],
        join_on => [id => '=' => {domain_id => $self->partner_db->owner_site}],
      )->join(
        table   => $self->partner_db->focus_domains_for_managers_mailing_lists,
        fields  => [qw(tier)],
        join_on => [
            'OR' => [
                # Точное совпадение
                [domain => '=' => {domain => $self->partner_db->site}],
                # Или поддомен
                [{domain => $self->partner_db->site} => 'LIKE' => {'CONCAT' => [\'%.', 'domain']}],
            ],
        ],
      )->limit($LIMIT_OF_ROWS_PER_QUERY)->get_all();

    my $video_an_site = $self->partner_db->query->select(
        table  => $self->partner_db->video_an_site,
        fields => {
            create_date => '',
            domain      => '',
            user_id     => 'owner_id',
        },
        filter => [
            AND => [
                [create_date => '>=' => \$from_date],    # включительно
                [create_date => '<'  => \$to_date],      # не включительно
            ]
        ],
      )->left_join( # Кроме тех пар домен-владелец, которые уже встречались
        table => $self->partner_db->query->select(
            table  => $self->partner_db->video_an_site,
            fields => [qw(domain owner_id)],
            filter => [create_date => '<' => \$from_date],
          )->group_by(qw(domain owner_id)),
        alias   => 'already',
        join_on => [
            AND => [
                [domain   => '=' => {domain   => $self->partner_db->video_an_site}],
                [owner_id => '=' => {owner_id => $self->partner_db->video_an_site}],
            ]
        ],
        filter => [domain => '=' => \undef],
      )->join(
        table   => $self->partner_db->users,
        fields  => [qw(login)],
        join_on => [id => '=' => {owner_id => $self->partner_db->video_an_site}],
      )->join(
        table   => $self->partner_db->focus_domains_for_managers_mailing_lists,
        fields  => [qw(tier)],
        join_on => [
            'OR' => [
                # Точное совпадение
                [domain => '=' => {domain => $self->partner_db->video_an_site}],
                # Или поддомен
                [{domain => $self->partner_db->video_an_site} => 'LIKE' => {'CONCAT' => [\'%.', 'domain']}],
            ],
        ],
      )->limit($LIMIT_OF_ROWS_PER_QUERY)->get_all();

    foreach my $dm (@$site, @$video_an_site) {
        $self->mailer->send(
            body => sprintf('Domain "%s" (id: %s, tier: %d) has new owner "%s" (id: %d) since %s',
                map {$_ // '--'} @{$dm}{qw(domain domain_id tier login user_id create_date)}),
            content_type => 'text/plain',
            from         => 'default',
            subject      => sprintf('tier: %d, domain: %s', $dm->{tier}, $dm->{domain}),
            to => sprintf($MANAGERS_MAILING_LIST_FOR_FOCUS_DOMAINS, $dm->{tier}),
        );
    }

    $self->kv_store->set($self->get_new_owners_kv_key(), $to_date);
}

sub get_host_from_content_block : CRON('*/5 * * * *') : LOCK {
    my ($self, $opts) = @_;

    my @params = (
        filter => {multistate => 'not deleted'},
        fields => ['news_params', 'bk_data', 'is_custom_bk_data', 'campaign_id', 'id']
    );

    my $ca = $self->app->context_on_site_content->get_all(@params);
    my $ci = $self->app->internal_context_on_site_content->get_all(@params);

    my %uniq_host;
    for my $row ((@$ca, @$ci)) {
        my $candidates;
        if ($row->{is_custom_bk_data}) {
            try {
                $candidates = defined $row->{bk_data}
                  && from_json($row->{bk_data})->{CustomBlockData}{WidgetInfo}{NewsParams}{candidates};
            }
            catch {
                my ($e) = @_;
                WARN {
                    exception => $e,
                    extra     => {
                        page_id  => $row->{campaign_id},
                        block_id => $row->{id},
                        message  => "Error parsing bk_data",
                    },
                };
            };
        } else {
            $candidates = $row->{news_params}{candidates};
        }
        for my $candidate (@{$candidates || []}) {
            $uniq_host{$candidate->{host}} = TRUE;
        }
    }

    my @hosts = map [$_], sort keys %uniq_host;

    $self->app->api_yt->rewrite_table_on_cluster(
        path         => '//home/partner/domains_for_news',
        data         => \@hosts,
        column_names => ['host'],
        order_by     => [
            {
                "name"       => "host",
                "type"       => "string",
                "sort_order" => "asc",
            }
        ],
        params => {
            timeout  => 60,
            attempts => 3,
            delay    => 0,
        }
    );

    Utils::MonitoringUtils::send_to_graphite(
        interval => 'five_min',
        path     => 'News.export_hosts',
        value    => scalar @hosts,
    );
}

1;
