package Cron::Methods::FastBan;

use qbit;

use base qw(QBit::Cron::Methods Cron::Methods);

use Utils::DB qw(fields_to_filter);
use Utils::Logger qw(INFOF);
use Utils::MonitoringUtils qw(send_to_graphite);

__PACKAGE__->model_accessors(
    partner_db => 'Application::Model::PartnerDB',
    api_yt     => 'QBit::Application::Model::API::Yandex::YT',
    fast_ban   => 'Application::Model::Block::Content::FastBan',
);

sub model_path {'fast_ban'}

sub export_fast_ban : CRON('*/5 * * * *') : LOCK {
    my ($self) = @_;

    my $bans = $self->fast_ban->get_all(
        fields => [
            qw(
              id
              create_date
              model
              page_id
              block_id
              url_md5_hex
              url
              )
        ],
        order_by => ['id']
    );

    my %bans_by_model = ();
    foreach (@$bans) {
        push(@{$bans_by_model{$_->{'model'}}}, {page_id => $_->{'page_id'}, id => $_->{block_id}});
    }

    my $domain_by_pk = {};
    foreach my $model (sort keys(%bans_by_model)) {
        my $blocks = $self->app->$model->get_all(
            fields => [qw(page_id id domain)],
            filter => fields_to_filter([qw(page_id id)], $bans_by_model{$model})
        );

        $domain_by_pk->{$_->{'page_id'}}{$_->{'id'}} = $_->{'domain'} foreach @$blocks;
    }

    $self->app->api_yt->rewrite_table_on_cluster(
        path         => '//home/partner/recommendation_widget_fast_ban',
        data         => [map {$_->{'domain'} = $domain_by_pk->{$_->{'page_id'}}{$_->{'block_id'}}; $_} @$bans],
        column_names => [
            qw(id
              create_date
              model
              page_id
              block_id
              url_md5_hex
              url
              domain
              )
        ],
        order_by => [
            {
                "name"       => "id",
                "type"       => "number",
                "sort_order" => "asc",
            }
        ],
        params => {
            timeout  => 60,
            attempts => 3,
            delay    => 0,
        }
    );

    send_to_graphite(
        interval => 'five_min',
        path     => 'export_fast_ban',
        value    => scalar @$bans,
    );
}

sub clear_old : CRON('0 5 * * *') : LOCK {
    my ($self) = @_;

    my $date = date_sub(curdate(), day => 30, oformat => 'db');
    INFOF 'Remove fast ban records older than %s', $date;
    my $count = $self->fast_ban->partner_db_table()->delete($self->partner_db->filter(['create_date', '<', \$date]));
    INFOF '%d were removed', $count;

    send_to_graphite(
        interval => 'one_day',
        path     => 'clear_fast_ban',
        value    => $count,
    );
}

TRUE;
