package Cron::Methods::Fias;

use qbit;
use Utils::Logger;
use File::Temp qw(tempfile);

use base qw(QBit::Cron::Methods);

use PiConstants qw($FIAS_UPDATE_DATABASES);

use Exception::Fias;

__PACKAGE__->model_accessors(
    api_yt     => 'QBit::Application::Model::API::Yandex::YT',
    partner_db => 'Application::Model::PartnerDB',
);

sub model_path {'fias'}

=head2 update_fias

=cut

sub update_fias : CRON('0 * * * *') : LOCK : FREQUENCY_LIMIT('1d') : TTL('2h') {
    my ($self) = @_;

    my $tmp_file_name = (
        tempfile(
            DIR    => $self->get_option('tmp_dir'),
            UNLINK => 1,
        )
    )[1];

    # Выкачиваем из YT справочник ФИАС (то что раньше называлось КЛАДР) и сохраняеи его в указнанный файл.
    # Ссылка на эту таблицу в вебинтерфейсе YT:
    # L<https://yt.yandex-team.ru/freud/#page=navigation&path=//home/balance-partners/fias/t_fias/latest&offsetMode=row>
    # Формат файла - jsonl, т.е. каждая строка файла - это json. Пример json из одной строки файла:
    #
    # {
    #     "SHORT_NAME": "ул",
    #     "SCN": 32,
    #     "POSTCODE": "350087",
    #     "CENTER_STATUS": "0",
    #     "CHUNK": 0,
    #     "FORMAL_NAME": "Инжирная",
    #     "GUID": "42d8b0d3-9e2c-4854-8824-668c7e3962bf",
    #     "KLADR_CODE": "230000010003337",
    #     "LIVE_STATUS": "1",
    #     "OBJ_LEVEL": "7",
    #     "PARENT_GUID": "6773f62e-6586-43ee-aa78-297928fc99ea"
    # }
    my $data = $self->api_yt->read_table(
        host    => 'freud',
        path    => '//home/balance-partners/fias/t_fias/latest',
        headers => {'X-YT-Parameters' => '{output_format=<encode_utf8=%false>json}',},
        params  => {
            timeout         => 300,
            attempts        => 1,
            delay           => 0,
            timeout_retry   => 0,
            ':content_file' => $tmp_file_name
        },
    );
    # На момент написания этого кода (2017-09-19) размер файла, который выкачивает эта саба составлял 365 мегабайт.
    # Код ниже проверят что размер файла не меньше чем 200 мегабайт. Цифра 200 взята с потолка. Если из YT начнут выгружаться
    # такие данные, то нужно разбираться - скорее всего это ошибка.
    my $file_size_bytes = -s $tmp_file_name;

    if (!$ENV{TAP_VERSION} && $file_size_bytes < 209_715_200) {
        throw Exception::Fias "Unexpected size of YT table t_fias",
          sentry => {extra => {file_size_bytes => $file_size_bytes}};
    } else {
        INFO "Size of file $tmp_file_name is $file_size_bytes.";
    }

    my ($lines_count) = `wc -l $tmp_file_name` =~ /^([0-9]+)/;

    INFO "Downloaded YT table t_fias to file $tmp_file_name Number of lines: $lines_count";

    for my $db (@$FIAS_UPDATE_DATABASES) {
        $self->app->send_heartbeat();

        # В штатном режеме работы этих таблиц быть не должно. Они появляются только если
        # в прошлый запуск этот скрипт упал.
        foreach my $table (qw(fias_tmp fias_previous)) {
            $self->$db->_do("drop table if exists $table");
        }

        $self->$db->_do($self->$db->fias_tmp->create_sql());

        my @rows;

        my $tree = $self->_build_parent_tree($tmp_file_name);

        open(my $fh, '<', $tmp_file_name) or throw "Can't open file $tmp_file_name";
        while (my $line = <$fh>) {
            my $data = from_json $line;

            if (_is_valid_fias_entry($data)) {
                delete @{$data}{qw(CENTER_STATUS KLADR_CODE CHUNK POSTCODE LIVE_STATUS SCN)};
                $data->{PARENT_GUID} = $self->_find_valid_parent($tree, $data->{PARENT_GUID})
                  if $data->{OBJ_LEVEL} != 1;
                $data->{lc($_)} = delete $data->{$_} for keys(%$data);
                push @rows, $data;
            }

            if (@rows > 10_000) {
                $self->$db->fias_tmp->add_multi(\@rows);
                @rows = ();
            }
        }
        close($fh) or throw "Can't close $tmp_file_name: $!";

        if (@rows > 0) {
            $self->$db->fias_tmp->add_multi(\@rows);
        }

        $self->$db->_do('rename table fias to fias_previous, fias_tmp to fias');
        $self->$db->_do('drop table fias_previous');
    }

    return 1;
}

sub _build_parent_tree {
    my ($self, $filename) = @_;

    my %parent_guid;

    open(my $fh, '<', $filename) or throw "Can't open file $filename";

    while (my $line = <$fh>) {
        my $data = from_json $line;
        if ($data->{OBJ_LEVEL} != 7) {
            $parent_guid{$data->{GUID}} = {map {$_ => $data->{$_}} (qw(GUID PARENT_GUID OBJ_LEVEL LIVE_STATUS))};
        }
    }

    close($fh) or throw "Can't close $filename: $!";

    return \%parent_guid;
}

my %valid_obj_level = map {$_ => 1} (qw(1 3 4 6 7));

sub _find_valid_parent {
    my ($self, $tree, $guid) = @_;

    while (!$valid_obj_level{$tree->{$guid}{OBJ_LEVEL}} || !$tree->{$guid}{LIVE_STATUS}) {
        $guid = $tree->{$guid}{PARENT_GUID};
    }

    return $guid;
}

sub _is_valid_guid {
    $_[0] =~ m/^[0-9a-f-]{36}$/;
}

sub _is_valid_fias_entry {
    my ($e) = @_;

    return
         _is_valid_guid($e->{GUID})
      && (!defined($e->{PARENT_GUID}) || _is_valid_guid($e->{PARENT_GUID}))
      && $e->{LIVE_STATUS} == 1
      && $valid_obj_level{$e->{OBJ_LEVEL}}
      && length($e->{FORMAL_NAME}) <= 255
      && length($e->{SHORT_NAME}) <= 40;
}

1;
