package Cron::Methods::MailNotification;

use qbit;

use base qw(QBit::Cron::Methods);

use Data::Dumper;
use Exception::TooManyRequests;
use Exception::SendMail::BadAddress;
use Exception::SysDie;
use Utils::Logger qw(ERROR INFOF WARNF);

use PiConstants qw($MAIL_NOTIFICATION_TYPES $MONEY_THRESHOLD);

my $DELETE_ROW_LIMIT = 20_000;

__PACKAGE__->model_accessors(
    mail_notification => 'Application::Model::MailNotification',
    partner_db        => 'Application::Model::PartnerDB',
    bk_statistics     => 'Application::Model::BKStatistics',
);

sub model_path {'mail_notification'}

sub check : CRON('*/1 * * * *') : LOCK : STAGE('PRODUCTION') {
    my ($self, %opts) = @_;

    $self->_check_new()   if $opts{check_new}   // 1;
    $self->_try_prepare() if $opts{try_prepare} // 1;
    $self->_try_send()    if $opts{try_send}    // 1;
}

sub find_incomplete : CRON('30 */6 * * *') : LOCK : FREQUENCY_LIMIT('1d') : TTL('7h') {
    my ($self, %opts) = @_;

    my $divider = 100;
    # Te пользователи, кому уже отсылаем письма
    my $user_with_notification_in_queue = [
        map {$_->{user_id}} @{
            $self->partner_db->query->select(
                fields => ['user_id'],
                filter => [
                    AND => [
                        [multistate => 'NOT IN' => \$self->mail_notification->get_multistates_by_filter('stopped')],
                        [type       => '='      => \$MAIL_NOTIFICATION_TYPES->{MONEY}],
                    ]
                ],
                table => $self->partner_db->mail_notification,
              )->get_all()
          }
    ];

    # Пользователи без второй части анкеты
    my $users = $self->partner_db->query->select(
        fields => [qw(id login)],
        filter => [
            AND => [
                [
                    {ifnull => [{json_unquote => [{json_extract => ['opts', \'$.is_form_done']}]}, \'null']},
                    'NOT IN', \['1', 'true']
                ],
                [id => 'NOT IN' => \$user_with_notification_in_queue],
            ],
        ],
        table => $self->partner_db->users,
    )->distinct()->get_all();

    my $cnt = 0;
    for (my $i = 0; $i < @$users; $i += $divider) {
        $self->app->send_heartbeat();

        my $min = $i;
        my $max = $i + $divider - 1;
        $max = $#$users if $max > $#$users;
        my %l2id = map {$_->{login} => $_->{id}} @$users[$min .. $max];
        my $statistics = $self->bk_statistics->get_statistics2(
            period        => [date_sub(curdate(), month => 6, oformat => 'db'), curdate(oformat => 'db')],
            fields        => ['partner_wo_nds'],
            entity_fields => ['login'],
            levels => [{id => 'payment', filter => ['AND', [[login => IN => [sort keys %l2id]]]]},],
            total => 0,
        );
        foreach my $r (@{$statistics->{points}}) {
            if ($r->{measures}[0]{partner_wo_nds} >= $MONEY_THRESHOLD) {
                $self->mail_notification->add_when_user_is_near_of_money_threshold(
                    {id => $l2id{$r->{dimensions}{login}}});
                $cnt++;
            }
        }
    }
    INFOF('Added %d users to mailing list', $cnt) if $cnt;
}

sub cleanup : CRON('*/11,*/41 * * * *') : LOCK {
    my ($self) = @_;

    my $multistate2prune = $self->mail_notification->get_multistates_by_filter('(stopped or sent) and not periodicity');
    my $dt2leave_after = date_sub(curdate(), day => 7, oformat => 'db_time');

    INFOF('will delete before dt: %s', $dt2leave_after);

    my $filter = $self->partner_db->filter(
        ['AND' => [['multistate' => 'IN' => \$multistate2prune], ['date_x' => '<' => \$dt2leave_after],]]);

    my $deleted_rows = $self->mail_notification->partner_db_table()->delete($filter, limit => $DELETE_ROW_LIMIT);

    INFOF('deleted count: %s', $deleted_rows);
}

sub _check_new {
    my ($self) = @_;

    my $mn          = $self->mail_notification;
    my $empty_state = $mn->get_all(
        fields => [qw(id)],
        filter => [
            AND => [
                [multistate  => '=' => '__EMPTY__'],
                [create_date => '<' => date_sub(curdate(), minute => 15, oformat => 'db_time',)],
            ],
        ],
    );

    my @stranges;
    foreach my $pk (@$empty_state) {
        try {
            $mn->do_action($pk, 'add');
            push @stranges, $pk->{id};
        }
        catch {
            my ($exception) = @_;
            ERROR $exception;
        };
    }

    WARNF('Following jobs were in empty state: %s', join ', ', @stranges)
      if scalar @stranges;
}

sub _try_prepare {
    my ($self) = @_;

    my $mn        = $self->mail_notification;
    my $preparing = $mn->get_all(
        fields => [qw(id)],
        filter => [
            AND => [
                [multistate => '=' => 'not stopped and ( preparing or sent and periodicity)'],
                [date_x     => '<' => curdate(oformat => 'db_time',)],
            ],
        ],
    );

    foreach my $pk (@$preparing) {
        try {
            $mn->maybe_do_action($pk, 'prepare');
        }
        catch {
            my ($exception) = @_;
            ERROR $exception;
        };
    }
}

sub _try_send {
    my ($self) = @_;

    my $mn         = $self->mail_notification;
    my $ready2send = $mn->get_all(
        fields => [qw(id opts type user_id)],
        filter => [
            AND => [[multistate => '=' => 'ready and not stopped'], [date_x => '<' => curdate(oformat => 'db_time',)],],
        ],
    );

    my %recipients;
    foreach my $task (@$ready2send) {
        $recipients{to_json($mn->get_to($task))} = TRUE;
    }

    my $timeouts = $mn->get_timeouts([sort keys %recipients]);

    foreach my $task (@$ready2send) {
        try {
            $mn->do_action($task, 'send', 'timeouts' => $timeouts);
        }
        catch Exception::TooManyRequests with {
            INFOF 'Task %s postponed due to timeout', $task->{id};
        }
        catch Exception::SendMail::BadAddress with {
            my ($exception) = @_;
            ERROR {
                message     => 'BadAddressEmailSend',
                fingerprint => ['mail_notification', 'try_send', 'BadAddress'],
                extra => {task_id => $task->{id},}
            };
            $mn->do_action($task, 'stop');
        }
        catch Exception::SysDie with {
            my ($exception) = @_;
            my $m = $exception->message;
            if ($m =~ /Bad recipient address syntax/ || $m =~ /No such user/) {
                ERROR {
                    message     => 'BadAddressEmailSend',
                    fingerprint => ['mail_notification', 'try_send', 'BadAddress'],
                    extra => {task_id => $task->{id},}
                };
                $mn->do_action($task, 'stop');
            } else {
                ERROR {
                    exception   => $exception,
                    fingerprint => ['mail_notification', 'try_send', 'Exception::SysDie'],
                    extra => {task_id => $task->{id},}
                };
            }
        }
        catch {
            my ($exception) = @_;
            ERROR {
                exception   => $exception,
                fingerprint => ['mail_notification', 'try_send', 'Suddenly'],
                extra => {task_id => $task->{id},}
            };
        };
    }
}

TRUE;
