package Cron::Methods::Metrika;

use qbit;

use Utils::Logger qw(ERRORF INFO);
use Utils::MonitoringUtils;

use base qw(QBit::Cron::Methods Cron::Methods);

use Utils::MonitoringUtils 'send_to_graphite';
use Utils::Logger qw(INFO INFOF);
use Exception::API::Metrika;
use Exception::Metrika;

__PACKAGE__->model_accessors(
    partner_db      => 'Application::Model::PartnerDB',
    product_manager => 'Application::Model::ProductManager',
    api_metrika     => 'QBit::Application::Model::API::Yandex::Metrika',
    api_yql         => 'QBit::Application::Model::API::Yandex::YQL',
);

sub model_path {'metrika'}

# Создем счетчики для новых площадок (#PI-10338)
sub create_page_counters : CRON('5 * * * *') : LOCK {
    my ($self) = @_;

    my $accessors = $self->product_manager->get_site_model_accessors();

    my $data = $self->partner_db->query->select(
        table  => $self->partner_db->all_pages,
        fields => [qw( page_id  model domain  owner_id )],
        filter => [AND => [[model => 'IN' => \$accessors], [is_balance_registered => '=' => \1],]],
      )->left_join(
        table   => $self->partner_db->metrika_counters,
        alias   => 'MK',
        fields  => [],
        filter  => ['counter_id' => 'IS' => \undef],
        join_on => ['page_id' => '=' => {'page_id' => $self->partner_db->all_pages}]
      )->order_by(qw(model page_id))->limit(500)->get_all();

    my @success_page_ids = ();
    my %failed_page_ids  = ();
    foreach my $row (@$data) {
        my ($domain, $page_id, $accessor, $owner_id) = @$row{qw(domain page_id model owner_id)};

        my $counter_id;
        my $exception;
        try {
            INFO "Execute: domain=$domain, page_id=$page_id, accessor=$accessor, owner_id=$owner_id";
            $counter_id =
              $self->api_metrika->create_or_get_page_counter_for_login($domain, $page_id, $accessor, $owner_id);
        }
        catch {
            ($exception) = @_;
        };

        if (defined($counter_id)) {
            push @success_page_ids,
              {
                page_id    => $page_id,
                counter_id => $counter_id,
              };
        } else {
            $failed_page_ids{$page_id} = defined($exception) ? $exception->{'text'} : 'API did not return counter_id';
        }
    }

    # flush success_page_ids before exit
    $self->partner_db->metrika_counters->add_multi(\@success_page_ids) if @success_page_ids;

    if (%failed_page_ids) {
        my $text = "Failed page_ids:\n";
        foreach my $page_id (keys %failed_page_ids) {
            $text .= "  page_id=$page_id, cause=$failed_page_ids{$page_id}\n";
        }
        throw Exception::Metrika $text, undef, undef, sentry => {
            fingerprint => ['Metrika', 'Cron create_page_counters'],
            extra => {failed_pages => \%failed_page_ids}
        };

    }

    return TRUE;
}

sub export_metrica_counters : CRON('*/10 * * * *') : LOCK {
    my ($self) = @_;

    my $stage = $self->app->get_option('stage');
    INFO "stage $stage";

    my $models = [
        @{$self->app->product_manager->get_site_model_accessors},
        @{$self->app->product_manager->get_video_model_accessors}
    ];

    my @data;
    foreach my $model_name (@$models) {
        my $model              = $self->app->$model_name;
        my $page_id_field_name = $model->get_page_id_field_name();

        my $pages = $model->get_all(
            fields => [$page_id_field_name, 'metrica_counters', 'domain'],
            filter => [
                AND => [
                    [metrica_counters => '<>' => ''],
                    (
                        $model->get_multistate_by_name('balance_registered')
                        ? {multistate => $model->get_multistates_by_filter('balance_registered')}
                        : ()
                    ),
                ]
            ],
        );

        foreach my $page (@$pages) {
            push @data, map {[$page->{$page_id_field_name}, $_, $page->{'domain'}]} @{$page->{'metrica_counters'}};
        }
    }

    # yt create table //home/partner/page_metrica_counters \
    #   --attributes $(cat <<EOF | tr "\n" " " | sed 's| ||g'
    #   {
    #       dynamic=%false;
    #       schema=[
    #         { name=page_id;     type=uint64; sort_order=ascending };
    #         { name=counter_id;  type=uint64; sort_order=ascending };
    #         {"name" = "domain"; "type" = "string"}
    #       ]
    #   }
    #   EOF
    # ) --proxy=hume

    $self->app->api_yt->rewrite_table_on_cluster(
        path         => '//home/partner/page_metrica_counters',
        data         => \@data,
        column_names => [qw(page_id  counter_id domain)],
        order_by     => [
            {
                "name"       => "page_id",
                "type"       => "number",
                "sort_order" => "asc",
            },
            {
                "name"       => "counter_id",
                "type"       => "number",
                "sort_order" => "asc",
            }
        ],
        params => {
            ':timeout'  => 300,
            ':attempts' => 3,
            ':delay'    => 0,
        }
    );

    send_to_graphite(
        interval => 'five_min',
        path     => 'Metrika.export_metrica_counters',
        value    => scalar @data
    );
}

sub export_metrica_counters_on_active_page : CRON('13,43 * * * *') : LOCK {
    my ($self) = @_;

    my $stage = $self->app->get_option('stage');
    INFO "stage $stage";

    my $models = $self->app->product_manager->get_site_model_accessors();

    my @inactive_ms_list       = qw(stopped deleted blocked rejected);
    my $domain_id_field_name_h = {
        'default'                           => 'domain_id',
        'internal_search_on_site_campaign'  => 'site_id',
        'internal_context_on_site_campaign' => 'site_id',
    };

    my @data;
    foreach my $model (@$models) {
        INFOF 'starting model [%s]', $model;

        my $page_id_field_name = $self->app->$model->get_page_id_field_name();
        my $domain_id_field_name = $domain_id_field_name_h->{$model} // $domain_id_field_name_h->{default};

        # internal/video pages dont have blocked|rejected
        my $ms_available_hash = $self->app->$model->get_multistate_names();
        my $ms_list_string = join(' or ', grep {exists $ms_available_hash->{$_}} @inactive_ms_list);

        # balance_registered есть во всех 4-х моделях
        my $multistate_active =
          $self->app->$model->get_multistates_by_filter("balance_registered and not ($ms_list_string)");

        my $q = $self->app->partner_db->query->select(
            table  => $self->app->partner_db->$model,
            fields => [$page_id_field_name],
            filter =>
              ['AND', [['multistate' => 'IN' => \$multistate_active], [$page_id_field_name => 'IS NOT' => \undef]]],
          )->join(
            table   => $self->app->partner_db->site,
            fields  => [qw( domain )],
            join_on => ['id' => '=' => {$domain_id_field_name => $self->app->partner_db->$model}],
          )->join(
            table   => $self->app->partner_db->metrika_counters,
            alias   => 'MK',
            fields  => [qw( counter_id )],
            join_on => ['page_id' => '=' => {$page_id_field_name => $self->app->partner_db->$model}]
          );

        push @data, map {[$_->{$page_id_field_name}, $_->{counter_id}, $_->{domain}]} @{$q->get_all()};
    }

    $self->app->api_yt->rewrite_table_on_cluster(
        path         => '//home/partner/page_metrica_counters_on_active',
        data         => \@data,
        column_names => [qw(page_id counter_id domain)],
        order_by     => [
            {
                "name"       => "page_id",
                "type"       => "number",
                "sort_order" => "asc",
            },
            {
                "name"       => "counter_id",
                "type"       => "number",
                "sort_order" => "asc",
            }
        ],
        params => {
            ':timeout'  => 300,
            ':attempts' => 3,
            ':delay'    => 0,
        }
    );

    send_to_graphite(
        interval => 'five_min',
        path     => 'Metrika.export_metrica_counters_on_active_page',
        value    => scalar @data
    );
}

=head2
    gets number of Metrika counters in our db that Metrika does not have

    perl -I./lib -MCron -e'Cron->new->do' metrika find_pages_with_deleted_metrika_counters
=cut

sub find_pages_with_deleted_metrika_counters : CRON('11 * * * *') : LOCK {
    my ($self) = @_;

    my $TABLE_PARTNER_COUNTERS = 'home/partner/page_metrica_counters_on_active';
    my $TABLE_METRIKA_COUNTERS = 'home/metrika/export/counters';

    my $query = sprintf(
        q(
$max_existing_counter = (SELECT MAX(counter_id) FROM `%s`);
SELECT count(partner_c.counter_id) AS deleted_counters
FROM `%s` AS partner_c
LEFT JOIN `%s` AS metrika_c ON partner_c.counter_id = metrika_c.counter_id
WHERE partner_c.counter_id < $max_existing_counter AND metrika_c.counter_id IS NULL
LIMIT 1;), $TABLE_METRIKA_COUNTERS, $TABLE_PARTNER_COUNTERS, $TABLE_METRIKA_COUNTERS
    );

    my $yql_operation_result = $self->api_yql->yql_start_operation_and_get_result(
        clusters     => $self->app->get_option('yt')->{replicas},
        start_params => {params => {content => $query}},
        get_params   => {
            format      => 'json',
            write_index => 0,
            limit       => 1
        },
    );

    my $operation_result = from_json($yql_operation_result);
    my $deleted_counters = $operation_result->{deleted_counters};

    if ($deleted_counters > 0) {
        ERRORF('Metrika counters missed count=[%d]', $deleted_counters);
    } else {
        INFO('no Metrika counters missed');
    }

    send_to_graphite(
        interval => "one_hour",
        path     => 'Metrika.counters.not_found',
        value    => $deleted_counters
    );
}

TRUE;
