package Cron::Methods::Moderation;

use qbit;

use base qw(QBit::Cron::Methods);

use PiConstants qw($MYSQL_MIN_DATETIME $MYSQL_MAX_DATETIME $SERVICE_NAME_IN_MODERATION);
use Utils::Logger qw(ERROR ERRORF INFOF WARN);
use QBit::StringUtils qw(from_jsonl);
use File::Slurp;

use Exception;
use Exception::BatchFail;
use Exception::IncorrectParams;
use Exception::Validation::BadArguments::InvalidJSON;
use Utils::MonitoringUtils 'send_to_graphite';

my $KV_STORE_KEY = 'moderation_check_verdicts';

__PACKAGE__->model_accessors(
    all_pages               => 'Application::Model::AllPages',
    api_selfservice         => 'Application::Model::API::Yandex::SelfService',
    api_yql                 => 'QBit::Application::Model::API::Yandex::YQL',
    context_on_site_mirrors => 'Application::Model::Product::AN::ContextOnSite::Campaign::Mirrors',
    kv_store                => 'QBit::Application::Model::KvStore',
    mobile_app              => 'Application::Model::Product::AN::MobileApp',
    search_on_site_mirrors  => 'Application::Model::Product::AN::SearchOnSite::Campaign::Mirrors',
    site                    => 'Application::Model::Product::AN::Site',
    outdoor_block           => 'Application::Model::Block::OutDoor',
    indoor_block            => 'Application::Model::Block::InDoor',
    indoor                  => 'Application::Model::Page::InDoor',
    outdoor                 => 'Application::Model::Page::OutDoor',
);

my %REQUEST_MAPPING;

sub model_path {'moderation'}

# will be set in get_need_approve when we have app
my $MDS_AVATARS_PUBLIC_URL;

my %MODELS = (
    context_on_site_mirrors => {
        convert                      => \&convert_mirror,
        list                         => \&get_mirrors,
        multistate                   => '__EMPTY__',
        return_to_moderation_for_no  => [qw(delete set_need_approve)],
        return_to_moderation_for_yes => undef,
        type                         => 'mirror',
        extra_fields                 => [],
    },
    mobile_app => {
        convert                      => \&convert_app,
        list                         => \&get_apps,
        multistate                   => 'need_approve',
        return_to_moderation_for_no  => undef,
        return_to_moderation_for_yes => [qw(set_need_approve)],
        type                         => 'bundle',
        extra_fields                 => [],
    },
    search_on_site_mirrors => {
        convert                      => \&convert_mirror,
        list                         => \&get_mirrors,
        multistate                   => '__EMPTY__',
        return_to_moderation_for_no  => [qw(delete set_need_approve)],
        return_to_moderation_for_yes => undef,
        type                         => 'mirror',
        extra_fields                 => [],
    },
    site => {
        convert                      => \&convert_site,
        list                         => \&get_sites,
        multistate                   => 'need_approve',
        return_to_moderation_for_no  => undef,
        return_to_moderation_for_yes => [qw(return_to_moderation)],
        type                         => 'site',
        extra_fields                 => ['is_graysite'],
    },
    indoor_block => {
        convert      => \&convert_dooh,
        list         => \&get_dooh_block_list,
        multistate   => 'need_approve and not deleted',
        type         => 'indoor_block',
        extra_fields => ['opts', 'moderation'],
        sfx_from     => [qw(caption photo_id_list)],
        is_block     => 1
    },
    outdoor_block => {
        convert      => \&convert_dooh,
        list         => \&get_dooh_block_list,
        multistate   => 'need_approve and not deleted',
        type         => 'outdoor_block',
        extra_fields => ['opts', 'moderation'],
        sfx_from     => [qw(caption photo_id_list address gps)],
        is_block     => 1
    },
    indoor => {
        convert      => \&convert_dooh,
        list         => \&get_dooh_page_list,
        list_fields  => [qw(page_id id caption address gps login owner_id client_id moderation opts)],
        multistate   => 'need_approve and not deleted',
        type         => 'indoor_page',
        extra_fields => ['opts', 'moderation'],
        sfx_from     => [qw(caption address gps)],
    },
    outdoor => {
        convert      => \&convert_dooh,
        list         => \&get_dooh_page_list,
        list_fields  => [qw(page_id id caption login owner_id client_id moderation opts)],
        multistate   => 'need_approve and not deleted',
        type         => 'outdoor_page',
        extra_fields => ['opts', 'moderation'],
        sfx_from     => [qw(caption)],
    },
);

my %DOOH_TYPE_DATA = (
    'caption' => {
        type_suffix => '_caption',
        get_data    => sub {
            my ($obj) = @_;

            return (caption => $obj->{caption},);
        },
    },
    'photo_id_list' => {
        type_suffix => '_image',
        get_data    => sub {
            my ($obj, $position) = @_;

            return (
                image_url  => $MDS_AVATARS_PUBLIC_URL . $obj->{photo_link_list}->[$position]->{links}->{orig}->{path},
                image_hash => $obj->{photo_link_list}->[$position]->{md5},
            );
        },
    },
    'address' => {
        type_suffix => '_address',
        get_data    => sub {
            my ($obj) = @_;

            return (
                address => $obj->{address},
                gps     => $obj->{gps},
            );
        },
    },
    'gps' => {
        type_suffix => '_address',
        get_data    => sub {
            my ($obj) = @_;

            return (
                address => $obj->{address},
                gps     => $obj->{gps},
            );
        },
    }
);

my %DOOH_FIELDS;
foreach (sort keys %DOOH_TYPE_DATA) {
    push @{$DOOH_FIELDS{$DOOH_TYPE_DATA{$_}{type_suffix}}}, $_;
}

my %TYPES;
for my $m (values %MODELS) {
    if (my $s = $m->{sfx_from}) {
        for my $t (@$s) {
            $TYPES{$m->{type} . $DOOH_TYPE_DATA{$t}{type_suffix}} = 1;
        }
    } else {
        $TYPES{$m->{type}} = 1;
    }
}

sub convert_app {
    my ($model, $row) = @_;

    return [
        {
            data => {
                client_id   => $row->{owners}[0]{client_id},
                create_date => $row->{create_date},
                store_id    => $row->{store_id},
                store_url   => $row->{store_url},
                type        => 0 + $row->{type},
                yuid        => $row->{owners}[0]{id},
            },
            meta => {
                id    => 0 + $row->{id},
                model => $model->accessor(),
            },
        }
    ];
}

sub convert_mirror {
    my ($model, $row) = @_;

    return [
        {
            data => {
                client_id   => $row->{client_id},
                create_date => $row->{create_date},
                domain      => $row->{domain},
                mirror      => $row->{mirror},
                yuid        => $row->{owner_id},
            },
            meta => {
                id    => 0 + $row->{id},
                model => $model->accessor(),
            },
        }
    ];
}

sub convert_site {
    my ($model, $row) = @_;

    return [
        {
            data => {
                client_id   => $row->{data_for_site_stat}[0]{owner}{client_id},
                create_date => $row->{create_date},
                domain      => $row->{domain},
                yuid        => $row->{data_for_site_stat}[0]{user_id},
            },
            meta => {
                id    => 0 + $row->{id},
                model => $model->accessor(),
            },
        }
    ];
}

sub convert_dooh {
    my ($model, $row) = @_;

    my @requests;
    my @extra;
    my %converted_groups;
    my $model_name       = $model->accessor();
    my $moderation       = $row->{moderation};
    my $moderated_fields = $model->get_fields_moderated();
    foreach my $field_name (sort keys %$moderated_fields) {
        unless (exists $DOOH_TYPE_DATA{$field_name}) {
            delete $moderation->{$field_name};
            next;
        }
        my $suffix = $DOOH_TYPE_DATA{$field_name}->{type_suffix};
        next if ($converted_groups{$suffix});
        my $field_moderation = $moderation->{$field_name};
        my $field_values     = $row->{$field_name};
        $field_values = [$field_values] if $moderated_fields->{$field_name} ne 'ARRAY';
        for (my $i = 0; $i < @$field_values; $i++) {
            my $value_moderation = $field_moderation->{$field_values->[$i]};
            if (!$value_moderation->{request_id} && !$value_moderation->{verdict}) {
                push @requests,
                  {
                    type => $MODELS{$model_name}->{type} . $suffix,
                    meta => {
                        model     => $model_name,
                        page_id   => 0 + $row->{page_id},
                        client_id => 0 +
                          ($MODELS{$model_name}->{is_block} ? $row->{owner_client_id} : $row->{client_id}),
                        user_id => 0 + $row->{owner_id},
                        ($MODELS{$model_name}->{is_block} ? (imp_id => 0 + $row->{id},) : ())
                    },
                    data => {$DOOH_TYPE_DATA{$field_name}->{get_data}->($row, $i),}
                  };
                push @extra,
                  {
                    id       => $row->{id},
                    fields   => $DOOH_FIELDS{$suffix},
                    page_id  => $row->{page_id},
                    position => $i,
                  };
                $converted_groups{$suffix} = TRUE;
            }
        }
    }
    return \@requests, \@extra;
}

sub get_apps {
    my ($model) = @_;

    # Получение приложений
    # находящихся в состоянии need_approve
    # и не проходившие обработку (дефолтное время)
    my $sites = $model->get_all(
        fields => [qw(id create_date type store_id store_url owners)],
        filter => [
            'AND',
            [
                ['multistate',         '=', 'need_approve'],         # Ожидающие модерацию
                ['waiting_moderation', '=', $MYSQL_MIN_DATETIME],    # Не обработанные
            ],
        ],
    );

    return $sites;
}

sub get_mirrors {
    my ($model) = @_;

    my $page_model_name = $model->page_model_name();
    my $page_model      = $model->$page_model_name;
    # Получение зеркал
    # находящихся в состоянии __EMPTY__ = need_approve
    # и не проходившие обработку (дефолтное время)
    my $mirrors = $model->partner_db->query->select(
        table  => $model->partner_db_table(),
        fields => {
            id          => '',
            mirror      => 'domain',
            campaign_id => '',
            create_date => '',
        },
        filter => [
            'AND',
            [
                # Ожидающие модерацию
                [multistate => IN => \$model->get_multistates_by_filter('__EMPTY__')],
                # Не обработанные
                [waiting_moderation => '=' => \$MYSQL_MIN_DATETIME],
            ],
        ],
      )->join(
        table   => $page_model->partner_db_table(),
        fields  => [qw(owner_id)],
        join_on => [id => '=' => {campaign_id => $model->partner_db_table()}],
      )->join(
        table   => $model->partner_db->site,
        fields  => [qw(domain)],
        join_on => [id => '=' => {domain_id => $page_model->partner_db_table()}],
      )->join(
        table   => $model->partner_db->users,
        fields  => [qw(client_id)],
        join_on => [id => '=' => {owner_id => $page_model->partner_db_table()}],
      )->get_all();

    return $mirrors;
}

sub get_sites {
    my ($model) = @_;

    # Получение сайтов
    # находящихся в состоянии need_approve
    # и не проходившие обработку (дефолтное время)
    my $sites = $model->get_all(
        fields => [qw(id domain create_date data_for_site_stat)],
        filter => [
            'AND',
            [
                ['multistate',         '=', 'need_approve'],         # Ожидающие модерацию
                ['waiting_moderation', '=', $MYSQL_MIN_DATETIME],    # Не обработанные
            ],
        ],
    );

    return $sites;
}

sub get_dooh_block_list {
    my ($model) = @_;

    my $block_list = $model->get_all(
        fields => [
            qw(page_id id caption address gps photo_id_list photo_link_list login owner_id owner_client_id moderation opts)
        ],
        filter => [
            'AND',
            [
                # Ожидающие модерацию
                ['multistate', '=', 'need_approve and not deleted and not testing'],
                ['waiting_moderation', '=', $MYSQL_MIN_DATETIME],    # Не обработанные
            ],
        ],
    );

    return $block_list;
}

sub get_dooh_page_list {
    my ($model, $list_fields) = @_;

    return $model->get_all(
        fields => $list_fields,
        filter => [
            'AND',
            [
                # Ожидающие модерацию
                ['multistate', '=', 'need_approve and not deleted and not testing'],
                ['waiting_moderation', '=', $MYSQL_MIN_DATETIME],    # Не обработанные
            ],
        ],
    );
}

sub prepare_moderation_objects_for_logbroker {
    my ($self, $type, @data) = @_;

    throw Exception::Validation::BadArguments gettext('Type is not specified') unless ($type);
    throw Exception::Validation::BadArguments gettext('Data is not specified') unless (@data);

    my $tm = int(Time::HiRes::time * 1_000_000);
    for (my $i = 0; $i < @data; $i++) {
        my $r = $data[$i];
        $r->{meta}{request_id} = $i + $tm;
        $r->{service} = $SERVICE_NAME_IN_MODERATION;
        $r->{type} = $type unless $r->{type};
    }

    return \@data;
}

=head2 check_need_approve

Метод определяет список сайтов, находящихся в состоянии need_approve и не проходивших обработку,
и отправляет их на модерацию

Метод написан для крона.
При необходимости старта вручную, запустить командой:
perl -I./lib -MCron -e 'Cron->new->do' moderation check_need_approve

=cut

sub check_need_approve : CRON('*/5 * * * *') : LOCK : STAGE('TEST') : STAGE('PRODUCTION') {
    my ($self) = @_;

    $self->_check_need_approve(\&send_requests_for_approval);
}

=head2 check_too_long_moderation

Метод определяет список сайтов, находящихся в состоянии need_approve,
проходшие обработку, но не получившие вердикт в течение долгого времени.

При их обнаружении кидает сообщение об ошибке в лог.

Метод написан для крона.
При необходимости старта вручную, запустить командой:
perl -I./lib -MCron -e 'Cron->new->do' moderation check_too_long_moderation

=cut

sub check_too_long_moderation : CRON('*/5 * * * *') : LOCK : STAGE('TEST') : STAGE('PRODUCTION') {
    my ($self) = @_;

    my $fail;
    # Получение количества сайтов и зеркал
    # находящихся в состоянии need_approve
    # поставленные в обработку (время не дефолтное)
    # и не прошедшие её за 15 часов
    foreach my $model_name (sort keys %MODELS) {
        my $model = $self->$model_name;
        my $count = $model->partner_db_table->get_all(
            fields => {
                count => {COUNT => ['id']},
                too_long =>
                  {if => [['waiting_moderation', '<', \date_sub(curdate(), hour => 15, oformat => 'db_time')], \1, \0]},
            },
            filter => [
                AND => [
                    [multistate         => '=' => \$model->get_multistates_by_filter($MODELS{$model_name}{multistate})],
                    [waiting_moderation => '>' => \$MYSQL_MIN_DATETIME],
                    (
                        $model->can('does')
                          && $model->does('Application::Model::Role::Has::Moderation')
                        ? [done_moderation => '=' => \$MYSQL_MIN_DATETIME]
                        : [waiting_moderation => '<' => \$MYSQL_MAX_DATETIME]
                    ),
                ],
            ],
            group_by => ['too_long'],
        );
        my $cnt     = 0;
        my $all_cnt = 0;
        foreach (@$count) {
            $all_cnt += $_->{count};
            $cnt += $_->{count} if ($_->{too_long});
        }
        if ($cnt) {
            # Поднятие тревоги, если они есть
            ERROR {
                message =>
                  sprintf('There are %d elements in %s have state need_approve for a long time', $cnt, $model_name),
                fingerprint => ['Cron', 'check_too_long_moderation', $model_name],
            };
            $fail = TRUE;
        }
        send_to_graphite(
            interval => 'five_min',
            path     => sprintf('Moderation.%s_outdated', $model_name),
            value    => $cnt,
            solomon  => {
                model  => $model_name,
                sensor => 'Moderation.outdated',
            }
        );
        send_to_graphite(
            interval => 'five_min',
            path     => sprintf('Moderation.%s_current', $model_name),
            value    => $all_cnt,
            solomon  => {
                model  => $model_name,
                sensor => 'Moderation.current',
            }
        );
    }

    throw Exception::BatchFail if $fail;
}

sub get_kv_key_data {
    my ($self) = @_;
    my $last_table = $self->kv_store->get($KV_STORE_KEY);
    return $last_table;
}

sub set_kv_key_data {
    my ($self, $last_table) = @_;
    $self->kv_store->set($KV_STORE_KEY, $last_table);
    return $last_table;
}

=head2 check_verdicts

Метод обращается к YT, определяет список сайтов, получивших вердикт,
и выполняет для них соответствующее действие (approve или reject).

Метод написан для крона.
При необходимости старта вручную, запустить командой:
perl -I./lib -MCron -e 'Cron->new->do' moderation check_verdicts

Если есть файлик со строками-json объектами, то можно скормить его крону:
perl -I./lib -I./local/lib/perl5 -MCron -e 'Cron->new->do' moderation check_verdicts --file_path /path/to/file.jsonl  2>&1  | tee check_verdicts.log

Длs дебага:
DEBUG_EXCEPTIONS=1  FORCE_LOGGER_TO_SCREEN=1 perl -I./lib -I./local/lib/perl5 -MCron -e 'Cron->new->do' moderation check_verdicts \
    --table_from='2022-04-22T23:10:00' --table_to='2022-04-22T23:10:00' \
    --limit 10
     2>&1  | tee check_verdicts.log

Подробнее см. PI-28072

=cut

sub check_verdicts : CRON('*/5 * * * *') : LOCK : STAGE('TEST') : STAGE('PRODUCTION') {
    my ($self, %opts) = @_;

    my $last_table = $self->get_kv_key_data();

    # кром можно запустить вручную передав путь к файлу c вердиктами, выгруженными из YT
    my $file_path = $opts{'--file_path'};

    my $verdicts = $self->get_verdicts($last_table, \%opts);

    my $counter = {};

    my $tables = {map {$_->{table_name} => 1} @$verdicts};
    INFOF(
        "Got %d verdicts to process from %d tables: - '%s'",
        scalar(@$verdicts),
        scalar(keys %$tables),
        join("'\n - '", (%$tables ? '' : ()), sort keys %$tables)
    );

    $self->app->send_heartbeat();

    if (@$verdicts) {

        $self->site->partner_db()->transaction(
            sub {
                for (my $i = 0; $i <= $#$verdicts; $i++) {
                    my $row = $verdicts->[$i];
                    $self->apply_verdict($row, $counter);

                    $self->app->send_heartbeat() if $i % 100 == 0;
                }
            }
        );

        INFOF '%d verdicts applied successfully', scalar @$verdicts;

        foreach my $model_name (keys %$counter) {
            foreach my $cnt_name (qw(all ignored Yes No)) {
                my $count = $counter->{$model_name}{$cnt_name} // 0;
                INFOF(' - %d %s verdicts for model "%s"', $count, $cnt_name, $model_name) if $count;
                unless ($file_path) {
                    send_to_graphite(
                        interval => 'five_min',
                        path     => sprintf('Moderation.%s_received_%s', $model_name, $cnt_name),
                        value    => $count,
                        solomon  => {
                            model  => $model_name,
                            sensor => 'Moderation.' . sprintf('received_%s', $cnt_name),
                        }
                    );
                }
            }
        }

        unless ($file_path) {
            my ($last_table) = @{$verdicts->[-1]}{qw( table_name )};
            $self->set_kv_key_data($last_table);

            send_to_graphite(
                interval => 'five_min',
                path     => 'Moderation.received_error',
                value    => 1,
                solomon  => {sensor => 'Moderation.received_error',}
            );
        }
    }

    return 1;
}

sub get_verdicts {
    my ($self, $last_table, $opts) = @_;

    my $verdicts = [];
    my $exception;

    if ($opts->{'--file_path'}) {
        my $text = read_file($opts->{'--file_path'});
        $verdicts = from_jsonl($text);
    } else {

        my $table_from = $opts->{'--table_from'}
          // date_sub($last_table, hour => 1, iformat => "db_time_t", oformat => "db_time_t");
        my $table_to = $opts->{'--table_to'} // curdate(oformat => 'db_time_t');
        my $limit = $opts->{'--limit'} // 1000;    # всего за день не более 1500

        # NOTE! Вытаскиваем данные с нахлестом на час от предыдущего запуска,
        #       потому что на другом кластере новые данные могут попасть в таблицу
        #       c меньшем временем (см MODADVERT-27072)
        #       Но потом мы должны взять тоолько последний вердикт, потому что
        #       в apply_verdict запись пропустится если только вердикт не изменился
        my $query = sprintf q[
               SELECT
                            TableName() as table_name,
                            TableRecordIndex() as table_record_index,
                            service,
                            type,
                            Yson::SerializeJson( meta ) AS meta,
                            Yson::SerializeJson( mod_results ) AS mod_results
                FROM        range(`logs/modadvert-pi-verdicts-log/stream/5min`,`%s`,`%s`)
                WHERE       service = '%s'
                            AND type IN (%s)
                            AND Yson::ConvertToString( meta.model ) IN (%s)
                ORDER BY    TableName(),
                            TableRecordIndex()
                limit       %d
            ], $table_from, $table_to,
          $SERVICE_NAME_IN_MODERATION,
          '"' . join('","', sort keys %TYPES) . '"',
          '"' . join('","', sort keys %MODELS) . '"',
          $limit;

        my $yql_operation_result = $self->api_yql->yql_start_operation_and_get_result(
            clusters => (
                $self->get_option('yt')->{'replicas'}
                  || throw Exception::IncorrectParams 'YT replicas should be specified'
            ),
            start_params => {params => {content => $query}},
            get_params   => {
                format      => 'json',
                write_index => 0,
                limit       => 1
            },
        );

        $verdicts = from_jsonl($yql_operation_result);
        INFOF("Got %d raw verdicts", scalar(@$verdicts));
    }

    # NOTE! Для каждой сущности оставляем только последний вердикт, так как
    #       выбиарем с нахлестом, и не хотим чтобы партнерам отсылаплись одни и те же письма
    my $last_verdicts = {
        # model => { id => <last_row> }
    };
    foreach my $row (@$verdicts) {
        foreach my $key (qw(meta mod_results)) {
            unless (ref($row->{$key})) {
                $row->{$key} = from_json($row->{$key});
            }
        }
        my ($model, $id) = @{$row->{meta}}{qw( model id )};
        $last_verdicts->{$model}->{$id} = $row;
    }

    $verdicts = [
        sort {$a->{table_name} cmp $b->{table_name} || $a->{table_record_index} <=> $b->{table_record_index}}
        map {values %$_} values %$last_verdicts
    ];

    return $verdicts;
}

sub _check_need_approve {
    my ($self, $process_requests) = @_;

    $MDS_AVATARS_PUBLIC_URL //= $self->app->get_option('api_media_storage_avatars')->{public_url};

    my $count = 0;
    foreach my $model_name (sort keys %MODELS) {
        my $model   = $self->$model_name;
        my $type    = $MODELS{$model_name}{type};
        my $convert = $MODELS{$model_name}{convert};
        my $data    = $MODELS{$model_name}{list}->($model, $MODELS{$model_name}{list_fields});
        my $cnt     = @$data;
        if ($cnt) {
            INFOF('Received %d elements in %s to proceed', $cnt, $model_name);
            # Конвертация данных
            my @converted_data;
            my @extra;
            foreach my $obj (@$data) {
                my ($converted, $ext) = $convert->($model, $obj);
                push @converted_data, @$converted;
                push @extra, @$ext if $ext;
            }
            $process_requests->($self, $type, $model, $data, \@converted_data, \@extra,);
            $count += $cnt;
        } else {
            INFOF('There is no elements in %s to proceed', $model_name);
        }
        send_to_graphite(
            interval => 'five_min',
            path     => sprintf('Moderation.%s_requested', $model_name),
            value    => $cnt,
            solomon  => {
                model  => $model_name,
                sensor => 'Moderation.requested',
            }
        );
    }
}

sub send_requests_for_approval {
    my ($self, $type, $model, $data, $converted_data, $extra) = @_;
    # Передача данных на модерацию через пуш-клиент
    my $topic = $self->get_option('moderation', {})->{logbroker_topic}
      or throw Exception::IncorrectParams 'Topic of moderation requests should be specified';
    $self->api_selfservice->logbroker(
        data  => $self->prepare_moderation_objects_for_logbroker($type, @$converted_data),
        topic => $topic,
    );

    # Обновление данных, если всё хорошо
    for (my $i = 0; $i < @$converted_data; $i++) {
        $converted_data->[$i]{extra} = $extra->[$i];
    }

    $model->on_moderation_request_sent($data, $converted_data);
}

sub apply_verdict {
    my ($self, $row, $counter) = @_;

    my $service = $row->{service} // 'undef';
    return if ($service ne $SERVICE_NAME_IN_MODERATION);

    my $type = $row->{type} // 'undef';
    throw Exception "Unknown type: $type" unless ($TYPES{$type});

    my $filter = _get_filter($row->{meta});
    throw Exception 'Incorrect id: ' . to_json($filter // {}) unless ([values %$filter]->[0]);

    my $model_name = $row->{meta}{model} // 'undef';
    throw Exception "Unknown model: $model_name" unless ($MODELS{$model_name});

    my $model   = $self->$model_name;
    my $result  = $row->{mod_results};
    my $verdict = $result->{verdict} // 'undef';
    throw Exception "Incorrect verdict: $verdict" if ($verdict ne 'Yes' && $verdict ne 'No');
    my $reason = $result->{reasons};

    my $obj = $model->get_all(
        filter => $filter,
        fields => [@{$model->get_pk_fields}, 'multistate', @{$MODELS{$model_name}{extra_fields}}]
    )->[0];

    throw Exception sprintf('Object not found in model "%s", filter: %s', $model_name, to_json($filter)), sentry => {
        fingerprint => ['Cron', 'Moderation', 'apply_verdict', 'Object not found'],
        extra => {moderation_data => $row},
      }
      unless $obj;

    $counter->{$model_name}{all}++;
    my $ignore;
    foreach my $state (qw(blocked deleted)) {
        if ($model->check_multistate_flag($obj->{multistate}, $state)) {
            $ignore = TRUE;
            WARN {
                message => sprintf('Model "%s" got verdict for "%s" state', $model_name, $state),
                extra   => {
                    moderation_data => $row,
                    object          => $obj,
                },
            };
        }
    }
    if ($ignore) {
        $counter->{$model_name}{ignored}++;
        return;
    }
    $counter->{$model_name}{$verdict}++;

    if ($model->can('on_verdict_received')) {
        $model->on_verdict_received(
            $obj,
            {
                request_id => $row->{meta}{request_id},
                verdict    => $verdict eq 'Yes' ? TRUE : FALSE,
            }
        );
        return;
    }

    my $id = $filter->{id};
    if ($model->check_multistate_flag($obj->{multistate}, 'approved')) {
        if ($verdict eq 'No') {
            if (my $actions = $MODELS{$model_name}{return_to_moderation_for_no}) {
                $model->do_action($id, $_) foreach (@$actions);
            }
            $model->do_action($id, 'reject', moderation_reason => $reason);
        }
    } elsif ($model->check_multistate_flag($obj->{multistate}, 'rejected')) {
        if ($verdict eq 'Yes') {
            if (my $actions = $MODELS{$model_name}{return_to_moderation_for_yes}) {
                $model->do_action($id, $_) foreach (@$actions);
            }
            $model->do_action($id, 'approve');
        }
    } else {
        if ($verdict eq 'Yes') {
            $model->do_action($id, 'approve');
        } elsif ($verdict eq 'No') {
            $model->do_action($id, 'reject', moderation_reason => $reason);
        }
    }
    if ($verdict eq 'Yes' && $model_name eq 'site') {
        my $is_graysite = (ref($result->{flags}) eq 'ARRAY' && in_array('mfa', $result->{flags}));
        if (!$obj->{is_graysite} && $is_graysite) {
            $model->do_action($id, 'set_is_graysite');
        } elsif ($obj->{is_graysite} && !$is_graysite) {
            $model->do_action($id, 'unset_is_graysite');
        }
    }
    if ($model->can('on_done_moderation')) {
        $model->on_done_moderation($id);
    } else {
        # Устанавливается в max, чтобы не срабатывали check_need_approve (min) и check_too_long_moderation (-12h)
        # Просто сбрасывать в NULL не удобно из-за последующих проверок в QBit::Application::Model::DBManager::Filter::date
        $model->partner_db_table()->edit($id, {'waiting_moderation' => $MYSQL_MAX_DATETIME});
    }
}

sub approve_data {
    my ($self, $converted_data) = @_;

    my %counter;
    $self->site->partner_db()->transaction(
        sub {
            foreach my $row (@$converted_data) {
                $self->apply_verdict(fake_approve_verdict($row), \%counter);
            }
        }
    );
}

sub fake_approve_verdict {
    my ($data) = @_;
    return {
        service     => $data->{service},
        mod_results => {
            flags   => [],
            reasons => [],
            verdict => 'Yes',
        },
        meta => $data->{meta},
        type => $data->{type},
    };
}

sub _get_filter {
    my ($meta) = @_;

    # site / mirror
    my $filter = {};
    # block
    if ($meta->{imp_id}) {
        $filter = {page_id => $meta->{page_id}, id => $meta->{imp_id}};
    }
    # page
    elsif ($meta->{page_id}) {
        $filter = {page_id => $meta->{page_id}};
    } else {
        $filter = {id => $meta->{id}};
    }
    return $filter;
}

TRUE;
