package Cron::Methods::Monitoring;

=encoding UTF-8

=cut

use qbit;

use Utils::MonitoringUtils;
use Utils::LogParser::Checks;

use Utils::Logger qw(ERRORF ERROR INFOF INFO);

use File::Slurp;
use File::Path;
use File::Copy;
use File::Temp qw(tempfile);
use Archive::Zip qw(AZ_OK);

use PiConstants qw($STAT_MONEY_SCALE $PARTNER_DSP_MONITORING_MAIL $PARTNER_MANAGERS_MAIL);

use base qw(QBit::Cron::Methods);

__PACKAGE__->model_accessors(
    partner_db         => 'Application::Model::PartnerDB',
    partner_monitoring => 'Application::Model::PartnerMonitoring',
    product_manager    => 'Application::Model::ProductManager',
    statistics         => 'Application::Model::Statistics',
    currency           => 'Application::Model::Currency',
    mailer             => 'Application::Model::SendMail',
    api_yql            => 'QBit::Application::Model::API::Yandex::YQL',
);

sub model_path {'monitoring'}

my $MAX_ATTACH_LENGTH = 1024 * 1024;

sub last_update_stat : CRON('*/5 * * * *') : LOCK {
    my ($self) = @_;

    my $products = $self->product_manager->get_statistics_products();

    my @kv_keys = ();
    my %key_to_product;
    foreach (@$products) {
        my $level = $self->app->$_;
        my $id    = $level->id();

        my $ch_key              = sprintf('last_update_stat_%s_clickhouse',           $id);
        my $ch_for_month_key    = sprintf('last_update_stat_for_month_%s_clickhouse', $id);
        my $mysql_key           = sprintf('last_update_stat_%s',                      $id);
        my $mysql_for_month_key = sprintf('last_update_stat_for_month_%s',            $id);

        my %common_labels = (
            product          => $_,
            statistics_level => $id,
        );

        if ($level->support_clickhouse()) {
            $key_to_product{$ch_key} = {
                %common_labels,
                database => 'clickhouse',
                period   => 'day',
            };

            $key_to_product{$ch_for_month_key} = {
                %common_labels,
                database => 'clickhouse',
                period   => 'month',
            };

            push @kv_keys, $ch_key, $ch_for_month_key;
        } else {
            $key_to_product{$mysql_key} = {
                %common_labels,
                database => 'mysql',
                period   => 'day',
            };

            $key_to_product{$mysql_for_month_key} = {
                %common_labels,
                database => 'mysql',
                period   => 'month',
            };

            push @kv_keys, $mysql_key, $mysql_for_month_key;
        }
    }

    my $data = $self->partner_db->kv_store->get_all(
        fields => [qw(key value last_change)],
        filter => {key => \@kv_keys},
    );

    my $now = curdate(oformat => 'db_time');

    foreach my $el (sort {$a->{'key'} cmp $b->{'key'}} @$data) {
        my $delta = dates_delta($el->{'last_change'}, $now, iformat => 'db_time', oformat => 'sec');

        # Рассчет секунд делается с допущением что в месяце всегда 30 дней, а
        # в году всегда 365 дней. Такое допущение тут возможно, так как таких
        # цифр тут никогда не должно быть.
        my $delta_seconds = $delta->[0] * 31536000    # 365 days
          + $delta->[1] * 2592000                     # 30 days
          + $delta->[2] * 86400                       # 1 day
          + $delta->[3] * 3600                        # 1 hour
          + $delta->[4] * 60                          # 1 minute
          + $delta->[5];

        my ($path) = $el->{'key'} =~ /^last_update_stat_(.*)$/;

        send_to_graphite(
            interval => "five_min",
            path     => "statistics_update.$path",
            value    => $delta_seconds,
            solomon  => {
                %{$key_to_product{$el->{key}}},
                type   => 'statistics_update',
                sensor => 'time_since',
            },
        );
    }

    return FALSE;
}

sub partner_monitoring_products : CRON('0 8 * * *') : LOCK : INSTANCES(10) {
    my ($self, %opts) = @_;

    # All products available to partners
    $self->partner_monitoring->send_partner_monitoring_products(
        login           => $opts{login},
        date            => $opts{date},
        instances_count => $opts{'instances'},
        instance_number => $opts{'instance_number'},
    );
}

sub project_version : CRON('*/5 * * * *') : FRONTEND : STAGE('TEST') : STAGE('PRODUCTION') {
    my ($self) = @_;

    # 2.14.5 => 2.014005
    my $num_version;

    my $version = get_version();

    if ($version =~ /^ ([0-9]+) \. ([0-9]+) \. ([0-9]+) $/x) {
        my ($v1, $v2, $v3) = ($1, $2, $3);
        $num_version = $1 . "." . sprintf('%03d', $2) . sprintf('%03d', $3);
    } else {
        throw "Error. Can't parse version $version.";
    }

    send_to_graphite(
        interval => "five_min",
        path     => "project_version",
        value    => $num_version,
    );

    return FALSE;
}

sub logs_monitoring : CRON('*/5 * * * *') : FRONTEND : NOLOCK : STAGE('TEST') : STAGE('PRODUCTION') {
    my ($self, %opts) = @_;

    Utils::LogParser::Checks::parse_logs($self, %opts);
}

sub notify_500 : CRON('*/5 * * * *') : STAGE('TEST') : STAGE('PRODUCTION') : FRONTEND : NOLOCK {
    my ($self) = @_;

    my $dir = $self->get_option('error_dump_dir');
    return unless defined($dir) && $dir ne '';

    my $cfg = $self->get_option('error_notify');
    return unless $cfg;

    my $sent_dir = $cfg->{'sent_dir'};
    return unless defined($sent_dir) && $sent_dir ne '';

    $dir      = $self->get_option('ApplicationPath') . '/' . $dir      unless $dir      =~ /^\//;
    $sent_dir = $self->get_option('ApplicationPath') . '/' . $sent_dir unless $sent_dir =~ /^\//;

    return unless -d $dir;

    # checks
    File::Path::make_path($sent_dir) or throw "Can't create dir [$sent_dir]" unless -d $sent_dir;
    throw "Can't write into dir [$dir, $sent_dir]" unless -w $dir && -w $sent_dir;

    # get all files
    opendir(my $dh, $dir || '') || return;
    my @files = map +{name => "$dir/$_", stat => [stat("$dir/$_")]}, grep {$_ !~ /last_error.html$/} readdir($dh);
    closedir($dh);

    # prepare mail body (newest first)
    my %errors  = ();
    my $content = '';    # to avoid extra malloc
    foreach my $file (sort({$b->{stat}->[9] <=> $a->{stat}->[9]} @files)) {
        next if !-f $file->{name};

        $content = read_file($file->{name}, binmode => ':utf8');

        # cut off the dump
        $content =~
          s/<pre id="application_dump_text">.*?<\/pre>/<pre id="application_dump_filename">$file->{name}<\/pre>/is;

        $content = substr($content, 0, $MAX_ATTACH_LENGTH) if length($content) > $MAX_ATTACH_LENGTH;

        my $app_name    = 'Notify 500';
        my $error_text  = 'Unknown Exception';
        my $error_block = $error_text . ' ' . $file->{name};

        # find the text
        # see Application::Model::ExceptionDumpe::exception_info()
        if (
            $content =~ /(<div\s+id="error_block"[^>]+>.+?
App:<\/strong>\s+(.+?)<br>.+?
<pre\s+id="exception_text">(.+?)<\/pre>\s*<\/h4>\s*<\/div>)/isx
           )
        {
            $error_block = $1;
            $app_name    = $2;
            $error_text  = $3;
        }

        if (!exists($errors{$error_text})) {
            $errors{$error_text} = {
                text        => $error_text,
                count       => 1,
                error_block => $error_block,
                app_name    => $app_name,
                content     => $content,
                dump_files  => [$file->{'name'}],
                latest_file => $file->{'name'}
            };
        } else {
            push(@{$errors{$error_text}{'dump_files'}}, $file->{'name'});
            $errors{$error_text}{'count'}++;
        }
    }

    # send mails and move
    foreach my $error (sort {$b->{'count'} <=> $a->{'count'}} values(%errors)) {
        if (
            $self->app->mailer->send(
                from => {$cfg->{'email_from'} => "PI2 $error->{'app_name'}"},
                to   => $cfg->{'email_to'},
                subject => $error->{'app_name'} . ' errors (' . substr(html_decode($error->{'text'}), 0, 50) . ' )',
                content_type => 'text/html',
                body         => '<html><body>'
                  . "$error->{'error_block'}<br>"
                  . "<strong>Errors count $error->{'count'}:</strong><br><br>"
                  . join('<br>', @{$error->{'dump_files'}})
                  . '<br><br>'
                  . '</body></html>',
                attachments =>
                  [{data => $error->{'content'}, content_type => 'text/html', filename => $error->{'latest_file'}}],
            )
           )
        {
            move($_, $sent_dir) foreach @{$error->{'dump_files'}};
        }
    }

    # cleanup (sent)
    opendir(my $sh, $sent_dir || '') || return;
    my @files_old = grep(-f $_ && time() - (stat($_))[9] > $cfg->{'sent_history'} * 60 * 60 * 24,
        map($sent_dir . '/' . $_, readdir($sh)));
    closedir($sh);
    unlink(@files_old) || throw "Can't delete old files" if @files_old;
}

sub _info_about_checks {
    my ($self, $level, $checks) = @_;

    foreach my $check (sort keys(%$checks)) {
        if ($check eq 'mysql_extra_pages') {
            INFOF('For level "%s" extra pages in MySQL: %s', $level, to_json([splice(@{$checks->{$check}}, 0, 10)]));
        } elsif ($check eq 'ch_extra_pages') {
            INFOF('For level "%s" extra pages in CH: %s', $level, to_json([splice(@{$checks->{$check}}, 0, 10)]));
        } elsif ($check eq 'metrics_not_idential') {
            foreach my $field (sort keys(%{$checks->{$check}})) {
                INFOF('For level "%s" value in field "%s" not idential: %s',
                    $level, $field, to_json([splice(@{$checks->{$check}{$field}}, 0, 10)]));
            }
        } elsif ($check eq 'metrics_not_equal') {
            foreach my $field (sort keys(%{$checks->{$check}})) {
                INFOF('For level "%s" value in field "%s" not equal: %s',
                    $level, $field, to_json([splice(@{$checks->{$check}{$field}}, 0, 10)]));
            }
        }
    }
}

sub empty_dsps_monitoring : CRON('0 6 * * *') : LOCK {
    my ($self, %opts) = @_;

    my $app = $self->app;

    my $blocks = _get_blocks_with_empty_dsps($app);

    my $count = @$blocks;

    $blocks = [
        map {
            my $model = $_->{model};
            $app->$model->public_id(
                {
                    $app->$model->get_page_id_field_name() => $_->{page_id},
                    id                                     => $_->{id},
                }
            );
          } @$blocks
    ];

    if ($count > 0) {
        $self->mailer->send(
            body         => join("\n", @$blocks),
            content_type => 'text/plain',
            from         => 'default',
            subject =>
              sprintf('Найдены блоки с пустым списком DSP. Количество: %s', $count),
            to => $PARTNER_DSP_MONITORING_MAIL,
        );
    }

    send_to_graphite(
        interval => "one_day",
        path     => "empty_dsps_monitoring.block_count",
        value    => $count,
    );
}

sub missing_tags : CRON('59 10 * * *') : LOCK : STAGE('PRODUCTION') {
    my ($self) = @_;

    my $yql_operation_result = $self->api_yql->yql_start_operation_and_get_result(
        clusters     => $self->app->get_option('yt')->{replicas},
        start_params => {
            params => {
                content => q(
SELECT COUNT(pages.page_id) as pages_missing_tags_count
FROM `home/partner/dict/pages` AS pages
LEFT JOIN `home/partner/dict/page_tags` AS tags ON pages.page_id = tags.page_id
WHERE
tags.page_id IS NULL
AND (
    pages.domain in (select domain from `home/partner/dict/inner_domain`)
    OR
    pages.login in (select login from `home/partner/dict/inner_login`)
)
LIMIT 1)
            },
        },
        get_params => {
            format      => 'json',
            write_index => 0,
            limit       => 1,
        },
    );

    my $operation_result = from_json($yql_operation_result);

    # https://solomon.yandex-team.ru/?project=PI&cluster=dev&service=perl_backend&l.type=tags&graph=auto
    send_to_graphite(
        interval => "one_hour",
        path     => 'yt_missing_page_tags',
        value    => $operation_result->{pages_missing_tags_count},
        solomon  => {
            type   => 'tags',
            sensor => 'yt_missing_page_tags',
        },
    );
}

sub _get_blocks_with_empty_dsps {
    my ($app) = @_;

    my @models =
      grep {$app->$_->DOES('Application::Model::Role::Has::DSPS')}
      @{$app->product_manager->get_block_model_accessors()};

    my $working_multistate = 2**$app->context_on_site_rtb->get_multistates_bits_hs()->{working}{bit};

    return $app->partner_db->query->select(
        alias  => 'ab',
        table  => $app->partner_db()->all_blocks,
        fields => ['page_id', 'id', 'model'],
        filter => [
            AND => [
                # TODO: multistate from func
                [{'multistate' => 'ab'} => '&'  => \$working_multistate],
                [{'model'      => 'ab'} => 'IN' => \\@models],
            ]
        ],
      )->left_join(
        alias  => 'dsps',
        table  => $app->partner_db()->block_dsps,
        fields => ['dsp_id'],
        filter => [
            'AND',
            [{isNULL => ['dsp_id']},]
        ],
        join_on => [
            'AND',
            [[{'page_id' => 'ab'} => '=' => {'page_id' => 'dsps'}], [{'id' => 'ab'} => '=' => {'block_id' => 'dsps'}],]
        ],
      )->order_by(
        ['model',   0],
        ['page_id', 0],
        ['id',      0]
      )->get_all();
}

sub page_id_difference : CRON('*/5 * * * *') : LOCK {
    my ($app) = @_;
    my $all_pages_max_id =
      $app->partner_db->all_pages->get_all(fields => {'max_page_id' => {'MAX' => ['id']}})->[0]{'max_page_id'};
    my $page_id_generator_max_id =
      $app->partner_db->page_id_generator->get_all(fields => {'max_page_id' => {'MAX' => ['page_id']}})
      ->[0]{'max_page_id'};

    INFO "MAX(all_pages.id) = $all_pages_max_id. MAX(page_id_generator.page_id) = $page_id_generator_max_id";

    my $result = $page_id_generator_max_id - $all_pages_max_id;

    send_to_graphite(
        interval => 'five_min',
        path     => 'page_id_difference',
        value    => $result,
    );
}

TRUE;
