package Cron::Methods::SelfEmployedCheck;

use qbit;

use base qw(QBit::Cron::Methods);

use Exception::SelfemployedOeBS;
use Exception::SelfemployedOeBS::Unbound;
use Exception::SelfemployedOeBS::Unregistered;
use PiConstants qw($SELFEMPLOYED_STATUS_NOT $SELFEMPLOYED_STATUS_AVAILABLE $SELFEMPLOYED_STATUS_READY);
use Utils::Logger qw(INFOF);
use Utils::MonitoringUtils qw(send_to_graphite);

my $LIMIT = 500;

sub model_path {'self_employed_check'}

__PACKAGE__->model_accessors(partner_db => 'Application::Model::PartnerDB',);

my $NOTIFICATION_SHORT_NAME = 'selfemployed_off';

sub get_from_balance : CRON('0 */2 * * *') : LOCK {
    my ($self) = @_;

    my $partner_list = $self->app->users->get_all(
        filter => {self_employed => $SELFEMPLOYED_STATUS_READY, multistate => 'not blocked'},
        fields => [qw(id login client_id)],
    );

    INFOF('Found %d self-employed partners', scalar @$partner_list);
    my $notification_id = $self->_get_notification_id($NOTIFICATION_SHORT_NAME);

    for my $partner (@$partner_list) {
        my %patch = $self->_get_selfemployed_status_patch($partner);
        if (keys %patch) {
            INFOF('self_employed = 0 for partner login=%s id=%d client_id=%d',
                $partner->{login}, $partner->{id}, $partner->{client_id});

            $self->app->users->do_action($partner->{id}, 'edit', %patch,);

            $self->app->user_notifications->add(
                notification_id => $notification_id,
                user_id         => $partner->{id},
            );
        }
    }
    return 1;
}

sub get_fns_notifications : CRON('5 * * * *') : LOCK {
    my ($self) = @_;

    my $tmp = $self->app->add_all_tmp_rights();

    my $FNS_NOTIFICATION_ID = $self->app->notification->get_all(filter => {short_name => 'fns_message'})->[0]->{id};

    my $db_filter = $self->app->users->get_db_filter({'self_employed' => $SELFEMPLOYED_STATUS_READY});
    my $db = $self->app->partner_db;

    my $partner_list = $self->app->users->get_all(
        filter => $db_filter,
        fields => [qw(id inn)],
    );

    my $inn_unbound_count = 0;
    while (my @partners = splice(@$partner_list, 0, $LIMIT)) {
        $self->app->send_heartbeat;

        my $partners_inn = {map {$_->{inn} => $_} @partners};
        my $inn_list = [sort keys %$partners_inn];

        my $notifications = {};
        my %inn_list_all = map {$_ => 1} @$inn_list;
        my ($is_all_inn_ok, $i) = (FALSE, scalar keys %inn_list_all);
        while (!$is_all_inn_ok && $i > 0) {
            try {
                $i--;
                $notifications =
                  $self->app->api_selfemployed_oebs->get_notifications([grep {$inn_list_all{$_}} keys %inn_list_all],
                    1, 1);
                $is_all_inn_ok = TRUE;
            }
            catch Exception::SelfemployedOeBS::Unbound with {
                my ($e) = @_;
                #'Ошибка платформы код:TAXPAYER_UNBOUND, сообщение:Партнер не привязан к налогоплательщику c ИНН 482413852764';
                my ($bad_inn) = $e->{inn};
                $inn_list_all{$bad_inn} = 0;
                INFOF('TAXPAYER_UNBOUND for INN=[%s]', $bad_inn);
            }
            catch Exception::SelfemployedOeBS::Unregistered with {
                my ($e) = @_;
                #'Ошибка платформы код:TAXPAYER_UNREGISTERED, сообщение:Партнер не привязан к налогоплательщику c ИНН 482413852764';
                my ($bad_inn) = $e->{inn};
                $inn_list_all{$bad_inn} = 0;
                INFOF('TAXPAYER_UNREGISTERED for INN=[%s]', $bad_inn);
            }
            catch Exception::SelfemployedOeBS with {
                # nothing
            };
        }

        $inn_unbound_count += scalar grep {!$inn_list_all{$_}} keys %inn_list_all;

        my $new_messages = {map {$_->{inn} => $_->{notifications}} @{$notifications->{innList} // []}};
        foreach my $inn (@$inn_list) {
            my $partner_messages = $self->app->user_notifications->get_all(
                filter =>
                  ['AND', [{user_id => $partners_inn->{$inn}->{id}}, {notification_id => $FNS_NOTIFICATION_ID}]],
                fields => '*'
            );
            my $exists_messages = {map {$_->{custom_data}->{id} => $_} @$partner_messages};
            foreach my $msg (@{$new_messages->{$inn}}) {
                unless ($exists_messages->{$msg->{id}}) {
                    #Добавляем новые сообщения
                    my $notification_id = $self->app->user_notifications->add(
                        notification_id => $FNS_NOTIFICATION_ID,
                        user_id         => $partners_inn->{$inn}->{id},
                        custom_data => {id => $msg->{id}, fns_message => $msg->{message}, fns_title => $msg->{title}}
                    );
                    $exists_messages->{$msg->{id}} = {
                        id         => $notification_id,
                        multistate => 0
                    };
                }
                #Синхронизируем статусы
                if (
                    $msg->{status} eq 'ACKNOWLEDGED'
                    && !$self->app->user_notifications->check_multistate_flag(
                        $exists_messages->{$msg->{id}}->{multistate}, 'viewed'
                    )
                   )
                {
                    $self->app->user_notifications->maybe_do_action({id => $exists_messages->{$msg->{id}}->{id}},
                        'set_viewed');
                }
                if (
                    $msg->{status} eq 'ARCHIVED'
                    && !$self->app->user_notifications->check_multistate_flag(
                        $exists_messages->{$msg->{id}}->{multistate},
                        'deleted_by_user'
                    )
                   )
                {
                    $self->app->user_notifications->maybe_do_action({id => $exists_messages->{$msg->{id}}->{id}},
                        'set_viewed');
                    $self->app->user_notifications->maybe_do_action({id => $exists_messages->{$msg->{id}}->{id}},
                        'delete');
                }
            }
        }
    }

    send_to_graphite(
        interval => "one_hour",
        path     => "self_employed_check",
        value    => $inn_unbound_count,
        solomon  => {
            type   => 'selfemployed_unbound',
            sensor => 'self_employed',
        },
    );
}

sub send_metric : CRON('30 * * * *') : LOCK {
    my ($self) = @_;

    my $counts = $self->partner_db->_get_all(
        qq(select
sum(case when opts->>'\$.cooperation_form' = 'ph' and opts->>'\$.inn' > '' then 1 else 0 end) as selfemployed_available,
sum(case when opts->>'\$.self_employed' = $SELFEMPLOYED_STATUS_AVAILABLE then 1 else 0 end) as selfemployed_started,
sum(case when opts->>'\$.self_employed' = $SELFEMPLOYED_STATUS_READY then 1 else 0 end) as selfemployed_already
from users
where opts->>'\$.cooperation_form' = 'ph' and opts->>'\$.inn' > '')
    )->[0];

    for my $metric (qw(selfemployed_available selfemployed_started selfemployed_already)) {
        send_to_graphite(
            interval => "one_hour",
            path     => "self_employed_check",
            value    => $counts->{$metric},
            solomon  => {
                type   => $metric,
                sensor => 'self_employed',
            },
        );
    }
}

sub _get_selfemployed_status_patch {
    my ($self, $partner) = @_;

    my %patch;
    if (!$self->_is_selfemployed_in_billing($partner)) {
        %patch = (
            'self_employed_request_id' => undef,
            'self_employed'            => $SELFEMPLOYED_STATUS_NOT,
        );
    }
    return %patch;
}

sub _is_selfemployed_in_billing {
    my ($self, $partner) = @_;

    throw Exception::Validation::BadArguments gettext('Missed required parameter "%s"', 'client_id')
      unless $partner->{client_id};

    my $contract = $self->app->documents->get_active_contract({client_id => $partner->{client_id}});
    return $contract->{Contract}->{selfemployed};
}

sub _get_notification_id {
    my ($self, $short_name) = @_;

    my $notification = $self->app->notification->partner_db->query->select(
        table  => $self->app->notification->partner_db_table(),
        fields => [qw(id)],
        filter => ['AND', [[type => '=' => \'auto'], [short_name => '=' => \$short_name]]]
    )->get_all()->[0];

    throw Exception::Validation::BadArguments gettext('Missed required parameter "%s"', 'notification')
      unless $notification;

    return $notification->{id};
}

1;
