package Cron::Methods::System;

use qbit;

use base qw(QBit::Cron::Methods);

use CHI;
use Date::Calc qw(Parse_Date);
use File::Temp qw(tempfile);

use Utils::Logger qw(INFO INFOF ERROR ERRORF WARN);
use Utils::MonitoringUtils qw(get_pjapi send_to_graphite);

use Exception::BatchFail;
use Exception::DB;
use Exception::Validation::BadArguments;

use PiConstants qw(
  $SUPPORT_MAIL
  $USE_JAVA_JSONAPI_KEY
  $TRANSFER_USER
  );

my $PROCESS_TIMEOUT = 900;

__PACKAGE__->model_accessors(
    intapi_acl             => 'Application::Model::IntAPI_ACL',
    migration              => 'Application::Model::Migration',
    moderation_bad_domains => 'Application::Model::Moderation::BadDomains',
    partner_db             => 'Application::Model::PartnerDB',
    product_manager        => 'Application::Model::ProductManager',
    queue                  => 'Application::Model::Queue',
    stat_download_data     => 'Application::Model::StatDownloadData',
    users                  => 'Application::Model::Users',
    clickhouse_db          => 'Application::Model::ClickhouseDB',
    kv_store               => 'QBit::Application::Model::KvStore',
    auto_stop              => 'Application::Model::AutoStop',
    partner_db             => 'Application::Model::PartnerDB',
);

sub model_path {'system'}

sub check_block_multistates : CRON('15 7 * * *') : LOCK {
    my ($self, %opts) = @_;
    # %opts = (
    #   page_ids   => '123,456',
    # );

    my ($page_ids) = $opts{'--page_ids'};
    INFOF('Filter --page_ids="%s" provided', $page_ids) if $page_ids;
    $page_ids = [split /\s*,\s*/, $page_ids // ''];

    my @page_models = qw(
      search_on_site_campaign
      mobile_app_settings
      internal_context_on_site_campaign
      internal_search_on_site_campaign
      internal_mobile_app
      video_an_site
      context_on_site_campaign
      );

    foreach my $page_model_name (@page_models) {

        my $model        = $self->app->$page_model_name;
        my $page_id_name = $model->get_page_id_field_name();

        my $model_something_wrong;

        #
        # Check that working or testing pages have only working or deleted blocks (but not deleted_with_page)
        #

        my $active_pages_stopped_blocks = {
            #   <block_accessor> => [
            #       {
            #           public_id        => 'R-A-218787-2',
            #           multistate       => <multistate>,
            #           multistate_names => 'Заархивирован, Нет статистики',
            #       }, ...
            #   ], ...
        };

        my $active_pages = $model->get_all(
            fields => [$page_id_name, qw(id multistate )],
            filter => {multistate => 'working or testing', (@$page_ids ? ($page_id_name => $page_ids) : ())},
            order_by => [$page_id_name],
        );

        foreach my $page (@$active_pages) {

            my $page_id = $page->{$page_id_name};
            next unless defined($page_id);

            foreach my $block_model ($model->get_block_models()) {
                my $block_model_name   = $block_model->accessor;
                my $block_page_id_name = $block_model->get_page_id_field_name();

                my $use_java_jsonapi =
                  from_json($self->kv_store->get($USE_JAVA_JSONAPI_KEY) // '{}')->{$block_model_name};
                my %query = (
                    fields => ['id', ($use_java_jsonapi ? $block_page_id_name : 'public_id'), 'multistate'],
                    filter => {
                        $block_page_id_name => $page_id,
                        multistate          => $use_java_jsonapi
                        ? $self->app->$block_model_name->get_multistates_by_filter(
                            'not (working or (deleted and not deleted_with_page))')
                        : 'not (working or (deleted and not deleted_with_page))',
                        ($use_java_jsonapi ? (model => $block_model_name) : ()),
                    },
                    order_by => [$block_page_id_name, 'id'],
                );

                my $blocks =
                    $use_java_jsonapi
                  ? $self->partner_db->query->select(table => $self->app->$block_model_name->partner_db_table(), %query)
                  ->get_all()
                  : $block_model->get_all(%query);

                next unless @$blocks;

                foreach my $block (@$blocks) {
                    next if $block->{'id'} == 0;    # don't touch 0-blocks
                    my $ar = $active_pages_stopped_blocks->{$block_model_name} //= [];

                    $block->{public_id} =
                      sprintf("%s%s-%s", $block_model->public_id_prefix, $block->{$block_page_id_name}, $block->{id})
                      if $use_java_jsonapi;

                    push(
                        @$ar,
                        {
                            public_id        => $block->{'public_id'},
                            multistate       => $block->{'multistate'},
                            multistate_names => join(', ',
                                $block_model->get_multistate_name_as_list($block->{'multistate'}, private => TRUE)),
                        }
                    );
                }
            }
        }

        $model_something_wrong++ if %$active_pages_stopped_blocks;

        #
        # Check that deleted pages have deleted blocks only
        #

        my $deleted_pages_active_blocks = {
            #   <block_accessor> => [
            #       {
            #           public_id        => 'R-A-218787-2',
            #           multistate       => <multistate>,
            #           multistate_names => 'Заархивирован, Нет статистики',
            #       }, ...
            #   ], ...
        };

        my $deleted_pages = $model->get_all(
            fields => [$page_id_name, qw(id multistate)],
            filter => {multistate => 'deleted', (@$page_ids ? ($page_id_name => $page_ids) : ())},
            order_by => [$page_id_name],
        );

        foreach my $page (@$deleted_pages) {

            my $page_id = $page->{$page_id_name};
            next unless defined($page_id);

            foreach my $block_model ($model->get_block_models()) {
                my $block_page_id_name = $block_model->get_page_id_field_name();
                my $block_model_name   = $block_model->accessor;

                my $use_java_jsonapi =
                  from_json($self->kv_store->get($USE_JAVA_JSONAPI_KEY) // '{}')->{$block_model_name};
                my %query = (
                    fields => ['id', ($use_java_jsonapi ? $block_page_id_name : 'public_id'), 'multistate'],
                    filter => {
                        $block_page_id_name => $page_id,
                        multistate          => $use_java_jsonapi
                        ? $self->app->$block_model_name->get_multistates_by_filter('not deleted')
                        : 'not deleted',
                        ($use_java_jsonapi ? (model => $block_model_name) : ()),
                    },
                    order_by => [$block_page_id_name, 'id'],
                );

                my $blocks =
                    $use_java_jsonapi
                  ? $self->partner_db->query->select(table => $self->$block_model_name->partner_db_table(), %query)
                  ->get_all()
                  : $block_model->get_all(%query);

                next unless @$blocks;

                foreach my $block (@$blocks) {
                    next if $block->{'id'} == 0;    # don't touch 0-blocks
                    my $ar = $deleted_pages_active_blocks->{$block_model_name} //= [];

                    $block->{public_id} =
                      sprintf("%s%s-%s", $block_model->public_id_prefix, $block->{$block_page_id_name}, $block->{id})
                      if $use_java_jsonapi;

                    push(
                        @$ar,
                        {
                            public_id        => $block->{'public_id'},
                            multistate       => $block->{'multistate'},
                            multistate_names => join(', ',
                                $block_model->get_multistate_name_as_list($block->{'multistate'}, private => TRUE)),
                        }
                    );
                }
            }
        }

        $model_something_wrong++ if %$deleted_pages_active_blocks;

        if ($model_something_wrong) {
            INFO('## ' . uc($page_model_name));

            if (%$active_pages_stopped_blocks) {

                INFO("\t# not (working or (deleted and not deleted_with_page)) blocks on working or testing pages");
                foreach my $block_model_name (keys %$active_pages_stopped_blocks) {
                    INFO("\t\t$block_model_name");
                    INFO(sprintf("\t\t\t%s - state: %d (%s)", @$_{qw( public_id  multistate  multistate_names)}))
                      foreach @{$active_pages_stopped_blocks->{$block_model_name}};
                }
            }

            if (%$deleted_pages_active_blocks) {

                INFO("\t# not deleted blocks on deleted pages");
                foreach my $block_model_name (keys %$deleted_pages_active_blocks) {
                    INFO("\t\t$block_model_name");
                    INFO(sprintf("\t\t\t%s - state: %d (%s)", @$_{qw( public_id  multistate  multistate_names)}))
                      foreach @{$deleted_pages_active_blocks->{$block_model_name}};
                }
            }

            throw Exception 'Found bad multistates while check block multistates', sentry => {
                extra => {
                    active_pages_stopped_blocks => $active_pages_stopped_blocks,
                    deleted_pages_active_blocks => $deleted_pages_active_blocks,
                },
                fingerprint => ['Cron', 'check_block_multistates', 'Result'],
            };
        }
    }
}

sub check_page_multistates : CRON('45 7 * * *') : LOCK {
    my ($self, %opts) = @_;
    # model   => 'search_on_site_campaign'
    # page_id => 102540,
    # id      => 42524,

    my %external_site_checks = (
        site_approved_page_need_approve => {
            sub         => \&_site_approved_page_need_approve,
            pretty_name => 'Site is approved, page is need_approve',
        },
        site_approved_page_rejected => {
            sub         => \&_site_approved_page_rejected,
            pretty_name => 'Site is approved, page is rejected',
        },
        site_rejected_page_need_approve => {
            sub         => \&_site_rejected_page_need_approve,
            pretty_name => 'Site is rejected, page is need_approve',
        },
        site_rejected_page_not_rejected => {
            sub         => \&_site_rejected_page_not_rejected,
            pretty_name => 'Site is rejected, page is not rejected',
        },
    );

    my %page_checks = (
        site_context_blocked_page_active => {
            sub         => \&_site_context_blocked_page_active,
            pretty_name => 'Context pages are blocked, page is active',
        },
        site_search_blocked_page_active => {
            sub         => \&_site_search_blocked_page_active,
            pretty_name => 'Search pages are blocked, page is active',
        },
    );

    my %models = (
        site => {
            fields => [qw(id multistate domain)],
            pages  => {
                context_on_site_campaign => {
                    fields             => [qw(id multistate domain_id)],
                    site_id_field_name => 'domain_id',
                    checks =>
                      {%external_site_checks, map {$_ => $page_checks{$_}} qw/ site_context_blocked_page_active /},
                },
                search_on_site_campaign => {
                    fields             => [qw(id multistate domain_id)],
                    site_id_field_name => 'domain_id',
                    checks =>
                      {%external_site_checks, map {$_ => $page_checks{$_}} qw/ site_search_blocked_page_active /},
                },
            },
        },
    );

    my @model_names = map {keys %{$_->{pages}}} values %models;
    my $filter = {multistate => 'not deleted'};

    if (defined $opts{model}) {
        my $page_model_name = $opts{model};
        @model_names = ($page_model_name);

        my $page_id_field = $self->app->$page_model_name->get_page_id_field_name();

        $filter = {$page_id_field => $opts{page_id}} if ($opts{page_id});
        $filter = {id             => $opts{id}}      if ($opts{id});

    }

    my $site_domains = {
        # <site_id> => <dimain>
    };

    my $data = {
        #  'Site' => {
        #    '<check_id>' => [
        #       '<page_accesor>' => [
        #          {
        #             site_id  => 33299,
        #             id       => ...,
        #             page_id  => 102540,
        #           }
        #       ]
        #    ]
        #  }
    };

    my $errors_count = 0;
    # 1. Get data
    foreach my $site_model_name (keys %models) {
        my $site_model  = $self->app->$site_model_name;
        my $site_fields = $models{$site_model_name}{fields};
        my %sites       = map {($_->{id} => $_)} @{$site_model->get_all(fields => $site_fields)};

        foreach my $page_model_name (@model_names) {
            my $page_model_info    = $models{$site_model_name}{pages}{$page_model_name};
            my $page_model         = $self->app->$page_model_name;
            my $page_fields        = $page_model_info->{fields};
            my $site_id_field_name = $page_model_info->{site_id_field_name};

            my $page_id_field = $self->app->$page_model_name->get_page_id_field_name();

            my $pages = $page_model->get_all(
                fields => [@$page_fields, $page_id_field],
                ($filter ? (filter => $filter) : ())
            );

            foreach my $page (@$pages) {
                my $site_id = $page->{$site_id_field_name};
                my $site    = $sites{$site_id};

                $site_domains->{$site_id} = $site->{domain};

                foreach my $check (keys %{$page_model_info->{checks}}) {
                    my $check_sub  = $page_model_info->{checks}{$check}{sub};
                    my $check_name = $page_model_info->{checks}{$check}{pretty_name};

                    if ($check_sub->($self, $site_model, $site, $page_model, $page)) {
                        push @{$data->{$site_model_name}{$check}{$page_model_name}},
                          {
                            site_id    => $site_id,
                            id         => $page->{id},
                            page_id    => $page->{$page_id_field},
                            multistate => $page->{multistate},
                          };
                        $errors_count++;
                    }
                }
            }
        }
    }

    # 2. Log to email
    foreach my $site_model_name (keys %$data) {
        foreach my $check (sort keys %{$data->{$site_model_name}}) {

            INFO('## ' . uc($external_site_checks{$check}->{pretty_name} // $page_checks{$check}->{pretty_name}));
            foreach my $page_model_name (sort keys %{$data->{$site_model_name}{$check}}) {
                INFO("\t" . $page_model_name);
                foreach my $page_data (sort {$a->{multistate} <=> $b->{multistate}}
                    @{$data->{$site_model_name}{$check}{$page_model_name}})
                {
                    INFO(
                        sprintf "\t\t %s (id=%d, multistate=%d), %s (id=%s)",
                        map {$_ // '-'} @$page_data{qw( page_id  id  multistate )},
                        $site_domains->{$page_data->{site_id}},
                        $page_data->{site_id}
                    );
                }
            }
        }
    }

    # 3. Send to graphite
    foreach my $site_model_name (keys %models) {
        foreach my $page_model_name (keys %{$models{$site_model_name}{pages}}) {
            my $page_model_info = $models{$site_model_name}{pages}{$page_model_name};
            foreach my $check (keys %{$page_model_info->{checks}}) {
                send_to_graphite(
                    interval => "one_day",
                    path     => "${site_model_name}_${page_model_name}_${check}",
                    value    => scalar(@{$data->{$site_model_name}{$check}{$page_model_name} // []}),
                    solomon  => {
                        sensor => 'site_page_status.' . $check,
                        model  => $page_model_name,
                    }
                );
            }
        }
    }

    if ($errors_count) {
        throw Exception "Found $errors_count bad page multistates", sentry => {
            extra => {data => $data},
            fingerprint => ['Cron', 'check_page_multistates', 'Result'],
        };
    }
}

sub _site_approved_page_need_approve {
    my ($self, $site_model, $site, $page_model, $page) = @_;

    $site_model->check_multistate_flag($site->{multistate}, 'approved')
      && $page_model->check_multistate_flag($page->{multistate}, 'need_approve');
}

sub _site_approved_page_rejected {
    my ($self, $site_model, $site, $page_model, $page) = @_;

    $site_model->check_multistate_flag($site->{multistate}, 'approved')
      && $page_model->check_multistate_flag($page->{multistate}, 'rejected');
}

sub _site_rejected_page_need_approve {
    my ($self, $site_model, $site, $page_model, $page) = @_;

    $site_model->check_multistate_flag($site->{multistate}, 'rejected')
      && $page_model->check_multistate_flag($page->{multistate}, 'need_approve');
}

sub _site_rejected_page_not_rejected {
    my ($self, $site_model, $site, $page_model, $page) = @_;

    $site_model->check_multistate_flag($site->{multistate}, 'rejected')
      && !$page_model->check_multistate_flag($page->{multistate}, 'rejected');
}

sub _site_context_blocked_page_active {
    my ($self, $site_model, $site, $page_model, $page) = @_;

    $site_model->check_multistate_flag($site->{multistate}, 'context_on_site_campaign_blocked')
      && ( $page_model->check_multistate_flag($page->{multistate}, 'working')
        || $page_model->check_multistate_flag($page->{multistate}, 'testing')
        || $page_model->check_multistate_flag($page->{multistate}, 'need_approve'));
}

sub _site_search_blocked_page_active {
    my ($self, $site_model, $site, $page_model, $page) = @_;

    $site_model->check_multistate_flag($site->{multistate}, 'search_on_site_campaign_blocked')
      && ( $page_model->check_multistate_flag($page->{multistate}, 'working')
        || $page_model->check_multistate_flag($page->{multistate}, 'testing')
        || $page_model->check_multistate_flag($page->{multistate}, 'need_approve'));
}

sub check_impossible_multistates : CRON('45 11 * * *') : LOCK : STAGE('TEST') : STAGE('PRODUCTION') {
    my ($self, %opts) = @_;
    # %opts = (
    #   page_ids   => '123,456',
    # );

    my $has_errors;
    my ($page_ids) = $opts{'--page_ids'};
    INFOF('Filter --page_ids="%s" provided', $page_ids) if $page_ids;
    $page_ids = [split /\s*,\s*/, $page_ids // ''];

    my @model_names;
    if ($opts{'--accessors'}) {
        @model_names = split /\s*,\s*/, $opts{'--accessors'};
    } else {
        @model_names = $self->_get_model_names_with_multistate_graph();
    }

    foreach my $model_name (@model_names) {

        my $model = $self->app->$model_name;
        my $page_id_name = eval {$model->get_page_id_field_name()};
        # не у всех моделей есть page_id, поэтому если передан page_ids, то модели без пропускаем
        next if @$page_ids and not $page_id_name;

        my $model_fields = $model->get_model_fields;
        my %pk = map {$_ => TRUE} grep {$model_fields->{$_}{'pk'}} keys(%$model_fields);

        my $multistates = $model->get_multistates();
        my @known_multistates = sort {$a <=> $b} keys(%$multistates);

        my $fields = {map {$_ => ''} keys(%pk), 'multistate'};

        # делаю через partner_db, а не через модель, так как нужно делать фильтр по multistate
        # (а в модели фильтр по multistate специальный)
        my $elements_with_impossible_multistates = $self->app->partner_db->query->select(
            table  => $model->partner_db_table(),
            fields => $fields,
            filter => [
                AND => [
                    [multistate => 'NOT IN' => \\@known_multistates],
                    (@$page_ids ? [$page_id_name => '=' => \$page_ids] : ())
                ]
            ],
        )->get_all();

        my @errors;

        foreach my $element (@$elements_with_impossible_multistates) {
            my $pk_value;
            foreach (keys(%pk)) {
                $pk_value->{$_} = $element->{$_};
            }

            my $multistate_name = $model->get_multistate_name($element->{'multistate'}, private => TRUE);
            $multistate_name =~ s/\n/ /g;

            push @errors,
              {
                multistate      => $element->{'multistate'},
                multistate_name => $multistate_name,
                pk              => to_json($pk_value),
              };
        }

        if (@errors) {
            $has_errors = TRUE;
            INFO("## " . $model_name);

            foreach my $error (@errors) {
                INFO(   $error->{'pk'} . ' - '
                      . $error->{'multistate_name'}
                      . ' (multistate: '
                      . $error->{'multistate'}
                      . ')');
            }

            ERROR {
                message =>
                  sprintf("## Model %s has %d objects with impossible multistate", $model_name, scalar @errors),
                fingerprint => ['Cron', 'check_impossible_multistates', $model_name],
                extra => {details => \@errors}
            };
        }
    }

    throw Exception::BatchFail if $has_errors;

    return 1;
}

sub generate_models_and_validation {
    my ($self, %opts) = @_;
    my ($accessors, $reminder_of_division, $page_ids, $random_start_position, $verbose, $instances, $instance_number,
        $file_path)
      = @opts{
        qw( --accessors  --every_nth  --page_ids --random_start_position --verbose --instances --instance_number --file_path)
      };

    INFOF('Filter --accessors="%s" provided',             $accessors)             if $accessors;
    INFOF('Filter --every_nth="%s" provided',             $reminder_of_division)  if $reminder_of_division;
    INFOF('Filter --page_ids="%s" provided',              $page_ids)              if $page_ids;
    INFOF('Filter --random_start_position="%s" provided', $random_start_position) if $random_start_position;
    INFOF('Option --instance_number="%s" provided',       $instance_number)       if $instance_number;

    $verbose //= defined($reminder_of_division) || defined($page_ids) || defined($random_start_position);

    $page_ids = [split /\s*,\s*/, $page_ids // ''];
    push @$page_ids, _get_page_id_list_from_file($file_path);

    my $count = 0;
    my @names =
        $accessors ? (split /\s*,\s*/, $accessors // '')
      : $opts{accessors} ? (@{$opts{accessors}})
      :                    $self->app->qbit_validator_checker->get_model_names_that_use_qbit_validator();

    my $time_of_start_validation = time();

    foreach my $name (sort @names) {
        INFOF('%d/%d. Start process model "%s"', ++$count, scalar(@names), $name);
        $self->validation(
            accessor        => $name,
            page_ids        => $page_ids,
            step            => ($reminder_of_division // $opts{every_nth}),
            start           => $random_start_position,
            verbose         => $verbose,
            key             => $opts{key},
            time_of_sending => $opts{time_of_sending},
            instances       => $opts{instances},
            instance_number => $opts{instance_number},
            heartbeat       => sub {
                my ($app) = @_;
                $app->send_heartbeat() if $app->can('send_heartbeat');
            },
        );
    }
    my $elapsed_time = time() - $time_of_start_validation;
    INFOF('Validation_time is : (%s ses)', $elapsed_time);
}

sub run_qbit_validator_checks_instream : CRON('10 */3 * * *') : LOCK : STAGE('TEST') : STAGE('PRODUCTION') :
  INSTANCES(2) : TTL('14h') {
    my ($self, %opts) = @_;
    $self->generate_models_and_validation(
        accessors       => ['video_an_site_instream'],
        key             => 'all',
        time_of_sending => 'one_hour',
        %opts
    );
}

sub run_qbit_validator_checks_1 : CRON('10 */3 * * *') : LOCK : STAGE('TEST') : STAGE('PRODUCTION') : TTL('5h') {
    my ($self, %opts) = @_;
    $self->generate_models_and_validation(
        accessors       => ['users'],
        key             => 'all',
        time_of_sending => 'one_hour',
        %opts
    );
}

sub run_qbit_validator_checks_2 : CRON('10 */3 * * *') : LOCK : STAGE('TEST') : STAGE('PRODUCTION') : INSTANCES(5) {
    my ($self, %opts) = @_;
    $self->generate_models_and_validation(
        accessors       => ['context_on_site_campaign'],
        key             => 'all',
        time_of_sending => 'one_hour',
        %opts
    );
}

sub run_qbit_validator_checks_3 : CRON('0 */3 * * *') : LOCK : STAGE('TEST') : STAGE('PRODUCTION') : TTL('7h') {
    my ($self, %opts) = @_;
    my %useless_models =
      map {$_ => 1} (
        'context_on_site_rtb',    'context_on_site_campaign', 'users', 'design_templates',
        'video_an_site_instream', 'mobile_app_rtb'
      );
    $self->generate_models_and_validation(
        accessors => [
            grep {not exists($useless_models{$_})}
              $self->app->qbit_validator_checker->get_model_names_that_use_qbit_validator()

        ],
        key             => 'all',
        time_of_sending => 'one_hour',
        %opts
    );
}

sub validation {
    my ($self, %opts) = @_;

    my ($accessor, $key, $time_of_sending, $instance_number, $verbose) =
      @opts{qw(accessor key time_of_sending instance_number verbose)};
    my $time               = time();
    my $is_send_to_juggler = 0;
    if (    !$ENV{'TAP_VERSION'}
        and in_array($self->get_option('stage', ''), [qw(test preprod production)])
        and ($key // '') ne 'random')
    {
        $is_send_to_juggler = 1;
    }
    my ($is_ok, $details) =
      $self->app->qbit_validator_checker->check_all_elements_in_model(is_send_to_juggler => $is_send_to_juggler, %opts);
    my $time_of_check_elements = time() - $time;
    if (defined($key) && defined($time_of_sending)) {
        send_to_graphite(
            interval => $time_of_sending,
            path     => "qbit_validation.${key}.time_of_validation.${accessor}",
            value    => $time_of_check_elements,
            solomon  => {
                type   => 'qbit_validation',
                subset => $key,
                model  => $accessor,
                sensor => 'time_spent',
            },
        );
    } else {
        send_to_graphite(
            interval => "one_day",
            path     => "${accessor}_qbit_validation_time",
            value    => $time_of_check_elements,
            # TODO: WTF?
        );
    }

    INFOF('Validation_time of "%s" model is : (%s ses)', $accessor, $time_of_check_elements);
    INFOF('End of validation "%s" model', $accessor);
    my $number_of_errors = 0;
    if (!$is_ok) {
        $number_of_errors = @{$details->{'error_elements'}};
    }
    if (defined($key) && defined($time_of_sending)) {
        send_to_graphite(
            interval => $time_of_sending,
            path     => "qbit_validation.${key}.error_items.${accessor}"
              . (defined $instance_number ? "_$instance_number" : ""),
            value   => $number_of_errors,
            solomon => {
                type            => 'qbit_validation',
                subset          => $key,
                model           => $accessor,
                sensor          => 'error_items',
                instance_number => $instance_number,
            },
        );
    }

}

sub start_task : CRON('*/1 * * * *') : LOCK : STAGE('TEST') : STAGE('PRODUCTION') : INSTANCES(10) {
    my ($self) = @_;

    while (defined(my $task = $self->queue->grab_next_task())) {
        $self->queue->start($task);
    }
}

sub clean_up_finished_tasks : CRON('*/30 * * * *') : LOCK : STAGE('TEST') : STAGE('PRODUCTION') {
    my ($self) = @_;

    $self->queue->clean_up_finished_tasks();
}

sub restart_lost_tasks : CRON('*/1 * * * *') : LOCK : STAGE('TEST') : STAGE('PRODUCTION') {
    my ($self) = @_;

    $self->queue->restart_lost_tasks();
}

sub check_stuck_tasks : CRON('* * * * *') : LOCK : STAGE('TEST') : STAGE('PRODUCTION') {
    my ($self) = @_;

    $self->queue->check_stuck_tasks();
}

sub _get_model_names_with_multistate_graph {
    my ($self) = @_;

    my $app = $self->app;

    my @names =
      sort
      grep {
             blessed($app->{$_})
          && $app->{$_}->isa('QBit::Application::Model::Multistate')
          && defined($app->$_->get_multistates_bits())
          && @{$app->$_->get_multistates_bits()}
      }
      keys %{$app};

    return @names;
}

sub update_intapi_acl_cache : CRON('17,47 * * * *') : LOCK : STAGE('TEST') : STAGE('PRODUCTION') {
    my ($self) = @_;

    my $intapi_acls = $self->intapi_acl->get_all(with_acl => TRUE);

    foreach my $intapi_acl (@$intapi_acls) {
        my @subnets = $self->intapi_acl->acl2subnets($intapi_acl->{acl});

        $self->partner_db->transaction(
            sub {
                $self->partner_db->intapi_acl->edit(
                    {
                        path   => $intapi_acl->{path},
                        method => $intapi_acl->{method},
                    },
                    {acl_cached => to_json(\@subnets),}
                );
            },
        );
    }

    return 1;
}

sub kill_long_db_queries : CRON('*/1 * * * *') : LOCK : STAGE('TEST') : STAGE('PRODUCTION') {
    my ($self, %opts) = @_;

    my $dry_run            = $opts{'--dry-run'} // FALSE;
    my $time_limit_seconds = $opts{'--longer-than-seconds'};
    my $time               = curdate();

    throw Exception 'Expected seconds' if defined($time_limit_seconds) && $time_limit_seconds !~ /^[1-9][0-9]*$/;

    my $sql_query = qq[
        SELECT
            `id`, `info`, `time`
        FROM
            information_schema.processlist
        WHERE
          `db` = "partner"
          AND `user` <> '$TRANSFER_USER'
          AND `info` IS NOT NULL
        ORDER BY
            `time` DESC;
        ];

    my $conditions = {
        web => {
            check => sub {
                my ($query, $system) = @_;

                return
                     $system eq 'rosetta'
                  || $system eq 'restapi'
                  || $system eq 'webinterface'
                  || $system eq 'java_jsonapi';
            },
            time_limit => $time_limit_seconds // 600,
        },
        rename_all_pages => {
            check => sub {
                my ($query, $system) = @_;

                return $system eq 'cron'
                  && $query->{'info'} =~ /RENAME TABLE `all_pages_new`/;
            },
            time_limit  => 30,
            before_kill => sub {
                my ($this) = @_;

                INFOF("Queries before kill rename: %s",
                    join("\n", map {$_->{'info'}} @{$this->partner_db->_get_all($sql_query)}));
            },
            after_kill => sub {
                my ($this) = @_;

                WARN("Kill rename all_pages. For more information see logs...");

                INFOF("Queries after kill rename: %s",
                    join("\n", map {$_->{'info'}} @{$this->partner_db->_get_all($sql_query)}));
            },
        },
        long_query => {
            check => sub {
                my ($query, $system) = @_;
                return $system eq 'cron' || $system eq 'java_hourglass';
            },
            time_limit  => $PROCESS_TIMEOUT,
            before_kill => sub {
                my ($this) = @_;

                INFOF("Queries before kill long_query: %s",
                    join("\n", map {$_->{'info'}} @{$this->partner_db->_get_all($sql_query)}));
            },
            after_kill => sub {
                my ($this) = @_;

                WARN("Kill long_query. For more information see logs...");

                INFOF("Queries after kill long_query: %s",
                    join("\n", map {$_->{'info'}} @{$this->partner_db->_get_all($sql_query)}));
            },
        }
    };

    my $qlist = $self->partner_db->_get_all($sql_query);
    INFOF('Found %d processes to check', scalar(@$qlist));

    my @kill;
    foreach my $query (@$qlist) {
        my $system;
        if ($query->{'info'} =~ m!^\s*/\*\s*(?<json>.*?)\s*\*/!) {
            my $json_info = eval {from_json($+{'json'})};
            $system = $json_info->{'system'} // 'cron';
        } else {
            $system = 'cron';
        }

        foreach (sort keys(%$conditions)) {
            my $condition = $conditions->{$_};
            if ($condition->{'check'}->($query, $system) && $query->{'time'} > $condition->{'time_limit'}) {
                push(@kill, {%$condition, query => $query});

                last;
            }
        }
    }

    if (@kill) {
        my $total = @kill;
        INFOF("Start killing %d processes", $total);

        my $counter = 1;
        foreach my $kill (sort {$a->{'time_limit'} <=> $b->{'time_limit'}} @kill) {
            my ($id, $time, $info) = @{$kill->{'query'}}{qw(id time info)};

            my ($days, $hours, $minutes, $seconds) = (gmtime($time))[7, 2, 1, 0];

            my $time_str =
                ($days    ? "${days}d "    : '')
              . ($hours   ? "${hours}h "   : '')
              . ($minutes ? "${minutes}m " : '')
              . "${seconds}s";

            INFOF("killing %d/%d: id: %d time: %s info: %s\n", $counter, $total, $id, $time_str, $info);

            try {
                if (my $before_kill = $kill->{'before_kill'}) {
                    $before_kill->($self);
                }

                $self->app->partner_db->_do("KILL ?;", $id) unless $dry_run;

                if (my $after_kill = $kill->{'after_kill'}) {
                    $after_kill->($self);
                }
            }
            catch Exception::DB with {
                my ($e) = @_;
                if ($e->message() =~ /Unknown thread id/) {
                    INFOF("Unknown thread id");
                } elsif ($e->message() =~ /You are not owner of thread/) {
                    INFO $e->message();
                } else {
                    throw $e;
                }
            };

            $counter++;
        }
    }

    return 1;
}

sub update_heavy_pages : CRON('0 */2 * * *') : LOCK {
    my ($self) = @_;

    my %db_heavy_pages = ();

    foreach my $block_model (@{$self->product_manager->get_block_model_names()}) {
        my $page_id_field_name = $self->app->$block_model->get_page_id_field_name();

        my $table = $self->app->$block_model->partner_db_table();

        my $query = $self->partner_db->query->select(
            table  => $table,
            fields => {cnt => {count => ['id']}, page_id => $page_id_field_name},
            filter => [
                'AND',
                [
                    ['multistate', 'IN', \$self->app->$block_model->get_multistates_by_filter('working')],
                    [
                        $page_id_field_name,
                        'NOT IN',
                        $self->partner_db->query->select(
                            table  => $self->partner_db->heavy_pages,
                            fields => [qw(page_id)]
                        )
                    ],
                    ($table->have_fields('model') ? ['model', '=', \$block_model] : ()),
                ]
            ]
        );

        $query->group_by($page_id_field_name);

        my $blocks = $query->get_all();

        foreach (@$blocks) {
            $db_heavy_pages{$_->{'page_id'}} += $_->{'cnt'};
        }
    }

    my $number = 200;

    my @page_ids = sort grep {$db_heavy_pages{$_} >= $number} keys(%db_heavy_pages);

    if (@page_ids) {
        INFOF "New heavy pages: %s", join(', ', sort {$a <=> $b} @page_ids);

        $self->partner_db->heavy_pages->add_multi([map {{page_id => $_}} @page_ids]);
    }
}

sub do_optimize_for_clickhouse_tables : CRON('0 21 * * *') : LOCK : FREQUENCY_LIMIT('1w') : TTL('25h') {
    my ($self) = @_;

    my @table_names = sort keys(%{$self->clickhouse_db->get_all_meta()->{'tables'}});

    foreach my $table_name (@table_names) {
        my $engine = $self->clickhouse_db->$table_name->engine;

        # это хэш у которого всегда один ключ
        if ([keys(%$engine)]->[0] eq 'ReplicatedSummingMergeTree') {
            $self->clickhouse_db->$table_name->optimize();
        }
    }
}

sub check_read_only_and_protected : CRON('0 */3 * * *') : LOCK {
    my ($self, %opts) = @_;

    $ENV{FORCE_EDIT_PROTECTED} = 1;

    my $page_accessors;
    if ($opts{model}) {
        $page_accessors = [split /\s*,\s*/, $opts{model}];
    } else {
        $page_accessors = $self->product_manager->get_page_model_accessors();
    }
    my $page_id;
    if ($opts{page_id}) {
        $page_id = [split /\s*,\s*/, $opts{page_id}];
    }

    my $has_failed;
    foreach my $model (sort @$page_accessors) {
        my $has_protected = $self->app->$model->get_multistate_by_name('protected');
        my $has_readonly  = $self->app->$model->get_multistate_by_name('read_only');

        next if $model =~ /^ssp_/ || (!$has_protected && !$has_readonly);

        INFOF 'Start process model "%s"', $model;
        my ($checked, $fixed, $has_error) = $self->app->$model->check_read_only_and_protected_status(
            page_id         => $page_id,
            check_ro_status => $has_readonly ? TRUE : FALSE
        );
        send_to_graphite(
            interval => "one_hour",
            path     => "check_read_only_and_protected.$model.checked",
            value    => $checked,
            solomon  => {
                model  => $model,
                sensor => 'check_read_only_and_protected.checked',
            }
        );
        send_to_graphite(
            interval => "one_hour",
            path     => "check_read_only_and_protected.$model.fixed",
            value    => $fixed,
            solomon  => {
                model  => $model,
                sensor => 'check_read_only_and_protected.fixed',
            }
        );
        INFOF 'Done process model "%s" checked=%d fixed=%d%s', $model, $checked, $fixed,
          ($has_error ? ' with_error' : '');
        $has_failed ||= $has_error;
    }

    # Turn on Juggler's CRIT for cron-job
    throw Exception::BatchFail if $has_failed;
}

sub create_juggler_checks : CRON('*/10 * * * *') : LOCK : STAGE('PRODUCTION') {
    my ($self) = @_;

    my $juggler_checks_version_key = 'juggler_checks_version';

    my $current_version = $self->kv_store->get($juggler_checks_version_key) // '';

    my $pja = get_pjapi($self);

    my $current_package_version = $self->app->get_option('version') // throw Exception 'Expected version';
    my $new_version = $current_package_version . '_' . $pja->get_option('md5_hex_config');

    if ($current_version ne $new_version) {
        $self->app->generate_juggler($pja);
        $self->app->qbit_validator_checker->generate_juggler($pja);

        $self->kv_store->set($juggler_checks_version_key, $new_version);
    }

    return TRUE;
}

sub clear_chi_file_cache : CRON('15 4 * * *') : USER('root') {
    my ($self, %opts) = @_;

    my $now = curdate('oformat' => 'db');

    my $path = $opts{'path'} // $self->app->get_option('chi_file_cache_dir');

    my $dh;
    my %data;
    local $; = ';';    # Разделитель для полей составного ключа
    if (opendir($dh, $path)) {
        while (my $name = readdir($dh)) {
            next if ($name eq '.' || $name eq '..');
            my ($date, $tm, $key) = ($name =~ /^(\d{4}-\d\d-\d\d)_(\d\d-\d\d-\d\d)_\d+_(.+)\.dat$/);
            # $key содержит название ручки и с какими параметрами её дёргали
            if ($key) {
                next if ($date eq $now);    # За текущий день не трогаем

                # Группировка на каждый день по каждой дате каждой ручки
                my $files = $data{$date, $key} //= [];
                my $r = {
                    'name' => $name,
                    'tm'   => $tm,
                };
                push @$files, $r;
            }
        }
        closedir $dh;
    } else {
        ERRORF(q[Can't open dir %s: %d - %s], $path, $!, $!);
    }

    INFOF(q[Found %d groups of files], scalar keys %data);

    foreach my $group (sort keys %data) {
        INFOF(q[Process: %s], $group);
        my @set = sort {$a->{'tm'} cmp $b->{'tm'}} @{$data{$group}};
        # Оставляем в каждой группе по одному файлу с максимальным временем
        my $save = (pop @set)->{'name'};
        @set = map {$_->{'name'}} @set;
        INFOF(q[Save: %s], $save);
        if (@set) {
            INFOF(q[Unlink: %s], join ', ', @set);
            my $cnt = unlink map {join('/', $path, $_)} @set;
            if ($cnt) {
                INFO('Unlink is success');
            } else {
                ERRORF(q[Not all files from group (%s) were deleted: %d - %s], $group, $!, $!);
            }
        } else {
            INFO('Nothing to remove');
        }
    }

    `find $path -type f -mtime +20 -delete`;
}

sub check_yamoney_cert_expiration : CRON('0 12 * * *') : NOLOCK : FRONTEND : STAGE('TEST') : STAGE('PRODUCTION') {
    my ($self, %opts) = @_;

    my $YAMONEY_CERT_FILE_PATH = '/etc/partners-ssl/yamoney.pem';
    my $days_to_expire         = _get_cert_expires_in_days($YAMONEY_CERT_FILE_PATH);

    return get_pjapi($self)->send(
        events => [
            {
                service => 'yamoney-ssl',
                status  => $days_to_expire > 28 ? 'OK' : ($days_to_expire >= 14 ? 'WARN' : 'CRIT'),
                host    => $self->app->get_option('hostname'),
            }
        ]
    );
}

sub clean_old_crons_raw_stat : CRON('0 * * * *') : LOCK {
    my ($self, %opts) = @_;

    my $count =
      $self->partner_db->crons_raw_stat->delete(
        $self->partner_db->filter([dt => '<' => \date_sub(curdate(), month => 3, oformat => 'db_time')]));

    INFOF 'Rows deleted: %d', $count;

    send_to_graphite(
        interval => "one_day",
        path     => "cron_raw_stat_deleted",
        value    => $count,
        solomon  => {
            sensor => 'deleted',
            type   => 'cron_raw_stat',
        }
    );
    send_to_graphite(
        interval => "one_day",
        path     => "cron_raw_stat_count",
        value    => $self->partner_db->crons_raw_stat->get_all(fields => {count => {COUNT => ['id']},},)->[0]{count},
        solomon  => {
            sensor => 'count',
            type   => 'cron_raw_stat',
        }
    );
}

sub clean_old_cron_auto_stop : CRON('0 3 * * *') : LOCK {
    my ($self, %opts) = @_;

    my $count =
      $self->partner_db->cron_auto_stop->delete(
        $self->partner_db->filter([dt => '<' => \date_sub(curdate(), month => 3, oformat => 'db')]));

    INFOF 'Rows deleted: %d', $count;
}

sub check_soon_auto_stop : CRON('*/5 * * * *') : LOCK {
    my ($self, %opts) = @_;

    my $after_date      = delete $opts{after_date}      // curdate(oformat => 'db');
    my $warn_days_left  = delete $opts{warn_days_left}  // 7;
    my $crit_days_left  = delete $opts{crit_days_left}  // 2;
    my $money_last_days = delete $opts{money_last_days} // 90;
    my $money_per_month = delete $opts{money_per_month} // 1_000_000;

    my $crit_date = date_add($after_date, iformat => 'db', oformat => 'db', day => $crit_days_left);
    my $warn_date = date_add($after_date, iformat => 'db', oformat => 'db', day => $warn_days_left);
    $warn_date = nearest_workday($warn_date, iformat => 'db', oformat => 'db');
    $warn_days_left = dates_delta_days($after_date, $warn_date, iformat => 'db');

    my ($warn, $crit) = $self->auto_stop->check_soon_auto_stop(
        after_date      => $after_date,
        warn_date       => $warn_date,
        crit_date       => $crit_date,
        money_last_days => $money_last_days,
        money_per_month => $money_per_month,
    );

    INFOF 'Soon auto_stop: warn=%d crit=%d', scalar(@$warn), scalar(@$crit);

    for my $service (
        (
            {
                service => 'soon_auto_stop.warn',
                data    => $warn,
                day     => $warn_days_left,
                status  => 'WARN',
            },
            {
                service => 'soon_auto_stop.crit',
                data    => $crit,
                day     => $crit_days_left,
                status  => 'CRIT',
            },
        )
      )
    {
        my $description;
        my $status;
        if (@{$service->{data}}) {
            $description = sprintf 'Stopped in %d day: %s', $service->{day}, join(',', @{$service->{data}});
            $status = $service->{status};
        } else {
            $status = 'OK';
        }
        $self->app->send_event(
            service => $service->{service},
            status  => $status,
            ($description ? (description => substr($description, 0, 1000)) : ()),
        );
    }
}

sub _get_cert_expires_in_days {
    my ($cert_file_path) = @_;

    my $exp_dt_f = _get_cert_expire_date_from_file($cert_file_path);
    my $exp_dt   = _parse_cert_expire_date($exp_dt_f);

    return dates_delta_days(curdate(), $exp_dt);
}

sub _get_cert_expire_date_from_file {
    my ($cert_file_path) = @_;
    return run_shell(qq(openssl x509 -in "$cert_file_path" -enddate), silent => TRUE);
}

sub _parse_cert_expire_date {
    my ($cert_str) = @_;

    # 'Oct 22 16:12:03 2020 GMT' = '%b %d %T %Y %Z'
    if ($cert_str =~ m/notAfter=([a-z]{3} \d{2} \d{2}:\d{2}:\d{2} \d{4} GMT)/igm) {
        return [Parse_Date($^N)];
    } else {
        throw Exception 'cannot parse notAfter date from certificate: ' . [split("\n", $cert_str)]->[0];
    }
}

sub _get_page_id_list_from_file {
    my ($file_path) = @_;
    my @page_ids;
    if ($file_path && -r $file_path) {
        if (open(my $FH, "<", $file_path)) {
            while (my $page_id = <$FH>) {
                push @page_ids, grep {$_ ne ""} map {trim($_)} split /,/, $page_id;
            }
            close $FH;
        } else {
            throw Exception "cannot open file: $file_path";
        }
    }
    return @page_ids;
}

TRUE;
