package Cron::Methods::UpdateBKAsync;

use qbit;

use base qw(QBit::Cron::Methods Cron::Methods);

use Utils::Logger qw(INFO INFOF ERROR);

__PACKAGE__->model_accessors(partner_db => 'Application::Model::PartnerDB',);

my $PAGE_LIMIT        = 100;
my $PAGE_UPDATE_LIMIT = 10;

sub model_path {'update_bk_async'}

# Помечаем все пейджи к отправке (раз в 2 недели)
#sub set_update_all_pages_in_bk : CRON('0 20 1,15 * *') : LOCK {
#    my ($self) = @_;
#    $self->app->all_pages->mark_pages_for_async_update(
#        'min_send_date' => date_sub(curdate(), day => 7, oformat => "db_time"));
#}

# Отправка очень больших пейджей
sub async_update_heavy_pages_in_bk : CRON('*/10 * * * *') : LOCK : STAGE('TEST') : STAGE('PRODUCTION') {
    my ($self) = @_;

    $self->app->api_bk->{'__SOAP__'}->proxy->timeout(7200);

    _async_update_in_bk(
        $self,
        'only_page_ids'     => [map {$_->{'page_id'}} @{$self->partner_db->heavy_pages->get_all()}],
        'limit'             => 1,
        'is_regular_update' => TRUE,
    );
}

# Отправляем все пейджи, кроме очень больших
sub async_update_light_pages_in_bk : CRON('*/1 * * * *') : LOCK : STAGE('TEST') : STAGE('PRODUCTION') : INSTANCES(10) {
    my ($self, %opts) = @_;

    # Все опции из командной строки, например --only_page_ids=275468,274310
    my %cmd_opts = map {my $orig = $_; s/^--//; $_ => $opts{$orig}} grep {$_ =~ /^--/} keys %opts;

    _async_update_in_bk(
        $self,
        'except_page_ids'   => [map {$_->{'page_id'}} @{$self->partner_db->heavy_pages->get_all()}],
        'instances_count'   => $opts{'instances'},
        'instance_number'   => $opts{'instance_number'},
        'limit'             => 100,
        'is_regular_update' => TRUE,
        %cmd_opts
    );

    return 1;
}

# Метод проверяет разницу во времени постановки на отправку и текущим времененем, если оно больше порога, ругается в почту
sub check_update_in_bk : CRON('*/5 * * * *') : LOCK : STAGE('TEST') : STAGE('PRODUCTION') {
    my ($self) = @_;

    my $page_accessors = $self->app->product_manager->get_page_model_accessors();

    foreach my $page_accessor (@$page_accessors) {
        my $page_field_name = $self->app->$page_accessor->get_page_id_field_name();

        my %temp_hash = (
            accessor => $page_accessor,
            fields   => [$page_field_name],
            filter   => {
                field    => $page_field_name,
                not_null => TRUE,
            }
        );

        $self->check_model_items_update_time(
            %temp_hash,
            graphite_metrics => sprintf('UpdateBKAsync.%s_pages_outdated', $page_accessor),
            solomon_metrics  => {
                type   => 'UpdateBKAsync',
                model  => $page_accessor,
                sensor => 'pages_outdated',
            },
            update_time_limit     => 60 * 60,
            warn_if_hang_too_long => TRUE,
            separate_metrics      => {
                graphite_metrics => sprintf('UpdateBKAsyncUserChanges.%s_pages_outdated', $page_accessor),
                solomon_metrics  => {
                    type   => 'UpdateBKAsyncUserChanges',
                    model  => $page_accessor,
                    sensor => 'pages_outdated',
                },
            }
        );

        $self->check_model_items_update_time(
            %temp_hash,
            graphite_metrics => sprintf('UpdateBKAsyncAll.%s_pages_outdated', $page_accessor),
            solomon_metrics  => {
                type   => 'UpdateBKAsyncAll',
                model  => $page_accessor,
                sensor => 'pages_outdated',
            },
            update_time_limit => 0,
            separate_metrics  => {
                graphite_metrics => sprintf('UpdateBKAsyncAllUserChanges.%s_pages_outdated', $page_accessor),
                solomon_metrics  => {
                    type   => 'UpdateBKAsyncAllUserChanges',
                    model  => $page_accessor,
                    sensor => 'pages_outdated',
                },
            }
        );
    }

    return TRUE;
}

# Метод осуществяет "асинхронную" отправку в БК - всех пейджей на площадках и блоках которых стоит флаг "set_need_update"
# NOTE! Ошибки не блочат всю отправку, но их нужно мониторть на борде отправки!
sub _async_update_in_bk {
    my ($self, %opts) = @_;

    $ENV{FORCE_EDIT_PROTECTED} = 1;

    my $page_with_blocks_accessors = $self->app->product_manager->get_page_model_accessors();

    my $rows_to_update_by_accessor = {
        #  <page_accessor>  => [
        #     { <page_id> => <id> },
        #     ...
        #  ],
        #  ...
    };
    foreach my $page_accessor (@$page_with_blocks_accessors) {
        my $page_field_name = $self->app->$page_accessor->get_page_id_field_name();

        my $found_rows = $self->get_model_items_to_update(
            accessor => $page_accessor,
            fields   => [$page_field_name, 'id'],
            order_by => $opts{order_by},
            filter   => {
                field           => $page_field_name,
                not_null        => !$self->app->$page_accessor->may_send_not_balance_registered(),
                only_ids        => $opts{'only_page_ids'},
                except_ids      => $opts{'except_page_ids'},
                instances_count => $opts{'instances_count'},
                instance_number => $opts{'instance_number'},
                limit           => $opts{'limit'},
                multistate      => $opts{multistate},
                model_filter    => $opts{model_filter},
            },
        );

        $rows_to_update_by_accessor->{$page_accessor} = $found_rows if $found_rows && @$found_rows;
    }

    INFOF('found %d pages to send to BK', scalar(map {@$_} values %$rows_to_update_by_accessor))
      if %$rows_to_update_by_accessor;

    foreach my $page_accessor (sort keys %$rows_to_update_by_accessor) {

        my $page_field_name = $self->app->$page_accessor->get_page_id_field_name();

        my @page_ids = sort map {$_->{$page_field_name}} @{$rows_to_update_by_accessor->{$page_accessor}};

        INFOF(qq[start sending "%s" (%s pages)], $page_accessor, scalar(@page_ids)) if @page_ids;

        my @failed_page_ids = ();
        foreach my $page_id (@page_ids) {

            INFOF("\tsending page=%s", $page_id);

            try {
                $self->app->$page_accessor->update_in_bk(
                    {$page_field_name => $page_id},
                    map {$_ => $opts{$_}} qw(do_not_change_multistate logbrocker_only protobuf_only is_regular_update)
                );
            }
            catch {
                my $exception = shift;
                my $msg       = $exception->message();

                push @failed_page_ids, $page_id;

                if (grep {$msg =~ /$_/} qr/Global Lock/, qr/Exists SSPID with this Token for mocked PageID=\d+/) {
                    # STDOUT - только в логи (без писем и без Sentry)
                    INFO($msg);
                } else {
                    # Заполнение по одному, так как там уже может быть структура, а это дополнение
                    $exception->{sentry}{extra}{original_message} = $msg;
                    $exception->{sentry}{extra}{page_id}          = $page_id;
                    $self->app->exception_dumper->dump_as_html_file($exception);
                }
            };
        }

        INFOF(
            "\t%s - %d pages sended successfully%s",
            $page_accessor,
            (scalar(@page_ids) - scalar(@failed_page_ids)),
            (@failed_page_ids ? ' (' . scalar(@failed_page_ids) . ' failed)' : '')
        );

    }

    return TRUE;
}

=begin comment

Саба с помощью которого можно определить, запускать ли указанный page на инстансе.
Все параметры обязательные.

    my $bool = _is_instance_for_page(
        page_id => $_->{page_id},
        instances_count => $opts{instances_count}, # Общее количество инстансов. Должно быть >= 1
        instance_number => $opts{instance_number}, # Порядковый номер инстанса. Должно было >= 1 и <= instances_count
    )

=end comment

=cut

sub _is_instance_for_page {
    my (%opts) = @_;

    return $opts{page_id} % $opts{instances_count} == $opts{instance_number} - 1;
}

sub async_logbrocker_light_pages_in_bk : CRON('*/1 * * * *') : LOCK : STAGE('TEST') : STAGE('PRODUCTION') {
    my ($self, %opts) = @_;

    # Все опции из командной строки, например --only_page_ids=275468,274310
    my %cmd_opts = map {my $orig = $_; s/^--//; $_ => $opts{$orig}} grep {$_ =~ /^--/} keys %opts;

    _async_update_in_bk(
        $self,
        except_page_ids => [map        {$_->{'page_id'}} @{$self->partner_db->heavy_pages->get_all()}],
        limit           => $PAGE_LIMIT,
        order_by        => [[send_time => 0]],
        multistate   => 'working and not updating and not need_update',
        model_filter => [send_time => '<' => \date_sub(curdate(), week => 2, oformat => 'db_time')],

        # PI-20292: Отправлять данные и в старый и в новый транспорт
        # В соответствии с этим тут игнорируем logbrocker_only
        # Отправка через EditPage в том числе отправляет и в ЛБ
        # Если потом снова понадобится отправлять только в ЛБ
        # Надо будет вернуть опцию logbrocker_only => TRUE
        do_not_change_multistate => TRUE,
        logbrocker_only          => FALSE,

        %cmd_opts
    );

    return 1;
}

sub async_logbrocker_heavy_pages_in_bk : CRON('*/10 * * * *') : LOCK : STAGE('TEST') : STAGE('PRODUCTION') {
    my ($self) = @_;

    $self->app->api_bk->{'__SOAP__'}->proxy->timeout(7200);

    _async_update_in_bk(
        $self,
        only_page_ids => [map        {$_->{'page_id'}} @{$self->partner_db->heavy_pages->get_all()}],
        limit         => 1,
        order_by      => [[send_time => 0]],
        multistate   => 'working and not updating and not need_update',
        model_filter => [send_time => '<' => \date_sub(curdate(), week => 2, oformat => 'db_time')],

        # PI-20292: Отправлять данные и в старый и в новый транспорт
        # В соответствии с этим тут игнорируем logbrocker_only
        # Отправка через EditPage в том числе отправляет и в ЛБ
        # Если потом снова понадобится отправлять только в ЛБ
        # Надо будет вернуть опцию logbrocker_only => TRUE
        do_not_change_multistate => TRUE,
        logbrocker_only          => FALSE,
    );
}

sub check_send_in_bk : CRON('*/5 * * * *') : LOCK : STAGE('TEST') : STAGE('PRODUCTION') {
    my ($self) = @_;

    my $accessors = $self->app->product_manager->get_page_model_accessors();

    foreach my $accessor (@$accessors) {
        my $field      = $self->app->$accessor->get_page_id_field_name();
        my $multistate = $self->app->$accessor->get_multistates_by_filter('working');
        my $list       = $self->app->partner_db->$accessor->get_all(
            fields => {cnt => {count => [$field]}},
            filter => [
                AND => [
                    [multistate => 'IN' => \$multistate],
                    [send_time  => '<'  => \date_sub(curdate(), week => 2, oformat => 'db_time')]
                ]
            ],
        );
        Cron::Methods::send_to_graphite(
            path    => "UpdateBKAsync.${accessor}_long_ago_send",
            value   => $list->[0]{cnt},
            solomon => {
                type   => 'UpdateBKAsync',
                model  => $accessor,
                sensor => 'long_ago_send',
            },
        );
        INFO "count $accessor: " . ($list->[0]{cnt} // '<undef>');
    }

    return TRUE;
}

sub send_pages_to_bk : CRON('* * * * *') : STAGE('PRODUCTION') : INSTANCES(2) {
    my ($self) = @_;

    my $curdate = curdate(oformat => 'db_time');
    my $curdate_id = trdate('db_time', 'date_time_only_numbers', $curdate);

    my $worker_id = sprintf('%s_%d_%d', $self->get_option('hostname'), $$, $curdate_id);

    my $need_update_pages_table = $self->partner_db->need_update_pages;

    my $sub_query = $self->partner_db->query->select(
        table => $self->partner_db->query->select(
            table  => $need_update_pages_table,
            fields => [qw(page_id)],
            filter => ['AND', [['worker', 'IS NOT', \undef], ['processed', '=', \0]]]
        ),
        fields => [qw(page_id)],
        alias  => 't'
    );

    $need_update_pages_table->edit(
        $self->partner_db->filter(['AND', [['worker', 'IS', \undef], ['page_id', 'NOT IN', $sub_query]]]),
        {
            worker       => $worker_id,
            start_update => $curdate,
        },
        limit => $PAGE_UPDATE_LIMIT,
    );

    my $pages_to_send = $need_update_pages_table->get_all(
        fields   => [qw(id page_id model)],
        filter   => ['AND', [[worker => '=' => \$worker_id], [processed => '=' => \0]]],
        order_by => ['model'],
    );

    unless (@$pages_to_send) {
        INFOF('Pages not found');
        return FALSE;
    }

    INFOF('found %d pages to send to BK', scalar(@$pages_to_send));

    my $pages_by_model = {};
    foreach my $row (@$pages_to_send) {
        my $page_id       = $row->{'page_id'};
        my $page_accessor = $row->{'model'};

        $pages_by_model->{$page_accessor}{$page_id} = $row;

        INFOF("Sending page=%d (%s)", $page_id, $page_accessor);

        try {
            $row->{'start_update'} = curdate(oformat => 'db_time');

            $self->app->$page_accessor->update_in_bk({page_id => $page_id}, do_not_change_multistate => TRUE,);

            $row->{'processed'} = 1;
        }
        catch {
            my $exception = shift;
            my $msg       = $exception->message();

            $row->{'processed'} = 0;

            if (grep {$msg =~ /$_/} qr/Global Lock/, qr/Exists SSPID with this Token for mocked PageID=\d+/) {
                # STDOUT - только в логи (без писем и без Sentry)
                INFO($msg);
            } else {
                # Заполнение по одному, так как там уже может быть структура, а это дополнение
                $exception->{sentry}{extra}{original_message} = $msg;
                $exception->{sentry}{extra}{page_id}          = $page_id;
                $self->app->exception_dumper->dump_as_html_file($exception);
            }
        }
        finally {
            $row->{'stop_update'} = curdate(oformat => 'db_time');
        };
    }

    my @sent_pages = ();
    my @failed_ids = ();
    foreach my $page_accessor (sort keys(%$pages_by_model)) {
        my $pages = $self->app->$page_accessor->get_all(
            fields => [qw(page_id update_time)],
            filter => {page_id => [keys(%{$pages_by_model->{$page_accessor}})]}
        );

        foreach my $page (@$pages) {
            my $row = $pages_by_model->{$page_accessor}{$page->{'page_id'}};

            if ($page->{'update_time'} ge $row->{'start_update'} && $page->{'update_time'} le $row->{'stop_update'}) {
                INFOF("Page %d (%s) was updated", $row->{'page_id'}, $page_accessor);

                $row->{'processed'} = 0;
            }

            if ($row->{'processed'} == 1) {
                delete(@$row{qw(page_id model)});
                push(@sent_pages, $row);
            } else {
                push(@failed_ids, $row->{'id'});
            }
        }
    }

    if (@sent_pages) {
        INFOF("Sent %d pages", scalar(@sent_pages));

        $need_update_pages_table->add_multi(\@sent_pages, duplicate_update => TRUE);
    }

    if (@failed_ids) {
        INFOF("Failed %d pages", scalar(@failed_ids));

        $need_update_pages_table->edit({id => \@failed_ids}, {worker => \undef, start_update => \undef});
    }

    return TRUE;
}

sub grab_lost_updating_pages : CRON('0 * * * *') : LOCK : STAGE('PRODUCTION') {
    my ($self) = @_;

    my $filter = $self->partner_db->filter(
        [
            'AND',
            [
                ['worker',    'IS NOT', \undef],
                ['processed', '=',      \0],
                ['start_update', '<', \date_sub(curdate(), hour => 3, oformat => 'db_time')],
            ]
        ]
    );

    $self->partner_db->need_update_pages->edit(
        $filter,
        {
            worker       => \undef,
            start_update => \undef,
        }
    );
}

sub clean_need_update_pages_table : CRON('0 9 * * *') : LOCK : STAGE('PRODUCTION') {
    my ($self) = @_;

    my $filter = $self->partner_db->filter(
        [
            'AND',
            [['processed', '=', \1], ['stop_update', '<', \date_sub(curdate(), month => 1, oformat => 'db_time')],]
        ]
    );

    $self->partner_db->need_update_pages->delete($filter);
}

TRUE;
