package Cron::Methods::UserNotifications;

use qbit;

use base qw(QBit::Cron::Methods Cron::Methods);

use Utils::Logger qw(INFOF);
use Utils::MonitoringUtils;
use PiConstants qw($NOTIFICATION_CLEANUP_EXCLUDE_LIST);

__PACKAGE__->model_accessors(
    notification       => 'Application::Model::Notification',
    user_notifications => 'Application::Model::UserNotifications',
    users              => 'Application::Model::Users',
);

sub model_path {'user_notifications'}

my $SEC_IN_DAY = 60 * 60 * 24;

=head2
    perl -I./lib -MCron -e 'Cron->new->do' user_notifications generate
=cut

sub generate_logins : CRON('*/5 * * * *') : LOCK : STAGE('PRODUCTION') {
    my ($self) = @_;

    my $app = $self->app;

    my $notification_list = $app->notification->get_all(
        filter => ['AND', [['type', '=', 'custom'], ['multistate', '=', 'working and need_update']]]);
    INFOF('found [%d] changed notifications', scalar @$notification_list);

    foreach my $notification (@$notification_list) {

        $app->partner_db->transaction(
            sub {
                my $tmp = $app->add_all_tmp_rights();
                $app->notification->do_action($notification->{id}, 'start_update');

                # delete user_notificatons that are unread and user_id no longer in notification.opts.roles or notification.opts.logins
                my $query_un2delete = _get_query_user_notifications_2delete($app, $notification);
                my $un_list2delete = $query_un2delete->get_all();

                if (@$un_list2delete) {
                    my $filter = $app->partner_db->filter([id => 'IN' => \[map {$_->{id}} @$un_list2delete]]);
                    my $deleted_cnt = $self->user_notifications->partner_db_table()->delete($filter);

                    if ($deleted_cnt > 0) {
                        INFOF('processing notification id[%d]; deleted user_notifications count: [%d]',
                            $notification->{id}, $deleted_cnt);

                        Utils::MonitoringUtils::send_to_graphite(
                            interval => 'five_min',
                            path     => 'user_notification.delete.' . $notification->{short_name},
                            value    => $deleted_cnt,
                            solomon  => {
                                notification_name => $notification->{short_name},
                                sensor            => 'user_notification.delete',
                            }
                        );
                    }
                }

                # add user_notificatons for users whose user_id is added to notification.opts.roles or notification.opts.logins
                # and that are not present in table atm
                my $query_un2add = _get_query_user_notifications_2add($app, $notification);
                my $un_list2add = $query_un2add->get_all();
                if (@$un_list2add) {
                    INFOF('processing notification id[%d]; found users to add notification count: [%d]',
                        $notification->{id}, scalar @$un_list2add);

                    my @un_list = map {
                        {
                            create_date     => curdate(oformat => 'db_time'),
                            notification_id => $notification->{id},
                            user_id         => $_->{uids_id},
                            opts            => to_json({}),
                        }
                    } @$un_list2add;

                    my $added = $app->partner_db->user_notifications->add_multi(\@un_list);

                    INFOF('processing notification id[%d]; added user_notifications count: [%d]',
                        $notification->{id}, $added)
                      if ($added > 0);
                }

                $app->notification->do_action($notification->{id}, 'stop_update');
            }
        );
    }
}

sub generate_roles : CRON('*/15 * * * *') : LOCK : STAGE('PRODUCTION') {
    my ($self) = @_;

    my $app = $self->app;

    my $notification_list = $app->notification->get_all(
        fields => [qw(id roles)],
        filter => ['AND', [['type', '=', 'custom'], ['multistate', '=', 'working']]],
    );
    INFOF('found [%d] changed notifications', scalar @$notification_list);

    foreach my $notification (@$notification_list) {
        my $query_un2add = _get_query_user_notifications_2add_by_roles($app, $notification);
        my $un_list2add = $query_un2add->get_all();
        next unless @$un_list2add;

        INFOF('processing notification id[%d]; found users to add notification count: [%d]',
            $notification->{id}, scalar @$un_list2add);

        my @un_list = map {
            {
                create_date     => curdate(oformat => 'db_time'),
                notification_id => $notification->{id},
                user_id         => $_->{user_id},
                opts            => '{}',
            }
        } @$un_list2add;

        my $added = $app->partner_db->user_notifications->add_multi(\@un_list);

        INFOF('processing notification id[%d]; added user_notifications count: [%d]', $notification->{id}, $added)
          if ($added > 0);
    }
}

=head2
    perl -I./lib -MCron -e 'Cron->new->do' user_notifications cleanup_expired
=cut

sub cleanup_expired : CRON('31 * * * *') : LOCK {
    my ($self) = @_;

    my $app = $self->app;

    my $multistate_list = $app->notification->get_multistates_by_filter('not deleted');

    my $notification_list = $app->partner_db->query->select(
        table  => $app->partner_db->notification,
        filter => [
            'AND',
            [
                ['multistate', 'IN', \$multistate_list],
                ['ttl',        '>',  \0],
                [
                    'create_date',
                    '<=',
                    {
                        'from_unixtime' =>
                          [['-' => [{'unix_timestamp' => [{'now' => []}]}, ['*' => ['ttl', \$SEC_IN_DAY]]]]]
                    }
                ],
            ]
        ],
        fields => ['id']
    )->get_all;

    my $notification_id_list = [map {$_->{id}} @$notification_list];
    INFOF(
        '[%d] expired notifications found; ids=[%s]',
        scalar @$notification_id_list,
        join(',', @$notification_id_list)
    );

    if (@$notification_id_list) {
        my $count = 0;
        for my $id (@$notification_id_list) {
            try {
                $self->notification->do_action($id, 'delete', message => 'expired');
                $count++;
            };
        }
        INFOF('[%d] expired notification deleted', $count);
    }
}

sub _get_query_user_notifications_2delete {
    my ($app, $notification) = @_;

    my $login_list = $notification->{logins} // [];

    my $unread_multistate_list = $app->user_notifications->get_multistates_by_filter('not deleted_by_user');

    return $app->partner_db->query->select(
        alias  => 'unfs',
        table  => $app->partner_db()->user_notifications,
        fields => ['id', 'user_id', 'notification_id'],
        filter => [
            AND => [
                [{'notification_id' => 'unfs'} => '='  => \$notification->{id}],
                [{'multistate'      => 'unfs'} => 'IN' => \$unread_multistate_list],
                [{'uids_id'         => 'uids'} => '='  => \undef],
            ]
        ]
      )->left_join(
        alias => 'uids',
        table => $app->partner_db->query->select(
            table  => $app->partner_db()->users,
            fields => {'uids_id' => 'id'},
            filter => [login => 'IN' => \$login_list],
        ),
        join_on => [{'uids_id' => 'uids'} => '=' => {'user_id' => 'unfs'}],
      );
}

sub _get_query_user_notifications_2add_by_roles {
    my ($app, $notification) = @_;

    my $roles_list = $notification->{roles} // [];

    return $app->partner_db->query->select(
        alias  => 'unfs',
        table  => $app->partner_db()->user_notifications,
        fields => [],
        filter => [{'user_id' => 'unfs'} => '=' => \undef],
      )->right_join(
        alias   => 'uids',
        table   => $app->partner_db()->user_role,
        fields  => ['user_id'],
        filter  => [role_id => 'IN' => \$roles_list],
        join_on => [
            AND => [
                [{'notification_id' => 'unfs'} => '=' => \$notification->{id}],
                [{'user_id'         => 'uids'} => '=' => {'user_id' => 'unfs'}],
            ]
        ],
      )->distinct;
}

sub _get_query_user_notifications_2add {
    my ($app, $notification) = @_;

    my $login_list = $notification->{logins} // [];

    return $app->partner_db->query->select(
        alias  => 'unfs',
        table  => $app->partner_db()->user_notifications,
        fields => ['id', 'user_id', 'notification_id'],
        filter => [{'user_id' => 'unfs'} => '=' => \undef],
      )->right_join(
        alias => 'uids',
        table => $app->partner_db->query->select(
            table  => $app->partner_db()->users,
            fields => {'uids_id' => 'id'},
            filter => [login => 'IN' => \$login_list],
        ),
        join_on => [
            AND => [
                [{'notification_id' => 'unfs'} => '=' => \$notification->{id}],
                [{'uids_id'         => 'uids'} => '=' => {'user_id' => 'unfs'}],
            ]
        ],
      );
}

sub cleanup_expired_auto : CRON('40 * * * *') : LOCK {
    my ($self) = @_;

    my $app = $self->app;

    my $ms_list = $app->user_notifications->get_multistates_by_filter('viewed');
    my @auto = map {$_->{id}} @{
        $app->notification->get_all(
            filter => [AND => [{'type' => 'auto'}, ['short_name' => 'NOT IN' => $NOTIFICATION_CLEANUP_EXCLUDE_LIST],]],
            fields => ['id'],
        )
      };
    my $date = date_sub(curdate(), day => 60, oformat => "db_time");
    my $filter = $app->partner_db->filter(
        [
            'AND',
            [
                ['create_date',     '<=', \$date],     # старше 2х месяцев
                ['notification_id', 'IN', \\@auto],    # только auto уведомления
            ]
        ]
    );

    my $list = $app->partner_db->query->select(
        table  => $app->partner_db->user_notifications,
        filter => $filter,
        fields => {
            'notification_id' => '',
            'cnt'             => {'count' => ['id']},
        },
      )->right_join(
        table  => $app->partner_db->notification,
        fields => ['short_name'],
        join_on =>
          [{notification_id => $app->partner_db->user_notifications} => '=' => {id => $app->partner_db->notification}]
      )->group_by('notification_id', 'short_name')->get_all;

    for my $row (@$list) {
        Utils::MonitoringUtils::send_to_graphite(
            interval => 'five_min',
            path     => 'user_notification.delete.' . $row->{short_name},
            value    => $row->{cnt},
            solomon  => {
                notification_name => $row->{short_name},
                sensor            => 'user_notification.delete',
            }
        );
    }

    my $count = $app->partner_db->user_notifications->delete($filter);

    INFOF('[%d] expired auto notifications deleted', $count);
}

sub monitor_create : CRON('*/5 * * * *') : LOCK {
    my ($self) = @_;

    my $app = $self->app;
    my $cur_date = round_timestamp(curdate(oformat => 'sec'));

    my $query = $app->partner_db->query->select(
        table  => $app->partner_db->user_notifications,
        filter => [
            AND => [
                ['create_date', '>=', \trdate('sec', 'db_time', $cur_date - (5 * 60))],
                ['create_date', '<', \trdate('sec', 'db_time', $cur_date)],
            ],
        ],
        fields => {
            'notification_id' => '',
            'cnt'             => {'count' => ['id']},
        },
      )->right_join(
        table  => $app->partner_db->notification,
        fields => ['short_name'],
        join_on =>
          [{notification_id => $app->partner_db->user_notifications} => '=' => {id => $app->partner_db->notification}]
      )->group_by('notification_id', 'short_name');

    for my $row (@{$query->get_all}) {
        next unless $row->{notification_id};
        Utils::MonitoringUtils::send_to_graphite(
            interval  => 'five_min',
            path      => 'user_notification.add.' . $row->{short_name},
            value     => $row->{cnt},
            timestamp => $cur_date,
            solomon   => {
                notification_name => $row->{short_name},
                sensor            => 'user_notification.add',
            }
        );
    }
}

sub monitor_view : CRON('*/5 * * * *') : LOCK {
    my ($self) = @_;

    my $app      = $self->app;
    my $cur_date = round_timestamp(curdate(oformat => 'sec'));
    my $ms_list  = $app->user_notifications->get_multistates_by_filter('not viewed');

    my $query = $app->partner_db->query->select(
        table  => $app->partner_db->user_notifications,
        filter => ['multistate', 'IN', \$ms_list],
        fields => {
            'notification_id' => '',
            'cnt'             => {'count' => ['id']},
        },
      )->right_join(
        table  => $app->partner_db->notification,
        fields => ['short_name'],
        join_on =>
          [{notification_id => $app->partner_db->user_notifications} => '=' => {id => $app->partner_db->notification}]
      )->group_by('notification_id', 'short_name');

    for my $row (@{$query->get_all}) {
        next unless $row->{notification_id};
        Utils::MonitoringUtils::send_to_graphite(
            interval  => 'five_min',
            path      => 'user_notification.view.' . $row->{short_name},
            value     => $row->{cnt},
            timestamp => $cur_date,
            solomon   => {
                notification_name => $row->{short_name},
                sensor            => 'user_notification.view',
            }
        );
    }
}

TRUE;
