package Curses::UI::Dialog::YandexTankerMessage;

use qbit;

use Curses;
use Curses::UI::Common;
use Curses::UI::Window;

use base qw(Curses::UI::Window Curses::UI::Common);

my %WIDGET_OPTIONS = (
    -x          => 0,
    -width      => -1,
    -border     => TRUE,
    -vscrollbar => 'right',
    -hscrollbar => FALSE,
    -wrapping   => TRUE,
    -singleline => FALSE,
);

sub new {
    my ($class, %args) = @_;

    my $self = $class->SUPER::new(%args, -border => FALSE);

    $self->set_binding(sub {exit 0}, "\cQ", "\cD");

    my $widgets_layout = $self->_get_widgets_layout();

    $self->add(
        '_ProgressBar', 'Progressbar',
        %WIDGET_OPTIONS,
        %{$widgets_layout->{'_ProgressBar'}},
        -max => $self->{'-total'},
        -pos => $self->{'-current'},
    ) if exists($widgets_layout->{'_ProgressBar'});

    $self->add(
        '_ContextView', 'TextViewer',
        %WIDGET_OPTIONS,
        %{$widgets_layout->{'_ContextView'}},
        -focusable => FALSE,
        -text      => $args{'-message_context'} // '',
        -title     => 'Context',
        -bg        => 'white',
        -fg        => 'black',
    ) if $widgets_layout->{'_ContextView'};

    $self->add(
        '_MessageView', 'TextViewer',
        %WIDGET_OPTIONS,
        %{$widgets_layout->{'_MessageView'}},
        -focusable => FALSE,
        -text      => $args{'-message'} // '',
        -title     => (!$widgets_layout->{'_PluralMessageView'} ? 'Key' : 'Key for plural form #1'),
        -bg        => 'white',
        -fg        => 'black',
    );

    $self->add(
        '_PluralMessageView', 'TextViewer',
        %WIDGET_OPTIONS,
        %{$widgets_layout->{'_PluralMessageView'}},
        -focusable => FALSE,
        -text      => $args{'-message_plural'} // '',
        -title     => 'Key for plural form #1, #5',
        -bg        => 'white',
        -fg        => 'black',
    ) if $widgets_layout->{'_PluralMessageView'};

    $self->add(
        '_CopyInputView', 'TextViewer',
        %WIDGET_OPTIONS,
        %{$widgets_layout->{'_CopyInputView'}},
        -focusable => FALSE,
        -text      => $args{'-translate1'} // '',
        -title     => 'Input copy',
    );

    my $count         = 0;
    my $on_change_sub = sub {
        my ($this) = @_;
        my $text = $this->parent->getfocusobj()->text();
        print STDERR "\r│$text", ' ' x 10;
        $count++;
        my $sep = $count % 2 ? '++' : '--';
        $self->getobj('_CopyInputView')->text("$sep$text$sep" . " " x 10);
    };

    # singular: просто перевод
    # plural: form #1 - хранит форму слова/фразы для числа 1 и других чисел, заканчивающихся на 1 (кроме 11);
    $self->add(
        '_RussianTextEditor1',
        'TextEditor',
        %WIDGET_OPTIONS,
        %{$widgets_layout->{'_RussianTextEditor1'}},
        -title => (
            $self->{'-message_plural'}
            ? 'Russian for plural form #1: 1 and other numbers ended by 1 (except 1)'
            : 'Russian'
        ),
        -text => $args{'-translate1'} // '',
        -onChange => $on_change_sub,
    );
    $self->{'__text_editor_bfg'} = $self->getobj('_RussianTextEditor1')->{'-bfg'};
    $self->getobj('_RussianTextEditor1')->focus();

    # plural: form #2 - хранит форму слова/фразы для чисел 2, 3, 4, а также для других чисел, которые заканчиваются на 2, 3, 4 (кроме 12, 13, 14);
    if ($widgets_layout->{'_RussianTextEditor2'}) {
        $self->add(
            '_RussianTextEditor2', 'TextEditor',
            %WIDGET_OPTIONS,
            %{$widgets_layout->{'_RussianTextEditor2'}},
            -title    => 'Russian for plural form #2: 2, 3, 4 and other numbers ended by 2, 3, 4 (except 12, 13, 14)',
            -text     => $args{'-translate2'} // '',
            -onChange => $on_change_sub,
        );
    }

    # plural: form #5 - хранит форму слова/фразы для всех остальных случаев;
    if ($widgets_layout->{'_RussianTextEditor5'}) {
        $self->add(
            '_RussianTextEditor5', 'TextEditor',
            %WIDGET_OPTIONS,
            %{$widgets_layout->{'_RussianTextEditor5'}},
            -title    => 'Russian for plural form #5 (all other cases)',
            -text     => $args{'-translate5'} // '',
            -onChange => $on_change_sub,
        );
    }

    $self->{'_Buttons'} = $self->add(
        '_Buttons',
        'Buttonbox',
        %{$widgets_layout->{'_Buttons'}},
        -buttons => [
            {
                -label => ' Save ',
                -value => 1
            },
            {
                -label => ' Skip ',
                -value => 0
            }
        ],
        -buttonalignment => 'middle',
    );

    $self->{'_Buttons'}->set_routine(
        'press-button' => sub {
            my ($this) = @_;

            my $button_value = $this->get();

            my $err_editor;
            if ($button_value == 1) {
                foreach my $widget_id (qw(_RussianTextEditor1 _RussianTextEditor2 _RussianTextEditor5)) {
                    my $editor = $self->getobj($widget_id);
                    next unless $editor;

                    my $text = $editor->text();

                    my $is_error = 0;
                    if (!$text) {
                        $is_error = 1;
                        $self->parent->dialog(
                            -title   => "Error",
                            -buttons => ['ok'],
                            -message => "Empty message not allowed"
                        );
                    } elsif ($text !~ /\S/) {
                        $is_error = 1;
                        $self->parent->dialog(
                            -title   => "Error",
                            -buttons => ['ok'],
                            -message => "Message cannot contain only whitespases"
                        );
                    }

                    if ($is_error) {
                        $err_editor = $editor;
                        $editor->focus();
                        $editor->set_color_bfg('red');
                    } else {
                        $editor->set_color_bfg($self->{'__text_editor_bfg'});
                    }
                }
                $err_editor->focus() if $err_editor;
            }

            unless ($err_editor) {
                $this->parent->loose_focus();
            }
        }
    );

    return $self;
}

sub layout() {
    my ($self) = @_;

    my $widgets_layout = $self->_get_widgets_layout();
    foreach my $widget (keys(%$widgets_layout)) {
        my $widget_obj = $self->getobj($widget);
        $widget_obj->{$_} = $widgets_layout->{$widget}{$_} foreach keys(%{$widgets_layout->{$widget}});
    }

    $self->SUPER::layout();

    return $self;
}

sub get() {
    my ($self) = @_;

    my $v = $self->{'_Buttons'}->get() || return undef;

    return [
        map  {$self->getobj($_)->text()}
        grep {$self->getobj($_)} qw(_RussianTextEditor1 _RussianTextEditor2 _RussianTextEditor5)
    ];
}

sub _get_widgets_layout() {
    my ($self) = @_;

    my $next_y = 0;
    my %res;

    if (defined($self->{'-total'}) && defined($self->{'-current'})) {
        $res{'_ProgressBar'} = {
            -height => 3,
            -y      => $next_y,
        };
        $next_y += 3;
    }

    if (defined($self->{'-message_context'})) {
        # ToDo: Линии могут быть длинные, тогда надо рубить по ширине TextView
        my $height = grep {/\S/} split(/\n/, $self->{'-message_context'});
        $height = 3 if $height > 3;
        $res{'_ContextView'} = {
            -height => $height + 2,
            -y      => $next_y,
        };

        $next_y += $res{'_ContextView'}->{'-height'};
    }

    my @widgets = (
        (
            defined($self->{'-message_plural'})
            ? qw(_MessageView _PluralMessageView _RussianTextEditor1 _RussianTextEditor2 _RussianTextEditor5)
            : qw(_MessageView _RussianTextEditor1)
        ),
        qw(_CopyInputView)
    );

    my $copy_widjet_height = 4;

    my $message_height = (($ENV{'LINES'} - 1 - $next_y - $copy_widjet_height) / (@widgets - 1));

    foreach my $widget_id (@widgets) {
        my $cur_message_height = $widget_id eq '_CopyInputView' ? $copy_widjet_height : $message_height;
        $res{$widget_id} = {
            -height => $cur_message_height,
            -y      => $next_y,
        };
        print STDERR $widget_id, ' h=', $cur_message_height, "\n";
        $next_y += $cur_message_height;
    }

    $res{'_Buttons'} = {-y => $next_y};

    return \%res;
}

TRUE;
