package Exception;

use strict;
use warnings;
use overload '""' => sub {shift->as_string()};

use Carp qw();
use Scalar::Util qw(blessed);
use List::Util qw(max);

sub as_string {
    my ($self) = @_;

    return
        ref($self)
      . ": $self->{'text'}\n"
      . "    Package: $self->{'package'}\n"
      . "    Filename: $self->{'filename'} (line $self->{'line'})\n"
      . "    CallStack:\n"
      . '        '
      . join("\n        ",
        map {$_->{'subroutine'} . "() called at '$_->{'filename'}' line $_->{'line'}"} @{$self->{'callstack'}})
      . "\n"
      . ($self->{'parent'} ? "\n$self->{'parent'}\n" : '');
}

sub catch {
    return \@_;
}

sub message {
    return shift->{'text'};
}

sub new {
    my ($this, $text, %data) = @_;
    my $class = ref($this) || $this;
    my $caller_depth = delete $data{caller_depth} // 0;

    $text = '' if !defined $text;

    my @call_stack = ();
    my $i          = 0;

    while (1) {

        package DB;
        my ($package, $filename, $line, $subroutine, $hasargs, $wantarray, $evaltext, $is_require, $hints, $bitmask) =
          caller($i);

        last if !defined($package);

        push(
            @call_stack,
            {
                package    => $package,
                filename   => $filename,
                line       => $line,
                subroutine => $subroutine,
                args       => [@DB::args],
            }
          )
          unless $package eq 'QBit::Exceptions'
              || $subroutine eq 'QBit::Exceptions::try'
              || $subroutine eq 'Exception::new'
              || $package    eq 'Coro::State';

        ++$i;
    }

    my $caller;
    ($caller, @call_stack) = @call_stack[max(0, $caller_depth - 1) .. $#call_stack];
    my $self = {
        %data,
        (
            blessed($text) && $text->isa('Exception')
            ? (text => $text->{'text'}, parent => $text)
            : (text => $text)
        ),
        filename  => $caller->{'filename'},
        package   => $caller->{'package'},
        line      => $caller->{'line'},
        callstack => \@call_stack,
    };

    bless $self, $class;
    return $self;
}

sub throw {
    my $exception = shift->new(@_);
    if ($ENV{'DEBUG_EXCEPTIONS'}) {
        my $message = $exception->can('as_string') ? $exception->as_string() : $_[0];
        if ($ENV{'FORCE_LOGGER_TO_SCREEN'}) {
            require Utils::Logger;
            Utils::Logger::ERROR($message);
        } else {
            Carp::cluck($message);
        }
    }
    QBit::Exceptions::throw($exception);
}

1;
