package Exception::API::HTTP;

use strict;
use warnings FATAL => 'all';
use utf8;

use base qw(Exception::API);

use Scalar::Util qw(blessed);

our $LIMIT_REQUEST          = 1_000;
our $LIMIT_RESPONSE         = 1_000;
our $LIMIT_RESPONSE_HEADERS = 1_000;

sub new {
    my ($class, $response, $msg, $extra_msg, %data) = @_;

    my $self;
    if (blessed($response) && $response->isa('Exception::API::HTTP')) {
        $self = $class->SUPER::new($response->{text}, %data);
        $self->{response} = $response->{response};
    } elsif (ref($response) eq 'HTTP::Response') {
        $self = $class->SUPER::new($response->status_line, %data);
        $self->{response} = $response;
    } else {
        $self = $class->SUPER::new($response, %data);
    }

    $self->{'text'} = $msg if defined $msg;
    $self->{'extra_msg'} = "\n" . $extra_msg if defined $extra_msg;
    return $self;
}

sub message {
    my ($self) = @_;

    my $text = $self->SUPER::message();

    if ($self->{response}) {
        $text .=
            $self->_get_extra_message() . "\n\n"
          . 'Request: '
          . $self->_get_request() . "\n"
          . 'Response: '
          . $self->_get_response() . "\n"
          . 'Response headers: '
          . $self->_get_response_headers();
    }

    return $text;
}

sub _get_extra_message {
    my ($self) = @_;

    return $self->{extra_msg} // '';
}

sub _get_request {
    my ($self) = @_;
    my $request = substr($self->{response}->request->as_string, 0, $LIMIT_REQUEST);
    $request =~ s|(Authorization: \S*) .*|$1 XXXX|;
    $request =~ s|\n|\n   |g;
    return $request;
}

sub _get_response {
    my ($self) = @_;
    my $response = substr(eval {$self->{response}->decoded_content()} // '', 0, $LIMIT_RESPONSE);
    $response =~ s|\n|\n   |g;
    utf8::decode($response) unless utf8::is_utf8($response);
    return $response;
}

sub _get_response_headers {
    my ($self) = @_;
    my $headers = substr($self->{response}->headers->as_string, 0, $LIMIT_RESPONSE_HEADERS);
    $headers =~ s|\n|\n   |g;
    return $headers;
}

1;
