package IntAPI::Method::AdFox;

use qbit;
use QBit::Validator;

use base qw(IntAPI::Method);

use Exception::Multistate::BadAction;
use Exception::Multistate::NotFound;
use Exception::Validation::BadArguments;

__PACKAGE__->model_accessors(
    users                             => 'Application::Model::Users',
    partner_db                        => 'Application::Model::PartnerDB',
    context_on_site_campaign          => 'Application::Model::Product::AN::ContextOnSite::Campaign',
    context_on_site_content           => 'Application::Model::Product::AN::ContextOnSite::Content',
    context_on_site_natural           => 'Application::Model::Product::AN::ContextOnSite::Natural',
    context_on_site_rtb               => 'Application::Model::Product::AN::ContextOnSite::RTB',
    internal_context_on_site_campaign => 'Application::Model::Product::InternalAN::InternalContextOnSite::Campaign',
    internal_context_on_site_content  => 'Application::Model::Product::InternalAN::InternalContextOnSite::Content',
    internal_context_on_site_natural  => 'Application::Model::Product::InternalAN::InternalContextOnSite::Natural',
    internal_context_on_site_rtb      => 'Application::Model::Product::InternalAN::InternalContextOnSite::RTB',
    mobile_app_settings               => 'Application::Model::Product::AN::MobileApp::Settings',
    mobile_app_rtb                    => 'Application::Model::Product::AN::MobileApp::RTB',
    product_manager                   => 'Application::Model::ProductManager',
    video_an_site                     => 'Application::Model::Page::Video',
    video_an_site_instream            => 'Application::Model::Product::VideoAN::Site::InStream',
    video_an_site_inpage              => 'Application::Model::Product::VideoAN::Site::InPage',
    video_an_site_fullscreen          => 'Application::Model::Product::VideoAN::Site::Fullscreen',

    statistics => 'Application::Model::Statistics',

    api_blackbox => 'QBit::Application::Model::API::Yandex::BlackBox',
);

sub link_user : METHOD : PARAMS(user_id, adfox_id) : TITLE('Link adfox user with pi user') : FORMATS(adfox) {
    my ($self, %params) = @_;
    my $app = $self->app;

    my $p_user_id  = $params{user_id};
    my $p_adfox_id = $params{adfox_id};

    throw Exception::Validation::BadArguments gettext("adfox_id is not numeric: %s", $p_adfox_id)
      unless $p_adfox_id =~ /^[0-9]+$/;
    throw Exception::Validation::BadArguments gettext("user_id is not numeric: %s", $p_user_id)
      unless $p_user_id =~ /^[0-9]+$/;

    my $tmp_rights = $app->add_tmp_rights(
        qw(
          users_view_all
          do_user_action_link_adfox_user
          )
    );

    unless ($app->users->get($p_user_id, fields => [qw(id)])) {
        throw Exception::Validation::BadArguments gettext('Unknown user_id');
    }

    try {
        $self->users->do_action($p_user_id, 'link_adfox_user', adfox_id => $p_adfox_id);
    }
    catch {
        my $msg = shift->message;
        throw Exception::Validation::BadArguments 'Can\'t link - ' . $msg
          unless $msg eq gettext('AdFox user is already linked');
    };

    return {user_id => $p_user_id, adfox_id => $p_adfox_id};
}

sub unlink_user : METHOD : PARAMS(user_id, adfox_id) : TITLE('Unlink adfox user with pi user') : FORMATS(adfox) {
    my ($self, %params) = @_;

    my $app = $self->app;

    for (qw(user_id adfox_id)) {
        throw Exception::Validation::BadArguments gettext("%s is not numeric: %s", $_, $params{$_})
          unless ($params{$_} && $params{$_} =~ /^[0-9]+$/);
    }

    my $p_user_id  = $params{user_id};
    my $p_adfox_id = $params{adfox_id};

    my $tmp_rights = $app->add_tmp_rights(
        qw(
          users_view_all
          do_user_action_unlink_adfox_user
          )
    );

    my $user = $app->users->get($p_user_id, fields => [qw(id adfox_info)]);

    throw Exception::Validation::BadArguments gettext('Unknown user_id')
      unless ($user);
    throw Exception::Validation::BadArguments gettext('Unknown adfox_id')
      unless 0 < grep {$p_adfox_id eq $_->{id}} @{$user->{adfox_info}};

    $self->users->do_action($p_user_id, 'unlink_adfox_user', adfox_id => $p_adfox_id);

    return TRUE;
}

sub delete_block : METHOD : PARAMS(!block_id) : TITLE('Delete ADFOX block') : FORMATS(json, xml, adfox) {
    my ($self, %params) = @_;

    my $tmp_rights = $self->app->add_all_tmp_rights;
    #my $tmp_rights = $self->app->add_tmp_rights(qw(
    #    users_view_all
    #
    #    mobile_app_rtb_view_all
    #    mobile_app_rtb_edit
    #
    #    video_an_site_instream_view_all
    #    video_an_site_instream_edit
    #    do_video_an_site_instream_delete_from_adfox
    #
    #    video_an_site_inpage_view_all
    #    video_an_site_inpage_edit
    #    do_video_an_site_inpage_delete_from_adfox
    #    ));

    my $public_id = $params{'block_id'};
    my ($page_id, $block_id) = $self->_relaxed_public_id_split($public_id);

    my ($found, $bad_action);

    foreach my $model (@{$self->product_manager->get_adfox_blocks_model_accessors()}) {
        my $page_id_name = $self->$model->get_page_id_field_name;
        try {
            # Let do_action read adfox_block field from the DB together with PK in one go
            # This DOES NOT set adfox_block to true because it is re-read from the DB
            $self->$model->do_action(
                {
                    $page_id_name => $page_id,
                    id            => $block_id,
                    adfox_block   => 1,
                },
                'delete_from_adfox'
            );
            $found = 1;
        }
        catch Exception::Multistate::NotFound with {
            # no op
          } catch Exception::Multistate::BadAction with {
            $bad_action = 1;
          };
        last if $found || $bad_action;
    }

    return {result => 'error', error_type => 'Bad Action', message => gettext('Cannot delete this block')}
      if $bad_action;
    return {result => 'error', error_type => 'Not Found', message => gettext('Block not found')} unless $found;
    return {result => 'ok'};
}

=head1 get_blocks

Возвращает данные по всем указанным блокам (блоки могут быть в любом
статусе).

Параметры:

    * block_id - публичный id блока (обязательный)

Пример:

    $ curl https://host/intapi/adfox/get_blocks.json?lang=ru&block_id=R-A-48398-4&block_id=R-A-48398-5

    {
        "data" : [
            {
                "public_id" : "R-A-48398-4",
                "text_blocked" : null,
                "strategy" : "1",
                "has_articles_settings" : "",
                "text_active" : null,
                "mincpm" : null,
                "percent_traffic" : null,
                "media_active" : null,
                "strategy_name" : "Maximum income",
                "media_blocked" : null,
                "has_brands_settings" : "",
                "media_cpm" : null,
                "has_geo_settings" : "",
                "text_cpm" : null
            },
            {
                "public_id" : "R-A-48398-5",
                "text_blocked" : null,
                "strategy" : "1",
                "has_articles_settings" : "",
                "text_active" : null,
                "mincpm" : null,
                "percent_traffic" : null,
                "media_active" : null,
                "strategy_name" : "Maximum income",
                "media_blocked" : null,
                "has_brands_settings" : "",
                "media_cpm" : null,
                "has_geo_settings" : "",
                "text_cpm" : null
            }
        ],
        "result" : "ok"
    }

=cut

sub get_blocks : METHOD : PARAMS(!block_id[]) : TITLE('Adfox block info') : FORMATS(json, xml, adfox) {
    my ($self, %params) = @_;

    my $block_ids = $self->_split_values($params{'block_id'});

    my $tmp_rights = $self->app->add_tmp_rights(qw(context_on_site_rtb_view_all));

    my @adfox_blocks = ();
    for my $model (qw(context_on_site_rtb)) {
        push @adfox_blocks, @{
            $self->$model->get_all(
                filter => {id => $block_ids, adfox_block => 1},
                fields => [
                    qw(
                      public_id
                      strategy
                      strategy_name
                      mincpm
                      media_active
                      media_blocked
                      media_cpm
                      text_active
                      text_blocked
                      text_cpm
                      has_articles_settings
                      has_brands_settings
                      has_geo_settings
                      )
                ],
            );
          };
    }

    return \@adfox_blocks;
}

sub get_deleted_blocks : METHOD : TITLE('Get deleted ADFOX blocks') : FORMATS(json, xml, adfox) {
    my ($self) = @_;

    my @data = ();
    for my $model (@{$self->product_manager->get_adfox_blocks_model_accessors()}) {
        my $blocks = $self->$model->partner_db_table()->get_all(
            fields => [$self->$model->get_page_id_field_name(), 'id'],
            filter => [
                'AND',
                [
                    ['adfox_block', '=',  \1],
                    ['multistate',  'IN', \$self->$model->get_multistates_by_filter('deleted')],
                    ($self->$model->is_block_table_with_multiple_models() ? ['model', '=', \$model] : ()),
                ]
            ],
        );
        push @data, map {$self->$model->public_id($_)} @$blocks;
    }

    return {
        result => 'ok',
        data   => \@data,
    };
}

=encoding UTF-8

=head1 get_pages

Список площадок для создания блока

url path: /adfox/get_pages, схема https, авторизация - нет (ALC by IP)
Формат: json
Параметры:

    adfox_id - некий ид. партнера, по которому прошла связка аккаунтов,

Параметры можно повторять в одном запросе, а также указывать через запятую.

Фильтрация: принадлежность площадки к adfox_id, статус (работают или тестируются).

Данные:

Плоский список площадок, сортировки нет (массив pages).

    login - строка, логин партнера на стороне ПИ
    page_id - число, id пейджа на стороно ПИ
    domain - строка, домен пейджа
    header - строка, кратко описывает площадку

Плюс плоский список логинов (массив users)

    login - логин партнера в ПИ

Запрос:

    $ GET 'https://host:port/adfox/get_pages.adfox?pretty=1&lang=ru&adfox_id=12345&adfox_id=67890'

или

    $ GET 'http://host:port/adfox/get_pages.adfox?pretty=1&lang=ru&adfox_id=12345,67890'

Ответ:

    {
       "users" : [
          {"login" : "xxxxx"},
          {"login" : "yyyyy"}
       ],
       "pages" : [
          {
             "domain" : "domain.ru",
             "page_id" : "1111",
             "header" : "Реклама на domain.ru, работает",
             "login" : "xxxxxx",
             "ProductType" : "context",
             "allowed_turbo" : 1,
             "allowed_amp" : 1
          }
       ]
    }

=cut

sub get_pages : METHOD : PARAMS(!adfox_id[]) : TITLE('Pages available for block creation') : FORMATS(json, xml, adfox) {
    my ($self, %params) = @_;

    my $adfox_ids = $self->_split_values($params{'adfox_id'});

    foreach (@$adfox_ids) {
        throw Exception::Validation::BadArguments gettext("adfox_id is not numeric: %s", $_) unless /^[0-9]+$/;
    }

    my $tmp_rights = $self->app->add_all_tmp_rights();

    my $user_adfox = $self->partner_db->query->select(
        table  => $self->partner_db->user_adfox,
        fields => [qw(adfox_id create_date)],
        filter => {adfox_id => $adfox_ids},
      )->join(
        table   => $self->partner_db->users,
        fields  => [qw(id login)],
        join_on => [id => '=' => {user_id => $self->partner_db->user_adfox}],
      )->get_all();

    return {users => [], pages => []} unless @$user_adfox;

    my %users;
    foreach (@$user_adfox) {
        $users{$_->{'id'}} //= {login => $_->{'login'}};

        push(@{$users{$_->{'id'}}{'links'}}, {adfox_id => $_->{'adfox_id'}, create_date => $_->{'create_date'}});
    }

    my ($multistate, $multi_names);

    my $context_pages = $self->context_on_site_campaign->get_all(
        fields => [qw(page_id domain caption login multistate allowed_turbo allowed_amp)],
        filter => {owner_id => [sort {$a <=> $b} keys(%users)], multistate => 'working or testing'},
    );

    foreach my $p (@$context_pages) {
        $p->{'ProductType'} = 'context';

        $multistate  = delete($p->{'multistate'});
        $multi_names = join(', ',
            $self->context_on_site_campaign->get_multistate_name_as_list($multistate, filter => ['working', 'testing'])
        );
        $p->{'header'} = delete($p->{'caption'}) . ', ' . $multi_names;
    }

    my $mobile_pages = $self->mobile_app_settings->get_all(
        fields => [qw(context_page_id store_id caption login multistate type_name store)],
        filter => {owner_id => [sort {$a <=> $b} keys(%users)], multistate => 'working or testing'},
    );

    foreach my $p (@$mobile_pages) {
        $p->{'ProductType'} = 'mobile';

        $p->{'page_id'} = delete($p->{'context_page_id'});
        $p->{'domain'}  = $p->{'store_id'};

        $multistate  = delete($p->{'multistate'});
        $multi_names = join(', ',
            $self->mobile_app_settings->get_multistate_name_as_list($multistate, filter => ['working', 'testing']));

        $p->{'header'} = delete($p->{'caption'}) . ', ' . $multi_names;

        $p->{'allowed_turbo'} = 0;
        $p->{'allowed_amp'}   = 0;
    }

    my $video_pages = $self->video_an_site->get_all(
        fields => [qw(page_id domain caption login multistate)],
        filter => {owner_id => [sort {$a <=> $b} keys(%users)], multistate => 'working or testing'},
    );

    foreach my $p (@$video_pages) {
        $p->{'ProductType'} = 'video';

        $multistate = delete($p->{'multistate'});
        $multi_names =
          join(', ', $self->video_an_site->get_multistate_name_as_list($multistate, filter => ['working', 'testing']));

        $p->{'header'} = delete($p->{'caption'}) . ', ' . $multi_names;

        $p->{'allowed_turbo'} = 0;
        $p->{'allowed_amp'}   = 0;
    }

    return {
        users => [values(%users)],
        pages => ([@$context_pages, @$mobile_pages, @$video_pages])
    };
}

#
# Allows whatever-<page_id>-<block_id> or just <page_id>-<block_id>
#
# bla-bla-bla-12345-6 - OK
# 12345-6             - OK
# -12345-6            - NOT OK
#
sub _relaxed_public_id_split {
    my ($self, $public_id) = @_;

    throw Exception::Validation::BadArguments gettext("Wrong format of block_id: %s", $public_id)
      unless $public_id =~ /(?:^|^.+-)([0-9]+)-([0-9]+)$/;

    return ($1, $2);
}

#
# For URL parameter specified as &var=a,b,c
#
sub _split_values {
    my ($self, $values) = @_;

    if (@$values == 1 && $values->[0] =~ /,/) {
        @$values = split(',', $values->[0]);
    }

    return $values;
}

TRUE;
