package IntAPI::Method::AntiFrod;

use qbit;

use base qw(IntAPI::Method);

=head1 page_owner_report

Возвращает список всех пейджей, логинов и id их владельцев.

==URL
%%http://partner2.yandex.ru/intapi/antifrod/page_owner_report%%

==Параметры
%%
<без параметров>
%%

==Пример
%%
$ curl -s https://partner2.yandex.ru/intapi/antifrod/page_owner_report.tsv
page_id    login    owner_id    assistants
41443    adv-vertis    356502796    20190389,23318590,45012113,199732229,310557976,378025741,440495993
134783    avitortb2013    198392540
200688    rubiconproject2015    345321935
155561    smaato    345331106
...
247590    interfax-aki    39137370
247756    tvigle-adv    309844779
247796    orionsites    423527495
247802    adinside    54519301
#END
%%

==Тело
Формат
  * tsv, json, xml
  * первая строка - заголовок "page_id login owner_id assistants" (только для tsv)
  * маркер конца - "#END" (только для tsv)

Поля
  * <page_id> - число, идентификатор площадки
  * <login> - строка, логин владельца площадки
  * <owner_id> - число, идентификатор владельца площадки
  * <assistants> - ничего, либо разделенные запятыми массив идентификаторов ассистентов

Данные
  * Выбираются все площадки с page_id

==Wiki
  https://wiki.yandex-team.ru/partner/w/partner2-intapi-antifrod-page-owner-report

==Тикеты

# https://st.yandex-team.ru/PI-10043
# https://st.yandex-team.ru/PI-10366
=cut

sub page_owner_report : METHOD : TITLE('Antifrod page owner report') : FORMATS(json, xml, tsv) {
    my ($self) = @_;

    my $tmp_rights = $self->app->add_all_tmp_rights();

    my $except_page_accessors = $self->app->product_manager->get_special_page_model_accessors();

    my $data = $self->app->all_pages->get_all(
        fields => [qw(page_id login owner_id)],
        filter => [AND => [{is_balance_registered => TRUE}, ['model', 'NOT IN', $except_page_accessors]]]
    );

    my %page_assistants = map {$_->{'page_id'} => []} @$data;

    my $assistants = $self->app->assistants->get_all(
        fields => [qw(page_id user_id)],
        filter => {'page_id' => [sort {$a <=> $b} keys %page_assistants]},
    );

    push @{$page_assistants{$_->{'page_id'}}}, $_->{'user_id'} for @$assistants;

    for my $row (@$data) {
        $row->{'assistants'} = join(',', @{$page_assistants{$row->{'page_id'}}}) || undef;
    }

    return {
        data   => $data,
        fields => ['page_id', 'login', 'owner_id', 'assistants'],
        options => {with_headers => 1},
    };
}

1;
