package IntAPI::Method::BK;

use qbit;

use base qw(IntAPI::Method);
use Exception::Validation::BadArguments;

__PACKAGE__->model_accessors(
    partner_genre   => 'Application::Model::Product::VideoAN::Site::Video::PartnerGenre',
    partner_content => 'Application::Model::Product::VideoAN::Site::Video::PartnerContent',
);

sub get_contents : METHOD : PARAMS(!fromdate, offset, limit) : TITLE('Contents list') : FORMATS(json, xml, tsv) {
    my ($self, %opts) = @_;

    throw gettext('option "fromdate" must consist of 14 digits') unless $opts{'fromdate'} =~ /^[0-9]{14}$/;

    foreach (qw(limit offset)) {
        throw gettext('option "%s" must consist of digits', $_) if defined($opts{$_}) && $opts{$_} !~ /^[0-9]+$/;
    }

    my $date = sprintf("%d%d%d%d-%d%d-%d%d %d%d:%d%d:%d%d", split('', $opts{'fromdate'}));

    my $tmp_rights = $self->app->add_tmp_rights('partner_content_view_all', 'internal_content_view_all');

    my $data = $self->partner_content->get_all(
        fields => [qw(modification_date site_id public_id internal_content)],
        filter => [modification_date => '>=' => $date],
        map {$_ => $opts{$_}} grep {defined($opts{$_})} qw(offset limit)
    );

    foreach (@$data) {
        my $internal_content = delete($_->{'internal_content'});
        if (%$internal_content) {
            $_->{'internal_id'} = $internal_content->{'id'};
            $_->{'yandex_name'} = $internal_content->{'name'};
        } else {
            $_->{'internal_id'} = '';
            $_->{'yandex_name'} = '';
        }
    }

    return {
        data    => $data,
        fields  => [qw(modification_date site_id public_id internal_id yandex_name)],
        options => {
            without_headers => TRUE,
            string_escape   => TRUE,
            end_marker      => FALSE,
        },
    };
}

sub get_genres : METHOD : PARAMS(!fromdate, offset, limit) : TITLE('Genres list') : FORMATS(json, xml, tsv) {
    my ($self, %opts) = @_;

    throw gettext('option "fromdate" must consist of 14 digits') unless $opts{'fromdate'} =~ /^[0-9]{14}$/;

    foreach (qw(limit offset)) {
        throw gettext('option "%s" must consist of digits', $_) if defined($opts{$_}) && $opts{$_} !~ /^[0-9]+$/;
    }

    my $date = sprintf("%d%d%d%d-%d%d-%d%d %d%d:%d%d:%d%d", split('', $opts{'fromdate'}));

    my $tmp_rights = $self->app->add_tmp_rights('partner_genre_view_all', 'internal_genre_view_all');

    my $data = $self->partner_genre->get_all(
        fields => [qw(modification_date site_id public_id internal_genres)],
        filter => [modification_date => '>=' => $date],
        map {$_ => $opts{$_}} grep {defined($opts{$_})} qw(offset limit)
    );

    my @result = ();
    foreach my $row (@$data) {
        if (@{$row->{'internal_genres'}}) {
            push(
                @result,
                map {
                    {
                        modification_date => $row->{'modification_date'},
                        site_id           => $row->{'site_id'},
                        partner_id        => $row->{'public_id'},
                        internal_id       => $_->{'public_id'}
                    }
                  } @{$row->{'internal_genres'}}
                );
        } else {
            push(
                @result,
                {
                    modification_date => $row->{'modification_date'},
                    site_id           => $row->{'site_id'},
                    partner_id        => $row->{'public_id'},
                    internal_id       => ''
                }
            );
        }
    }

    return {
        data    => \@result,
        fields  => [qw(modification_date site_id partner_id internal_id)],
        options => {
            without_headers => TRUE,
            string_escape   => TRUE,
            end_marker      => FALSE,
        },
    };
}

sub list_dsp : METHOD : PARAMS(!login) : TITLE('List DSP for login') : FORMATS(json, xml) {
    my ($self, %opts) = @_;

    my $tmp_rights = $self->app->add_all_tmp_rights;
    my $user = $self->users->get_by_login($opts{'login'}, fields => ['id']);
    throw Exception::Validation::BadArguments gettext('User with login "%s" not found', $opts{'login'}) unless $user;
    my $list = $self->app->dsp->get_all(
        fields => [qw(id short_caption url)],
        filter => {owner_id => $user->{id}},
    );

    return $list;
}

sub add_notification : METHOD : TITLE('Add user notification for login') : FORMATS(json) {
    my ($self) = @_;

    throw Exception::Validation::BadArguments 'Incorrect Content-Type'
      unless $self->app->request->http_header('content-type') =~ /application\/json/;

    my $body = $self->get_body();
    throw Exception::Validation::BadArguments 'Must specify body' unless $body;

    my $opts = eval {from_json($body)};

    throw Exception::Validation::BadArguments 'Body must be a valid json' unless $opts;

    foreach (qw(login notification)) {
        throw Exception::Validation::BadArguments "Must specify $_" unless $opts->{$_};
    }

    my $tmp_rights = $self->app->add_all_tmp_rights;
    my $user = $self->users->get_by_login($opts->{login}, fields => ['id']);
    throw Exception::Validation::BadArguments sprintf('User with login "%s" not found', $opts->{login})
      unless $user;

    my $notification =
      $self->app->notification->get_all(fields => ['id'], filter => {short_name => $opts->{notification}})->[0];
    throw Exception::Validation::BadArguments sprintf('Notification "%s" not found', $opts->{notification})
      unless $notification;

    my $notification_id = $self->app->user_notifications->add(
        "user_id"         => $user->{id},
        "notification_id" => $notification->{id},
        "custom_data"     => $opts->{custom_data}
    );

    return $notification_id;

}

sub delete_notification : METHOD : PARAMS(!id) : TITLE('Delete user notification by id') : FORMATS(json) {
    my ($self, %opts) = @_;

    my $tmp_rights = $self->app->add_all_tmp_rights;
    $tmp_rights = $self->app->add_tmp_rights('user_notifications_view_all', 'do_user_notifications_delete');
    my $notification = $self->app->user_notifications->get({id => $opts{id}});
    throw Exception::Validation::BadArguments sprintf('User notification with id "%s" not found', $opts{id})
      unless $notification;
    throw Exception::Validation::BadArguments sprintf("Can't delete user notification with id %s", $opts{id})
      unless $self->app->user_notifications->check_action({id => $opts{id}}, 'delete');
    $self->app->user_notifications->do_action({id => $opts{id}}, 'delete');
}

TRUE;
