package IntAPI::Method::Devel;

=encoding UTF-8

=cut

use qbit;

use base qw(IntAPI::Method);

use Exception::Validation::BadArguments;
use Exception::Validator::Fields;

use Utils::DB qw(fields_to_filter);

use PiConstants qw($DSP_DIRECT_ID);

__PACKAGE__->model_accessors(queue => 'Application::Model::Queue');

sub version : METHOD : TITLE('Deb package version') : FORMATS(json) {
    my ($self) = @_;

    return $self->get_option('version');
}

sub readiness_probe : METHOD : ALLOWHTTP : NOAUTH : FORMATS(json) : TITLE('check INTAPI alive') {
    my ($self) = @_;

    return $self->app->check_alive->is_alive;
}

=head2 page_ids_in_several_models

L<https://wiki.yandex-team.ru/partner/w/partner2-intapi-devel-page-ids-in-several-models/>

=cut

sub page_ids_in_several_models : METHOD : FORMATS(json) {
    my ($self) = @_;

    my $result = {};

    my @page_ids = map {$_->{page_id}} @{
        $self->app->partner_db->_get_all(
'select page_id from (select page_id, count(page_id) as c from all_pages group by page_id) as t where c > 1;'
        )
      };

    if (@page_ids) {

        my $data = $self->app->partner_db->all_pages->get_all(
            fields => [qw(page_id model)],
            filter => {page_id => \@page_ids,}
        );

        foreach my $el (@$data) {
            push @{$result->{$el->{'page_id'}}->{models}}, $el->{'model'};
        }
    }

    return $result;
}

sub create_task_db_dump : METHOD : PARAMS(page_id[], login[], date) : TITLE('Dump database') : FORMATS(json) {
    my ($self, %params) = @_;

    my $qv = $self->app->validator->check(
        data     => \%params,
        template => {
            type   => 'hash',
            fields => {
                page_id => {
                    type => 'array',
                    all  => {type => 'int_un', msg => gettext('Page ID must be a whole number')},
                },
                login => {
                    type => 'array',
                    all =>
                      {type => 'scalar', len_min => 1, msg => gettext('Login must contain at least one character')},
                },
                date => {
                    type     => 'date',
                    optional => TRUE,
                    format   => 'db',
                    msg      => gettext('Date must be in format "db"')
                },
            },
            check => sub {
                my ($qv, $data) = @_;

                throw Exception::Validator::Fields gettext('Expected "page_id" or "login"')
                  unless @{$data->{'page_id'}} || @{$data->{'login'}};
              }
        },
    );

    throw Exception::Validation::BadArguments $qv->get_all_errors if $qv->has_errors;

    my $tmp_rights = $self->app->add_tmp_rights('queue_add_db_dump');

    my $id = $self->queue->add(
        method_name => 'db_dump',
        params      => {page_ids => $params{'page_id'}, logins => $params{'login'}, date => $params{'date'}}
    );

    return $id;
}

sub get_task_db_dump_results : METHOD : PARAMS(!id) : TITLE('Get info') : FORMATS(json) {
    my ($self, %params) = @_;

    my $tmp_rights = $self->app->add_tmp_rights('queue_db_dump');

    my $task = $self->queue->get($params{'id'}, fields => [qw(multistate data_result)]);

    throw Exception::Validation::BadArguments gettext('Task with id %s not found', $params{'id'}) unless defined($task);

    my $result = {id => $params{'id'}};

    if ($self->queue->check_multistate_flag($task->{'multistate'}, 'finished_with_success')) {
        $result->{'status'} = 'finished';

        $result->{'hostname'} = $task->{'data_result'}[0]{'hostname'};
        $result->{'file'}     = $task->{'data_result'}[0]{'file'};
    } elsif ($self->queue->check_multistate_flag($task->{'multistate'}, 'finished_with_error')) {
        $result->{'status'} = 'error';
    } else {
        $result->{'status'} = 'processing';
    }

    return $result;
}

TRUE;
