package IntAPI::Method::Form;

=encoding UTF-8

=cut

use qbit;

use base qw(IntAPI::Method);

use Digest::MD5 qw(md5_base64);

use Utils::Fias;
use Utils::Logger qw(ERROR);

use Exception::CommonOffer;
use Exception::API::AdFoxGraphQL;
use Exception::Balance::IncorrectAnswer::KnownErrors;
use Exception::Balance::NotFound;
use Exception::Validation::BadArguments;

use PiConstants qw(
  @ADFOX_PRESET_ALL
  @ADFOX_PRESET_DEFAULT
  @ADFOX_PRESET_PAIDPRODUCTS
  $ASSESSOR_ROLE_ID
  $CPM_AVAILABLE_CURRENCIES
  $CONTRACT_OFFER_ERROR_MAIL
  $DEFAULT_CPM_CURRENCY
  $MOBILE_PARTNER_ROLE_ID
  $PARTNER_FORM_S3_PREFIX
  $PARTNER2_COMMON_OFFER_ERRORS
  $SELFEMPLOYED_COOPERATION_FORM
  $SELFEMPLOYED_STATUS_NOT
  $SITE_PARTNER_ROLE_ID
  $VIDEO_PARTNER_ROLE_ID
  );

my $LIGHT_FORM_PROBABILITY_KEY_NAME = 'light_form_percent';
my $DEFAULT_FORM_VERSION            = 1;
my $LIGHT_FORM_VERSION              = 2;

__PACKAGE__->model_accessors(
    api_adfox_graphql    => 'Application::Model::API::Yandex::AdFoxGraphQL',
    api_balance          => 'Application::Model::API::Yandex::Balance',
    api_blackbox         => 'QBit::Application::Model::API::Yandex::BlackBox',
    api_media_storage_s3 => 'Application::Model::API::Yandex::MediaStorage::S3',
    api_yamoney          => 'Application::Model::API::Yandex::YaMoney',
    exception_dumper     => 'Application::Model::ExceptionDumper',
    kv_store             => 'QBit::Application::Model::KvStore',
    mail_notification    => 'Application::Model::MailNotification',
    partner_db           => 'Application::Model::PartnerDB',
    rbac                 => 'Application::Model::RBAC',
    users                => 'Application::Model::Users',
    user_features        => 'Application::Model::Users::Features',
);

=head2 form_version

L<https://wiki.yandex-team.ru/partner/w/dev/infrastructure/systems/applications/partner/partner2-intapi/form/form-version/>

=cut

sub form_version : METHOD : PARAMS(!user_id) {
    my ($self, %params) = @_;

    my $user_id = $params{user_id};
    throw Exception::Validation::BadArguments sprintf("user_id is not numeric: %s", $user_id)
      unless $user_id =~ /^[0-9]+\z/;

    my $user_form_version = ($self->partner_db->form_version->get($user_id, fields => ['version']) // {})->{version};
    unless ($user_form_version) {
        my $probability = $self->kv_store->get($LIGHT_FORM_PROBABILITY_KEY_NAME);
        if ($probability && $probability > rand(100)) {
            $user_form_version = $LIGHT_FORM_VERSION;
        } else {
            $user_form_version = $DEFAULT_FORM_VERSION;
        }
        $self->partner_db->form_version->add(
            {
                dt      => curdate(oformat => 'db_time'),
                user_id => $user_id,
                version => $user_form_version,
            }
        );
    }

    return +{version => $user_form_version + 0};
}

=head2 client_id

L<https://wiki.yandex-team.ru/partner/w/partner2-intapi-form-client-id/>

=cut

sub client_id : METHOD : PARAMS(!user_id) {
    my ($self, %params) = @_;

    throw Exception::Validation::BadArguments sprintf("user_id is not numeric: %s", $params{user_id})
      unless $params{user_id} =~ /^[0-9]+\z/;

    my $tmp_rights = $self->app->add_tmp_rights(qw(users_view_field__client_id users_view_all));

    my $user = $self->users->get($params{user_id}, fields => [qw(client_id)],);

    if ($user && $user->{client_id}) {
        return {client_id => $user->{client_id},};
    } elsif ($user && !$user->{client_id}) {
        throw sprintf 'Internal error. user_id "%s" in database, but there is no client_id', $params{user_id};
    } else {
        my $result;
        try {
            my $client_id = $self->users->get_client_id($params{user_id});
            $result = +{client_id => $client_id,};
        }
        catch Exception::Balance::IncorrectAnswer::KnownErrors with {
            my ($exception) = @_;
            $result = +{
                'error' => {
                    'id'  => $exception->get_error_id(),
                    'txt' => $exception->get_error_text(),
                },
            };
            ERROR $exception;
        };

        return $result;
    }
}

sub check_pi_adfox_contracts : METHOD : FORMATS(json) {
    my ($self, %params) = @_;

    my $body = $self->get_body();
    my $opts = eval {from_json($body)};
    my $result;
    try {
        $result = $self->users->check_pi_adfox_contracts(%$opts) &&{contracts => 'ok'};
    }
    catch Exception::CommonOffer with {
        my ($exception) = @_;

        $result = {
            error_message => $exception->{text},
            error_token   => $exception->{error_token},
        };
        $self->exception_dumper->dump_as_html_file($exception);
    }
    catch {
        my ($exception) = @_;
        $self->exception_dumper->dump_as_html_file($exception);
        throw $exception;
    };

    return $result;
}

=head2 create_offer

L<https://wiki.yandex-team.ru/partner/w/partner2-intapi-form-create-offer/>

Пример ответа:

    {
      "data": {
        "contract": {
          "EXTERNAL_ID": "РС-64233-05/18",
          "ID": "611822"
        }
      },
      "result": "ok"
    }

=cut

sub create_offer : METHOD : FORMATS(json) {
    my ($self, %params) = @_;

    @_ = ($self, '[FILTERED]');    # Защита от сохранения пароля в стектрейсе sentry

    throw Exception::Validation::BadArguments 'Incorrect Content-Type'
      unless $self->app->request->http_header('content-type') =~ /application\/json/;

    my $body = $self->get_body();
    throw Exception::Validation::BadArguments 'Must specify body' unless $body;

    my $opts = eval {from_json($body)};

    throw Exception::Validation::BadArguments 'Body must be a valid json' unless $opts;

    my $contract;
    my $exception;
    try {
        my $adfox                             = delete $opts->{adfox_account};
        my $user_id                           = $opts->{operator_uid};
        my $login                             = (delete($opts->{login}) || $user_id);
        my $allow_existing_test_mode_contract = $opts->{allow_existing_test_mode_contract};
        my $contracts                         = $self->users->check_pi_adfox_contracts(
            allow_existing_test_mode_contract => $allow_existing_test_mode_contract,
            allow_paid_services               => $adfox->{adfox_offer},
            from_offer                        => 0,
            user_id                           => $user_id,
            (
                $adfox && $adfox->{has_account}
                ? (
                    adfox_login    => $adfox->{login},
                    adfox_password => $adfox->{password},
                  )
                : ()
            )
        );
        unless (defined($contracts->{pi_contract}) && $allow_existing_test_mode_contract) {
            $self->app->set_cur_user({id => $user_id});
            my $new_contract = $self->users->accept_common_offer(
                ($adfox && $adfox->{has_account} ? (adfox_login => $adfox->{login},) : ()),
                client_id => $opts->{client_id},
                contracts => $contracts,
                params    => $opts,
                user_id   => $user_id,
            );
            $contract = $new_contract;
        } else {
            $contract = {
                EXTERNAL_ID => $contracts->{pi_contract}{Contract}{external_id},
                ID          => $contracts->{pi_contract}{Contract}{contract2_id},
            };
        }
    }
    catch {
        $exception = shift;

        # Для простоты и удобства разработки мы отдаем информацию про ошибку в ответе ручки
        # Но так же и формируем файл с описанем ощибки, чтобы видеть его в почте
        $self->exception_dumper->dump_as_html_file($exception);
    };

    return {
        ($contract ? (contract => $contract) : ()),
        ($exception ? (error_message => $exception->message(), error_token => $exception->{error_token},) : ()),
    };
}

=head2 update_contract

L<https://wiki.yandex-team.ru/partner/w/partner2-intapi-form-update-contract/>

=cut

sub update_contract : METHOD : FORMATS(json) {
    my ($self, %params) = @_;

    throw Exception::Validation::BadArguments 'Incorrect Content-Type'
      unless $self->app->request->http_header('content-type') =~ /application\/json/;

    my $body = $self->get_body();
    throw Exception::Validation::BadArguments 'Must specify body' unless $body;

    my $opts = eval {from_json($body)};

    throw Exception::Validation::BadArguments 'Body must be a valid json' unless $opts;

    QBit::Validator->new(
        data     => $opts,
        template => {
            type   => 'hash',
            fields => {
                operator_uid => {type => 'int_un',},
                contract_id  => {type => 'int_un',},
            },
            extra => TRUE,
        },
        throw => 1,
    );

    my $contract;
    my $error_message;
    try {
        $contract = $self->api_balance->update_contract(%$opts);
    }
    catch {
        my $exception = shift;

        # Для простоты и удобства разработки мы отдаем информацию про ошибку в ответе ручки
        $error_message = $exception->message();

        # Но так же и формируем файл с описанем ощибки, чтобы видеть его в почте
        $self->exception_dumper->dump_as_html_file($exception);
    };

    return {($contract ? (contract => $contract) : ()), ($error_message ? (error_message => $error_message) : ()),};
}

=head2 create_person

L<https://wiki.yandex-team.ru/partner/w/partner2-intapi-form-create-person/>

=cut

sub create_person : METHOD : FORMATS(json) {
    my ($self, %params) = @_;

    throw Exception::Validation::BadArguments 'Incorrect Content-Type'
      unless $self->app->request->http_header('content-type') =~ /application\/json/;

    my $body = $self->get_body();
    throw Exception::Validation::BadArguments 'Must specify body' unless $body;

    my $opts = eval {from_json($body)};

    throw Exception::Validation::BadArguments 'Body must be a valid json' unless $opts;

    my $person_id;
    my $error_message;
    try {
        $person_id = $self->api_balance->create_person(%$opts);
    }
    catch {
        my $exception = shift;

        # Для простоты и удобства разработки мы отдаем информацию про ошибку в ответе ручки
        $error_message = $exception->message();

        # Но так же и формируем файл с описанем ощибки, чтобы видеть его в почте
        $self->exception_dumper->dump_as_html_file($exception);
    };

    return {($person_id ? (person_id => $person_id) : ()), ($error_message ? (error_message => $error_message) : ()),};
}

=head2 is_yamoney_account_identificated

L<https://wiki.yandex-team.ru/partner/w/partner2-intapi-form-is-yamoney-account-identificated/>

=cut

sub is_yamoney_account_identificated : METHOD : PARAMS(!account, !lastname, !firstname, !middlename, !passport) :
  FORMATS(json) {
    my ($self, %params) = @_;

    my $is_identificated = $self->api_yamoney->is_account_identificated(
        account    => $params{account},
        lastname   => $params{lastname},
        firstname  => $params{firstname},
        middlename => $params{middlename},
        passport   => $params{passport},
    );

    return {is_identificated => $is_identificated,};
}

=head2 save_data

L<https://wiki.yandex-team.ru/partner/w/partner2-intapi-form-save_data/>

=cut

sub save_data : METHOD : FORMATS(json) {
    my ($self) = @_;

    throw Exception::Validation::BadArguments 'Incorrect Content-Type'
      unless $self->app->request->http_header('content-type') =~ /application\/json/;

    my $body = $self->get_body();
    throw Exception::Validation::BadArguments 'Must specify body' unless $body;

    my $opts = eval {from_json($body)};

    throw Exception::Validation::BadArguments 'Body must be a valid json' unless $opts;

    QBit::Validator->new(
        data     => $opts,
        template => {
            type   => 'hash',
            fields => {
                user_id    => {type => 'int_un',},
                client_id  => {type => 'int_un',},
                person_id  => {type => 'int_un', optional => TRUE,},
                country_id => {type => 'int_un',},
                branch_id  => {type => 'scalar',},
                data => {
                    type  => 'hash',
                    extra => TRUE,
                },
            }
        },
        throw => 1,
    );

    $self->partner_db->form_data->add(
        {
            dt         => curdate(oformat => 'db_time'),
            user_id    => $opts->{user_id},
            client_id  => $opts->{client_id},
            person_id  => $opts->{person_id},
            country_id => $opts->{country_id},
            branch_id  => $opts->{branch_id},
            data       => to_json($opts->{data}),
        }
    );

    return 'ok';
}

=head2 create_or_update_partner

L<https://wiki.yandex-team.ru/partner/w/partner2-intapi-form-create-or-update-partner/>

=cut

sub create_or_update_partner : METHOD : FORMATS(json) {
    my ($self) = @_;

    throw Exception::Validation::BadArguments 'Incorrect Content-Type'
      unless $self->app->request->http_header('content-type') =~ /application\/json/;

    my $body = $self->get_body();
    throw Exception::Validation::BadArguments 'Must specify body' unless $body;

    my $opts = eval {from_json($body)};

    throw Exception::Validation::BadArguments 'Body must be a valid json' unless $opts;

    my $fields = $self->users->get_model_fields();
    my %fields = (features => {type => 'array', optional => TRUE});
    foreach my $id (keys(%{$fields})) {
        next if $id eq 'id';
        next if $id eq 'login';
        next if (!$fields->{$id}{db} && !$fields->{$id}{from_opts});
        $fields{$id} = {optional => TRUE};
    }

    my $is_assessor = delete $opts->{data}{is_assessor};
    my $roles       = delete $opts->{data}{roles};
    if ($opts->{api_mobile_mediation_data}) {
        $opts->{api_mobile_mediation_data}{user_id} = $opts->{user_id};
        $opts->{api_mobile_mediation_data}{presets} //= ['MOBILEMEDIATION'];
    }

    $opts->{data}{has_common_offer} //= 1
      if (
           225 eq ($opts->{data}{country_id} // 0)
        && !$opts->{data}{is_video_blogger}
        && !$opts->{data}{is_efir_blogger}
        && (   $opts->{api_mobile_mediation_data}
            || $opts->{adfox_link} && $opts->{adfox_link}{type} && in_array($opts->{adfox_link}{type}, [qw(bind new)]))
         );
    $opts->{data}{adfox_offer} ||= 1
      if ( $opts->{data}{has_common_offer}
        && $opts->{adfox_link}
        && $opts->{adfox_link}{adfox_offer}
        && $opts->{adfox_link}{type}
        && in_array($opts->{adfox_link}{type}, [qw(bind new)]));
    $opts->{data}{self_employed} //= $SELFEMPLOYED_STATUS_NOT
      if (($opts->{data}{cooperation_form} // '') eq $SELFEMPLOYED_COOPERATION_FORM);

    QBit::Validator->new(
        data     => $opts,
        template => {
            type   => 'hash',
            fields => {
                user_id => {type => 'int_un',},
                data    => {
                    type   => 'hash',
                    fields => \%fields,
                },
                adfox_link => {
                    fields => {
                        adfox_offer => {
                            optional => TRUE,
                            type     => 'scalar',
                        },
                        login => {
                            len_max  => 32,
                            len_min  => 1,
                            optional => TRUE,
                            type     => 'scalar',
                        },
                        presets => {
                            all => {
                                in   => \@ADFOX_PRESET_ALL,
                                type => 'scalar',
                            },
                            optional => TRUE,
                            type     => 'array',
                        },
                        type => {
                            in   => [qw(bind new none)],
                            type => 'scalar',
                        },
                    },
                    optional => TRUE,
                    type     => 'hash',
                },
                api_mobile_mediation_data => {
                    extra    => TRUE,
                    optional => TRUE,
                    type     => 'hash',
                },
            },
            one_of => [qw(adfox_link api_mobile_mediation_data)],
        },
        throw => 1,
    );

    my $tmp_rights = $self->app->add_all_tmp_rights();

    my $user = $self->users->get_by_uid($opts->{user_id}, fields => [qw(id), keys(%{$opts->{data}})]);
    my $adfox_user;
    my $user_info;
    my $result = 'ok';
    try {
        $self->partner_db->transaction(
            sub {
                if ($user) {
                    # Пользователь есть в базе - редактируем если данные отличаются от того что в базе
                    delete $opts->{data}->{current_currency};
                    my $features = delete $opts->{data}->{features} // [];
                    # we dont want to cmp features to determine is there something to edit
                    delete $user->{features};
                    require Test::Deep::NoTest;
                    Test::Deep::NoTest->import(qw(eq_deeply));
                    if (!eq_deeply($user, {id => $user->{id}, %{$opts->{data}}})) {
                        $self->users->do_action($user->{id}, 'edit', %{$opts->{data}},);
                    }
                    $self->user_features->replace_multi($user->{id}, $features, only_add => TRUE) if @$features;
                } else {
                    # Пользователь в базе нет - создаем
                    $user_info = $self->api_blackbox->get_user_info($opts->{user_id});

                    throw Exception::Validation::BadArguments 'yndx login can\'t be partner'
                      if $user_info->{canonical_login} =~ /^yndx-/;

                    my $tmp_rights = $self->app->add_tmp_rights('do_user_action_add');

                    if (defined $opts->{data}->{current_currency}
                        && !in_array($opts->{data}->{current_currency}, $CPM_AVAILABLE_CURRENCIES))
                    {
                        $opts->{data}->{current_currency} = $DEFAULT_CPM_CURRENCY;
                    }
                    $self->users->add(
                        uid   => $opts->{user_id},
                        login => $user_info->{canonical_login},

                        # По умолчанию используем данные из паспорта
                        name     => $user_info->{name},
                        lastname => $user_info->{lastname},
                        midname  => $user_info->{midname},
                        email    => $user_info->{email},

                        # Но если в ручку передали эти же поля, то будут использованы их значения
                        %{$opts->{data}},
                    );
                    $user = $self->users->get_by_uid($opts->{user_id}, fields => [qw(cooperation_form id inn login)]);
                    if ($is_assessor) {
                        $self->users->do_action(
                            $user->{id},
                            'set_user_role',
                            role_id => [
                                $ASSESSOR_ROLE_ID,
                                @{
                                    $self->rbac->get_roles(
                                        assessor_applicable => TRUE,
                                        id_only             => TRUE,
                                    )
                                  }
                            ]
                        );
                    } elsif ($opts->{data}{is_video_blogger} || $opts->{data}{is_efir_blogger}) {
                        $self->users->do_action($user->{id}, 'set_user_role', role_id => [$VIDEO_PARTNER_ROLE_ID]);
                    } else {
                        $roles ||= [$SITE_PARTNER_ROLE_ID, $MOBILE_PARTNER_ROLE_ID];
                        my %roles = map {$_ => TRUE} @$roles;
                        my $mail_sub;
                        if ($roles{$SITE_PARTNER_ROLE_ID}) {
                            $mail_sub =
                              $roles{$MOBILE_PARTNER_ROLE_ID}
                              ? 'add_when_partner_registered_w_site_and_app'
                              : 'add_when_partner_registered_w_site';

                        } elsif ($roles{$MOBILE_PARTNER_ROLE_ID}) {
                            $mail_sub = 'add_when_partner_registered_w_mobile_app';
                        }
                        $self->mail_notification->$mail_sub({user_id => $user->{id}, login => $user->{login}})
                          if $mail_sub;
                        $self->users->do_action($user->{id}, 'set_user_role', role_id => $roles);
                    }
                }

                if ($opts->{api_mobile_mediation_data}) {
                    $adfox_user = $self->api_adfox_graphql->create_user(%{$opts->{api_mobile_mediation_data}});
                } elsif ($opts->{adfox_link}) {
                    if ($opts->{adfox_link}{type} eq 'bind') {

                        my %args4bind_user = (
                            user_id     => $user->{id},
                            adfox_login => $opts->{adfox_link}{login},
                            presets     => ($opts->{adfox_link}{presets} // \@ADFOX_PRESET_DEFAULT),
                        );

                        # partner has AdFox paid products
                        if ($opts->{data}{adfox_offer}) {
                            my $adfox_contract_created = $self->users->create_contract_adfox_paid_products(
                                user_id     => $user->{id},
                                adfox_login => $opts->{adfox_link}{login},
                                mail_info   => {
                                    user_id    => $user->{id},
                                    adfox_info => {
                                        id    => '',
                                        login => $opts->{adfox_link}{login},
                                    },
                                },
                            );
                            $args4bind_user{presets} = array_uniq($args4bind_user{presets}, @ADFOX_PRESET_PAIDPRODUCTS);
                            if ($adfox_contract_created) {
                                $args4bind_user{billing_data} = {
                                    client_id   => $adfox_contract_created->{client_id},
                                    contract_id => $adfox_contract_created->{id},
                                };
                            }
                        }

                        $adfox_user = $self->api_adfox_graphql->bind_user(%args4bind_user);

                    } elsif ($opts->{adfox_link}{type} eq 'new') {
                        $adfox_user = $self->api_adfox_graphql->create_user(
                            user_id => $user->{id},
                            presets => ($opts->{adfox_link}{presets} // \@ADFOX_PRESET_DEFAULT),
                        );
                    }
                }
                if ($adfox_user) {
                    # Так как у нас уже есть логин пользователя АФ, то передаётся флаг mobile_mediation
                    # который отвечает за то, чтобы не ходить в ручку АФ за инфой о пользователе
                    $self->users->do_action(
                        $user->{id},
                        'link_adfox_user',
                        adfox_id         => $adfox_user->{id},
                        adfox_login      => $adfox_user->{login},
                        mobile_mediation => TRUE,
                    );
                }
            }
        );
    }
    catch Exception with {
        my ($exception) = @_;

        if (  !$exception->{'__mailed'}
            && $opts->{adfox_link}{type}
            && in_array($opts->{adfox_link}{type}, [qw(bind new)]))
        {
            local $Data::Dumper::Sortkeys = 1;
            my $message = join(
                "\n",
                'Пользователь не смог перейти на оферту',
                '-' x 40,
                'Error: ' . $exception->message(),
                sprintf(
                    'PI: login=%s, id=%s',
                    ($user_info && $user_info->{canonical_login} // ''),
                    ($opts->{user_id} // ''),
                ),
                'AdFox: login=' . ($adfox_user && $adfox_user->{login} // ''),
                '',
                'Технические детали:',
                '-' x 40,
                $exception,
                Dumper {opts => $opts, user_info => $user_info, adfox_user => $adfox_user,}
            );
            $self->mail_notification->add_common_offer_error(
                to      => [$PARTNER2_COMMON_OFFER_ERRORS, $CONTRACT_OFFER_ERROR_MAIL],
                subject => sprintf(
                    'Common offer failed - Оферта, login=%s, adfox_login=%s',
                    ($user_info && $user_info->{canonical_login} || $opts->{user_id} // ''),
                    ($adfox_user && $adfox_user->{login} // '')
                ),
                message => $message,
            );
        }

        my $err = $exception->message();
        if ($err =~ /AdFox/i || $exception->isa('Exception::API::AdFoxGraphQL')) {
            $result = {'error' => $err,};
            ERROR $exception;
        } else {
            throw $exception;
        }
    };

    return $result;
}

=head2 save_attachment

L<https://wiki.yandex-team.ru/partner/w/partner2-intapi-form-save-attachment/>

=cut

sub save_attachment : METHOD : PARAMS(!uid, !data, file_name) : FORMATS(json) {
    my ($self, %params) = @_;

    my $body = $params{data}{content};
    my $key = _build_key($params{uid}, $body, $params{file_name});

    $self->api_media_storage_s3->put($key, $params{data}{'content-type'}, $body);

    return {key => $key};
}

=head2 list_attachments

L<https://wiki.yandex-team.ru/partner/w/partner2-intapi-form-list-attachments/>

=cut

sub list_attachments : METHOD : PARAMS(!uid) : FORMATS(json) {
    my ($self, %params) = @_;

    return $self->api_media_storage_s3->list(_uid_prefix($params{uid}));
}

=head2 user

L<https://wiki.yandex-team.ru/partner/w/partner2-intapi-form-user/>

=cut

sub user : METHOD : PARAMS(!user_id, db, rep) : FORMATS(json) {
    my ($self, %params) = @_;

    my $tmp_rights = $self->app->add_all_tmp_rights();

    my $user_id = $params{user_id};

    my $user = $self->users->get($user_id, fields => [qw(active_contract roles)]);
    my $db;

    if ($params{db}) {
        $db = {
            user      => $self->partner_db->users->get($user_id),
            form_data => $self->partner_db->form_data->get_all(
                filter => {user_id => $user_id},
                order_by => [['id', 1]],
                limit    => 1,
              )->[0],
        };
        delete $db->{user}->{opts} if $db->{user};
    }
    my $representative;
    if ($params{rep}) {
        my $client_id = $self->api_balance->get_client_id_by_uid($user_id);
        $representative = $self->partner_db->users->get_all(
            fields => ['id'],
            filter => [AND => [[client_id => '=' => \$client_id], [id => '<>' => \$user_id]]]
          )->[0]{id}
          if ($client_id);
    }

    return {
        has_role_in_partner2 => (@{$user->{roles} // []} > 0 ? TRUE : FALSE),
        can_fill_part2 => ($user->{active_contract}{Contract}{allows_to_fill_part2} ? TRUE : FALSE),
        ($db ? (db => $db) : ()),
        ($params{rep} ? (representative => $representative,) : ()),
    };
}

=head2 query_fias

L<https://wiki.yandex-team.ru/partner/w/partner2-intapi-form-query-fias/>

=cut

sub query_fias : METHOD : PARAMS(parent_guid, guid) : FORMATS(json) {
    return Utils::Fias::query_fias(@_);
}

=head2 fias_hierarchy

L<https://wiki.yandex-team.ru/partner/w/partner2-intapi-form-fias-hierarchy/>

=cut

sub fias_hierarchy : METHOD : PARAMS(!guid) : FORMATS(json) {
    return Utils::Fias::fias_hierarchy(@_);
}

=head2 send_email

L<https://wiki.yandex-team.ru/partner/w/partner2-intapi-form-send-email/>

=cut

sub send_email : METHOD : FORMATS(json) {
    my ($self, %params) = @_;

    throw Exception::Validation::BadArguments 'Incorrect Content-Type'
      unless $self->app->request->http_header('content-type') =~ /application\/json/;

    my $body = $self->get_body();
    throw Exception::Validation::BadArguments 'Must specify body' unless $body;

    my $opts = eval {from_json($body)};

    throw Exception::Validation::BadArguments 'Body must be a valid json' unless $opts;

    my $for_sender = ($opts->{transport} && $opts->{transport} eq 'sender' && $opts->{template} && $opts->{user_id});

    QBit::Validator->new(
        data     => $opts,
        template => {
            type   => 'hash',
            fields => {
                from         => {type => 'scalar', optional => TRUE},
                to           => {type => 'scalar', optional => TRUE},
                subject      => {type => 'scalar', optional => $for_sender ? TRUE : FALSE},
                body         => {type => 'scalar', optional => $for_sender ? TRUE : FALSE},
                cc           => {type => 'scalar', optional => TRUE},
                bcc          => {type => 'scalar', optional => TRUE},
                content_type => {type => 'scalar', optional => TRUE},
                user_id      => {type => 'scalar', optional => TRUE},
                transport    => {type => 'scalar', optional => TRUE},
                template     => {type => 'scalar', optional => TRUE},
                vars         => {type => 'hash',   optional => TRUE},
            }
        },
        throw => 1,
    );

    my %mail_notification = (
        type    => 0,
        user_id => ($opts->{user_id} // 0),
        period  => undef,
        opts    => {
            ($opts->{user_id} ? () : (check_alive => FALSE,)),
            values => {message_body => $opts->{body},},
            map {$opts->{$_} ? ($_ => $opts->{$_}) : ()} qw(from to subject cc bcc),
        },
    );

    if ($for_sender) {
        $mail_notification{type}                   = 12;
        $mail_notification{opts}{values}{template} = $opts->{template};
        $mail_notification{opts}{vars}             = $opts->{vars};
    }

    $self->mail_notification->add(%mail_notification);

    return 'ok';
}

=head2 rm_user_in_db

L<https://wiki.yandex-team.ru/partner/w/partner2-intapi-form-rm-user-in-db/>

=cut

sub rm_user_in_db : METHOD : PARAMS(!user_id) : FORMATS(json) {
    my ($self, %params) = @_;

    throw Exception::Validation::BadArguments sprintf("Can't use this endpoint on this stage")
      unless in_array($self->get_option('stage', 'unknown'), [qw(dev test)]);

    my @sqls = (
        'delete from form_data where user_id = ?',
        'delete from users_action_log where elem_id = ?',
        'delete from user_role where user_id = ?',
'delete from context_on_site_campaign_action_log where elem_id in (select id from context_on_site_campaign where owner_id = ?)',
        'delete from context_on_site_campaign where owner_id = ?',
        'delete from owner_site where user_id = ?',
        'delete from site_action_log where user_id = ?',
        'delete from user_adfox where user_id = ?',
        'delete from mail_notification where user_id = ?',
        'delete from common_offer_allowed_users where user_id = ?',
        'delete from user_features where user_id = ?',
        'delete from users where id = ?',
    );

    $self->partner_db->transaction(
        sub {
            foreach my $sql (@sqls) {
                $self->partner_db->_do($sql, $params{user_id});
            }
        }
    );

    return 'ok';

}

=head2 person

L<https://wiki.yandex-team.ru/partner/w/partner2-intapi-form-person/>

=cut

sub person : METHOD : PARAMS(!person_id, !client_id) {
    my ($self, %params) = @_;

    my $client_id = $params{client_id};
    my $person_id = $params{person_id};

    my @persons = @{$self->api_balance->get_client_persons($client_id, 1,)};

    my @p = grep {$_->{id} eq $person_id} @persons;

    return $p[0];
}

=head2 bank

L<https://wiki.yandex-team.ru/partner/w/partner2-intapi-form-bank/>

=cut

sub bank : METHOD : PARAMS(bik, swift) : FORMATS(json) {
    my ($self, %params) = @_;

    return $self->api_balance->check_bank(%params);
}

=head2 remove_user_client_association

L<https://wiki.yandex-team.ru/partner/w/partner2-intapi-form-remove-user-client-association/>

=cut

sub remove_user_client_association : METHOD : PARAMS(!operator_user_id, !client_id, !user_id) : FORMATS(json) {
    my ($self, %params) = @_;

    QBit::Validator->new(
        data     => \%params,
        template => {
            type   => 'hash',
            fields => {
                operator_user_id => {type => 'int_un'},
                client_id        => {type => 'int_un'},
                user_id          => {type => 'int_un'},
            }
        },
        throw => 1,
    );

    try {
        $self->api_balance->remove_user_client_association($params{operator_user_id},
            $params{client_id}, $params{user_id});
    }
    catch {
        my ($e) = @_;

        unless (ref($e) eq 'Exception::Balance::IncorrectAnswer'
            && $e->message() =~ /Passport .* is NOT linked to client/)
        {
            throw $e;
        }
    };

    return 'ok';
}

sub _build_key {
    my ($uid, $data, $file_name) = @_;

    $file_name //= '';

    my $extension = '';
    if ($file_name =~ /(\.[^\.]+)\z/) {
        $extension = lc($1);
    }

    my $checksum  = md5_base64($data);
    my $timestamp = time;

    return "${PARTNER_FORM_S3_PREFIX}_${uid}_${checksum}_${timestamp}${extension}";
}

sub _uid_prefix {
    my ($uid) = @_;

    return "${PARTNER_FORM_S3_PREFIX}_${uid}_";
}

=head2 bank

L<https://wiki.yandex-team.ru/partner/w/partner2-intapi-form-rubank_account/>

=cut

sub rubank_account : METHOD : PARAMS(!bik, account) : FORMATS(json) {
    my ($self, %params) = @_;

    QBit::Validator->new(
        data     => \%params,
        template => {
            type   => 'hash',
            fields => {
                bik     => {type => 'scalar', len => 9},
                account => {type => 'scalar', len => 20, optional => TRUE},
            }
        },
        throw => 1,
    );

    return $self->api_balance->check_rubank_account(%params);
}

1;
