package IntAPI::Method::IDM;

use qbit;
use Exception::Denied;
use Utils::Logger qw(INFOF INFO);
use PiConstants qw($IDM_SLUG $INTERNAL_YAN_MANAGER_ROLE_ID $PARTNER2_IDM_MAIL);
use Exception::Validation::BadArguments;
use Utils::MonitoringUtils;

use base qw(IntAPI::Method);

__PACKAGE__->model_accessors(
    partner_db => 'Application::Model::PartnerDB',
    rbac       => 'Application::Model::RBAC',
);

sub info : METHOD : ALLOWHTTP : TVM(tvm_idm) : FORMATS(json) : TITLE('IDM info') {
    my ($self, %params) = @_;

    my %roles;
    for my $role (@{$self->rbac->get_roles(idm => TRUE)}) {
        my %f = $role->{idm_fields} ? (fields => $role->{idm_fields}) : ();
        $roles{$role->{idm}} = {
            name => $role->{name},
            %f,
        };
    }

    return format_result(
        {
            code  => 0,
            roles => {
                slug   => $IDM_SLUG,
                name   => "Роль",
                values => \%roles,
            },
            fields => [
                {
                    slug => "passport-login",
                    name => {
                        ru => "Паспортный логин",
                        en => "Passport login",
                    },
                    type     => "passportlogin",
                    required => \1
                }
            ]
        }
    );
}

sub get_roles {
    my ($self) = @_;
    my %roles = map {
        $_->{idm} => {
            id => $_->{id},
            ($_->{idm_fields} ? (fields => [map {$_->{slug}} @{$_->{idm_fields}}]) : ())
          }
    } @{$self->rbac->get_roles(idm => TRUE)};
    return \%roles;
}

sub get_all_roles : METHOD : ALLOWHTTP : TVM(tvm_idm) : FORMATS(json) : TITLE('IDM list roles') {
    my ($self, %params) = @_;

    my %roles_id = map {$_->{id} => $_->{idm}} @{$self->rbac->get_roles(idm => TRUE)};
    my $list = $self->partner_db->query->select(
        table  => $self->partner_db->user_role,
        fields => [qw(user_id role_id)],
        filter => [role_id => 'IN' => \[sort {$a <=> $b} keys %roles_id]],
        order_by => ['user_id', 'role_id']
      )->join(
        table   => $self->partner_db->users,
        fields  => [qw(login domain_login)],
        join_on => [id => '=' => {user_id => $self->partner_db->user_role}],
      )->get_all;

    my $manager_id;
    my $manager_query = $self->partner_db->query->select(
        table  => $self->partner_db->managers,
        fields => ['manager_id', 'page_id'],
        filter => [manager_id => '=' => \$manager_id],
    );
    my %logins;
    for my $row (@$list) {
        $row->{domain_login} ||= $row->{login};
        if ($row->{role_id} == $INTERNAL_YAN_MANAGER_ROLE_ID) {
            $manager_id = $row->{user_id};
            for my $manager_row (@{$manager_query->get_all}) {
                push @{$logins{$row->{domain_login}}},
                  [
                    {$IDM_SLUG => $roles_id{$row->{role_id}}},
                    {
                        'passport-login' => $row->{login},
                        'page_id'        => "$manager_row->{page_id}",
                    },
                  ];
            }
        } else {
            push @{$logins{$row->{domain_login}}},
              [{$IDM_SLUG => $roles_id{$row->{role_id}}}, {'passport-login' => $row->{login}}];
        }
    }

    return format_result(
        {
            code  => 0,
            users => [
                map +{
                    login => $_,
                    roles => $logins{$_},
                },
                sort keys %logins,
            ]
        }
    );
}

sub add_role : METHOD : ALLOWHTTP : PARAMS(!login, !role, !path, !fields) : TVM(tvm_idm) : FORMATS(json) :
  TITLE('IDM add role') {
    my ($self, %params) = @_;

    my $tmp_rights = $self->app->add_all_tmp_rights();
    my $data;
    my $result;
    try {
        my $extra_message = '';
        my %extra_data;
        $data = $self->check_params(%params);

        my $user_id = $self->get_user_id($data);
        unless ($user_id) {
            $user_id = $self->add_passport_login($data);
        }

        if ($data->{role_id} == $INTERNAL_YAN_MANAGER_ROLE_ID) {
            $self->internal_manager_role_add($data, $user_id);
            $extra_data{data}{page_id} = $data->{fields}{page_id};
            $extra_message = sprintf "На площадку %d\n", $data->{fields}{page_id};
        } else {
            $self->users->do_action($user_id, 'set_user_role', role_id => $data->{role_id}, allow_idm => TRUE);
        }
        $extra_data{data}{'passport-login'} = $data->{'passport-login'};

        $result = {
            code => 0,
            %extra_data
        };

        INFOF "ADD_ROLE_OK '%d' for '%s'", $data->{role_id}, $data->{'passport-login'};

        Utils::MonitoringUtils::send_to_graphite(
            interval => 'one_min',
            path     => 'idm.add_role.' . $data->{role_name},
            value    => 1,
            solomon  => {
                role   => $data->{role_name},
                sensor => 'idm.add_role',
            }
        );

        $self->send_mail_to_list(
            subject =>
              sprintf('Добавлена роль %s для %s', $data->{role_name}, $data->{'passport-login'}),
            message => sprintf(
                "Добавлена роль '%s' - %d\n"
                  . "на пользователя PI '%s' - %d\n"
                  . $extra_message
                  . "на login '%s'\n",
                $data->{role_name}, $data->{role_id}, $data->{'passport-login'},
                $user_id, $data->{login}
            ),
        );

    }
    catch {
        my ($e) = @_;
        unless ($result) {
            my $type_error = $e->{idm_warning} ? 'warning' : 'fatal';
            $result = {
                code        => 1000,
                $type_error => $e->message
            };
            if ($data) {
                INFOF "ADD_ROLE_FAIL '%d' for '%s'", $data->{role_id}, $data->{'passport-login'};
            } else {
                INFOF "ADD_ROLE_ERROR %s", to_json(\%params, pretty => TRUE);
            }
        }
    };

    return format_result($result);
}

sub remove_role : METHOD : ALLOWHTTP : PARAMS(!login, !role, !path, !fields, fired) : TVM(tvm_idm) : FORMATS(json) :
  TITLE('IDM remove role') {
    my ($self, %params) = @_;

    my $tmp_rights = $self->app->add_all_tmp_rights();
    my $data;
    my $result;
    try {
        my $extra_message = '';
        $data = $self->check_params(%params);

        my $user_id = $self->get_user_id($data);
        throw Exception 'User not found' unless $user_id;

        if ($data->{role_id} == $INTERNAL_YAN_MANAGER_ROLE_ID) {
            $self->internal_manager_role_revoke($data, $user_id);
            $extra_message = sprintf "На площадке %d\n", $data->{fields}{page_id};
        } else {
            $self->users->do_action($user_id, 'revoke_roles', roles_id => $data->{role_id}, allow_idm => TRUE);
        }

        $result = {code => 0};

        INFOF "REMOVE_ROLE_OK '%d' for '%s'", $data->{role_id}, $data->{'passport-login'};

        Utils::MonitoringUtils::send_to_graphite(
            interval => 'one_min',
            path     => 'idm.remove_role.' . $data->{role_name},
            value    => 1,
            solomon  => {
                role   => $data->{role_name},
                sensor => 'idm.remove_role',
            }
        );

        $self->send_mail_to_list(
            subject => sprintf('Отобрана роль %s у %s', $data->{role_name}, $data->{'passport-login'}),
            message => sprintf(
                "Отобрана роль '%s' - %d\n"
                  . "у пользователя PI '%s' - %d\n"
                  . $extra_message
                  . "на login '%s'\n",
                $data->{role_name}, $data->{role_id}, $data->{'passport-login'},
                $user_id, $data->{login}
            ),
        );
    }
    catch {
        my ($e) = @_;
        unless ($result) {
            $result = {
                code  => 1000,
                fatal => $e->message
            };
            if ($data) {
                INFOF "REMOVE_ROLE_FAIL '%d' from '%s'", $data->{role_id}, $data->{'passport-login'};
            } else {
                INFOF "REMOVE_ROLE_ERROR %s", to_json(\%params, pretty => TRUE);
            }
        }
    };

    return format_result($result);
}

sub check_params {
    my ($self, %params) = @_;

    my $roles = $self->get_roles;
    my (undef, $role_slug, $role_name, @rest_path) = split /\//, $params{path};
    my $role_hash = from_json($params{role});
    my $fields    = from_json($params{fields});

    my $error;
    unless ($fields->{'passport-login'}) {
        $error = 'No passport login';
    } elsif (!$role_slug or !$role_name or @rest_path) {
        $error = 'Invalid path';
    } elsif ($role_slug ne $IDM_SLUG or !exists $role_hash->{$IDM_SLUG}) {
        $error = 'Invalid slug';
    } elsif ($role_name ne $role_hash->{$IDM_SLUG}) {
        $error = 'Different role';
    } elsif (!$roles->{$role_name}) {
        $error = 'Unexists role';
    } elsif (
        $roles->{$role_name}{fields} and @{$roles->{$role_name}{fields}} != grep {
            defined $fields->{$_}
        } @{$roles->{$role_name}{fields}}
      )
    {
        $error = 'Some required fields are absent: '
          . join(',', grep {!defined $fields->{$_}} @{$roles->{$role_name}{fields}});
    }

    throw Exception::Validation::BadArguments $error if $error;

    my %fields =
      $roles->{$role_name}{fields} ? (fields => {map {$_ => $fields->{$_}} @{$roles->{$role_name}{fields}}}) : ();

    return {
        'login'          => $params{login},
        'passport-login' => $fields->{'passport-login'},
        'role_id'        => $roles->{$role_name}{id},
        'role_name'      => $role_name,
        %fields
    };
}

sub get_user_id {
    my ($self, $data) = @_;
    my $user_id;
    try {
        $user_id = $self->partner_db->users->get_all(
            fields => ['id'],
            filter => {login => fix_login($data->{'passport-login'})}
        )->[0]{id};
        if ($user_id) {
            $self->partner_db->users->edit({id => $user_id}, {domain_login => $data->{login}});
        }
    };
    return $user_id;
}

sub format_result {
    my ($data) = @_;
    return sub {
        if (defined $data) {
            my $result = to_json($data);
            undef $data;
            return [$result];
        } else {
            return [];
        }
    };
}

sub add_passport_login {
    my ($self, $data) = @_;

    my $user_id = $self->app->api_blackbox->get_uid_by_login($data->{'passport-login'});
    my $staff   = $self->app->api_staff->persons(
        login  => $data->{login},
        _one   => 1,
        fields => [qw(work_email name)],
    );

    $self->app->users->add(
        lastname => $staff->{name}{last}{ru}  || $staff->{name}{last}{en}  || '',
        name     => $staff->{name}{first}{ru} || $staff->{name}{first}{en} || '',
        email    => $staff->{work_email}      || '',
        uid      => $user_id,
        login    => fix_login($data->{'passport-login'}),
    );

    return $self->get_user_id($data);
}

sub internal_manager_role_add {
    my ($self, $data, $user_id) = @_;

    throw Exception::Validation::BadArguments(gettext('Invalid page_id "%s"', $data->{fields}{page_id}))
      unless $data->{fields}{page_id} =~ /^\d+$/;

    if (
        $self->partner_db->managers->get_all(
            filter => {
                manager_id => $user_id,
                page_id    => $data->{fields}{page_id},
            }
        )->[0]
       )
    {
        throw Exception::Validation::BadArguments(gettext('User "%s" already has roles', $data->{'passport-login'}),
            idm_warning => TRUE);
    }
    unless (
        $self->partner_db->all_pages->get_all(
            fields => ['page_id'],
            filter => {
                is_internal => 1,
                page_id     => $data->{fields}{page_id},
            }
        )->[0]
      )
    {
        throw Exception::Validation::BadArguments(gettext('Pages "%s" is not internal', $data->{fields}{page_id}));
    }
    my $has_role = $self->rbac->get_roles_by_user_id($user_id)->{$data->{role_id}};
    $self->partner_db->transaction(
        sub {
            $self->users->do_action($user_id, 'set_user_role', role_id => $data->{role_id}, allow_idm => TRUE)
              unless $has_role;
            $self->partner_db->managers->add({manager_id => $user_id, page_id => $data->{fields}{page_id}});
        }
    );
}

sub internal_manager_role_revoke {
    my ($self, $data, $user_id) = @_;

    throw Exception::Validation::BadArguments(gettext('Invalid page_id "%s"', $data->{fields}{page_id}))
      unless $data->{fields}{page_id} =~ /^\d+$/;

    $self->partner_db->transaction(
        sub {
            $self->partner_db->managers->delete({manager_id => $user_id, page_id => $data->{fields}{page_id}});
            if (0 == @{$self->partner_db->managers->get_all(filter => {manager_id => $user_id,})}) {
                $self->users->do_action($user_id, 'revoke_roles', roles_id => $data->{role_id}, allow_idm => TRUE);
            }
        }
    );
}

sub send_mail_to_list {
    my ($self, %params) = @_;
    $self->app->mail_notification->add(
        type    => 0,
        user_id => 0,
        opts    => {
            check_alive => FALSE,
            subject     => $params{subject},
            to          => $PARTNER2_IDM_MAIL,
            values      => {
                message_body       => $params{message},
                plain_text_wrapper => TRUE,
            },
        },
    );
}

1;
