package MenuHierarchy;

use qbit;

use base qw( Exporter );

our @EXPORT_OK = qw(
  get_menu
  );

=head1
    For menu format see https://wiki.yandex-team.ru/partner/w/adfox/united-menu/
=cut

sub get_menu {
    my ($app) = @_;

    return [
        {
            'caption'  => gettext('Dashboard'),
            'children' => [],
            'icon' =>
'data:image/svg+xml,<svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg"><path fill-rule="evenodd" clip-rule="evenodd" d="M6 4.7H9.6125V9.6125H4.7V6C4.7 5.28203 5.28203 4.7 6 4.7ZM3 6C3 4.34315 4.34315 3 6 3H11.3125V11.3125H3V6ZM4.7 15.3875H9.6125V20.3H6C5.28203 20.3 4.7 19.718 4.7 19V15.3875ZM3 19V13.6875H11.3125V22H6C4.34315 22 3 20.6569 3 19ZM19 4.7H15.3875V9.6125H20.3V6C20.3 5.28203 19.718 4.7 19 4.7ZM13.6875 3V11.3125H22V6C22 4.34315 20.6569 3 19 3H13.6875ZM15.3875 15.3875H20.3V19C20.3 19.718 19.718 20.3 19 20.3H15.3875V15.3875ZM13.6875 22V13.6875H22V19C22 20.6569 20.6569 22 19 22H13.6875Z" fill="white"/></svg>',
            'origin'            => 'partner',
            'resource'          => undef,
            'type'              => 'item',
            'url'               => '/v2/dashboard/',
            'only_with_feature' => 'simple_menu',
        },
        {
            'caption'  => gettext('notification'),
            'children' => [],
            'icon' =>
'data:image/svg+xml;base64,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',
            'icon_has_notifications' =>
'data:image/svg+xml;base64,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',
            'origin'            => 'partner',
            'resource'          => 'user_notifications',
            'type'              => 'item',
            'url'               => '/v2/notifications/',
            'only_with_feature' => 'simple_menu',
        },
        # context_on_site
        {
            'caption'  => gettext('Website ads'),
            'children' => [
                {
                    'caption'  => gettext('Sites'),
                    'children' => [],
                    'origin'   => 'partner',
                    'resource' => 'site',
                    'type'     => 'item',
                    'url'      => '/v2/sites/'
                },
                {
                    'caption'    => gettext('Content sites'),
                    'children'   => [],
                    'origin'     => 'partner',
                    'resource'   => 'context_on_site_campaign',
                    'type'       => 'item',
                    'url'        => '/v2/context/campaigns/',
                    'additional' => [
                        {
                            'caption'  => gettext('Archive'),
                            'children' => [],
                            'origin'   => 'partner',
                            'resource' => 'context_on_site_campaign',
                            'type'     => 'item',
                            'url'      => '/v2/context/campaigns/archive/'
                        },
                    ],
                },
                {
                    'caption'    => gettext('RTB blocks'),
                    'children'   => [],
                    'origin'     => 'partner',
                    'resource'   => 'context_on_site_rtb',
                    'type'       => 'item',
                    'url'        => '/v2/context/rtb/',
                    'additional' => [
                        {
                            'caption'  => gettext('Archive'),
                            'children' => [],
                            'origin'   => 'partner',
                            'resource' => 'context_on_site_rtb',
                            'type'     => 'item',
                            'url'      => '/v2/context/rtb/archive/'
                        },
                    ],
                },
                {
                    'caption'    => gettext('Content blocks'),
                    'children'   => [],
                    'origin'     => 'partner',
                    'resource'   => 'context_on_site_content',
                    'type'       => 'item',
                    'url'        => '/v2/context/content/',
                    'additional' => [
                        {
                            'caption'  => gettext('Archive'),
                            'children' => [],
                            'origin'   => 'partner',
                            'resource' => 'context_on_site_content',
                            'type'     => 'item',
                            'url'      => '/v2/context/content/archive/'
                        },
                    ],
                },
                {
                    'caption'    => gettext('Natural blocks'),
                    'children'   => [],
                    'origin'     => 'partner',
                    'resource'   => 'context_on_site_natural',
                    'type'       => 'item',
                    'url'        => '/v2/context/natural/',
                    'additional' => [
                        {
                            'caption'  => gettext('Archive'),
                            'children' => [],
                            'origin'   => 'partner',
                            'resource' => 'context_on_site_natural',
                            'type'     => 'item',
                            'url'      => '/v2/context/natural/archive/'
                        },
                    ],
                },
                {
                    'caption'    => gettext('Adblock blocks'),
                    'children'   => [],
                    'origin'     => 'partner',
                    'resource'   => 'context_on_site_adblock',
                    'type'       => 'item',
                    'url'        => '/v2/context/adblock/',
                    'additional' => [
                        {
                            'caption'  => gettext('Archive'),
                            'children' => [],
                            'origin'   => 'partner',
                            'resource' => 'context_on_site_adblock_deleted',
                            'type'     => 'item',
                            'url'      => '/v2/context/adblock/archive/'
                        },
                    ],
                },
            ],
            'icon' =>
'data:image/svg+xml,<svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg"><path fill-rule="evenodd" clip-rule="evenodd" d="M20 4.7H5C4.83431 4.7 4.7 4.83431 4.7 5V15C4.7 15.1657 4.83431 15.3 5 15.3H20C20.1657 15.3 20.3 15.1657 20.3 15V5C20.3 4.83431 20.1657 4.7 20 4.7ZM5 3C3.89543 3 3 3.89543 3 5V15C3 16.1046 3.89543 17 5 17H8.15V20H7.85C7.38056 20 7 20.3806 7 20.85C7 21.3194 7.38056 21.7 7.85 21.7H17.15C17.6194 21.7 18 21.3194 18 20.85C18 20.3806 17.6194 20 17.15 20H16.85V17H20C21.1046 17 22 16.1046 22 15V5C22 3.89543 21.1046 3 20 3H5ZM15.15 20V17H9.85V20H15.15Z" fill="white"/></svg>',
            'origin'            => 'partner',
            'resource'          => 'context_on_site_campaign',
            'type'              => 'group',
            'url'               => '/v2/sites/',
            'only_with_feature' => 'simple_menu',
        },
        # search_on_site
        {
            'caption'  => gettext('Search ads'),
            'children' => [
                {
                    'caption'    => gettext('Search ad platforms'),
                    'children'   => [],
                    'origin'     => 'partner',
                    'resource'   => 'search_on_site_campaign',
                    'type'       => 'item',
                    'url'        => '/v2/search/campaigns/',
                    'additional' => [
                        {
                            'caption'  => gettext('Archive'),
                            'children' => [],
                            'origin'   => 'partner',
                            'resource' => 'search_on_site_campaign_deleted',
                            'type'     => 'item',
                            'url'      => '/v2/search/campaigns/archive/'
                        },
                    ],
                },
                {
                    'caption'    => gettext('Premium Placement'),
                    'children'   => [],
                    'origin'     => 'partner',
                    'resource'   => 'search_on_site_premium',
                    'type'       => 'item',
                    'url'        => '/v2/search/premium/',
                    'additional' => [
                        {
                            'caption'  => gettext('Archive'),
                            'children' => [],
                            'origin'   => 'partner',
                            'resource' => 'search_on_site_premium_deleted',
                            'type'     => 'item',
                            'url'      => '/v2/search/premium/archive/'
                        },
                    ],
                },
                {
                    'caption'    => gettext('Search Direct'),
                    'children'   => [],
                    'origin'     => 'partner',
                    'resource'   => 'search_on_site_direct',
                    'type'       => 'item',
                    'url'        => '/v2/search/direct/',
                    'additional' => [
                        {
                            'caption'  => gettext('Archive'),
                            'children' => [],
                            'origin'   => 'partner',
                            'resource' => 'search_on_site_direct_deleted',
                            'type'     => 'item',
                            'url'      => '/v2/search/direct/archive/'
                        },
                    ],
                },
            ],
            'icon' =>
'data:image/svg+xml,<svg width="19" height="19" viewBox="0 0 19 19" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M7.5 15C9.21054 15 10.7873 14.4274 12.0491 13.4633L16.7929 18.2071C17.1834 18.5976 17.8166 18.5976 18.2071 18.2071C18.5976 17.8166 18.5976 17.1834 18.2071 16.7929L13.4633 12.0491C14.4274 10.7873 15 9.21054 15 7.5C15 3.35786 11.6421 0 7.5 0C3.35786 0 0 3.35786 0 7.5C0 11.6421 3.35786 15 7.5 15ZM13 7.5C13 10.5376 10.5376 13 7.5 13C4.46243 13 2 10.5376 2 7.5C2 4.46243 4.46243 2 7.5 2C10.5376 2 13 4.46243 13 7.5Z" fill="white"/></svg>',
            'origin'            => 'partner',
            'resource'          => 'search_on_site_campaign',
            'type'              => 'group',
            'url'               => '/v2/search/campaigns/',
            'only_with_feature' => 'simple_menu',
        },
        # video_an_site
        {
            'caption'  => gettext('Video ads'),
            'children' => [
                {
                    'caption'    => gettext('Video resources'),
                    'children'   => [],
                    'origin'     => 'partner',
                    'resource'   => 'video_an_site',
                    'type'       => 'item',
                    'url'        => '/v2/video/sites/',
                    'additional' => [
                        {
                            'caption'  => gettext('Archive'),
                            'children' => [],
                            'origin'   => 'partner',
                            'resource' => 'video_an_site_deleted',
                            'type'     => 'item',
                            'url'      => '/v2/video/sites/archive/'
                        }
                    ],
                },
                {
                    'caption'    => gettext('Instream blocks'),
                    'children'   => [],
                    'origin'     => 'partner',
                    'resource'   => 'video_an_site_instream',
                    'type'       => 'item',
                    'url'        => '/v2/video/instream/',
                    'additional' => [
                        {
                            'caption'  => gettext('Archive'),
                            'children' => [],
                            'origin'   => 'partner',
                            'resource' => 'video_an_site_instream_deleted',
                            'type'     => 'item',
                            'url'      => '/v2/video/instream/archive/'
                        }
                    ],
                },
                {
                    'caption'    => gettext('Inpage blocks'),
                    'children'   => [],
                    'origin'     => 'partner',
                    'resource'   => 'video_an_site_inpage',
                    'type'       => 'item',
                    'url'        => '/v2/video/inpage/',
                    'additional' => [
                        {
                            'caption'  => gettext('Archive'),
                            'children' => [],
                            'origin'   => 'partner',
                            'resource' => 'video_an_site_inpage_deleted',
                            'type'     => 'item',
                            'url'      => '/v2/video/inpage/archive/'
                        }
                    ],
                },
                {
                    'caption'    => gettext('Fullscreen&#13;blocks'),
                    'children'   => [],
                    'origin'     => 'partner',
                    'resource'   => 'video_an_site_fullscreen',
                    'type'       => 'item',
                    'url'        => '/v2/video/fullscreen/',
                    'additional' => [
                        {
                            'caption'  => gettext('Archive'),
                            'children' => [],
                            'origin'   => 'partner',
                            'resource' => 'video_an_site_fullscreen_deleted',
                            'type'     => 'item',
                            'url'      => '/v2/video/fullscreen/archive/'
                        }
                    ],
                },
                {
                    'caption'  => gettext('Video scenaries'),
                    'children' => [],
                    'origin'   => 'partner',
                    'resource' => 'video_scenaries',
                    'type'     => 'group',
                    'url'      => '/v2/video/scenaries/'
                },
                {
                    'caption'    => gettext('Category'),
                    'children'   => [],
                    'origin'     => 'partner',
                    'resource'   => undef,
                    'type'       => 'group',
                    'url'        => '/v2/video/reference/',
                    'additional' => [
                        {
                            'caption'  => gettext('Content'),
                            'children' => [],
                            'origin'   => 'partner',
                            'resource' => 'internal_content',
                            'type'     => 'item',
                            'url'      => '/v2/video/reference/content/'
                        },
                        {
                            'caption'  => gettext('Content moderation'),
                            'children' => [],
                            'origin'   => 'partner',
                            'resource' => 'partner_content',
                            'type'     => 'item',
                            'url'      => '/v2/video/reference/content/moderation/'
                        },
                        {
                            'caption'  => gettext('Genres'),
                            'children' => [],
                            'origin'   => 'partner',
                            'resource' => 'internal_genre',
                            'type'     => 'item',
                            'url'      => '/v2/video/reference/genre/'
                        },
                        {
                            'caption'  => gettext('Genres moderation'),
                            'children' => [],
                            'origin'   => 'partner',
                            'resource' => 'partner_genre',
                            'type'     => 'item',
                            'url'      => '/v2/video/reference/genre/moderation/'
                        },
                    ],
                },
            ],
            'icon' =>
'data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iMjQiIGhlaWdodD0iMjQiIHZpZXdCb3g9IjAgMCAyNCAyNCIgZmlsbD0ibm9uZSIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj4KPHBhdGggZmlsbC1ydWxlPSJldmVub2RkIiBjbGlwLXJ1bGU9ImV2ZW5vZGQiIGQ9Ik0xOS44NzQ5IDkuNzkxNjFDMjEuNTc0OSAxMC43NzMxIDIxLjU3NDkgMTMuMjI2OCAxOS44NzQ5IDE0LjIwODNMOC40NzQ5MSAyMC43OTAxQzYuNzc0OTEgMjEuNzcxNiA0LjY0OTkgMjAuNTQ0OCA0LjY0OTkgMTguNTgxOEw0LjY0OTkgNS40MTgxOUM0LjY0OTkgMy40NTUyIDYuNzc0OSAyLjIyODMyIDguNDc0OSAzLjIwOTgyTDE5Ljg3NDkgOS43OTE2MVpNMTkuMDI0OSAxMi43MzYxQzE5LjU5MTYgMTIuNDA4OSAxOS41OTE2IDExLjU5MSAxOS4wMjQ5IDExLjI2MzlMNy42MjQ5IDQuNjgyMDZDNy4wNTgyNCA0LjM1NDkgNi4zNDk5IDQuNzYzODUgNi4zNDk5IDUuNDE4MTlMNi4zNDk5IDE4LjU4MThDNi4zNDk5IDE5LjIzNjEgNy4wNTgyNCAxOS42NDUxIDcuNjI0OSAxOS4zMTc5TDE5LjAyNDkgMTIuNzM2MVoiIGZpbGw9IndoaXRlIi8+Cjwvc3ZnPg==',
            'origin'            => 'partner',
            'resource'          => 'video_an_site',
            'type'              => 'group',
            'url'               => '',
            'only_with_feature' => 'simple_menu',
        },
        # mobile_app
        {
            'caption' => gettext('In-app ads'),
            'icon' =>
'data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iMjQiIGhlaWdodD0iMjQiIHZpZXdCb3g9IjAgMCAyNCAyNCIgZmlsbD0ibm9uZSIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj4KPHBhdGggZmlsbC1ydWxlPSJldmVub2RkIiBjbGlwLXJ1bGU9ImV2ZW5vZGQiIGQ9Ik04LjcwMDA1IDEuMjk5OTlIMTUuM0MxNy4xNzc4IDEuMjk5OTkgMTguNyAyLjgyMjIyIDE4LjcgNC42OTk5OVYxOS4zQzE4LjcgMjEuMTc3OCAxNy4xNzc4IDIyLjcgMTUuMyAyMi43SDguNzAwMDVDNi44MjIyOCAyMi43IDUuMzAwMDUgMjEuMTc3OCA1LjMwMDA1IDE5LjNWNC42OTk5OUM1LjMwMDA1IDIuODIyMjIgNi44MjIyOCAxLjI5OTk5IDguNzAwMDUgMS4yOTk5OVpNOC43MDAwNSAyLjk5OTk5QzcuNzYxMTYgMi45OTk5OSA3LjAwMDA1IDMuNzYxMTEgNy4wMDAwNSA0LjY5OTk5VjE5LjNDNy4wMDAwNSAyMC4yMzg5IDcuNzYxMTYgMjEgOC43MDAwNSAyMUgxNS4zQzE2LjIzODkgMjEgMTcgMjAuMjM4OSAxNyAxOS4zVjQuNjk5OTlDMTcgMy43NjExIDE2LjIzODkgMi45OTk5OSAxNS4zIDIuOTk5OTlIMTRDMTQgMy41NTIyNyAxMy41NTIzIDMuOTk5OTkgMTMgMy45OTk5OUgxMUMxMC40NDc4IDMuOTk5OTkgMTAgMy41NTIyNyAxMCAyLjk5OTk5SDguNzAwMDVaTTEzIDE4QzEzIDE4LjU1MjMgMTIuNTUyMyAxOSAxMiAxOUMxMS40NDc4IDE5IDExIDE4LjU1MjMgMTEgMThDMTEgMTcuNDQ3NyAxMS40NDc4IDE3IDEyIDE3QzEyLjU1MjMgMTcgMTMgMTcuNDQ3NyAxMyAxOFoiIGZpbGw9IndoaXRlIi8+Cjwvc3ZnPg==',
            'origin'            => 'partner',
            'resource'          => 'mobile_app_settings',
            'type'              => 'item',
            'url'               => '/v2/inapp/',
            'only_with_feature' => 'simple_menu',
            'only_no_feature'   => ['new_mobmed', 'inapp_upd_metrics', 'inapp_ab_exp'],
        },
        {
            'caption'  => gettext('In-app ads'),
            'children' => [
                {
                    'caption'  => gettext('Apps and Ad units'),
                    'children' => [],
                    'origin'   => 'partner',
                    'resource' => 'mobile_app_settings',
                    'type'     => 'item',
                    'url'      => '/v2/inapp/',
                },
                {
                    'caption'           => gettext('Networks'),
                    'children'          => [],
                    'origin'            => 'partner',
                    'resource'          => 'mobile_app_settings',
                    'type'              => 'item',
                    'url'               => '/v2/inapp/network',
                    'only_with_feature' => 'inapp_upd_metrics',
                },
                {
                    'caption'           => gettext('A/B tests'),
                    'children'          => [],
                    'origin'            => 'partner',
                    'resource'          => 'mobile_app_settings',
                    'type'              => 'item',
                    'url'               => '/v2/inapp/experiments',
                    'only_with_feature' => 'inapp_ab_exp',
                },
            ],
            'icon' =>
'data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iMjQiIGhlaWdodD0iMjQiIHZpZXdCb3g9IjAgMCAyNCAyNCIgZmlsbD0ibm9uZSIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj4KPHBhdGggZmlsbC1ydWxlPSJldmVub2RkIiBjbGlwLXJ1bGU9ImV2ZW5vZGQiIGQ9Ik04LjcwMDA1IDEuMjk5OTlIMTUuM0MxNy4xNzc4IDEuMjk5OTkgMTguNyAyLjgyMjIyIDE4LjcgNC42OTk5OVYxOS4zQzE4LjcgMjEuMTc3OCAxNy4xNzc4IDIyLjcgMTUuMyAyMi43SDguNzAwMDVDNi44MjIyOCAyMi43IDUuMzAwMDUgMjEuMTc3OCA1LjMwMDA1IDE5LjNWNC42OTk5OUM1LjMwMDA1IDIuODIyMjIgNi44MjIyOCAxLjI5OTk5IDguNzAwMDUgMS4yOTk5OVpNOC43MDAwNSAyLjk5OTk5QzcuNzYxMTYgMi45OTk5OSA3LjAwMDA1IDMuNzYxMTEgNy4wMDAwNSA0LjY5OTk5VjE5LjNDNy4wMDAwNSAyMC4yMzg5IDcuNzYxMTYgMjEgOC43MDAwNSAyMUgxNS4zQzE2LjIzODkgMjEgMTcgMjAuMjM4OSAxNyAxOS4zVjQuNjk5OTlDMTcgMy43NjExIDE2LjIzODkgMi45OTk5OSAxNS4zIDIuOTk5OTlIMTRDMTQgMy41NTIyNyAxMy41NTIzIDMuOTk5OTkgMTMgMy45OTk5OUgxMUMxMC40NDc4IDMuOTk5OTkgMTAgMy41NTIyNyAxMCAyLjk5OTk5SDguNzAwMDVaTTEzIDE4QzEzIDE4LjU1MjMgMTIuNTUyMyAxOSAxMiAxOUMxMS40NDc4IDE5IDExIDE4LjU1MjMgMTEgMThDMTEgMTcuNDQ3NyAxMS40NDc4IDE3IDEyIDE3QzEyLjU1MjMgMTcgMTMgMTcuNDQ3NyAxMyAxOFoiIGZpbGw9IndoaXRlIi8+Cjwvc3ZnPg==',
            'origin'            => 'partner',
            'resource'          => 'mobile_app_settings',
            'type'              => 'item',
            'url'               => '/v2/inapp/',
            'only_with_feature' => ['simple_menu', 'new_mobmed'],
        },
        # internal_context_on_site
        {
            'caption'  => gettext('Website ads (Yandex)'),
            'children' => [
                {
                    'caption'    => gettext('Content sites'),
                    'children'   => [],
                    'origin'     => 'partner',
                    'resource'   => 'internal_context_on_site_campaign',
                    'type'       => 'item',
                    'url'        => '/v2/internal/context/campaigns/',
                    'additional' => [
                        {
                            'caption'  => gettext('Archive'),
                            'children' => [],
                            'origin'   => 'partner',
                            'resource' => 'internal_context_on_site_campaign',
                            'type'     => 'item',
                            'url'      => '/v2/internal/context/campaigns/archive/'
                        },
                    ],
                },
                {
                    'caption'    => gettext('RTB blocks'),
                    'children'   => [],
                    'origin'     => 'partner',
                    'resource'   => 'internal_context_on_site_rtb',
                    'type'       => 'item',
                    'url'        => '/v2/internal/context/rtb/',
                    'additional' => [
                        {
                            'caption'  => gettext('Archive'),
                            'children' => [],
                            'origin'   => 'partner',
                            'resource' => 'internal_context_on_site_rtb',
                            'type'     => 'item',
                            'url'      => '/v2/internal/context/rtb/archive/'
                        },
                    ],
                },
                {
                    'caption'    => gettext('Content blocks'),
                    'children'   => [],
                    'origin'     => 'partner',
                    'resource'   => 'internal_context_on_site_content',
                    'type'       => 'item',
                    'url'        => '/v2/internal/context/content/',
                    'additional' => [
                        {
                            'caption'  => gettext('Archive'),
                            'children' => [],
                            'origin'   => 'partner',
                            'resource' => 'internal_context_on_site_content',
                            'type'     => 'item',
                            'url'      => '/v2/internal/context/content/archive/'
                        },
                    ],
                },
                {
                    'caption'    => gettext('Natural blocks'),
                    'children'   => [],
                    'origin'     => 'partner',
                    'resource'   => 'internal_context_on_site_natural',
                    'type'       => 'item',
                    'url'        => '/v2/internal/context/natural/',
                    'additional' => [
                        {
                            'caption'  => gettext('Archive'),
                            'children' => [],
                            'origin'   => 'partner',
                            'resource' => 'internal_context_on_site_natural',
                            'type'     => 'item',
                            'url'      => '/v2/internal/context/natural/archive/'
                        },
                    ],
                },
                {
                    'caption'    => gettext('Direct blocks'),
                    'children'   => [],
                    'origin'     => 'partner',
                    'resource'   => 'internal_context_on_site_direct',
                    'type'       => 'item',
                    'url'        => '/v2/internal/context/direct/',
                    'additional' => [
                        {
                            'caption'  => gettext('Archive'),
                            'children' => [],
                            'origin'   => 'partner',
                            'resource' => 'internal_context_on_site_direct_deleted',
                            'type'     => 'item',
                            'url'      => '/v2/internal/context/direct/archive/'
                        },
                    ],
                },
                {
                    'caption'    => gettext('Stripe'),
                    'children'   => [],
                    'origin'     => 'partner',
                    'resource'   => 'internal_context_on_site_stripe',
                    'type'       => 'item',
                    'url'        => '/internal_context_on_site_stripe/list/',
                    'additional' => [
                        {
                            'caption'  => gettext('Archive'),
                            'children' => [],
                            'origin'   => 'partner',
                            'resource' => 'internal_context_on_site_stripe_deleted',
                            'type'     => 'sub-item',
                            'url'      => '/internal_context_on_site_stripe/list/archive/'
                        },
                    ],
                },
                {
                    'caption'    => gettext('Search pages (Yandex)'),
                    'children'   => [],
                    'origin'     => 'partner',
                    'resource'   => 'internal_search_on_site_campaign',
                    'type'       => 'item',
                    'url'        => '/v2/internal/search/campaigns/',
                    'additional' => [
                        {
                            'caption'  => gettext('Archive'),
                            'children' => [],
                            'origin'   => 'partner',
                            'resource' => 'internal_search_on_site_campaign_deleted',
                            'type'     => 'item',
                            'url'      => '/v2/internal/search/campaigns/archive/'
                        },
                    ],
                },
            ],
            'icon' =>
'data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iMjQiIGhlaWdodD0iMjQiIHZpZXdCb3g9IjAgMCAyNCAyNCIgZmlsbD0ibm9uZSIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj4KPHBhdGggZmlsbC1ydWxlPSJldmVub2RkIiBjbGlwLXJ1bGU9ImV2ZW5vZGQiIGQ9Ik0yMCA0LjdINUM0LjgzNDMxIDQuNyA0LjcgNC44MzQzMSA0LjcgNVYxNUM0LjcgMTUuMTY1NyA0LjgzNDMxIDE1LjMgNSAxNS4zSDIwQzIwLjE2NTcgMTUuMyAyMC4zIDE1LjE2NTcgMjAuMyAxNVY1QzIwLjMgNC44MzQzMSAyMC4xNjU3IDQuNyAyMCA0LjdaTTUgM0MzLjg5NTQzIDMgMyAzLjg5NTQzIDMgNVYxNUMzIDE2LjEwNDYgMy44OTU0MyAxNyA1IDE3SDguMTVWMjBINy44NUM3LjM4MDU2IDIwIDcgMjAuMzgwNiA3IDIwLjg1QzcgMjEuMzE5NCA3LjM4MDU2IDIxLjcgNy44NSAyMS43SDE3LjE1QzE3LjYxOTQgMjEuNyAxOCAyMS4zMTk0IDE4IDIwLjg1QzE4IDIwLjM4MDYgMTcuNjE5NCAyMCAxNy4xNSAyMEgxNi44NVYxN0gyMEMyMS4xMDQ2IDE3IDIyIDE2LjEwNDYgMjIgMTVWNUMyMiAzLjg5NTQzIDIxLjEwNDYgMyAyMCAzSDVaTTE1LjE1IDIwVjE3SDkuODVWMjBIMTUuMTVaIiBmaWxsPSJ3aGl0ZSIvPgo8cmVjdCB4PSIxNCIgd2lkdGg9IjEwIiBoZWlnaHQ9IjEwIiByeD0iNSIgZmlsbD0iI0REMDAwMCIvPgo8cGF0aCBkPSJNMTkuMTYwNiA1LjU3MjVDMTkuNDY0OSA2LjIzOTE3IDE5LjU2NjQgNi40NzEwNiAxOS41NjY0IDcuMjcxNzhWOC4zMzMzN0gxOC40Nzk0VjYuNTQzNTJMMTYuNDI4NyAyLjA4MzM3SDE3LjU2MjhMMTkuMTYwNiA1LjU3MjVaTTIwLjUwMTIgMi4wODMzN0wxOS4xNzE1IDUuMTA1MTFIMjAuMjc2NUwyMS42MDk5IDIuMDgzMzdIMjAuNTAxMloiIGZpbGw9IndoaXRlIi8+Cjwvc3ZnPg==',
            'origin'            => 'partner',
            'resource'          => 'internal_context_on_site_campaign',
            'type'              => 'group',
            'url'               => '',
            'only_with_feature' => 'simple_menu',
        },
        # internal_mobile_app
        # simple
        {
            'caption' => gettext('In-app Adds (Yandex)'),
            'icon' =>
'data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iMjQiIGhlaWdodD0iMjQiIHZpZXdCb3g9IjAgMCAyNCAyNCIgZmlsbD0ibm9uZSIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj48cGF0aCBmaWxsLXJ1bGU9ImV2ZW5vZGQiIGNsaXAtcnVsZT0iZXZlbm9kZCIgZD0iTTguNzAwMDUgMS4zMDAwNUgxNS4zQzE3LjE3NzggMS4zMDAwNSAxOC43IDIuODIyMjggMTguNyA0LjcwMDA1VjE5LjNDMTguNyAyMS4xNzc4IDE3LjE3NzggMjIuNzAwMSAxNS4zIDIyLjcwMDFIOC43MDAwNUM2LjgyMjI4IDIyLjcwMDEgNS4zMDAwNSAyMS4xNzc4IDUuMzAwMDUgMTkuM1Y0LjcwMDA1QzUuMzAwMDUgMi44MjIyOCA2LjgyMjI4IDEuMzAwMDUgOC43MDAwNSAxLjMwMDA1Wk04LjcwMDA1IDMuMDAwMDVDNy43NjExNiAzLjAwMDA1IDcuMDAwMDUgMy43NjExNyA3LjAwMDA1IDQuNzAwMDVWMTkuM0M3LjAwMDA1IDIwLjIzODkgNy43NjExNiAyMSA4LjcwMDA1IDIxSDE1LjNDMTYuMjM4OSAyMSAxNyAyMC4yMzg5IDE3IDE5LjNWNC43MDAwNUMxNyAzLjc2MTE2IDE2LjIzODkgMy4wMDAwNSAxNS4zIDMuMDAwMDVIMTVDMTUgMy41NTIzMyAxNC41NTIzIDQuMDAwMDUgMTQgNC4wMDAwNUgxMUMxMC40NDc4IDQuMDAwMDUgMTAgMy41NTIzMyAxMCAzLjAwMDA1SDguNzAwMDVaTTEzIDE4QzEzIDE4LjU1MjMgMTIuNTUyMyAxOSAxMiAxOUMxMS40NDc4IDE5IDExIDE4LjU1MjMgMTEgMThDMTEgMTcuNDQ3OCAxMS40NDc4IDE3IDEyIDE3QzEyLjU1MjMgMTcgMTMgMTcuNDQ3OCAxMyAxOFoiIGZpbGw9IndoaXRlIi8+PHJlY3QgeD0iMTQiIHdpZHRoPSIxMCIgaGVpZ2h0PSIxMCIgcng9IjUiIGZpbGw9IiNERDAwMDAiLz48cGF0aCBkPSJNMTkuMTYwNiA1LjU3MjVDMTkuNDY0OSA2LjIzOTE3IDE5LjU2NjQgNi40NzEwNiAxOS41NjY0IDcuMjcxNzhWOC4zMzMzN0gxOC40Nzk0VjYuNTQzNTJMMTYuNDI4NyAyLjA4MzM3SDE3LjU2MjhMMTkuMTYwNiA1LjU3MjVaTTIwLjUwMTIgMi4wODMzN0wxOS4xNzE1IDUuMTA1MTFIMjAuMjc2NUwyMS42MDk5IDIuMDgzMzdIMjAuNTAxMloiIGZpbGw9IndoaXRlIi8+PC9zdmc+',
            'origin'            => 'partner',
            'resource'          => 'internal_mobile_app',
            'type'              => 'item',
            'url'               => '/v2/internal/inapp/',
            'only_with_feature' => ['simple_menu', 'internal_simple_inapp'],
        },
        # non-simple
        {
            'caption'  => gettext('In-app Adds (Yandex)'),
            'children' => [
                {
                    'caption'    => gettext('Applications'),
                    'children'   => [],
                    'origin'     => 'partner',
                    'resource'   => 'internal_mobile_app',
                    'type'       => 'item',
                    'url'        => '/v2/internal/mobile/applications/',
                    'additional' => [
                        {
                            'caption'  => gettext('Archive'),
                            'children' => [],
                            'origin'   => 'partner',
                            'resource' => 'internal_mobile_app_deleted',
                            'type'     => 'item',
                            'url'      => '/v2/internal/mobile/applications/archive/'
                        }
                    ]
                },
                {
                    'caption'    => gettext('Mobile RTB blocks'),
                    'children'   => [],
                    'origin'     => 'partner',
                    'resource'   => 'internal_mobile_app_rtb',
                    'type'       => 'item',
                    'url'        => '/v2/internal/mobile/rtb/',
                    'additional' => [
                        {
                            'caption'  => gettext('Archive'),
                            'children' => [],
                            'origin'   => 'partner',
                            'resource' => 'internal_mobile_app_rtb_deleted',
                            'type'     => 'item',
                            'url'      => '/v2/internal/mobile/rtb/archive/'
                        }
                    ]
                },
            ],
            'icon' =>
'data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iMjQiIGhlaWdodD0iMjQiIHZpZXdCb3g9IjAgMCAyNCAyNCIgZmlsbD0ibm9uZSIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj48cGF0aCBmaWxsLXJ1bGU9ImV2ZW5vZGQiIGNsaXAtcnVsZT0iZXZlbm9kZCIgZD0iTTguNzAwMDUgMS4zMDAwNUgxNS4zQzE3LjE3NzggMS4zMDAwNSAxOC43IDIuODIyMjggMTguNyA0LjcwMDA1VjE5LjNDMTguNyAyMS4xNzc4IDE3LjE3NzggMjIuNzAwMSAxNS4zIDIyLjcwMDFIOC43MDAwNUM2LjgyMjI4IDIyLjcwMDEgNS4zMDAwNSAyMS4xNzc4IDUuMzAwMDUgMTkuM1Y0LjcwMDA1QzUuMzAwMDUgMi44MjIyOCA2LjgyMjI4IDEuMzAwMDUgOC43MDAwNSAxLjMwMDA1Wk04LjcwMDA1IDMuMDAwMDVDNy43NjExNiAzLjAwMDA1IDcuMDAwMDUgMy43NjExNyA3LjAwMDA1IDQuNzAwMDVWMTkuM0M3LjAwMDA1IDIwLjIzODkgNy43NjExNiAyMSA4LjcwMDA1IDIxSDE1LjNDMTYuMjM4OSAyMSAxNyAyMC4yMzg5IDE3IDE5LjNWNC43MDAwNUMxNyAzLjc2MTE2IDE2LjIzODkgMy4wMDAwNSAxNS4zIDMuMDAwMDVIMTVDMTUgMy41NTIzMyAxNC41NTIzIDQuMDAwMDUgMTQgNC4wMDAwNUgxMUMxMC40NDc4IDQuMDAwMDUgMTAgMy41NTIzMyAxMCAzLjAwMDA1SDguNzAwMDVaTTEzIDE4QzEzIDE4LjU1MjMgMTIuNTUyMyAxOSAxMiAxOUMxMS40NDc4IDE5IDExIDE4LjU1MjMgMTEgMThDMTEgMTcuNDQ3OCAxMS40NDc4IDE3IDEyIDE3QzEyLjU1MjMgMTcgMTMgMTcuNDQ3OCAxMyAxOFoiIGZpbGw9IndoaXRlIi8+PHJlY3QgeD0iMTQiIHdpZHRoPSIxMCIgaGVpZ2h0PSIxMCIgcng9IjUiIGZpbGw9IiNERDAwMDAiLz48cGF0aCBkPSJNMTkuMTYwNiA1LjU3MjVDMTkuNDY0OSA2LjIzOTE3IDE5LjU2NjQgNi40NzEwNiAxOS41NjY0IDcuMjcxNzhWOC4zMzMzN0gxOC40Nzk0VjYuNTQzNTJMMTYuNDI4NyAyLjA4MzM3SDE3LjU2MjhMMTkuMTYwNiA1LjU3MjVaTTIwLjUwMTIgMi4wODMzN0wxOS4xNzE1IDUuMTA1MTFIMjAuMjc2NUwyMS42MDk5IDIuMDgzMzdIMjAuNTAxMloiIGZpbGw9IndoaXRlIi8+PC9zdmc+',
            'origin'            => 'partner',
            'resource'          => 'internal_mobile_app',
            'type'              => 'group',
            'url'               => '/v2/internal/mobile/applications/',
            'only_with_feature' => 'simple_menu',
            'only_no_feature'   => 'internal_simple_inapp',
        },
        # DSP
        {
            'caption'  => gettext('Advertising platforms'),
            'children' => [],
            'icon' =>
'data:image/svg+xml,<svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M15.7999 4.32502C15.4999 4.62502 14.9999 4.62502 14.7999 4.32502L13.0999 2.32502V6.82502C13.0999 7.22502 12.7999 7.52502 12.3999 7.52502C11.9999 7.52502 11.6999 7.22502 11.6999 6.82502V2.32502L9.9999 4.32502C9.6999 4.62502 9.1999 4.62502 8.9999 4.32502C8.6999 4.02502 8.6999 3.52502 8.9999 3.32502L11.8999 0.425024C12.1999 0.125024 12.6999 0.125024 12.8999 0.425024L15.7999 3.32502C16.0999 3.62502 16.0999 4.02502 15.7999 4.32502Z" fill="white"/><path d="M18.8 14.2C19.2418 14.2 19.6 13.8418 19.6 13.4C19.6 12.9581 19.2418 12.6 18.8 12.6C18.3582 12.6 18 12.9581 18 13.4C18 13.8418 18.3582 14.2 18.8 14.2Z" fill="white"/><path d="M18.8 20.1C19.2418 20.1 19.6 19.7418 19.6 19.3C19.6 18.8582 19.2418 18.5 18.8 18.5C18.3582 18.5 18 18.8582 18 19.3C18 19.7418 18.3582 20.1 18.8 20.1Z" fill="white"/><path fill-rule="evenodd" clip-rule="evenodd" d="M5 11.2H20C20.7 11.2 21.3 11.8 21.3 12.5V14.1C21.3 14.8 20.7 15.4 20 15.4H5C4.3 15.4 3.7 14.8 3.7 14.1V12.5C3.7 11.8 4.3 11.2 5 11.2ZM2 12.5C2 10.8 3.3 9.5 5 9.5H20C21.7 9.5 23 10.8 23 12.5V14.1C23 14.9 22.6 15.7 22.1 16.3C22.7 16.8 23 17.6 23 18.5V20.1C23 21.8 21.7 23.1 20 23.1H5C3.3 23.1 2 21.8 2 20.1V18.5C2 17.7 2.4 16.9 2.9 16.3C2.4 15.8 2 15 2 14.2V12.5ZM21.3 18.5V20.1C21.3 20.8 20.7 21.4 20 21.4H5C4.3 21.4 3.7 20.8 3.7 20.1V18.5C3.7 17.8 4.3 17.2 5 17.2H12.5H20C20.7 17.2 21.3 17.8 21.3 18.5Z" fill="white"/></svg>',
            'origin'            => 'partner',
            'resource'          => 'dsp',
            'type'              => 'item',
            'url'               => '/v2/dsp/connections/',
            'only_with_feature' => 'simple_menu',
        },
        # SSP
        {
            'caption'  => gettext('SSP'),
            'children' => [
                {
                    'caption'  => gettext('Mobile applications'),
                    'children' => [],
                    'origin'   => 'partner',
                    'resource' => 'ssp_link_mobile_app',
                    'type'     => 'item',
                    'url'      => '/v2/ssp/moderation/applications/'
                },
                {
                    'caption'  => gettext('Content sites'),
                    'children' => [],
                    'origin'   => 'partner',
                    'resource' => 'ssp_link_context_rtb',
                    'type'     => 'item',
                    'url'      => '/v2/ssp/moderation/campaigns/'
                },
                {
                    'caption'  => gettext('Video platforms'),
                    'children' => [],
                    'origin'   => 'partner',
                    'resource' => 'ssp_link_video_app',
                    'type'     => 'item',
                    'url'      => '/v2/ssp/moderation/video/'
                }
            ],
            'icon' =>
'data:image/svg+xml,<svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M15.7999 3.39995C15.4999 3.09995 14.9999 3.09995 14.7999 3.39995L13.0999 5.39995V0.899951C13.0999 0.499951 12.7999 0.199951 12.3999 0.199951C11.9999 0.199951 11.6999 0.499951 11.6999 0.899951V5.39995L9.9999 3.39995C9.6999 3.09995 9.1999 3.09995 8.9999 3.39995C8.6999 3.69995 8.6999 4.19995 8.9999 4.39995L11.8999 7.29995C12.1999 7.59995 12.6999 7.59995 12.8999 7.29995L15.7999 4.39995C16.0999 4.09995 16.0999 3.69995 15.7999 3.39995Z" fill="white"/><path d="M18.8 14.2C19.2418 14.2 19.6 13.8418 19.6 13.4C19.6 12.9581 19.2418 12.6 18.8 12.6C18.3582 12.6 18 12.9581 18 13.4C18 13.8418 18.3582 14.2 18.8 14.2Z" fill="white"/><path d="M18.8 20.1C19.2418 20.1 19.6 19.7418 19.6 19.3C19.6 18.8582 19.2418 18.5 18.8 18.5C18.3582 18.5 18 18.8582 18 19.3C18 19.7418 18.3582 20.1 18.8 20.1Z" fill="white"/><path fill-rule="evenodd" clip-rule="evenodd" d="M5 11.2H20C20.7 11.2 21.3 11.8 21.3 12.5V14.1C21.3 14.8 20.7 15.4 20 15.4H5C4.3 15.4 3.7 14.8 3.7 14.1V12.5C3.7 11.8 4.3 11.2 5 11.2ZM2 12.5C2 10.8 3.3 9.5 5 9.5H20C21.7 9.5 23 10.8 23 12.5V14.1C23 14.9 22.6 15.7 22.1 16.3C22.7 16.8 23 17.6 23 18.5V20.1C23 21.8 21.7 23.1 20 23.1H5C3.3 23.1 2 21.8 2 20.1V18.5C2 17.7 2.4 16.9 2.9 16.3C2.4 15.8 2 15 2 14.2V12.5ZM21.3 18.5V20.1C21.3 20.8 20.7 21.4 20 21.4H5C4.3 21.4 3.7 20.8 3.7 20.1V18.5C3.7 17.8 4.3 17.2 5 17.2H12.5H20C20.7 17.2 21.3 17.8 21.3 18.5Z" fill="white"/></svg>',
            'origin'            => 'partner',
            'resource'          => undef,
            'type'              => 'group',
            'url'               => '',
            'only_with_feature' => 'simple_menu',
        },
        # statistics
        {
            'caption'  => gettext('Statistics'),
            'children' => [],
            'icon' =>
'data:image/svg+xml,<svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg"><path fill-rule="evenodd" clip-rule="evenodd" d="M10.8 3.7C10.5746 3.7 10.3584 3.78955 10.199 3.94896C10.0396 4.10837 9.95 4.32457 9.95 4.55V12.29V19.88H14.36V6.53H14.36V4.55C14.36 4.32457 14.2704 4.10837 14.111 3.94896C13.9516 3.78955 13.7354 3.7 13.51 3.7H10.8ZM16.06 6.53V4.55C16.06 3.8737 15.7913 3.2251 15.3131 2.74688C14.8349 2.26866 14.1863 2 13.51 2H10.8C10.1237 2 9.4751 2.26866 8.99688 2.74688C8.51866 3.2251 8.25 3.8737 8.25 4.55V12.29H4.55C3.8737 12.29 3.2251 12.5587 2.74688 13.0369C2.26866 13.5151 2 14.1637 2 14.84V18.73C2 19.4859 2.30027 20.2108 2.83475 20.7453C3.36922 21.2797 4.09413 21.58 4.85 21.58L8.25 21.58L9.95 21.58L14.36 21.58L16.06 21.58L18.8 21.58C19.5559 21.58 20.2808 21.2797 20.8152 20.7453C21.3497 20.2108 21.65 19.4859 21.65 18.73V9.08C21.65 8.4037 21.3813 7.7551 20.9031 7.27688C20.4249 6.79866 19.7763 6.53 19.1 6.53H16.06ZM16.06 19.88V8.23H19.1C19.3254 8.23 19.5416 8.31955 19.701 8.47896C19.8604 8.63836 19.95 8.85456 19.95 9.08V18.73C19.95 19.035 19.8288 19.3275 19.6132 19.5432C19.3975 19.7588 19.105 19.88 18.8 19.88H16.06ZM4.55 13.99H8.25V19.88H4.85C4.545 19.88 4.25249 19.7588 4.03683 19.5432C3.82116 19.3275 3.7 19.035 3.7 18.73V14.84C3.7 14.6146 3.78955 14.3984 3.94896 14.239C4.10837 14.0796 4.32457 13.99 4.55 13.99Z" fill="white"/></svg>',
            'origin'            => 'partner',
            'resource'          => 'statistics',
            'type'              => 'item',
            'url'               => '/v2/statistics/constructor/',
            'only_with_feature' => 'simple_menu',
        },
        # Extras
        {
            'caption'  => gettext('Extras'),
            'children' => [
                {
                    'caption'  => gettext('Business rules'),
                    'children' => [],
                    'icon' =>
'data:image/svg+xml;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHdpZHRoPSIyNCIgaGVpZ2h0PSIyNCIgZmlsbD0ibm9uZSI+PHBhdGggZmlsbD0iI2ZmZiIgZmlsbC1ydWxlPSJldmVub2RkIiBkPSJNNSA1YzAtMS4xLjktMiAyLTJoNy4xN2EyIDIgMCAwIDEgMS40Mi41OUwxOC40IDYuNGEyIDIgMCAwIDEgLjYgMS40M3YyLjI0QTcgNyAwIDAgMCAxMi4yNSAyMUg3YTIgMiAwIDAgMS0yLTJWNXptMTIuNTcgMi41N2wtMy4xNC0zLjE0YS4yNS4yNSAwIDAgMC0uNDMuMTd2My4xNWMwIC4xNC4xMS4yNS4yNS4yNWgzLjE1Yy4yMiAwIC4zMy0uMjcuMTctLjQzek0yMyAxN2E1IDUgMCAxIDEtMTAgMCA1IDUgMCAwIDEgMTAgMHptLTUuNzggMi41YS44My44MyAwIDAgMS0uNTgtLjI1Yy0uOTItLjk0LTIuMi0yLjItMi4yLTIuMmEuNjIuNjIgMCAxIDEgLjg3LS44OGwxLjkxIDEuOTEgMy40LTMuNGEuNjIuNjIgMCAxIDEgLjg4Ljg4bC0zLjcgMy42OWEuNzcuNzcgMCAwIDEtLjU4LjI1eiIgY2xpcC1ydWxlPSJldmVub2RkIi8+PC9zdmc+',
                    'origin'   => 'partner',
                    'resource' => 'business_rules',
                    'type'     => 'item',
                    'url'      => '/v2/business-rules/'
                },
                {
                    'caption'  => gettext('Mass editing'),
                    'children' => [],
                    'icon' =>
'data:image/svg+xml;base64,PHN2ZyB4bWxucz0naHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmcnIHdpZHRoPScyNCcgaGVpZ2h0PScyNCcgZmlsbD0nbm9uZSc+PHBhdGggZmlsbD0nI2ZmZicgZmlsbC1ydWxlPSdldmVub2RkJyBkPSdNNSAzYTIgMiAwIDAgMC0yIDJ2MTRjMCAxLjEuOSAyIDIgMmgxNGEyIDIgMCAwIDAgMi0yVjVhMiAyIDAgMCAwLTItMkg1em02Ljg3IDUuODFhMy4wNiAzLjA2IDAgMSAwIDAgNi4xMiAzLjA2IDMuMDYgMCAwIDAgMC02LjEyem01LjI1IDIuODhsLS42Ni4xMnYuMDZjMCAuNy0uMTUgMS4zNS0uNDMgMS45NGwuNTYuMzhhLjc2Ljc2IDAgMCAxLS44NyAxLjI2bC0uNTYtLjM4YTQuNiA0LjYgMCAwIDEtMS43MSAxLjExbC4xMi42NmEuNzcuNzcgMCAwIDEtMS41LjI4bC0uMTMtLjY2aC0uMDdjLS42OSAwLTEuMzQtLjE1LTEuOTMtLjQzbC0uMzguNTZhLjc2Ljc2IDAgMCAxLTEuMjYtLjg3bC4zOC0uNTZhNC41OCA0LjU4IDAgMCAxLTEuMTEtMS43MmwtLjY3LjEzYS43Ni43NiAwIDEgMS0uMjctMS41bC42Ni0uMTN2LS4wN2MwLS42OS4xNS0xLjM0LjQzLTEuOTNsLS41Ni0uMzhhLjc2Ljc2IDAgMCAxIC44Ny0xLjI2bC41NS4zOGE0LjYgNC42IDAgMCAxIDEuNzItMS4xMWwtLjEyLS42N2EuNzcuNzcgMCAwIDEgMS41LS4yN2wuMTMuNjZoLjA2Yy43IDAgMS4zNS4xNSAxLjk0LjQzbC4zOC0uNTZhLjc2Ljc2IDAgMCAxIDEuMjYuODdsLS4zOC41NWMuNDkuNDguODcgMS4wNyAxLjExIDEuNzJsLjY2LS4xMmEuNzcuNzcgMCAwIDEgLjI4IDEuNXonIGNsaXAtcnVsZT0nZXZlbm9kZCcvPjwvc3ZnPg==',
                    'origin'   => 'partner',
                    'resource' => 'block_presets',
                    'type'     => 'item',
                    'url'      => '/v2/block-presets/'
                },
                {
                    'caption'  => gettext('Users'),
                    'children' => [],
                    'icon' =>
'data:image/svg+xml;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHdpZHRoPSIyNCIgaGVpZ2h0PSIyNCIgZmlsbD0ibm9uZSI+PHBhdGggZmlsbD0iI2ZmZiIgZmlsbC1ydWxlPSJldmVub2RkIiBkPSJNNS4yNiAxMy44M2MtLjIgMC0uNi0uMTgtLjg3LS43OS0uMjYtLjYxLS4zLS44NC0uMzgtMS4zNS0uMDctLjUuMTgtLjczLjQyLS43OC4yMy0uMDUuNC0uMDIuNS4wOS0uMDMtLjMtLjUyLTMuMTEtLjIxLTMuNzJhMi41NSAyLjU1IDAgMCAxIDIuMDQtMS40IDguOSA4LjkgMCAwIDEgMi4zMSAwYy0uNS40Ni0uNTggMS4zLS41NiAxLjgzLjAyLjU0LjIgMS40OC4yNiAxLjg4LTEuMDIuMjUtMS4xNSAxLjQ5LS44NiAyLjk2LjMgMS40NyAxLjY2IDMuMjUgMi40MyAzLjFsLjIuNTNjLS4zMy41OS0uNTcuNzUtMS4xLjg3LS4zNy4wOC0uOS4wNC0xLjMtLjA3YTQuNTIgNC41MiAwIDAgMS0yLjMyLTIuMzNjLS4xMi0uMzEtLjI5LS45NS0uMjktLjk1cy0uMDguMTMtLjI3LjEzem01LjQ3LjM4Yy0xLjMxLS4zNy0xLjc1LTMuMTQtMS4xNy0zLjMzLjU4LS4yLjg0IDAgLjg0LjEzLS4wMy0uMTUtLjktMy4wOS0uMy00LjI4LjYtMS4xOSAxLjM1LTEuMzMgMi4zNS0xLjU4Ljc3LS4yIDMuMDgtLjI2IDQuMjMuMTggMS42LjYxIDEuOTguMzggMS45OC4zOHMuNjYuOC4zNiAzLjI2Yy0uMDMuMi0uMjIgMS43My0uMjIgMi4wNCAwIC4wNi42Mi0uNDIgMS4xIDAgLjQ2LjQyLS43MyAzLjItMS4yNCAzLjItLjUgMC0uNDUtLjIyLS40OC0uMjUtLjAyLS4wMi0uMDMtLjAyLS4wMyAwLS4xLjUxLS4yMS43OC0uMzIgMS4wNmwtLjA1LjEyYy0uNjMgMS40My0yIDIuNDYtMi42IDIuNTktLjQzLjA5LS42OS4xMi0xLjE1IDBhNS4xNyA1LjE3IDAgMCAxLTIuNjYtMi42NCAxOS42IDE5LjYgMCAwIDEtLjMzLTEuMDlzLS4wOS4xNC0uMzEuMnoiIGNsaXAtcnVsZT0iZXZlbm9kZCIvPjwvc3ZnPg==',
                    'origin'   => 'partner',
                    'resource' => 'users',
                    'type'     => 'item',
                    'url'      => '/v2/users/'
                },
                {
                    'caption'  => gettext('Invitations'),
                    'children' => [],
                    'icon' =>
'data:image/svg+xml;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHdpZHRoPSIyNCIgaGVpZ2h0PSIyNCIgZmlsbD0ibm9uZSI+PHBhdGggZmlsbD0iI2ZmZiIgZmlsbC1ydWxlPSJldmVub2RkIiBkPSJNOC43OCAxMS4xNWMuMjQuNjEuNjIuOS44Ljg0LjE4LS4wNi4yNS0uMTguMjUtLjE4cy4xNi42NS4yNi45NmMuMzMgMS4wMSAxLjQ5IDIuMTYgMi4xNCAyLjM1LjM3LjExLjU3LjA4LjkyIDAgLjQ4LS4xMSAxLjU4LTEuMDMgMi4wOS0yLjNsLjA0LS4xYy4wOS0uMjYuMTctLjUuMjYtLjk1IDAtLjAyIDAtLjAxLjAxIDAgLjAzLjAzLjEuMS4yNy4xMS4yLjAzLjMxLS4yLjQ4LS42bC4xMy0uMjdjLjQ2LTEuMTMuNC0xLjkuMDgtMS45Ni0uMTUtLjAzLS4yNS4wNS0uMzUuMTUtLjA1IDAtLjEgMC0uMS0uMDUgMC0uMTEuMDEtLjI1LjA0LS40LjA0LS4zMy4xLS43Ni4wNS0xLjM0LjYxLTEuNjItLjU2LTIuNzctLjU2LTIuNzdzLS42NC4yLTEuOTItLjM1YTQuMzQgNC4zNCAwIDAgMC0zLjA2LS4wMmMtLjUuMTctLjguNjItLjczIDEuMS0uNDYtLjE5LS44OC4zLS45Ny41Ny40LjU2LjM1IDEuMTIuMzIgMS42YTUuNTIgNS41MiAwIDAgMCAuMTMgMS43OXMtLjEuMS0uMTkuMDFjLS4wOC0uMDgtLjMyLS4zMi0uNDYtLjMyLS4xNSAwLS4zNS4yNS0uMjguNzYuMDYuNTIuMS43NS4zNSAxLjM3ek0xNiAxNS42bC0uNzYuMzlIMTZ2LS4zOXptLTIgMS4wMXYzLjJjLS41LjA5LTEuMDQuMTMtMS42Mi4xMy0zLjQ2IDAtNS45OC0xLjY2LTYuOTUtMi40MmEuNC40IDAgMCAxIC4wNC0uNjhjLjYzLS4zOCAxLjg4LS45OSAzLjY1LTEuMjcuMS0uMDEuMiAwIC4yOC4wNC45LjQzIDIuMzkgMS4yNiAyLjk2IDEuNThhLjUuNSAwIDAgMCAuNDYgMGwxLjE4LS41OHptMy0xLjEyYzAtLjI4LjIyLS41LjUtLjVoMWMuMjggMCAuNS4yMi41LjVWMTdoMS41Yy4yOCAwIC41LjIyLjUuNXYxYS41LjUgMCAwIDEtLjUuNUgxOXYxLjVhLjUuNSAwIDAgMS0uNS41aC0xYS41LjUgMCAwIDEtLjUtLjVWMTloLTEuNWEuNS41IDAgMCAxLS41LS41di0xYzAtLjI4LjIyLS41LjUtLjVIMTd2LTEuNXoiIGNsaXAtcnVsZT0iZXZlbm9kZCIvPjwvc3ZnPg==',
                    'origin'   => 'partner',
                    'resource' => 'inviter',
                    'type'     => 'item',
                    'url'      => '/v2/inviter/'
                },
                {
                    'caption'  => gettext('Documents'),
                    'children' => [],
                    'icon' =>
'data:image/svg+xml;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHdpZHRoPSIyNCIgaGVpZ2h0PSIyNCIgZmlsbD0ibm9uZSI+PHBhdGggZmlsbD0iI2ZmZiIgZmlsbC1ydWxlPSJldmVub2RkIiBkPSJNNyAzYTIgMiAwIDAgMC0yIDJ2MTRjMCAxLjEuOSAyIDIgMmgxMGEyIDIgMCAwIDAgMi0yVjcuODNhMiAyIDAgMCAwLS41OS0xLjQyTDE1LjYgMy42YTIgMiAwIDAgMC0xLjQzLS42SDd6bTcuNDMgMS40M2wzLjE0IDMuMTRjLjE2LjE2LjA1LjQzLS4xNy40M2gtMy4xNWEuMjUuMjUgMCAwIDEtLjI1LS4yNVY0LjZjMC0uMjIuMjctLjMzLjQzLS4xN3ptLTMuMTcgOS4xOHYtMi41aC42M2MuNDcgMCAuOC4xIDEgLjI5LjIuMTguMzEuNDkuMzEuOTEgMCAuODctLjQ1IDEuMy0xLjM0IDEuM2gtLjZ6bTAgMy44OXYtMS4yM0gxM2EuNDcuNDcgMCAwIDAgMC0uOTVoLTEuNzV2LS43NmguNTljLjQgMCAuNzgtLjA0IDEuMTItLjEyLjM0LS4xLjYzLS4yMy44OC0uNC4yNS0uMTkuNDMtLjQyLjU3LS43LjEzLS4yOS4yLS42NC4yLTEuMDRzLS4wNy0uNzMtLjItMS4wMmExLjggMS44IDAgMCAwLS41NS0uNzFjLS4yMy0uMi0uNTItLjM0LS44Ni0uNDItLjMzLS4xLS43LS4xNS0xLjEyLS4xNWgtMS41NWEuNS41IDAgMCAwLS41LjV2My4xaC0uMzdhLjQ3LjQ3IDAgMCAwIDAgLjk2aC4zN3YuNzZoLS4zN2EuNDcuNDcgMCAwIDAgMCAuOTVoLjM3djEuMjNjMCAuMjguMjIuNS41LjVoLjQyYS41LjUgMCAwIDAgLjUtLjV6IiBjbGlwLXJ1bGU9ImV2ZW5vZGQiLz48L3N2Zz4=',
                    'origin'   => 'partner',
                    'resource' => 'partner_acts',
                    'type'     => 'item',
                    'url'      => '/v2/financial/acts/'
                },
                {
                    'caption'  => gettext('Cookie match'),
                    'children' => [],
                    'icon' =>
'data:image/svg+xml;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHdpZHRoPSIyNCIgaGVpZ2h0PSIyNCIgZmlsbD0ibm9uZSI+PHBhdGggZmlsbD0iI2ZmZiIgZmlsbC1ydWxlPSJldmVub2RkIiBkPSJNMTIgMjBhOCA4IDAgMSAwIDAtMTYgOCA4IDAgMCAwIDAgMTZ6bTQuNTctOGExLjE0IDEuMTQgMCAxIDAgMC0yLjI5IDEuMTQgMS4xNCAwIDAgMCAwIDIuMjl6TTE0LjMgOC41N2ExLjE0IDEuMTQgMCAxIDAgMC0yLjI4IDEuMTQgMS4xNCAwIDAgMCAwIDIuMjh6bS0uNTggOS4xNGExLjcxIDEuNzEgMCAxIDAgMC0zLjQyIDEuNzEgMS43MSAwIDAgMCAwIDMuNDJ6bS01LjE0LTIuMjhhMS4xNCAxLjE0IDAgMSAwIDAtMi4yOSAxLjE0IDEuMTQgMCAwIDAgMCAyLjI5em0uNTctNC41N2ExLjcxIDEuNzEgMCAxIDAgMC0zLjQzIDEuNzEgMS43MSAwIDAgMCAwIDMuNDN6IiBjbGlwLXJ1bGU9ImV2ZW5vZGQiLz48L3N2Zz4=',
                    'origin'   => 'partner',
                    'resource' => 'cookie_match',
                    'type'     => 'item',
                    'url'      => '/v2/cookie-matching/'
                },
                {
                    'caption'  => gettext('Moderation'),
                    'children' => [],
                    'icon' =>
'data:image/svg+xml;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHdpZHRoPSIyNCIgaGVpZ2h0PSIyNCIgZmlsbD0ibm9uZSI+PHBhdGggZmlsbD0iI2ZmZiIgZD0iTTEyLjM0IDIxLjg1YzEuNS0uNzQgNy40My00LjMzIDcuNjUtMTUuMjJhMSAxIDAgMCAwLS41NC0uOWwtNy0zLjVhMSAxIDAgMCAwLS45IDBsLTcgMy41YTEgMSAwIDAgMC0uNTQuOWMuMjIgMTAuODkgNi4xNSAxNC40OCA3LjY1IDE1LjIyLjIyLjEuNDYuMS42OCAweiIvPjwvc3ZnPg==',
                    'origin'   => 'partner',
                    'resource' => 'moderation',
                    'type'     => 'item',
                    'url'      => '/v2/moderation/?search=%7B%22multistate%22%3A%5B%22%3D%22%2C%22new%22%5D%7D'
                },
            ],
            'icon' =>
'data:image/svg+xml,<svg width="20" height="20" viewBox="0 0 20 20" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M18.3334 10C18.3334 14.6024 14.6025 18.3334 10.0001 18.3334C5.39771 18.3334 1.66675 14.6024 1.66675 10C1.66675 5.39765 5.39771 1.66669 10.0001 1.66669C14.6025 1.66669 18.3334 5.39765 18.3334 10ZM16.6667 10C16.6667 6.31812 13.682 3.33335 10.0001 3.33335C6.31818 3.33335 3.33341 6.31812 3.33341 10C3.33341 13.6819 6.31818 16.6667 10.0001 16.6667C13.682 16.6667 16.6667 13.6819 16.6667 10ZM5.83324 10.833C5.37301 10.8329 4.99999 10.4597 5.00008 9.9995C5.00018 9.53926 5.37335 9.16624 5.83359 9.16633L9.16675 9.16702V5.83335C9.16675 5.37312 9.53984 5.00002 10.0001 5.00002C10.4603 5.00002 10.8334 5.37312 10.8334 5.83335V9.16736L14.1669 9.16805C14.6272 9.16815 15.0002 9.54132 15.0001 10.0016C15 10.4618 14.6268 10.8348 14.1666 10.8347L10.8334 10.834V14.1667C10.8334 14.6269 10.4603 15 10.0001 15C9.53984 15 9.16675 14.6269 9.16675 14.1667V10.8337L5.83324 10.833Z" fill="white"/></svg>',
            'origin'            => 'partner',
            'resource'          => undef,
            'type'              => 'group',
            'url'               => '',
            'only_with_feature' => 'simple_menu',
        },
        {
            'caption'  => gettext('For Developers'),
            'children' => [
                {
                    'caption'  => gettext('Messages'),
                    'children' => [],
                    'origin'   => 'partner',
                    'resource' => 'devel_view_mail',
                    'type'     => 'item',
                    'url'      => '/devel/view_mail/'
                },
                {
                    'caption'  => gettext('Cron tasks'),
                    'children' => [],
                    'origin'   => 'partner',
                    'resource' => 'devel_cron_jobs',
                    'type'     => 'item',
                    'url'      => '/devel/cron_jobs/'
                },
                {
                    'caption'  => gettext('Statuses'),
                    'children' => [],
                    'origin'   => 'partner',
                    'resource' => 'devel_multistates',
                    'type'     => 'item',
                    'url'      => '/devel/multistates/'
                },
                {
                    'caption'  => gettext('Queue'),
                    'children' => [],
                    'origin'   => 'partner',
                    'resource' => 'queue',
                    'type'     => 'item',
                    'url'      => '/queue/list/'
                },
                {
                    'caption'  => gettext('Class graph'),
                    'children' => [],
                    'origin'   => 'partner',
                    'resource' => 'devel_class_graph',
                    'type'     => 'item',
                    'url'      => '/devel/class_graph/'
                },
                {
                    'caption'  => gettext('Show contracts'),
                    'children' => [],
                    'origin'   => 'partner',
                    'resource' => 'devel_show_agreements',
                    'type'     => 'item',
                    'url'      => '/devel/show_agreements/'
                },
                {
                    'caption'  => gettext('Text templates'),
                    'children' => [],
                    'origin'   => 'partner',
                    'resource' => 'text_template',
                    'type'     => 'item',
                    'url'      => '/devel/text_template_list/'
                },
                {
                    'caption'  => gettext('Internal API ACLs'),
                    'children' => [],
                    'origin'   => 'partner',
                    'resource' => 'intapi_acl',
                    'type'     => 'item',
                    'url'      => '/devel/intapi_acl_list/'
                },
                {
                    'caption'  => gettext('Simple notification'),
                    'children' => [],
                    'origin'   => 'partner',
                    'resource' => 'simple_notification_tool',
                    'type'     => 'item',
                    'url'      => '/devel/simple_notification_list/'
                },
                {
                    'caption'  => gettext('Notification'),
                    'children' => [],
                    'origin'   => 'partner',
                    'resource' => 'notification_tool',
                    'type'     => 'item',
                    'url'      => '/devel/notification_list/'
                },
                {
                    'caption'  => gettext('moderation_reason'),
                    'children' => [],
                    'origin'   => 'partner',
                    'resource' => 'moderation_reason_tool',
                    'type'     => 'item',
                    'url'      => '/devel/moderation_reason_list/'
                },
                {
                    'caption'  => gettext('Pages tags'),
                    'children' => [],
                    'origin'   => 'partner',
                    'resource' => 'pages_tags',
                    'type'     => 'item',
                    'url'      => '/v2/devel/pages_tags/'
                },
                {
                    'caption'  => gettext('Support queue task'),
                    'children' => [],
                    'origin'   => 'partner',
                    'resource' => 'queue_support_task',
                    'type'     => 'item',
                    'url'      => '/queue/list/'
                },
                {
                    'caption'  => gettext('Delete user'),
                    'children' => [],
                    'origin'   => 'partner',
                    'resource' => 'devel_drop_user',
                    'type'     => 'item',
                    'url'      => '/devel/drop_user/'
                },
                {
                    'caption'  => gettext('Create client'),
                    'children' => [],
                    'origin'   => 'partner',
                    'resource' => 'devel_create_client',
                    'type'     => 'item',
                    'url'      => '/devel/create_client/'
                },
                {
                    'caption'  => gettext('Remove client from user'),
                    'children' => [],
                    'origin'   => 'partner',
                    'resource' => 'devel_remove_user_client_association',
                    'type'     => 'item',
                    'url'      => '/devel/remove_user_client_association/'
                },
                {
                    'caption'  => gettext('CreateOrUpdatePlace switch'),
                    'children' => [],
                    'origin'   => 'partner',
                    'resource' => 'devel_create_or_update_place_switch',
                    'type'     => 'item',
                    'url'      => '/devel/create_or_update_place_switch/'
                },
                {
                    'caption'  => gettext('BS and PI data comparison'),
                    'children' => [],
                    'origin'   => 'partner',
                    'resource' => 'devel_bk_pi_comparison',
                    'type'     => 'item',
                    'url'      => '/devel/bk_pi_comparison/'
                },
                {
                    'caption'  => gettext('View BS send log'),
                    'children' => [],
                    'origin'   => 'partner',
                    'resource' => 'devel_bk_send_log',
                    'type'     => 'item',
                    'url'      => '/devel/bk_send_log/'
                },
                {
                    'caption'  => gettext('Change options on Page'),
                    'children' => [],
                    'origin'   => 'partner',
                    'resource' => 'devel_update_options',
                    'type'     => 'item',
                    'url'      => '/devel/update_options/'
                },
                {
                    'caption'  => gettext('<nobr>lang-detect</nobr> test tool'),
                    'children' => [],
                    'origin'   => 'partner',
                    'resource' => 'devel_lang_detect_check',
                    'type'     => 'item',
                    'url'      => '/devel/lang_detect_check/'
                },
                {
                    'caption'  => gettext('Fields check'),
                    'children' => [],
                    'origin'   => 'partner',
                    'resource' => 'devel_fields',
                    'type'     => 'item',
                    'url'      => '/devel/fields/'
                }
            ],
            'icon' =>
'data:image/svg+xml,<svg width="20" height="20" viewBox="0 0 20 20" fill="none" xmlns="http://www.w3.org/2000/svg"><path fill-rule="evenodd" clip-rule="evenodd" d="M17.9163 10L13.9927 7.64565C13.8421 7.55529 13.75 7.39254 13.75 7.21692V5.8831C13.75 5.49445 14.174 5.25439 14.5072 5.45435L19.5145 8.4587C19.8157 8.63942 20 8.96493 20 9.31619V10.6838C20 11.0351 19.8157 11.3606 19.5145 11.5413L14.5072 14.5457C14.174 14.7456 13.75 14.5056 13.75 14.1169V12.7831C13.75 12.6075 13.8421 12.4447 13.9927 12.3543L17.9163 10ZM2.08375 10L6.00727 12.3543C6.15786 12.4447 6.25 12.6075 6.25 12.7831V14.1169C6.25 14.5056 5.82601 14.7456 5.49275 14.5457L0.485504 11.5413C0.184298 11.3606 0 11.0351 0 10.6838V9.31619C0 8.96493 0.184299 8.63942 0.485504 8.4587L5.49275 5.45435C5.82601 5.25439 6.25 5.49445 6.25 5.8831V7.21692C6.25 7.39254 6.15786 7.55529 6.00727 7.64565L2.08375 10ZM9.93037 2.9178C9.97055 2.67671 10.1791 2.5 10.4236 2.5H11.9098C12.2187 2.5 12.4538 2.77743 12.403 3.0822L10.0696 17.0822C10.0295 17.3233 9.82086 17.5 9.57644 17.5H8.09023C7.78126 17.5 7.54624 17.2226 7.59703 16.9178L9.93037 2.9178Z" fill="white"/></svg>',
            'origin'            => 'partner',
            'resource'          => undef,
            'type'              => 'group',
            'url'               => '/v2/dashboard/',
            'only_with_feature' => 'simple_menu',
        },
        # NO simple_menu starts
        {
            'caption'  => gettext('Dashboard'),
            'children' => [],
            'icon' =>
'data:image/svg+xml;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHdpZHRoPSIyNCIgaGVpZ2h0PSIyNCIgZmlsbD0ibm9uZSI+PHBhdGggZmlsbD0iI2ZmZiIgZmlsbC1ydWxlPSJldmVub2RkIiBkPSJNNCAxNWExIDEgMCAwIDAtMSAxdjRhMSAxIDAgMCAwIDEgMWg2YTEgMSAwIDAgMCAxLTF2LTRhMSAxIDAgMCAwLTEtMUg0em0xMC00YTEgMSAwIDAgMC0xIDF2OGExIDEgMCAwIDAgMSAxaDZhMSAxIDAgMCAwIDEtMXYtOGExIDEgMCAwIDAtMS0xaC02ek0zIDRhMSAxIDAgMCAxIDEtMWg2YTEgMSAwIDAgMSAxIDF2OGExIDEgMCAwIDEtMSAxSDRhMSAxIDAgMCAxLTEtMVY0em0xMS0xYTEgMSAwIDAgMC0xIDF2NGExIDEgMCAwIDAgMSAxaDZhMSAxIDAgMCAwIDEtMVY0YTEgMSAwIDAgMC0xLTFoLTZ6IiBjbGlwLXJ1bGU9ImV2ZW5vZGQiLz48L3N2Zz4=',
            'origin'          => 'partner',
            'resource'        => undef,
            'type'            => 'item',
            'url'             => '/v2/dashboard/',
            'only_no_feature' => 'simple_menu',
        },
        {
            'caption'  => gettext('notification'),
            'children' => [],
            'icon' =>
'data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iMjQiIGhlaWdodD0iMjQiIHZpZXdCb3g9IjAgMCAyNCAyNCIgZmlsbD0ibm9uZSIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj4KPHBhdGggZmlsbC1ydWxlPSJldmVub2RkIiBjbGlwLXJ1bGU9ImV2ZW5vZGQiIGQ9Ik0xMCAxOEgxNVYxOS41QzE1IDE5Ljg5NzggMTQuODQyIDIwLjI3OTQgMTQuNTYwNyAyMC41NjA3QzE0LjI3OTQgMjAuODQyIDEzLjg5NzggMjEgMTMuNSAyMUgxMS41QzExLjEwMjIgMjEgMTAuNzIwNiAyMC44NDIgMTAuNDM5MyAyMC41NjA3QzEwLjE1OCAyMC4yNzk0IDEwIDE5Ljg5NzggMTAgMTkuNVYxOFoiIGZpbGw9IndoaXRlIi8+CjxwYXRoIGQ9Ik0xNSAxNi41QzE1IDEzLjY1NjIgMTggMTEuNCAxOCA4LjZDMTggNS41MDc0NCAxNS41Mzc0IDMgMTIuNSAzQzkuNDYyNjcgMyA3IDUuNTA3NCA3IDguNkM3IDExLjQgMTAgMTMuNzQzNyAxMCAxNi41SDE1WiIgZmlsbD0id2hpdGUiLz4KPC9zdmc+',
            'icon_has_notifications' =>
"data:image/svg+xml,%3Csvg width='24' height='24' viewBox='0 0 24 24' fill='none' xmlns='http://www.w3.org/2000/svg'%3E%3Cpath fill-rule='evenodd' clip-rule='evenodd' d='M10 18H15V19.5C15 19.8978 14.842 20.2794 14.5607 20.5607C14.2794 20.842 13.8978 21 13.5 21H11.5C11.1022 21 10.7206 20.842 10.4393 20.5607C10.158 20.2794 10 19.8978 10 19.5V18Z' fill='white'/%3E%3Cpath d='M15 16.5C15 13.6562 18 11.4 18 8.6C18 5.50744 15.5374 3 12.5 3C9.46267 3 7 5.5074 7 8.6C7 11.4 10 13.7437 10 16.5H15Z' fill='white'/%3E%3Crect x='13.5' y='1.5' width='9' height='9' rx='4.5' fill='%23FF3333' stroke='%23434445'/%3E%3C/svg%3E%0A",
            'origin'          => 'partner',
            'resource'        => 'user_notifications',
            'type'            => 'item',
            'url'             => '/v2/notifications/',
            'only_no_feature' => 'simple_menu',
        },
        {
            'caption'  => gettext('Products'),
            'children' => [
                {
                    'caption'  => gettext('Ads on&nbsp;websites'),
                    'children' => [
                        {
                            'caption'  => gettext('Sites'),
                            'children' => [],
                            'origin'   => 'partner',
                            'resource' => 'site',
                            'type'     => 'item',
                            'url'      => '/v2/sites/'
                        },
                        {
                            'caption'  => gettext('Content sites'),
                            'children' => [
                                {
                                    'caption'  => gettext('Content sites'),
                                    'children' => [],
                                    'origin'   => 'partner',
                                    'resource' => 'context_on_site_campaign',
                                    'type'     => 'item',
                                    'url'      => '/v2/context/campaigns/'
                                },
                                {
                                    'caption'  => gettext('Archive'),
                                    'children' => [],
                                    'origin'   => 'partner',
                                    'resource' => 'context_on_site_campaign',
                                    'type'     => 'item',
                                    'url'      => '/v2/context/campaigns/archive/'
                                }
                            ],
                            'origin'   => 'partner',
                            'resource' => 'context_on_site_campaign',
                            'type'     => 'group',
                            'url'      => '/v2/context/campaigns/'
                        },
                        {
                            'caption'  => gettext('undef'),
                            'children' => [
                                {
                                    'caption'  => gettext('RTB blocks'),
                                    'children' => [
                                        {
                                            'caption'  => gettext('RTB blocks'),
                                            'children' => [],
                                            'origin'   => 'partner',
                                            'resource' => 'context_on_site_rtb',
                                            'type'     => 'item',
                                            'url'      => '/v2/context/rtb/'
                                        },
                                        {
                                            'caption'  => gettext('Archive'),
                                            'children' => [],
                                            'origin'   => 'partner',
                                            'resource' => 'context_on_site_rtb',
                                            'type'     => 'item',
                                            'url'      => '/v2/context/rtb/archive/'
                                        }
                                    ],
                                    'origin'   => 'partner',
                                    'resource' => undef,
                                    'type'     => 'group',
                                    'url'      => ''
                                },
                                {
                                    'caption'  => gettext('Content blocks'),
                                    'children' => [
                                        {
                                            'caption'  => gettext('Content blocks'),
                                            'children' => [],
                                            'origin'   => 'partner',
                                            'resource' => 'context_on_site_content',
                                            'type'     => 'item',
                                            'url'      => '/v2/context/content/'
                                        },
                                        {
                                            'caption'  => gettext('Archive'),
                                            'children' => [],
                                            'origin'   => 'partner',
                                            'resource' => 'context_on_site_content',
                                            'type'     => 'item',
                                            'url'      => '/v2/context/content/archive/'
                                        }
                                    ],
                                    'origin'   => 'partner',
                                    'resource' => undef,
                                    'type'     => 'group',
                                    'url'      => ''
                                },
                                {
                                    'caption'  => gettext('Natural blocks'),
                                    'children' => [
                                        {
                                            'caption'  => gettext('Natural blocks'),
                                            'children' => [],
                                            'origin'   => 'partner',
                                            'resource' => 'context_on_site_natural',
                                            'type'     => 'item',
                                            'url'      => '/v2/context/natural/'
                                        },
                                        {
                                            'caption'  => gettext('Archive'),
                                            'children' => [],
                                            'origin'   => 'partner',
                                            'resource' => 'context_on_site_natural',
                                            'type'     => 'item',
                                            'url'      => '/v2/context/natural/archive/'
                                        }
                                    ],
                                    'origin'   => 'partner',
                                    'resource' => undef,
                                    'type'     => 'group',
                                    'url'      => ''
                                },
                                {
                                    'caption'  => gettext('Yandex.Direct blocks'),
                                    'children' => [
                                        {
                                            'caption'  => gettext('Direct blocks'),
                                            'children' => [],
                                            'origin'   => 'partner',
                                            'resource' => 'context_on_site_direct',
                                            'type'     => 'item',
                                            'url'      => '/v2/context/direct/'
                                        },
                                        {
                                            'caption'  => gettext('Archive'),
                                            'children' => [],
                                            'origin'   => 'partner',
                                            'resource' => 'context_on_site_direct_deleted',
                                            'type'     => 'item',
                                            'url'      => '/v2/context/direct/archive/'
                                        }
                                    ],
                                    'origin'   => 'partner',
                                    'resource' => undef,
                                    'type'     => 'group',
                                    'url'      => ''
                                },
                                {
                                    'caption'  => gettext('Stripe'),
                                    'children' => [
                                        {
                                            'caption'  => gettext('Stripe'),
                                            'children' => [],
                                            'origin'   => 'partner',
                                            'resource' => 'context_on_site_stripe',
                                            'type'     => 'item',
                                            'url'      => '/v2/context/stripe/'
                                        },
                                        {
                                            'caption'  => gettext('Archive'),
                                            'children' => [],
                                            'origin'   => 'partner',
                                            'resource' => 'context_on_site_stripe_deleted',
                                            'type'     => 'item',
                                            'url'      => '/v2/context/stripe/archive/'
                                        }
                                    ],
                                    'origin'   => 'partner',
                                    'resource' => undef,
                                    'type'     => 'group',
                                    'url'      => ''
                                },
                                {
                                    'caption'  => gettext('Adblock blocks'),
                                    'children' => [
                                        {
                                            'caption'  => gettext('Adblock blocks'),
                                            'children' => [],
                                            'origin'   => 'partner',
                                            'resource' => 'context_on_site_adblock',
                                            'type'     => 'item',
                                            'url'      => '/v2/context/adblock/'
                                        },
                                        {
                                            'caption'  => gettext('Archive'),
                                            'children' => [],
                                            'origin'   => 'partner',
                                            'resource' => 'context_on_site_adblock_deleted',
                                            'type'     => 'item',
                                            'url'      => '/v2/context/adblock/archive/'
                                        }
                                    ],
                                    'origin'   => 'partner',
                                    'resource' => undef,
                                    'type'     => 'group',
                                    'url'      => ''
                                }
                            ],
                            'origin'   => 'partner',
                            'resource' => undef,
                            'type'     => 'sub-group',
                            'url'      => ''
                        },
                        {
                            'caption'  => gettext('Site search ad platforms'),
                            'children' => [
                                {
                                    'caption'  => gettext('Search ad platforms'),
                                    'children' => [],
                                    'origin'   => 'partner',
                                    'resource' => 'search_on_site_campaign',
                                    'type'     => 'item',
                                    'url'      => '/v2/search/campaigns/'
                                },
                                {
                                    'caption'  => gettext('Archive'),
                                    'children' => [],
                                    'origin'   => 'partner',
                                    'resource' => 'search_on_site_campaign_deleted',
                                    'type'     => 'item',
                                    'url'      => '/v2/search/campaigns/archive/'
                                }
                            ],
                            'origin'   => 'partner',
                            'resource' => 'search_on_site_campaign',
                            'type'     => 'group',
                            'url'      => '/v2/search/campaigns/'
                        },
                        {
                            'caption'  => gettext('undef'),
                            'children' => [
                                {
                                    'caption'  => gettext('Premium&#13;Placement'),
                                    'children' => [
                                        {
                                            'caption'  => gettext('Premium Placement'),
                                            'children' => [],
                                            'origin'   => 'partner',
                                            'resource' => 'search_on_site_premium',
                                            'type'     => 'item',
                                            'url'      => '/v2/search/premium/'
                                        },
                                        {
                                            'caption'  => gettext('Archive'),
                                            'children' => [],
                                            'origin'   => 'partner',
                                            'resource' => 'search_on_site_premium_deleted',
                                            'type'     => 'item',
                                            'url'      => '/v2/search/premium/archive/'
                                        }
                                    ],
                                    'origin'   => 'partner',
                                    'resource' => undef,
                                    'type'     => 'group',
                                    'url'      => ''
                                },
                                {
                                    'caption'  => gettext('Search Direct'),
                                    'children' => [
                                        {
                                            'caption'  => gettext('Search Direct'),
                                            'children' => [],
                                            'origin'   => 'partner',
                                            'resource' => 'search_on_site_direct',
                                            'type'     => 'item',
                                            'url'      => '/v2/search/direct/'
                                        },
                                        {
                                            'caption'  => gettext('Archive'),
                                            'children' => [],
                                            'origin'   => 'partner',
                                            'resource' => 'search_on_site_direct_deleted',
                                            'type'     => 'item',
                                            'url'      => '/v2/search/direct/archive/'
                                        }
                                    ],
                                    'origin'   => 'partner',
                                    'resource' => undef,
                                    'type'     => 'group',
                                    'url'      => ''
                                }
                            ],
                            'origin'   => 'partner',
                            'resource' => undef,
                            'type'     => 'sub-group',
                            'url'      => ''
                        }
                    ],
                    'origin'   => 'partner',
                    'resource' => undef,
                    'type'     => 'group',
                    'url'      => ''
                },
                {
                    'caption'  => gettext('Ads in apps'),
                    'children' => [
                        {
                            'caption'         => gettext('Applications'),
                            'children'        => [],
                            'origin'          => 'partner',
                            'resource'        => 'mobile_app',
                            'type'            => 'item',
                            'url'             => '/v2/mobile/applications/',
                            'only_no_feature' => 'simple_inapp',
                        },
                        {
                            'caption'  => gettext('Mobile platforms'),
                            'children' => [
                                {
                                    'caption'  => gettext('Mobile platforms'),
                                    'children' => [],
                                    'origin'   => 'partner',
                                    'resource' => 'mobile_app_settings',
                                    'type'     => 'item',
                                    'url'      => '/v2/mobile/campaigns/'
                                },
                                {
                                    'caption'  => gettext('Archive'),
                                    'children' => [],
                                    'origin'   => 'partner',
                                    'resource' => 'mobile_app_settings_deleted',
                                    'type'     => 'item',
                                    'url'      => '/v2/mobile/campaigns/archive/'
                                }
                            ],
                            'origin'   => 'partner',
                            'resource' => 'mobile_app_settings',
                            'type'     => 'group',
                            'url'      => '/v2/mobile/campaigns/'
                        },
                        {
                            'caption'  => gettext('Mobile RTB blocks'),
                            'children' => [
                                {
                                    'caption'  => gettext('Mobile RTB blocks'),
                                    'children' => [],
                                    'origin'   => 'partner',
                                    'resource' => 'mobile_app_rtb',
                                    'type'     => 'item',
                                    'url'      => '/v2/mobile/rtb/'
                                },
                            ],
                            'origin'   => 'partner',
                            'resource' => undef,
                            'type'     => 'sub-group',
                            'url'      => '',
                        }
                    ],
                    'origin'   => 'partner',
                    'resource' => undef,
                    'type'     => 'group',
                    'url'      => ''
                },
                {
                    'caption'  => gettext('Ads in videos'),
                    'children' => [
                        {
                            'caption'  => gettext('Resources'),
                            'children' => [
                                {
                                    'caption'  => gettext('Video resources'),
                                    'children' => [],
                                    'origin'   => 'partner',
                                    'resource' => 'video_an_site',
                                    'type'     => 'item',
                                    'url'      => '/v2/video/sites/'
                                },
                                {
                                    'caption'  => gettext('Archive'),
                                    'children' => [],
                                    'origin'   => 'partner',
                                    'resource' => 'video_an_site_deleted',
                                    'type'     => 'item',
                                    'url'      => '/v2/video/sites/archive/'
                                }
                            ],
                            'origin'   => 'partner',
                            'resource' => 'video_an_site',
                            'type'     => 'group',
                            'url'      => '/v2/video/sites/'
                        },
                        {
                            'caption'  => gettext('undef'),
                            'children' => [
                                {
                                    'caption'  => gettext('Instream blocks'),
                                    'children' => [
                                        {
                                            'caption'  => gettext('Video blocks'),
                                            'children' => [],
                                            'origin'   => 'partner',
                                            'resource' => 'video_an_site_instream',
                                            'type'     => 'item',
                                            'url'      => '/v2/video/instream/'
                                        },
                                        {
                                            'caption'  => gettext('Archive'),
                                            'children' => [],
                                            'origin'   => 'partner',
                                            'resource' => 'video_an_site_instream_deleted',
                                            'type'     => 'item',
                                            'url'      => '/v2/video/instream/archive/'
                                        }
                                    ],
                                    'origin'   => 'partner',
                                    'resource' => undef,
                                    'type'     => 'group',
                                    'url'      => ''
                                },
                                {
                                    'caption'  => gettext('Inpage blocks'),
                                    'children' => [
                                        {
                                            'caption'  => gettext('Video blocks'),
                                            'children' => [],
                                            'origin'   => 'partner',
                                            'resource' => 'video_an_site_inpage',
                                            'type'     => 'item',
                                            'url'      => '/v2/video/inpage/'
                                        },
                                        {
                                            'caption'  => gettext('Archive'),
                                            'children' => [],
                                            'origin'   => 'partner',
                                            'resource' => 'video_an_site_inpage_deleted',
                                            'type'     => 'item',
                                            'url'      => '/v2/video/inpage/archive/'
                                        }
                                    ],
                                    'origin'   => 'partner',
                                    'resource' => undef,
                                    'type'     => 'group',
                                    'url'      => ''
                                },
                                {
                                    'caption'  => gettext('Fullscreen&#13;blocks'),
                                    'children' => [
                                        {
                                            'caption'  => gettext('Video blocks'),
                                            'children' => [],
                                            'origin'   => 'partner',
                                            'resource' => 'video_an_site_fullscreen',
                                            'type'     => 'item',
                                            'url'      => '/v2/video/fullscreen/'
                                        },
                                        {
                                            'caption'  => gettext('Archive'),
                                            'children' => [],
                                            'origin'   => 'partner',
                                            'resource' => 'video_an_site_fullscreen_deleted',
                                            'type'     => 'item',
                                            'url'      => '/v2/video/fullscreen/archive/'
                                        }
                                    ],
                                    'origin'   => 'partner',
                                    'resource' => undef,
                                    'type'     => 'group',
                                    'url'      => ''
                                }
                            ],
                            'origin'   => 'partner',
                            'resource' => undef,
                            'type'     => 'sub-group',
                            'url'      => ''
                        },

                        {
                            'caption'  => gettext('Video scenaries'),
                            'children' => [],
                            'origin'   => 'partner',
                            'resource' => 'video_scenaries',
                            'type'     => 'group',
                            'url'      => '/v2/video/scenaries/'
                        },
                        {
                            'caption'  => gettext('Category'),
                            'children' => [
                                {
                                    'caption'  => gettext('Content'),
                                    'children' => [],
                                    'origin'   => 'partner',
                                    'resource' => 'internal_content',
                                    'type'     => 'item',
                                    'url'      => '/v2/video/reference/content/'
                                },
                                {
                                    'caption'  => gettext('Content moderation'),
                                    'children' => [],
                                    'origin'   => 'partner',
                                    'resource' => 'partner_content',
                                    'type'     => 'item',
                                    'url'      => '/v2/video/reference/content/moderation/'
                                },
                                {
                                    'caption'  => gettext('Genres'),
                                    'children' => [],
                                    'origin'   => 'partner',
                                    'resource' => 'internal_genre',
                                    'type'     => 'item',
                                    'url'      => '/v2/video/reference/genre/'
                                },
                                {
                                    'caption'  => gettext('Genres moderation'),
                                    'children' => [],
                                    'origin'   => 'partner',
                                    'resource' => 'partner_genre',
                                    'type'     => 'item',
                                    'url'      => '/v2/video/reference/genre/moderation/'
                                }
                            ],
                            'origin'   => 'partner',
                            'resource' => undef,
                            'type'     => 'group',
                            'url'      => ''
                        },
                    ],
                    'origin'   => 'partner',
                    'resource' => undef,
                    'type'     => 'group',
                    'url'      => ''
                },
                {
                    'caption'  => gettext('Yandex Services'),
                    'children' => [
                        {
                            'caption'  => gettext('Content sites'),
                            'children' => [
                                {
                                    'caption'  => gettext('Content sites'),
                                    'children' => [],
                                    'origin'   => 'partner',
                                    'resource' => 'internal_context_on_site_campaign',
                                    'type'     => 'item',
                                    'url'      => '/v2/internal/context/campaigns/'
                                },
                                {
                                    'caption'  => gettext('Archive'),
                                    'children' => [],
                                    'origin'   => 'partner',
                                    'resource' => 'internal_context_on_site_campaign',
                                    'type'     => 'item',
                                    'url'      => '/v2/internal/context/campaigns/archive/'
                                }
                            ],
                            'origin'   => 'partner',
                            'resource' => 'internal_context_on_site_campaign',
                            'type'     => 'group',
                            'url'      => '/v2/internal/context/campaigns/'
                        },
                        {
                            'caption'  => gettext('undef'),
                            'children' => [
                                {
                                    'caption'  => gettext('RTB blocks'),
                                    'children' => [
                                        {
                                            'caption'  => gettext('RTB blocks'),
                                            'children' => [],
                                            'origin'   => 'partner',
                                            'resource' => 'internal_context_on_site_rtb',
                                            'type'     => 'item',
                                            'url'      => '/v2/internal/context/rtb/'
                                        },
                                        {
                                            'caption'  => gettext('Archive'),
                                            'children' => [],
                                            'origin'   => 'partner',
                                            'resource' => 'internal_context_on_site_rtb',
                                            'type'     => 'item',
                                            'url'      => '/v2/internal/context/rtb/archive/'
                                        }
                                    ],
                                    'origin'   => 'partner',
                                    'resource' => undef,
                                    'type'     => 'group',
                                    'url'      => ''
                                },
                                {
                                    'caption'  => gettext('Content blocks'),
                                    'children' => [
                                        {
                                            'caption'  => gettext('Content blocks'),
                                            'children' => [],
                                            'origin'   => 'partner',
                                            'resource' => 'internal_context_on_site_content',
                                            'type'     => 'item',
                                            'url'      => '/v2/internal/context/content/'
                                        },
                                        {
                                            'caption'  => gettext('Archive'),
                                            'children' => [],
                                            'origin'   => 'partner',
                                            'resource' => 'internal_context_on_site_content',
                                            'type'     => 'item',
                                            'url'      => '/v2/internal/context/content/archive/'
                                        }
                                    ],
                                    'origin'   => 'partner',
                                    'resource' => undef,
                                    'type'     => 'group',
                                    'url'      => ''
                                },
                                {
                                    'caption'  => gettext('Natural blocks'),
                                    'children' => [
                                        {
                                            'caption'  => gettext('Natural blocks'),
                                            'children' => [],
                                            'origin'   => 'partner',
                                            'resource' => 'internal_context_on_site_natural',
                                            'type'     => 'item',
                                            'url'      => '/v2/internal/context/natural/'
                                        },
                                        {
                                            'caption'  => gettext('Archive'),
                                            'children' => [],
                                            'origin'   => 'partner',
                                            'resource' => 'internal_context_on_site_natural',
                                            'type'     => 'item',
                                            'url'      => '/v2/internal/context/natural/archive/'
                                        }
                                    ],
                                    'origin'   => 'partner',
                                    'resource' => undef,
                                    'type'     => 'group',
                                    'url'      => ''
                                },
                                {
                                    'caption'  => gettext('Yandex.Direct blocks'),
                                    'children' => [
                                        {
                                            'caption'  => gettext('Direct blocks'),
                                            'children' => [],
                                            'origin'   => 'partner',
                                            'resource' => 'internal_context_on_site_direct',
                                            'type'     => 'item',
                                            'url'      => '/v2/internal/context/direct/'
                                        },
                                        {
                                            'caption'  => gettext('Archive'),
                                            'children' => [],
                                            'origin'   => 'partner',
                                            'resource' => 'internal_context_on_site_direct_deleted',
                                            'type'     => 'item',
                                            'url'      => '/v2/internal/context/direct/archive/'
                                        }
                                    ],
                                    'origin'   => 'partner',
                                    'resource' => undef,
                                    'type'     => 'group',
                                    'url'      => ''
                                },
                                {
                                    'caption'  => gettext('Stripe'),
                                    'children' => [
                                        {
                                            'caption'  => gettext('Stripe'),
                                            'children' => [],
                                            'origin'   => 'partner',
                                            'resource' => 'internal_context_on_site_stripe',
                                            'type'     => 'item',
                                            'url'      => '/internal_context_on_site_stripe/list/'
                                        },
                                        {
                                            'caption'  => gettext('Archive'),
                                            'children' => [],
                                            'origin'   => 'partner',
                                            'resource' => 'internal_context_on_site_stripe_deleted',
                                            'type'     => 'sub-item',
                                            'url'      => '/internal_context_on_site_stripe/list/archive/'
                                        }
                                    ],
                                    'origin'   => 'partner',
                                    'resource' => undef,
                                    'type'     => 'group',
                                    'url'      => ''
                                }
                            ],
                            'origin'   => 'partner',
                            'resource' => undef,
                            'type'     => 'sub-group',
                            'url'      => ''
                        },
                        {
                            'caption'  => gettext('Site search ad platforms'),
                            'children' => [
                                {
                                    'caption'  => gettext('Search ad platforms'),
                                    'children' => [],
                                    'origin'   => 'partner',
                                    'resource' => 'internal_search_on_site_campaign',
                                    'type'     => 'item',
                                    'url'      => '/v2/internal/search/campaigns/'
                                },
                                {
                                    'caption'  => gettext('Archive'),
                                    'children' => [],
                                    'origin'   => 'partner',
                                    'resource' => 'internal_search_on_site_campaign_deleted',
                                    'type'     => 'item',
                                    'url'      => '/v2/internal/search/campaigns/archive/'
                                }
                            ],
                            'origin'   => 'partner',
                            'resource' => 'internal_search_on_site_campaign',
                            'type'     => 'group',
                            'url'      => '/v2/internal/search/campaigns/'
                        },
                        {
                            'caption'  => gettext('undef'),
                            'children' => [
                                {
                                    'caption'  => gettext('Premium&#13;Placement'),
                                    'children' => [
                                        {
                                            'caption'  => gettext('Premium Placement'),
                                            'children' => [],
                                            'origin'   => 'partner',
                                            'resource' => 'internal_search_on_site_premium',
                                            'type'     => 'item',
                                            'url'      => '/internal_search_on_site_premium/list/'
                                        }
                                    ],
                                    'origin'   => 'partner',
                                    'resource' => 'internal_search_on_site_premium',
                                    'type'     => 'group',
                                    'url'      => '/internal_search_on_site_premium/list/'
                                },
                                {
                                    'caption'  => gettext('Search Direct'),
                                    'children' => [
                                        {
                                            'caption'  => gettext('Search Direct'),
                                            'children' => [],
                                            'origin'   => 'partner',
                                            'resource' => 'internal_search_on_site_direct',
                                            'type'     => 'item',
                                            'url'      => '/internal_search_on_site_direct/list/'
                                        }
                                    ],
                                    'origin'   => 'partner',
                                    'resource' => 'internal_search_on_site_direct',
                                    'type'     => 'group',
                                    'url'      => '/internal_search_on_site_direct/list/'
                                }
                            ],
                            'origin'   => 'partner',
                            'resource' => undef,
                            'type'     => 'sub-group',
                            'url'      => ''
                        },
                        {
                            'caption'  => gettext('Applications'),
                            'children' => [
                                {
                                    'caption'  => gettext('Applications'),
                                    'children' => [],
                                    'origin'   => 'partner',
                                    'resource' => 'internal_mobile_app',
                                    'type'     => 'item',
                                    'url'      => '/v2/internal/mobile/applications/'
                                },
                                {
                                    'caption'  => gettext('Archive'),
                                    'children' => [],
                                    'origin'   => 'partner',
                                    'resource' => 'internal_mobile_app_deleted',
                                    'type'     => 'item',
                                    'url'      => '/v2/internal/mobile/applications/archive/'
                                }
                            ],
                            'origin'   => 'partner',
                            'resource' => 'internal_mobile_app',
                            'type'     => 'group',
                            'url'      => '/v2/internal/mobile/applications/'
                        },
                        {
                            'caption'  => gettext('Mobile RTB blocks'),
                            'children' => [
                                {
                                    'caption'  => gettext('Mobile RTB blocks'),
                                    'children' => [],
                                    'origin'   => 'partner',
                                    'resource' => 'internal_mobile_app_rtb',
                                    'type'     => 'item',
                                    'url'      => '/v2/internal/mobile/rtb/'
                                },
                            ],
                            'origin'   => 'partner',
                            'resource' => undef,
                            'type'     => 'sub-group',
                            'url'      => '',
                        }
                    ],
                    'origin'   => 'partner',
                    'resource' => undef,
                    'type'     => 'group',
                    'url'      => ''
                },
                {
                    'caption'  => gettext('Advertising platforms'),
                    'children' => [
                        {
                            'caption'  => gettext('Connections'),
                            'children' => [
                                {
                                    'caption'  => gettext('Connections'),
                                    'children' => [],
                                    'origin'   => 'partner',
                                    'resource' => 'dsp',
                                    'type'     => 'item',
                                    'url'      => '/v2/dsp/connections/'
                                },
                                {
                                    'caption'  => gettext('Archive'),
                                    'children' => [],
                                    'origin'   => 'partner',
                                    'resource' => 'dsp_deleted',
                                    'type'     => 'item',
                                    'url'      => '/v2/dsp/connections/archive/'
                                },
                                {
                                    'caption'  => gettext('Tools'),
                                    'children' => [
                                        {
                                            'caption'  => gettext('Sandbox'),
                                            'children' => [],
                                            'origin'   => 'partner',
                                            'resource' => 'dsp_testing',
                                            'type'     => 'sub-item',
                                            'url'      => '/dsp/test/'
                                        },
                                        {
                                            'caption'  => gettext('Response statistics'),
                                            'children' => [],
                                            'origin'   => 'partner',
                                            'resource' => 'statistics_dsp_response',
                                            'type'     => 'sub-item',
                                            'url'      => '/dsp/statistics_responses/'
                                        }
                                    ],
                                    'origin'   => 'partner',
                                    'resource' => undef,
                                    'type'     => 'group',
                                    'url'      => ''
                                }
                            ],
                            'origin'   => 'partner',
                            'resource' => undef,
                            'type'     => 'group',
                            'url'      => ''
                        },
                        {
                            'caption'  => gettext('Documentation'),
                            'children' => [],
                            'origin'   => 'partner',
                            'resource' => 'dsp',
                            'type'     => 'item',
                            'url'      => 'https://tech.yandex.com/rtb/doc/api/index-docpage/'
                        }
                    ],
                    'origin'   => 'partner',
                    'resource' => undef,
                    'type'     => 'group',
                    'url'      => ''
                },
                {
                    'caption'  => gettext('SSP'),
                    'children' => [
                        {
                            'caption'  => gettext('Moderation'),
                            'children' => [
                                {
                                    'caption'  => gettext('Mobile applications'),
                                    'children' => [],
                                    'origin'   => 'partner',
                                    'resource' => 'ssp_link_mobile_app',
                                    'type'     => 'item',
                                    'url'      => '/v2/ssp/moderation/applications/'
                                },
                                {
                                    'caption'  => gettext('Content sites'),
                                    'children' => [],
                                    'origin'   => 'partner',
                                    'resource' => 'ssp_link_context_rtb',
                                    'type'     => 'item',
                                    'url'      => '/v2/ssp/moderation/campaigns/'
                                },
                                {
                                    'caption'  => gettext('Video platforms'),
                                    'children' => [],
                                    'origin'   => 'partner',
                                    'resource' => 'ssp_link_video_app',
                                    'type'     => 'item',
                                    'url'      => '/v2/ssp/moderation/video/'
                                }
                            ],
                            'origin'   => 'partner',
                            'resource' => undef,
                            'type'     => 'group',
                            'url'      => ''
                        }
                    ],
                    'origin'   => 'partner',
                    'resource' => undef,
                    'type'     => 'group',
                    'url'      => ''
                },
            ],
            'icon' =>
'data:image/svg+xml;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHdpZHRoPSIyNCIgaGVpZ2h0PSIyNCIgZmlsbD0ibm9uZSI+PHBhdGggZmlsbD0iI2ZmZiIgZmlsbC1ydWxlPSJldmVub2RkIiBkPSJNMjEgMTJhOSA5IDAgMSAwLTE4IDAgOSA5IDAgMCAwIDE4IDB6TTkuNCAxNi4zN2EuNS41IDAgMCAwIC41LS41di03LjNhLjUuNSAwIDAgMC0uNS0uNWgtLjk3bC0yLjQgMS43NWEuNS41IDAgMCAwLS4yLjR2LjMzYzAgLjQxLjQ2LjY1Ljc5LjRsMS4yMi0uODljLjE3LS4xMi40IDAgLjQuMnY1LjYxYzAgLjI4LjIyLjUuNS41aC42NnptNS4yOC4xMmEyLjcgMi43IDAgMCAxLTEuMTctLjI1IDIuMzQgMi4zNCAwIDAgMS0uOTEtLjc1IDMuODcgMy44NyAwIDAgMS0uNi0xLjMzIDcuNyA3LjcgMCAwIDEtLjIxLTEuOTRjMC0uNzYuMDctMS40LjIxLTEuOTQuMTQtLjU0LjM0LS45OC42LTEuMzIuMjUtLjM0LjU2LS42LjkxLS43NWEyLjg1IDIuODUgMCAwIDEgMi4zMiAwYy4zNS4xNi42Ni40LjkxLjc0LjI1LjM0LjQ1Ljc4LjU5IDEuMzIuMTQuNTMuMiAxLjE4LjIgMS45MiAwIC43Ni0uMDYgMS40LS4yIDEuOTUtLjE0LjU0LS4zNC45OC0uNTkgMS4zM2EyLjQzIDIuNDMgMCAwIDEtMi4wNiAxLjAyem0uNTItMS40NGEuOTguOTggMCAwIDEtLjUzLjE0Ljk4Ljk4IDAgMCAxLS41NC0uMTQgMS4xIDEuMSAwIDAgMS0uMzgtLjVjLS4xLS4yMy0uMTgtLjUzLS4yMy0uOWExMS4yNiAxMS4yNiAwIDAgMSAwLTIuODJjLjA1LS4zOC4xMi0uNjguMjMtLjkyLjEtLjIzLjIzLS40LjM5LS41LjE1LS4xLjMzLS4xNC41NC0uMTQuMiAwIC4zOS4wNS41NC4xNS4xNS4xLjI3LjI2LjM3LjUuMS4yMy4xNy41My4yMi45LjA1LjM5LjA3Ljg1LjA3IDEuNCAwIC41Ny0uMDIgMS4wNC0uMDcgMS40MmEzLjUgMy41IDAgMCAxLS4yMy45MWMtLjEuMjQtLjIyLjQtLjM4LjV6IiBjbGlwLXJ1bGU9ImV2ZW5vZGQiLz48L3N2Zz4=',
            'origin'          => 'partner',
            'resource'        => undef,
            'type'            => 'group',
            'url'             => '/v2/context/campaigns/?limit=100',
            'only_no_feature' => 'simple_menu',
        },
        {
            'caption'  => gettext('Business rules'),
            'children' => [],
            'icon' =>
'data:image/svg+xml;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHdpZHRoPSIyNCIgaGVpZ2h0PSIyNCIgZmlsbD0ibm9uZSI+PHBhdGggZmlsbD0iI2ZmZiIgZmlsbC1ydWxlPSJldmVub2RkIiBkPSJNNSA1YzAtMS4xLjktMiAyLTJoNy4xN2EyIDIgMCAwIDEgMS40Mi41OUwxOC40IDYuNGEyIDIgMCAwIDEgLjYgMS40M3YyLjI0QTcgNyAwIDAgMCAxMi4yNSAyMUg3YTIgMiAwIDAgMS0yLTJWNXptMTIuNTcgMi41N2wtMy4xNC0zLjE0YS4yNS4yNSAwIDAgMC0uNDMuMTd2My4xNWMwIC4xNC4xMS4yNS4yNS4yNWgzLjE1Yy4yMiAwIC4zMy0uMjcuMTctLjQzek0yMyAxN2E1IDUgMCAxIDEtMTAgMCA1IDUgMCAwIDEgMTAgMHptLTUuNzggMi41YS44My44MyAwIDAgMS0uNTgtLjI1Yy0uOTItLjk0LTIuMi0yLjItMi4yLTIuMmEuNjIuNjIgMCAxIDEgLjg3LS44OGwxLjkxIDEuOTEgMy40LTMuNGEuNjIuNjIgMCAxIDEgLjg4Ljg4bC0zLjcgMy42OWEuNzcuNzcgMCAwIDEtLjU4LjI1eiIgY2xpcC1ydWxlPSJldmVub2RkIi8+PC9zdmc+',
            'origin'          => 'partner',
            'resource'        => 'business_rules',
            'type'            => 'item',
            'url'             => '/v2/business-rules/',
            'only_no_feature' => 'simple_menu',
        },
        {
            'caption'  => gettext('Mass editing'),
            'children' => [],
            'icon' =>
'data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iMjQiIGhlaWdodD0iMjQiIGZpbGw9Im5vbmUiIHhtbG5zPSJodHRwOi8vd3d3LnczLm9yZy8yMDAwL3N2ZyI+CiAgICA8cGF0aCBkPSJNNyAyYTEgMSAwIDAgMCAwIDJoOWE0IDQgMCAwIDEgNCA0djlhMSAxIDAgMSAwIDIgMFY4YTYgNiAwIDAgMC02LTZIN1oiIGZpbGw9IiNmZmYiLz4KICAgIDxwYXRoIGZpbGwtcnVsZT0iZXZlbm9kZCIgY2xpcC1ydWxlPSJldmVub2RkIgogICAgICAgICAgZD0iTTIgMTBhNCA0IDAgMCAxIDQtNGg4YTQgNCAwIDAgMSA0IDR2OGE0IDQgMCAwIDEtNCA0SDZhNCA0IDAgMCAxLTQtNHYtOFptMi40OSA0LjM4NSAxLjY0NC0xLjY0NCAyLjc0IDIuNzQuODIyLjgyMi44MjMuODIzLTEuNjQ1IDEuNjQ0LTQuMzg0LTQuMzg1Wm00LjM4NCAxLjA5NkwxNC4zNTYgMTAgMTYgMTEuNjQ0bC01LjQ4MSA1LjQ4Mi0uODIzLS44MjMtLjgyMi0uODIyWiIKICAgICAgICAgIGZpbGw9IiNmZmYiLz4KPC9zdmc+Cg==',
            'origin'          => 'partner',
            'resource'        => 'block_presets',
            'type'            => 'item',
            'url'             => '/v2/block-presets/',
            'only_no_feature' => 'simple_menu',
        },
        {
            'caption'  => gettext('Mobile mediation'),
            'children' => [],
            'icon' =>
'data:image/svg+xml;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHdpZHRoPSIyNCIgaGVpZ2h0PSIyNCIgZmlsbD0ibm9uZSI+PHBhdGggZmlsbD0iI2ZmZiIgZmlsbC1ydWxlPSJldmVub2RkIiBkPSJNMyA0YTEgMSAwIDAgMSAxLTFoMmExIDEgMCAwIDEgMSAxdjJhMSAxIDAgMCAxLTEgMUg0YTEgMSAwIDAgMS0xLTFWNHptMCA3YTEgMSAwIDAgMSAxLTFoMTZhMSAxIDAgMCAxIDEgMXYyYTEgMSAwIDAgMS0xIDFINGExIDEgMCAwIDEtMS0xdi0yem0xNSA2YTEgMSAwIDAgMC0xIDF2MmExIDEgMCAwIDAgMSAxaDJhMSAxIDAgMCAwIDEtMXYtMmExIDEgMCAwIDAtMS0xaC0yeiIgY2xpcC1ydWxlPSJldmVub2RkIi8+PC9zdmc+',
            'origin'          => 'partner',
            'resource'        => 'mobile_mediation',
            'type'            => 'item',
            'url'             => '/v2/mobile-mediation/',
            'only_no_feature' => 'new_mobmed'
        },
        {
            'caption'  => gettext('Statistics'),
            'children' => [],
            'icon' =>
'data:image/svg+xml;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHdpZHRoPSIyNCIgaGVpZ2h0PSIyNCIgZmlsbD0ibm9uZSI+PHBhdGggZmlsbD0iI2ZmZiIgZmlsbC1ydWxlPSJldmVub2RkIiBkPSJNOS41IDVhLjUuNSAwIDAgMC0uNS41djEzYzAgLjI4LjIyLjUuNS41aDFhLjUuNSAwIDAgMCAuNS0uNXYtMTNhLjUuNSAwIDAgMC0uNS0uNWgtMXptLTQgMTBhLjUuNSAwIDAgMC0uNS41djNjMCAuMjguMjIuNS41LjVoMWEuNS41IDAgMCAwIC41LS41di0zYS41LjUgMCAwIDAtLjUtLjVoLTF6TTEzIDkuNWMwLS4yOC4yMi0uNS41LS41aDFjLjI4IDAgLjUuMjIuNS41djlhLjUuNSAwIDAgMS0uNS41aC0xYS41LjUgMCAwIDEtLjUtLjV2LTl6bTQuNSAzLjVhLjUuNSAwIDAgMC0uNS41djVjMCAuMjguMjIuNS41LjVoMWEuNS41IDAgMCAwIC41LS41di01YS41LjUgMCAwIDAtLjUtLjVoLTF6IiBjbGlwLXJ1bGU9ImV2ZW5vZGQiLz48L3N2Zz4=',
            'origin'          => 'partner',
            'resource'        => 'statistics',
            'type'            => 'item',
            'url'             => '/v2/statistics/constructor/',
            'only_no_feature' => 'simple_menu',
        },
        {
            'caption'  => gettext('Invitations'),
            'children' => [],
            'icon' =>
'data:image/svg+xml;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHdpZHRoPSIyNCIgaGVpZ2h0PSIyNCIgZmlsbD0ibm9uZSI+PHBhdGggZmlsbD0iI2ZmZiIgZmlsbC1ydWxlPSJldmVub2RkIiBkPSJNOC43OCAxMS4xNWMuMjQuNjEuNjIuOS44Ljg0LjE4LS4wNi4yNS0uMTguMjUtLjE4cy4xNi42NS4yNi45NmMuMzMgMS4wMSAxLjQ5IDIuMTYgMi4xNCAyLjM1LjM3LjExLjU3LjA4LjkyIDAgLjQ4LS4xMSAxLjU4LTEuMDMgMi4wOS0yLjNsLjA0LS4xYy4wOS0uMjYuMTctLjUuMjYtLjk1IDAtLjAyIDAtLjAxLjAxIDAgLjAzLjAzLjEuMS4yNy4xMS4yLjAzLjMxLS4yLjQ4LS42bC4xMy0uMjdjLjQ2LTEuMTMuNC0xLjkuMDgtMS45Ni0uMTUtLjAzLS4yNS4wNS0uMzUuMTUtLjA1IDAtLjEgMC0uMS0uMDUgMC0uMTEuMDEtLjI1LjA0LS40LjA0LS4zMy4xLS43Ni4wNS0xLjM0LjYxLTEuNjItLjU2LTIuNzctLjU2LTIuNzdzLS42NC4yLTEuOTItLjM1YTQuMzQgNC4zNCAwIDAgMC0zLjA2LS4wMmMtLjUuMTctLjguNjItLjczIDEuMS0uNDYtLjE5LS44OC4zLS45Ny41Ny40LjU2LjM1IDEuMTIuMzIgMS42YTUuNTIgNS41MiAwIDAgMCAuMTMgMS43OXMtLjEuMS0uMTkuMDFjLS4wOC0uMDgtLjMyLS4zMi0uNDYtLjMyLS4xNSAwLS4zNS4yNS0uMjguNzYuMDYuNTIuMS43NS4zNSAxLjM3ek0xNiAxNS42bC0uNzYuMzlIMTZ2LS4zOXptLTIgMS4wMXYzLjJjLS41LjA5LTEuMDQuMTMtMS42Mi4xMy0zLjQ2IDAtNS45OC0xLjY2LTYuOTUtMi40MmEuNC40IDAgMCAxIC4wNC0uNjhjLjYzLS4zOCAxLjg4LS45OSAzLjY1LTEuMjcuMS0uMDEuMiAwIC4yOC4wNC45LjQzIDIuMzkgMS4yNiAyLjk2IDEuNThhLjUuNSAwIDAgMCAuNDYgMGwxLjE4LS41OHptMy0xLjEyYzAtLjI4LjIyLS41LjUtLjVoMWMuMjggMCAuNS4yMi41LjVWMTdoMS41Yy4yOCAwIC41LjIyLjUuNXYxYS41LjUgMCAwIDEtLjUuNUgxOXYxLjVhLjUuNSAwIDAgMS0uNS41aC0xYS41LjUgMCAwIDEtLjUtLjVWMTloLTEuNWEuNS41IDAgMCAxLS41LS41di0xYzAtLjI4LjIyLS41LjUtLjVIMTd2LTEuNXoiIGNsaXAtcnVsZT0iZXZlbm9kZCIvPjwvc3ZnPg==',
            'origin'          => 'partner',
            'resource'        => 'inviter',
            'type'            => 'item',
            'url'             => '/v2/inviter/',
            'only_no_feature' => 'simple_menu',
        },
        {
            'caption'  => gettext('Documents'),
            'children' => [],
            'icon' =>
'data:image/svg+xml;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHdpZHRoPSIyNCIgaGVpZ2h0PSIyNCIgZmlsbD0ibm9uZSI+PHBhdGggZmlsbD0iI2ZmZiIgZmlsbC1ydWxlPSJldmVub2RkIiBkPSJNNyAzYTIgMiAwIDAgMC0yIDJ2MTRjMCAxLjEuOSAyIDIgMmgxMGEyIDIgMCAwIDAgMi0yVjcuODNhMiAyIDAgMCAwLS41OS0xLjQyTDE1LjYgMy42YTIgMiAwIDAgMC0xLjQzLS42SDd6bTcuNDMgMS40M2wzLjE0IDMuMTRjLjE2LjE2LjA1LjQzLS4xNy40M2gtMy4xNWEuMjUuMjUgMCAwIDEtLjI1LS4yNVY0LjZjMC0uMjIuMjctLjMzLjQzLS4xN3ptLTMuMTcgOS4xOHYtMi41aC42M2MuNDcgMCAuOC4xIDEgLjI5LjIuMTguMzEuNDkuMzEuOTEgMCAuODctLjQ1IDEuMy0xLjM0IDEuM2gtLjZ6bTAgMy44OXYtMS4yM0gxM2EuNDcuNDcgMCAwIDAgMC0uOTVoLTEuNzV2LS43NmguNTljLjQgMCAuNzgtLjA0IDEuMTItLjEyLjM0LS4xLjYzLS4yMy44OC0uNC4yNS0uMTkuNDMtLjQyLjU3LS43LjEzLS4yOS4yLS42NC4yLTEuMDRzLS4wNy0uNzMtLjItMS4wMmExLjggMS44IDAgMCAwLS41NS0uNzFjLS4yMy0uMi0uNTItLjM0LS44Ni0uNDItLjMzLS4xLS43LS4xNS0xLjEyLS4xNWgtMS41NWEuNS41IDAgMCAwLS41LjV2My4xaC0uMzdhLjQ3LjQ3IDAgMCAwIDAgLjk2aC4zN3YuNzZoLS4zN2EuNDcuNDcgMCAwIDAgMCAuOTVoLjM3djEuMjNjMCAuMjguMjIuNS41LjVoLjQyYS41LjUgMCAwIDAgLjUtLjV6IiBjbGlwLXJ1bGU9ImV2ZW5vZGQiLz48L3N2Zz4=',
            'origin'          => 'partner',
            'resource'        => 'partner_acts',
            'type'            => 'item',
            'url'             => '/v2/financial/acts/',
            'only_no_feature' => 'simple_menu',
        },
        {
            'caption'  => gettext('Offer'),
            'children' => [],
            'origin'   => 'partner',
            'resource' => 'common_offer',
            'type'     => 'item',
            'url'      => '/v2/offer/'
        },
        {
            'caption'  => gettext('Users'),
            'children' => [],
            'icon' =>
'data:image/svg+xml;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHdpZHRoPSIyNCIgaGVpZ2h0PSIyNCIgZmlsbD0ibm9uZSI+PHBhdGggZmlsbD0iI2ZmZiIgZmlsbC1ydWxlPSJldmVub2RkIiBkPSJNNS4yNiAxMy44M2MtLjIgMC0uNi0uMTgtLjg3LS43OS0uMjYtLjYxLS4zLS44NC0uMzgtMS4zNS0uMDctLjUuMTgtLjczLjQyLS43OC4yMy0uMDUuNC0uMDIuNS4wOS0uMDMtLjMtLjUyLTMuMTEtLjIxLTMuNzJhMi41NSAyLjU1IDAgMCAxIDIuMDQtMS40IDguOSA4LjkgMCAwIDEgMi4zMSAwYy0uNS40Ni0uNTggMS4zLS41NiAxLjgzLjAyLjU0LjIgMS40OC4yNiAxLjg4LTEuMDIuMjUtMS4xNSAxLjQ5LS44NiAyLjk2LjMgMS40NyAxLjY2IDMuMjUgMi40MyAzLjFsLjIuNTNjLS4zMy41OS0uNTcuNzUtMS4xLjg3LS4zNy4wOC0uOS4wNC0xLjMtLjA3YTQuNTIgNC41MiAwIDAgMS0yLjMyLTIuMzNjLS4xMi0uMzEtLjI5LS45NS0uMjktLjk1cy0uMDguMTMtLjI3LjEzem01LjQ3LjM4Yy0xLjMxLS4zNy0xLjc1LTMuMTQtMS4xNy0zLjMzLjU4LS4yLjg0IDAgLjg0LjEzLS4wMy0uMTUtLjktMy4wOS0uMy00LjI4LjYtMS4xOSAxLjM1LTEuMzMgMi4zNS0xLjU4Ljc3LS4yIDMuMDgtLjI2IDQuMjMuMTggMS42LjYxIDEuOTguMzggMS45OC4zOHMuNjYuOC4zNiAzLjI2Yy0uMDMuMi0uMjIgMS43My0uMjIgMi4wNCAwIC4wNi42Mi0uNDIgMS4xIDAgLjQ2LjQyLS43MyAzLjItMS4yNCAzLjItLjUgMC0uNDUtLjIyLS40OC0uMjUtLjAyLS4wMi0uMDMtLjAyLS4wMyAwLS4xLjUxLS4yMS43OC0uMzIgMS4wNmwtLjA1LjEyYy0uNjMgMS40My0yIDIuNDYtMi42IDIuNTktLjQzLjA5LS42OS4xMi0xLjE1IDBhNS4xNyA1LjE3IDAgMCAxLTIuNjYtMi42NCAxOS42IDE5LjYgMCAwIDEtLjMzLTEuMDlzLS4wOS4xNC0uMzEuMnoiIGNsaXAtcnVsZT0iZXZlbm9kZCIvPjwvc3ZnPg==',
            'origin'          => 'partner',
            'resource'        => 'users',
            'type'            => 'item',
            'url'             => '/v2/users/',
            'only_no_feature' => 'simple_menu',
        },
        {
            'caption'  => gettext('News'),
            'children' => [],
            'icon' =>
'data:image/svg+xml;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHdpZHRoPSIyNCIgaGVpZ2h0PSIyNCIgZmlsbD0ibm9uZSI+PHBhdGggZmlsbD0iI2ZmZiIgZmlsbC1ydWxlPSJldmVub2RkIiBkPSJNNC41IDRhLjUuNSAwIDAgMC0uNS41djJjMCAuMjguMjIuNS41LjVoMTVhLjUuNSAwIDAgMCAuNS0uNXYtMmEuNS41IDAgMCAwLS41LS41aC0xNXptMCA1YS41LjUgMCAwIDAtLjUuNXY5YzAgLjI4LjIyLjUuNS41aDZhLjUuNSAwIDAgMCAuNS0uNXYtOWEuNS41IDAgMCAwLS41LS41aC02em04LjUuNWMwLS4yOC4yMi0uNS41LS41aDZhLjUuNSAwIDAgMSAwIDFoLTZhLjUuNSAwIDAgMS0uNS0uNXptLjUgMi41YS41LjUgMCAwIDAgMCAxaDZhLjUuNSAwIDAgMCAwLTFoLTZ6bS0uNSAzLjVjMC0uMjguMjItLjUuNS0uNWg2YS41LjUgMCAwIDEgMCAxaC02YS41LjUgMCAwIDEtLjUtLjV6bS41IDIuNWEuNS41IDAgMCAwIDAgMWg0YS41LjUgMCAwIDAgMC0xaC00eiIgY2xpcC1ydWxlPSJldmVub2RkIi8+PC9zdmc+',
            'origin'          => 'partner',
            'resource'        => 'news',
            'type'            => 'item',
            'url'             => 'https://bunker.yandex-team.ru/pi-promo/news/',
            'only_no_feature' => 'simple_menu',
        },
        {
            'caption'  => gettext('For Developers'),
            'children' => [
                {
                    'caption'  => gettext('Information'),
                    'children' => [
                        {
                            'caption'  => gettext('Messages'),
                            'children' => [],
                            'origin'   => 'partner',
                            'resource' => 'devel_view_mail',
                            'type'     => 'item',
                            'url'      => '/devel/view_mail/'
                        },
                        {
                            'caption'  => gettext('Cron tasks'),
                            'children' => [],
                            'origin'   => 'partner',
                            'resource' => 'devel_cron_jobs',
                            'type'     => 'item',
                            'url'      => '/devel/cron_jobs/'
                        },
                        {
                            'caption'  => gettext('Statuses'),
                            'children' => [],
                            'origin'   => 'partner',
                            'resource' => 'devel_multistates',
                            'type'     => 'item',
                            'url'      => '/devel/multistates/'
                        },
                        {
                            'caption'  => gettext('Queue'),
                            'children' => [],
                            'origin'   => 'partner',
                            'resource' => 'queue',
                            'type'     => 'item',
                            'url'      => '/v2/queue/list/'
                        },
                        {
                            'caption'  => gettext('Class graph'),
                            'children' => [],
                            'origin'   => 'partner',
                            'resource' => 'devel_class_graph',
                            'type'     => 'item',
                            'url'      => '/devel/class_graph/'
                        },
                        {
                            'caption'  => gettext('Show contracts'),
                            'children' => [],
                            'origin'   => 'partner',
                            'resource' => 'devel_show_agreements',
                            'type'     => 'item',
                            'url'      => '/devel/show_agreements/'
                        }
                    ],
                    'origin'   => 'partner',
                    'resource' => undef,
                    'type'     => 'group',
                    'url'      => ''
                },
                {
                    'caption'  => gettext('Management'),
                    'children' => [
                        {
                            'caption'  => gettext('Text templates'),
                            'children' => [],
                            'origin'   => 'partner',
                            'resource' => 'text_template',
                            'type'     => 'item',
                            'url'      => '/devel/text_template_list/'
                        },
                        {
                            'caption'  => gettext('Internal API ACLs'),
                            'children' => [],
                            'origin'   => 'partner',
                            'resource' => 'intapi_acl',
                            'type'     => 'item',
                            'url'      => '/devel/intapi_acl_list/'
                        },
                        {
                            'caption'  => gettext('Simple notification'),
                            'children' => [],
                            'origin'   => 'partner',
                            'resource' => 'simple_notification_tool',
                            'type'     => 'item',
                            'url'      => '/devel/simple_notification_list/'
                        },
                        {
                            'caption'  => gettext('Notification'),
                            'children' => [],
                            'origin'   => 'partner',
                            'resource' => 'notification_tool',
                            'type'     => 'item',
                            'url'      => '/devel/notification_list/'
                        },
                        {
                            'caption'  => gettext('moderation_reason'),
                            'children' => [],
                            'origin'   => 'partner',
                            'resource' => 'moderation_reason_tool',
                            'type'     => 'item',
                            'url'      => '/devel/moderation_reason_list/'
                        },
                        {
                            'caption'  => gettext('Pages tags'),
                            'children' => [],
                            'origin'   => 'partner',
                            'resource' => 'pages_tags',
                            'type'     => 'item',
                            'url'      => '/v2/devel/pages_tags/'
                        },
                        {
                            'caption'  => gettext('Support queue task'),
                            'children' => [],
                            'origin'   => 'partner',
                            'resource' => 'queue_support_task',
                            'type'     => 'item',
                            'url'      => '/v2/queue/list/'
                        },
                    ],
                    'origin'   => 'partner',
                    'resource' => undef,
                    'type'     => 'group',
                    'url'      => ''
                },
                {
                    'caption'  => gettext('Balance'),
                    'children' => [
                        {
                            'caption'  => gettext('Delete user'),
                            'children' => [],
                            'origin'   => 'partner',
                            'resource' => 'devel_drop_user',
                            'type'     => 'item',
                            'url'      => '/devel/drop_user/'
                        },
                        {
                            'caption'  => gettext('Create client'),
                            'children' => [],
                            'origin'   => 'partner',
                            'resource' => 'devel_create_client',
                            'type'     => 'item',
                            'url'      => '/devel/create_client/'
                        },
                        {
                            'caption'  => gettext('Remove client from user'),
                            'children' => [],
                            'origin'   => 'partner',
                            'resource' => 'devel_remove_user_client_association',
                            'type'     => 'item',
                            'url'      => '/devel/remove_user_client_association/'
                        },
                        {
                            'caption'  => gettext('CreateOrUpdatePlace switch'),
                            'children' => [],
                            'origin'   => 'partner',
                            'resource' => 'devel_create_or_update_place_switch',
                            'type'     => 'item',
                            'url'      => '/devel/create_or_update_place_switch/'
                        }
                    ],
                    'origin'   => 'partner',
                    'resource' => undef,
                    'type'     => 'group',
                    'url'      => ''
                },
                {
                    'caption'  => gettext('BS'),
                    'children' => [
                        {
                            'caption'  => gettext('BS and PI data comparison'),
                            'children' => [],
                            'origin'   => 'partner',
                            'resource' => 'devel_bk_pi_comparison',
                            'type'     => 'item',
                            'url'      => '/devel/bk_pi_comparison/'
                        },
                        {
                            'caption'  => gettext('View BS send log'),
                            'children' => [],
                            'origin'   => 'partner',
                            'resource' => 'devel_bk_send_log',
                            'type'     => 'item',
                            'url'      => '/devel/bk_send_log/'
                        },
                        {
                            'caption'  => gettext('Change options on Page'),
                            'children' => [],
                            'origin'   => 'partner',
                            'resource' => 'devel_update_options',
                            'type'     => 'item',
                            'url'      => '/devel/update_options/'
                        }
                    ],
                    'origin'   => 'partner',
                    'resource' => undef,
                    'type'     => 'group',
                    'url'      => ''
                },
                {
                    'caption'  => gettext('Check'),
                    'children' => [
                        {
                            'caption'  => gettext('<nobr>lang-detect</nobr> test tool'),
                            'children' => [],
                            'origin'   => 'partner',
                            'resource' => 'devel_lang_detect_check',
                            'type'     => 'item',
                            'url'      => '/devel/lang_detect_check/'
                        },
                        {
                            'caption'  => gettext('Fields check'),
                            'children' => [],
                            'origin'   => 'partner',
                            'resource' => 'devel_fields',
                            'type'     => 'item',
                            'url'      => '/devel/fields/'
                        }
                    ],
                    'origin'   => 'partner',
                    'resource' => undef,
                    'type'     => 'group',
                    'url'      => ''
                }
            ],
            'icon' =>
'data:image/svg+xml;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHdpZHRoPSIyNCIgaGVpZ2h0PSIyNCIgZmlsbD0ibm9uZSI+PHBhdGggZmlsbD0iI2ZmZiIgZmlsbC1ydWxlPSJldmVub2RkIiBkPSJNMTQuMyA0LjA0YTEgMSAwIDAgMSAuNjYgMS4yNWwtNCAxM2ExIDEgMCAwIDEtMS45Mi0uNThsNC0xM2ExIDEgMCAwIDEgMS4yNS0uNjd6TTYuNyA4LjNhMSAxIDAgMCAxIDAgMS40Mkw0LjQyIDEybDIuMyAyLjNhMSAxIDAgMSAxLTEuNDIgMS40bC0yLjY0LTIuNjRhMS41IDEuNSAwIDAgMSAwLTIuMTJMNS4zIDguMjlhMSAxIDAgMCAxIDEuNDIgMHptMTIgMGExIDEgMCAxIDAtMS40IDEuNDJMMTkuNTggMTJsLTIuMyAyLjNhMSAxIDAgMCAwIDEuNDIgMS40bDIuNjQtMi42NGExLjUgMS41IDAgMCAwIDAtMi4xMkwxOC43IDguMjl6IiBjbGlwLXJ1bGU9ImV2ZW5vZGQiLz48L3N2Zz4=',
            'origin'          => 'partner',
            'resource'        => undef,
            'type'            => 'group',
            'url'             => '/v2/dashboard/',
            'only_no_feature' => 'simple_menu',
        },
        {
            'caption'  => gettext('Cookie match'),
            'children' => [
                {
                    'caption'  => gettext('List of sync sources'),
                    'children' => [],
                    'origin'   => 'partner',
                    'resource' => 'cookie_match',
                    'type'     => 'item',
                    'url'      => '/v2/cookie-matching/'
                }
            ],
            'icon' =>
'data:image/svg+xml;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHdpZHRoPSIyNCIgaGVpZ2h0PSIyNCIgZmlsbD0ibm9uZSI+PHBhdGggZmlsbD0iI2ZmZiIgZmlsbC1ydWxlPSJldmVub2RkIiBkPSJNMTIgMjBhOCA4IDAgMSAwIDAtMTYgOCA4IDAgMCAwIDAgMTZ6bTQuNTctOGExLjE0IDEuMTQgMCAxIDAgMC0yLjI5IDEuMTQgMS4xNCAwIDAgMCAwIDIuMjl6TTE0LjMgOC41N2ExLjE0IDEuMTQgMCAxIDAgMC0yLjI4IDEuMTQgMS4xNCAwIDAgMCAwIDIuMjh6bS0uNTggOS4xNGExLjcxIDEuNzEgMCAxIDAgMC0zLjQyIDEuNzEgMS43MSAwIDAgMCAwIDMuNDJ6bS01LjE0LTIuMjhhMS4xNCAxLjE0IDAgMSAwIDAtMi4yOSAxLjE0IDEuMTQgMCAwIDAgMCAyLjI5em0uNTctNC41N2ExLjcxIDEuNzEgMCAxIDAgMC0zLjQzIDEuNzEgMS43MSAwIDAgMCAwIDMuNDN6IiBjbGlwLXJ1bGU9ImV2ZW5vZGQiLz48L3N2Zz4=',
            'origin'          => 'partner',
            'resource'        => undef,
            'type'            => 'group',
            'url'             => '/v2/cookie-matching/',
            'only_no_feature' => 'simple_menu',
        },
        {
            'caption'  => gettext('Moderation'),
            'children' => [],
            'icon' =>
'data:image/svg+xml;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHdpZHRoPSIyNCIgaGVpZ2h0PSIyNCIgZmlsbD0ibm9uZSI+PHBhdGggZmlsbD0iI2ZmZiIgZD0iTTEyLjM0IDIxLjg1YzEuNS0uNzQgNy40My00LjMzIDcuNjUtMTUuMjJhMSAxIDAgMCAwLS41NC0uOWwtNy0zLjVhMSAxIDAgMCAwLS45IDBsLTcgMy41YTEgMSAwIDAgMC0uNTQuOWMuMjIgMTAuODkgNi4xNSAxNC40OCA3LjY1IDE1LjIyLjIyLjEuNDYuMS42OCAweiIvPjwvc3ZnPg==',
            'origin'          => 'partner',
            'resource'        => 'moderation',
            'type'            => 'item',
            'url'             => '/v2/moderation/?search=%7B%22multistate%22%3A%5B%22%3D%22%2C%22new%22%5D%7D',
            'only_no_feature' => 'simple_menu',
        },
        {
            'caption'  => gettext('ADFOX'),
            'children' => [],
            'icon' =>
'data:image/svg+xml;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHdpZHRoPSIyNCIgaGVpZ2h0PSIyNCIgZmlsbD0ibm9uZSI+PHBhdGggZmlsbD0iI2ZmZiIgZmlsbC1ydWxlPSJldmVub2RkIiBkPSJNNi45NiAxNi4xQzMuNSAxMi4xNSAyLjIgOC4yMyAzLjUgM2M0LjM0IDAgNy43IDIuNjIgNy43IDIuNjJzLTIuNDUgMS4zLTMuODYgMy45M2gxLjI5cy0yLjU3IDMuOTItMS42NiA2LjU0em0xMi40NC0yLjU3QzIyLjUyIDguMTUgMjAuMTIgMyAyMC4xMiAzcy0yIDAtNS4yMiAxLjI5Yy0zLjIyIDEuMjgtNC43IDUuMTQtNC43IDUuMTRoMS4zMnMtMS45NyAwLTEuOTcgNS4xNGMuNjUgNS4xNCAxLjk3IDYuNDMgMS45NyA2LjQzczQuNzYtMi4wOCA3Ljg4LTcuNDd6IiBjbGlwLXJ1bGU9ImV2ZW5vZGQiLz48L3N2Zz4=',
            'origin'          => 'adfox',
            'resource'        => undef,
            'type'            => 'item',
            'url'             => get_menu_adfox_url($app),
            'only_no_feature' => 'simple_menu',
        },
        {
            'caption'  => gettext('ADFOX'),
            'children' => [],
            'icon' =>
'data:image/svg+xml,<svg width="20" height="20" viewBox="0 0 20 20" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M9.93282 18.3076C8.34356 15.5636 7.72472 11.7788 9.41462 8.35909C11.8152 3.50109 16.2251 3.11053 17.8603 3.11112C17.9078 3.24983 17.9602 3.42149 18.0135 3.62447C18.2003 4.33478 18.3792 5.35394 18.4144 6.5334C18.4851 8.9011 17.9763 11.8019 15.9308 14.1884C13.553 16.9625 11.4267 17.8129 9.93282 18.3076ZM18.4622 3.12587C18.4487 3.12587 18.4354 3.1256 18.4223 3.12507C18.4383 3.12557 18.4516 3.12587 18.4622 3.12587ZM5.13516 8.10918C4.52468 9.36827 4.21493 10.8306 4.0719 12.1147C3.46046 11.1915 2.83596 10.0335 2.42116 8.74295C1.75122 6.65868 2.00612 4.38822 2.32626 3.06055C3.26628 3.15612 4.58417 3.35619 5.53839 3.69699C6.24293 3.94861 6.8817 4.27712 7.39142 4.58717C7.18481 4.82155 6.97602 5.08477 6.77034 5.36606C6.20033 6.14562 5.61742 7.11456 5.13516 8.10918Z" stroke="white" stroke-width="2"/></svg>',
            'origin'            => 'adfox',
            'resource'          => undef,
            'type'              => 'item',
            'url'               => get_menu_adfox_url($app),
            'only_with_feature' => 'simple_menu',
        },
        {
            'caption'  => gettext('Settings'),
            'children' => [],
            'icon' =>
'data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iMjQiIGhlaWdodD0iMjQiIGZpbGw9Im5vbmUiIHhtbG5zPSJodHRwOi8vd3d3LnczLm9yZy8yMDAwL3N2ZyI+CiAgICA8cGF0aCBmaWxsLXJ1bGU9ImV2ZW5vZGQiIGNsaXAtcnVsZT0iZXZlbm9kZCIKICAgICAgICAgIGQ9Im00LjkyOSA0LjkzLjAwMS0uMDAyLjAwMi4wMDEuNTI3LS41MjhhLjU3NS41NzUgMCAwIDEgLjc4Ni0uMDI1bDEuMjEgMS4wNjFhMS44NTYgMS44NTYgMCAwIDAgMy4xMTUtMS4yOTFoLjAwNGwuMTA1LTEuNjA3YS41NzUuNTc1IDAgMCAxIC41NzQtLjUzN2guNzQ2VjJ2LjAwMmguNzQ3Yy4zMDMgMCAuNTU0LjIzNS41NzQuNTM3bC4xMDUgMS42MDdoLjAwNWMuMDE5LjQ4NC4yMjMuOTIuNTQ0IDEuMjRhMS44NTQgMS44NTQgMCAwIDAgMi41ODQuMDM5bDEuMTk2LTEuMDVhLjU3NS41NzUgMCAwIDEgLjc4Ni4wMjZsLjUyOC41MjguMDAyLS4wMDJ2LjAwMmwtLjAwMS4wMDIuNTI4LjUyN2EuNTc1LjU3NSAwIDAgMSAuMDI2Ljc4NmwtMS4wNiAxLjIxMmExLjg1NCAxLjg1NCAwIDAgMCAuMDU2IDIuNTc1Yy4zMi4zMTguNzUzLjUyIDEuMjM1LjUzOXYuMDA0bDEuNjA2LjEwNWMuMzAzLjAyLjUzOC4yNzEuNTM4LjU3NFYxMkgyMnYuMDAyaC0uMDAydi43NDZhLjU3NS41NzUgMCAwIDEtLjUzNy41NzRsLTEuNjA3LjEwN3YuMDAxYy0uNDg0LjAyLS45Mi4yMjMtMS4yNC41NDRhMS44NTQgMS44NTQgMCAwIDAtLjA1IDIuNTcyaC0uMDAybDEuMDYyIDEuMjExYy4yLjIyOC4xODguNTcyLS4wMjYuNzg2bC0uNTI4LjUyOHYuMDAyaC0uMDAxbC0uNTI4LjUyN2EuNTc1LjU3NSAwIDAgMS0uNzg1LjAyNmwtMS4xNjgtMS4wMjFhMS44NTQgMS44NTQgMCAwIDAtMi42MTkuMDE0IDEuODYgMS44NiAwIDAgMC0uNTQgMS4yMzhoLS4wMDRsLS4xMDUgMS42MDdhLjU3NS41NzUgMCAwIDEtLjU0LjUzNmgtMS41NmEuNTc1LjU3NSAwIDAgMS0uNTQtLjUzNmwtLjEwNS0xLjYwN2gtLjAwNGExLjg1MSAxLjg1MSAwIDAgMC0xLjE0Ni0xLjY0NSAxLjg1IDEuODUgMCAwIDAtMi4wMDQuMzg1bC0xLjE3NyAxLjAzYS41NzUuNTc1IDAgMCAxLS43ODUtLjAyN2wtLjUyOC0uNTI4LS4wMDEtLjAwMS0uNTI4LS41MjhhLjU3NS41NzUgMCAwIDEtLjAyNi0uNzg2bDEuMDYyLTEuMjEtLjAwMS0uMDAxYTEuODU0IDEuODU0IDAgMCAwLTEuMjkxLTMuMTE2di0uMDAxbC0xLjYwNy0uMTA3YS41NzUuNTc1IDAgMCAxLS41MzctLjU3NHYtLjc0NkgyVjEyaC4wMDF2LS43NDdjMC0uMzAzLjIzNS0uNTU0LjUzOC0uNTc0bDEuNjA2LS4xMDV2LS4wMDRhMS44NSAxLjg1IDAgMCAwIDEuMjQyLS41NDUgMS44NTQgMS44NTQgMCAwIDAgLjA0My0yLjU3N0w0LjM3NiA2LjI0NGEuNTc1LjU3NSAwIDAgMSAuMDI2LS43ODZsLjUyOC0uNTI3LS4wMDEtLjAwMnYuMDAxWk0xNi4yODYgMTJhNC4yODYgNC4yODYgMCAxIDEtOC41NzMgMCA0LjI4NiA0LjI4NiAwIDAgMSA4LjU3MyAwWiIKICAgICAgICAgIGZpbGw9IiNmZmYiLz4KPC9zdmc+Cg==',
            'origin'            => 'partner',
            'resource'          => 'settings_in_menu',
            'type'              => 'item',
            'url'               => '/v2/settings',
            'only_with_feature' => 'editable_requisites'
        },
    ];
}

sub get_menu_adfox_url {
    my ($app) = @_;

    my $url = $app->get_option('adfox_web');
    if ($url =~ /\[%TLD%\]/) {
        my $tld        = 'ru';
        my $controller = $app->get_option('restapi_controller');
        if ($controller) {
            my $request = $controller->req;
            if ($request) {
                my $env    = $request->env;
                my $domain = $env->{HTTP_HOST};
                $tld = 'com' if ($domain && $domain =~ /\.yandex\.com/i);
            }
        }
        $url =~ s/\[%TLD%\]/$tld/;
    }

    return $url;
}

TRUE;
