package Partner::BlockPreset;

use qbit;

use base qw(QBit::Class);

use Partner::BlockPreset::Component::Strategies;
use Partner::BlockPreset::Component::Brands;
use Partner::BlockPreset::Component::Geo;
use Partner::BlockPreset::Component::Articles;
use Partner::BlockPreset::Component::PICategories;

use Exception::Validation::BadArguments;

use PiConstants qw($DEFAULT_CPM_CURRENCY);

__PACKAGE__->mk_ro_accessors(qw(app settings));

my %CLASS_BY_NAME = (
    strategies_group   => 'Partner::BlockPreset::Component::Strategies',
    brands_group       => 'Partner::BlockPreset::Component::Brands',
    geo_group          => 'Partner::BlockPreset::Component::Geo',
    articles_group     => 'Partner::BlockPreset::Component::Articles',
    picategories_group => 'Partner::BlockPreset::Component::PICategories',
);

sub build_and_validate {
    my ($class, $app, $settings) = @_;

    return $class->new(app => $app, settings => $settings);
}

sub init {
    my ($self) = @_;

    my $settings = $self->settings();

    my @errors = ();
    foreach my $name (sort keys(%$settings)) {
        unless (exists($CLASS_BY_NAME{$name})) {
            push(
                @errors,
                {
                    msgs => [gettext('Unknown component "%s"', $name)],
                    path => [$name]
                }
            );

            next;
        }

        my $component = $CLASS_BY_NAME{$name}->new(app => $self->app, settings => $settings->{$name});

        push(@{$self->{'__COMPONENTS__'}}, $component);

        my $qv = QBit::Validator->new(
            data     => $component->settings,
            template => {
                type   => 'hash',
                fields => $component->specification->get_validation_template(),
            },
            app     => $component,
            pre_run => $component->specification->get_validation_pre_run(),
        );

        if ($qv->has_errors) {
            push(@errors, map {unshift(@{$_->{'path'}}, $name); $_} $qv->get_fields_with_error);
        }
    }

    $self->{'__ERRORS__'} = \@errors;
}

sub get_errors {
    my ($self) = @_;

    return @{$self->{'__ERRORS__'}} ? $self->{'__ERRORS__'} : undef;
}

sub get_filter {
    my ($self, $model) = @_;

    my @filters = ();
    foreach my $component (@{$self->{'__COMPONENTS__'}}) {
        if (my $filter = $component->get_filter($model)) {
            #TODO: заиспользовать фильтры из Partner::DSP
            return $filter if ref($filter) eq '' && $filter eq FALSE;

            push(@filters, $filter);
        }
    }

    my $multistates = $model->get_multistate_names();
    if (exists($multistates->{'deleted'})) {
        push(@filters, ['multistate', '=', 'not deleted']);
    }

    if ($model->DOES('Application::Model::Role::Has::CustomBkData')) {
        push(@filters, ['is_custom_bk_data', '=', FALSE]);
    }

    if ($model->DOES('Application::Model::Role::Has::IsMobileMediationBase')) {
        push(@filters, ['is_mobile_mediation', '=', FALSE]);
    }

    if ($model->DOES('Application::Model::Role::Block::Has::CPMCurrency')) {
        push(@filters, ['OR', [['current_currency', 'IS', undef], ['current_currency', '=', $DEFAULT_CPM_CURRENCY]]]);
    }

    my $page_multistates = $model->page->get_multistate_names();
    my @statuses         = ();
    if (exists($page_multistates->{'protected'})) {
        push(@statuses, 'protected');
    }

    if (exists($page_multistates->{'blocked'})) {
        push(@statuses, 'blocked');
    }

    if (@statuses) {
        push(@filters, ['page', 'MATCH', ['multistate', '=', 'not (' . join(' or ', @statuses) . ')']]);
    }

    if ($model->check_rights('is_assistant') && $model->page->is_external_page()) {
        my $user_id = $model->app->get_option('cur_user')->{'id'};

        my @page_ids =
          map {$_->{'page_id'}} @{$model->page->get_all(fields => [qw(page_id)], filter => {owner_id => $user_id})};

        push(
            @page_ids,
            map {$_->{'page_id'}} @{
                $model->app->assistants->get_all(
                    fields => [qw(page_id)],
                    filter => {user_id => $user_id, can_edit => TRUE}
                )
              }
            );

        push(@filters, ['page_id', 'IN', \@page_ids]);
    }

    return ['AND', \@filters];
}

sub get_patch {
    my ($self, $model) = @_;

    my $patch = {};

    foreach my $component (@{$self->{'__COMPONENTS__'}}) {
        $patch = {%$patch, $component->get_patch($model)};
    }

    return $patch;
}

TRUE;
