package Partner::BlockPreset::Specification::Strategies;

use qbit;

use base qw(Partner::BlockPreset::Specification);

use Exception::Validator::Fields;

use PiConstants qw(
  $MAX_CPM
  $MAX_REVENUE_STRATEGY_ID
  $MIN_CPM_STRATEGY_ID
  $SEPARATE_CPM_STRATEGY_ID
  );

__PACKAGE__->mk_ro_accessors(qw(all_strategies));

our $STRATEGIES = [
    {id => $MAX_REVENUE_STRATEGY_ID, name => d_gettext('Maximum revenue'), fields => []},
    {id => $MIN_CPM_STRATEGY_ID,     name => d_gettext('Minimum CPM'),     fields => ['mincpm']},
    {
        id     => $SEPARATE_CPM_STRATEGY_ID,
        name   => d_gettext('Separate CPM'),
        fields => []
    },
];

sub init {
    my ($self) = @_;

    my $specification = $self->specification;

    my $strategies = clone($STRATEGIES);

    my @strategies = ();
    foreach my $row (@$strategies) {
        next unless $specification->{$row->{'id'}};

        if ($row->{'id'} == $SEPARATE_CPM_STRATEGY_ID) {
            $row->{'fields'} =
              [map {("${_}_active", "${_}_blocked", "${_}_cpm")}
                  @{$specification->{$SEPARATE_CPM_STRATEGY_ID}{'fields'}}];
        }

        if ($specification->{$row->{'id'}}{'applicable_to'}) {
            $row->{'applicable_to'} = $specification->{$row->{'id'}}{'applicable_to'};
        }

        push(@strategies, $row);
    }

    $self->{'all_strategies'} = \@strategies;
}

sub get_strategies_types {
    my ($self, $model, $obj) = @_;

    my $strategies = clone($self->get_filtered_strategies_types($model, $obj));

    foreach (@$strategies) {
        delete($_->{'applicable_to'});

        $_->{'name'} = $_->{'name'}->();
    }

    return $strategies;
}

sub get_strategies_types_as_hash {
    my ($self, $model, $obj) = @_;

    return {map {$_->{'id'} => $_->{'fields'}} @{$self->get_filtered_strategies_types($model, $obj)}};
}

sub get_filtered_strategies_types {
    my ($self, $model, $obj) = @_;

    return [grep {!$_->{'applicable_to'}{'sub_filter'} || $_->{'applicable_to'}{'sub_filter'}->($model, $obj)}
          @{$self->all_strategies()}];
}

sub get_validation_template {
    my ($self) = @_;

    my $strategies = $self->specification;

    return {
        strategy => {
            check => sub {
                my ($qv, $strategy) = @_;

                my $data = $qv->data();

                my $strategy_types = $self->get_strategies_types_as_hash($qv->app, $data);

                throw Exception::Validator::Fields gettext('Incorrect strategy value')
                  unless $strategy_types->{$strategy};

                if ($strategy == $SEPARATE_CPM_STRATEGY_ID) {
                    my $active         = 0;
                    my $blocked        = 0;
                    my $blocked_fields = 0;

                    foreach my $field (@{$strategy_types->{$strategy}}) {
                        if ($field =~ /_active$/) {
                            $active += $data->{$field} // 0;
                        } elsif ($field =~ /_blocked$/) {
                            my ($ad_type) = split('_', $field);
                            $blocked_fields++
                              if exists($data->{$ad_type . '_active'}) && defined $data->{$ad_type . '_active'};
                            $blocked += $data->{$field} // 0;
                        }
                    }

                    throw Exception::Validator::Fields gettext('You should set CPM or block ads')
                      if $active == 0;

                    throw Exception::Validator::Fields gettext('You cannot blocked all ads')
                      if $blocked == $blocked_fields;
                }
            },
        },
        (
            $strategies->{$MIN_CPM_STRATEGY_ID}
            ? (
                mincpm => {
                    type => 'cpm_zero',
                    max  => $MAX_CPM,
                    msg  => d_gettext("mincpm value must be from '0' to '%d'", $MAX_CPM),
                },
              )
            : ()
        ),
        (
            $strategies->{$SEPARATE_CPM_STRATEGY_ID} ? map {
                (
                    "${_}_active"  => {type => 'boolean'},
                    "${_}_blocked" => {type => 'boolean'},
                    "${_}_cpm"     => {
                        type => 'cpm',
                        msg  => d_gettext("%s value must be from '0.001' to '%d'", "${_}_cpm", $MAX_CPM),
                    },
                )
              } @{$strategies->{$SEPARATE_CPM_STRATEGY_ID}{'fields'}}
            : ()
        ),
    };
}

sub get_validation_pre_run {
    my ($self) = @_;

    return sub {
        my ($qv) = @_;

        my $template = $qv->template;
        my $fields   = $template->{'fields'};

        my $deps_fields_from_strategy = $self->get_strategies_types_as_hash($qv->app);

        foreach my $strategy (
            grep {defined($qv->data->{'strategy'}) && $_ ne $qv->data->{'strategy'}}
            keys(%$deps_fields_from_strategy)
          )
        {
            foreach (@{$deps_fields_from_strategy->{$strategy}}) {
                $fields->{$_} = {eq => undef};
            }
        }

        # TODO:
        # make two separate required methods instead of just one
        # - one returns all possible strategies for model (get_all_strategies)
        # - another limits the set of all possible strategies
        #   to those available for specific block (filter_strategies_for_block)
        # - in addition to that define third method that calls
        #   filter_strategies_for_block(get_all_strategies()) (get_strategies_for_block)
        my $strategies = $self->get_strategies_types_as_hash($qv->app, $qv->data);

        if ($strategies->{$SEPARATE_CPM_STRATEGY_ID}) {
            foreach (@{$self->specification->{$SEPARATE_CPM_STRATEGY_ID}{'fields'}}) {
                if (defined($qv->data->{"${_}_active"}) && !$qv->data->{"${_}_active"}) {
                    $fields->{"${_}_blocked"} = {eq => undef};
                    $fields->{"${_}_cpm"}     = {eq => undef};
                }

                if (defined($qv->data->{"${_}_blocked"}) && $qv->data->{"${_}_blocked"}) {
                    $fields->{"${_}_cpm"} = {eq => undef};
                }
            }

            if ($qv->app->can('get_non_applicable_seperate_cpm_strategy_fields')) {
                foreach ($qv->app->get_non_applicable_seperate_cpm_strategy_fields($qv->data)) {
                    $fields->{$_} = {eq => undef, optional => 1};
                }
            }

        } else {
            foreach (@{$deps_fields_from_strategy->{$SEPARATE_CPM_STRATEGY_ID}}) {
                $fields->{$_} = {eq => undef, optional => 1};
            }
        }
    };
}

sub get_seperate_cpm_strategy_fields {
    my ($self, $ad_type_list) = @_;

    my @fields;
    for my $suffix (qw(active blocked cpm)) {
        push @fields, map {$_ . '_' . $suffix} @$ad_type_list;
    }
    return @fields;
}

TRUE;
