package Partner2::Code;

=encoding UTF-8

=cut

use strict;
use warnings FATAL => 'all';

use File::Find;
use Exporter;

our @ISA       = qw(Exporter);
our @EXPORT_OK = qw(
  get_expected_sub_order
  get_regex_subs_to_order
  get_pm_files
  );
our @EXPORT = @EXPORT_OK;

=head2 get_expected_sub_order

    my @expected_sub_order = get_expected_sub_order();

Возвращает список скаляров. Этот список - порядок в котором должны
следовать сабы в .pm файлах.

=cut

sub get_expected_sub_order {
    my @expected_sub_order = (
        # "хедер" - однострочные ф-ции
        'accessor',
        'db_table_name',
        'parent_accessor',    # - для статистики
        'page_model_name',
        'get_campaign_model_name',
        'get_product_name',
        'get_page_id_field_name',
        'get_block_model_names',
        'db_banner_lang',
        'block_seq_db_table',
        'get_need_update_in_bk_block_models',

        # может быть либо public_id либо public_id_prefix
        'public_id_prefix',
        'public_id',

        # Традиционно: аксессоры, декларация прав, поля, фильтры, граф состояний.
        'model_accessors',
        'get_structure_model_accessors',

        'register_rights',
        'get_structure_rights_to_register',

        'model_fields',
        'get_structure_model_fields',

        'model_filter',
        'get_structure_model_filter',
        'get_db_filter_simple_fields'
        ,    # возможно позже сольется с get_structure_model_filter в экстазе

        'multistates_graph',
        'get_structure_multistates_graph',

        'pre_process_fields',

        # Потроха для вычислимых полей

        # get_campaigns_*_cnt:
        'get_campaigns_adfox_cnt',
        'get_campaigns_block_cnt',
        'get_campaigns_direct_cnt',
        'get_campaigns_for_add',
        'get_campaigns_premium_cnt',
        'get_campaigns_rtb_cnt',
        'get_campaigns_stripe_cnt',

        'get_available_fields',    # (в модели должно быть поле available_fields)
        'get_add_fields',
        'get_editable_fields',     # (в модели должно быть поле editable_fields)
        '_get_common_add_edit_fields',

        # Изменение данных - add
        'add',
        'can_action_add',
        'on_action_add',

        # Изменение данных - edit
        'edit',
        'can_action_edit',
        'on_action_edit',

        # Изменение данных - delete
        'delete',
        'can_action_delete',
        'on_action_delete',

        # Изменение данных - duplicate
        'duplicate',
        'can_action_duplicate',
        'on_action_duplicate',

        qr/^(can|on)_action_([^\s]+)/,

        # Валидатор данных для изменения
        'get_template',

        # Основной запрос к баз для модели
        'query',

        # "BK"
        'get_need_update_in_bk_fields',
        'get_bk_data',
        'update_in_bk',

        # API
        'api_available_fields',
        'api_available_actions',
        'api_can_edit',
    );

    my %h;
    foreach my $sub_name (@expected_sub_order) {
        $h{$sub_name}++;
    }

    my $has_error;
    foreach my $sub_name (keys(%h)) {
        if ($h{$sub_name} > 1) {
            warn "Several usage of '$sub_name'";
            $has_error = 1;
        }
    }

    die "Incorrect list of subs" if $has_error;

    return @expected_sub_order;
}

=head2 get_pm_files

    my @files = get_pm_files

Возвращает массив со списком .pm файлов, которые находятся в папке lib/

Пордок файлов не определен.

Пример того что может вернуть эта саба:

    lib/Application.pm
    lib/Application/Model/Product/AN/Site.pm

=cut

sub get_pm_files {
    my @files;

    find(
        {
            wanted => sub {
                if (-f $File::Find::name && $File::Find::name =~ /\.pm\z/) {
                    push @files, $File::Find::name;
                }
            },
            no_chdir => 1,
        },
        'lib',
    );

    return @files;
}

sub get_regex_subs_to_order {
    return {qr/^(can|on)_action_([^\s]+)/, => \&_order_can_on_actions,};
}

sub _order_can_on_actions {
    my $subs = shift // [];

    my $h = {};
    foreach my $sub_name (@$subs) {
        my ($prefix, $action) = ($sub_name =~ /^(can|on)_action_([^\s]+)/);
        $h->{$action}->{$sub_name} = 1;
    }

    my @ordered_subs = ();
    foreach my $action (sort keys %$h) {
        push @ordered_subs, sort keys %{$h->{$action}};
    }

    return \@ordered_subs;
}

1;
