
=head1 Name

QBit::Application - base class for create applications.

=head1 Description

It union all project models.

=cut

package QBit::Application;

use qbit;

use base qw(QBit::Class);

use lib::abs;
use File::Basename qw( basename );
use File::Slurp qw(read_file);

use QBit::Application::_Utils::TmpLocale;
use QBit::Application::_Utils::TmpRights;

use Exception::Validation::BadArguments;

use Utils::Logger qw(ERROR);
use Utils::Safe qw(fail_connect_on_production);

=head2 add_tmp_rights

Short method description

B<Arguments:>

=over

=item

B<@rights> - type, description

=back

B<Return value:> type, description

=cut

sub add_tmp_rights {
    my ($self, @rights) = @_;

    return QBit::Application::_Utils::TmpRights->new(app => $self, rights => \@rights);
}

=head2 check_rights

    my $bool = $app->check_rights('right_1');

Вернет true значение если у текущего пользователя есть право 'right_1'. Если
права нет, то вернет false.

=cut

sub check_rights {$_[0]->{'__CURRENT_USER_RIGHTS__'}{$_[1]} ? TRUE : FALSE}

=head1 Package methods

=head2 config_opts

Short method description

B<Arguments:>

=over

=item

B<%opts> - additional arguments:

=back

B<Return value:> type, description

=cut

sub config_opts {
    my ($self, %opts) = @_;

    my $class = ref($self) || $self;

    my $pkg_name = $class;
    $pkg_name =~ s/::/\//g;
    $pkg_name .= '.pm';

    $self->_push_pkg_opts($INC{$pkg_name} || $pkg_name => \%opts);
}

=head2 delete_field_by_rights

Удаляет ключи полей их хэша филдов, если нет соотв. права
В качестве права может передаваться маска
Одно право может удалять несколько ключей

    $app->delete_field_by_rights( $fields, {
        # <Право или маска под право>                     <Поля которые оно закрывает>
         'do_context_on_site_rtb_save_dsps'           => 'dsp_list',
         'context_on_site_campaign_view_field__owner' => [qw(login owner_id owner_client_id)],
         'context_on_site_rtb_view_field__%s'         => [qw(comment blind)],
    });

=cut

sub delete_field_by_rights {
    my $self = shift;
    my $fields = shift || {
        #    <field_name> => ...,
    };
    my $fields_by_right = shift || {
        #   '..view_field__%s' => [qw( <field_name>, ... )],
    };

    foreach my $right_templ (keys %$fields_by_right) {
        my $fields_to_check = $fields_by_right->{$right_templ} // [];
        $fields_to_check = [$fields_to_check] unless ref($fields_to_check) eq 'ARRAY';

        foreach my $field_name (@$fields_to_check) {
            my $right_name =
              ($right_templ =~ m/%/)
              ? sprintf($right_templ, $field_name)
              : $right_templ;

            delete $fields->{$field_name}
              unless $self->check_rights($right_name);
        }
    }

    return 1;
}

=head2 get_models

Short method description

B<No arguments.>

B<Return value:> type, description

=cut

sub get_models {
    my ($self) = @_;

    my $models = {};

    package_merge_isa_data(
        ref($self),
        $models,
        sub {
            my ($package, $res) = @_;

            my $pkg_models = package_stash($package)->{'__MODELS__'} || {};
            $models->{$_} = $pkg_models->{$_} foreach keys(%$pkg_models);
        },
        __PACKAGE__
    );

    return $models;
}

=head1 Methods

=head2 get_option

Short method description

B<Arguments:>

=over

=item

B<$name> - type, description

=item

B<$default> - type, description

=back

B<Return value:> type, description

=cut

sub get_option {
    my ($self, $name, $default) = @_;

    my $res = $self->{'__OPTIONS__'}{$name} // $default;

    if (defined($res) && (!ref($res) || ref($res) eq 'ARRAY')) {
        foreach my $str (ref($res) eq 'ARRAY' ? @$res : $res) {
            while ($str =~ /^(.*?)(?:\[%\s+([^ ]+)\s+%\])(.*)$/) {
                $str = ($1 || '') . ($self->get_option($2) || '') . ($3 || '');
            }
            while ($str =~ /^(.*?)(?:\$\{\s*([^ ]+)\s*\})(.*)$/) {
                $str = ($1 || '') . ($ENV{$2} || '') . ($3 || '');
            }
        }
    }

    return $res;
}

=head2 get_registered_right_groups

Short method description

B<No arguments.>

B<Return value:> type, description

=cut

sub get_registered_right_groups {
    my ($self) = @_;

    my $rights = {};
    package_merge_isa_data(
        ref($self),
        $rights,
        sub {
            my ($ipackage, $res) = @_;

            my $ipkg_stash = package_stash($ipackage);
            $res->{'__RIGHT_GROUPS__'} =
              {%{$res->{'__RIGHT_GROUPS__'} || {}}, %{$ipkg_stash->{'__RIGHT_GROUPS__'} || {}}};
        },
        __PACKAGE__
    );

    return $rights->{'__RIGHT_GROUPS__'};
}

=head2 get_registered_rights

Short method description

B<No arguments.>

B<Return value:> type, description

=cut

sub get_registered_rights {
    my ($self) = @_;

    my $rights = {};
    package_merge_isa_data(
        ref($self),
        $rights,
        sub {
            my ($ipackage, $res) = @_;

            my $ipkg_stash = package_stash($ipackage);
            $res->{'__RIGHTS__'} = {%{$res->{'__RIGHTS__'} || {}}, %{$ipkg_stash->{'__RIGHTS__'} || {}}};
        },
        __PACKAGE__
    );

    return $rights->{'__RIGHTS__'};
}

sub get_registred_right_groups {&get_registered_right_groups;}

sub get_registred_rights {&get_registered_rights;}

sub init {
    my ($self) = @_;

    $self->SUPER::init();
    my $orig_opts = $self->{'__ORIG_OPTIONS__'} = {};
    package_merge_isa_data(
        ref($self),
        $orig_opts,
        sub {
            my ($package, $res) = @_;

            my $pkg_stash = package_stash($package);

            foreach my $cfg (@{$pkg_stash->{'__OPTIONS__'} || []}) {
                foreach (keys %{$cfg->{'config'}}) {
                    warn gettext('%s: option "%s" replaced', $cfg->{'filename'}, $_)
                      if exists($res->{$_});
                    $res->{$_} = $cfg->{'config'}{$_};
                }
            }
        },
        __PACKAGE__
    );

    my $app_module = ref($self) . '.pm';
    $app_module =~ s/::/\//g;

    $orig_opts->{'FrameworkPath'} = $INC{'QBit/Class.pm'} =~ /(.+?)QBit\/Class\.pm$/ ? $1 : './';
    $orig_opts->{'ApplicationPath'} =
        ($INC{$app_module} || '') =~ /(.*?\/?)(?:lib\/*)?$app_module$/
      ? ($1 || './')
      : './';

    $orig_opts->{'hostname'} = get_hostname();

    if (defined($orig_opts->{'stage'}) && grep {$orig_opts->{'stage'} eq $_} (qw(autotest test preprod production))) {
        $orig_opts->{'version'} = get_version();
    } else {
        $orig_opts->{'version'} = undef;
    }

    # Увеличиваем таймаут для Cron задач (#INFRASTRUCTUREPI-1942)
    $orig_opts->{'http_timeout'} =
        $self->isa('QBit::Cron')
      ? $orig_opts->{'http_timeout_cron'}
      : $orig_opts->{'http_timeout_interface'};

    # To set global options
    $self->{'__OPTIONS__'} = $orig_opts;

    my $locales = $self->get_option('locales', {});
    if (%$locales) {
        my ($locale) = grep {$locales->{$_}{'default'}} keys(%$locales);
        ($locale) = keys(%$locales) unless $locale;

        $self->set_app_locale($locale);
    }

    $self->init_accessors();

    if (!$ENV{'LAZY_LOAD'} && $self->get_option('preload_accessors')) {
        $self->$_ foreach keys(%{$self->get_models()});
    }

    # Doesn't work with LAZY_LOAD because Coro will override $SIG{__DIE__}
    # during lazy import
    if ($self->get_option('install_die_handler')) {
        if ($self->get_option('use_coro')) {
            $Coro::State::DIEHOOK = \&QBit::Exceptions::die_handler;
        } else {
            $SIG{__DIE__} = \&QBit::Exceptions::die_handler;
        }
    }

    delete($self->{'__OPTIONS__'});    # Options initializing in pre_run

}

=head2 post_run

Short method description

B<No arguments.>

B<Return value:> type, description

=cut

sub post_run {
    my ($self) = @_;

    foreach (keys(%{$self->get_models()})) {
        if (exists($self->{$_}) && blessed($self->{$_})) {
            $self->$_->finish()   if $self->{$_}->can('finish');
            $self->$_->post_run() if $self->{$_}->can('post_run');
        }
    }

    my $is_check_cycles = $self->get_option('find_app_mem_cycle') && rand(1) > 0.3;
    if ($is_check_cycles) {
        if (eval {require 'Devel/Cycle.pm'}) {
            Devel::Cycle->import();

            # Some workaround about old not fixed bug https://rt.cpan.org/Public/Bug/Display.html?id=56681
            {
                my $old_get_type = \&Devel::Cycle::_get_type;
                no warnings 'redefine';
                *Devel::Cycle::_get_type = sub {
                    # https://perldoc.perl.org/functions/ref.html
                    # If the unblessed referent is a scalar,
                    # then the return value will be one of the strings
                    # SCALAR, VSTRING, REF, GLOB, LVALUE, or REGEXP,
                    # depending on the kind of value the scalar currently has.
                    return 'SCALAR' if UNIVERSAL::isa($_[0], 'REGEXP');
                    return 'SCALAR' if UNIVERSAL::isa($_[0], 'GLOB');
                    goto &$old_get_type;
                };
            }

            my @cycles;
            Devel::Cycle::find_cycle($self, sub {push(@cycles, shift)});
            $self->process_mem_cycles(\@cycles) if @cycles;
        } else {
            l(gettext('Devel::Cycle is not installed'));
        }
    }
}

=head2 pre_run

Short method description

B<No arguments.>

B<Return value:> type, description

=cut

sub pre_run {
    my ($self) = @_;

    fail_connect_on_production();

    $self->{'__CURRENT_USER_RIGHTS__'} = {};

    $self->{'__OPTIONS__'} = clone($self->{'__ORIG_OPTIONS__'});

    foreach (keys(%{$self->get_models()})) {
        $self->$_->pre_run() if exists($self->{$_}) && $self->{$_}->can('pre_run');
    }
}

=head2 process_mem_cycles

Short method description

B<Arguments:>

=over

=item

B<$cycles> - type, description

=back

B<Return value:> type, description

=cut

sub process_mem_cycles {
    my ($self, $cycles) = @_;

    my $counter = 0;
    my $text    = '';
    foreach my $path (@$cycles) {
        $text .= gettext('Cycle (%s):', ++$counter) . "\n";
        foreach (@$path) {
            my ($type, $index, $ref, $value, $is_weak) = @$_;
            $text .= gettext(
                "\t%30s => %-30s\n",
                ($is_weak ? 'w-> ' : '') . Devel::Cycle::_format_reference($type, $index, $ref, 0),
                Devel::Cycle::_format_reference(undef, undef, $value, 1)
            );
        }
        $text .= "\n";
    }

    l($text);
    return $text;
}

=head2 set_app_locale

Short method description

B<Arguments:>

=over

=item

B<$locale_id> - type, description

=back

B<Return value:> type, description

=cut

sub set_app_locale {
    my ($self, $locale_id) = @_;

    my $locale = $self->get_option('locales', {})->{$locale_id};
    throw Exception::Validation::BadArguments gettext('Unknown language "%s"', $locale) unless defined($locale);
    throw gettext('Undefined locale code for locale "%s"', $locale_id) unless $locale->{'code'};

    set_locale(
        project => $self->get_option('locale_domain', 'application'),
        path    => $self->get_option('ApplicationPath') . 'locale',
        lang    => $locale->{'code'},
    );

    $self->set_option(locale => $locale_id);
}

sub set_accessors {
    my ($package, %opts) = @_;

    my $app_stash = package_stash(__PACKAGE__);
    $app_stash->{'__NEED_ACCESSORS__'} //= {};
    $app_stash->{'__TAGS_ACCESSORS__'} //= {};

    my $need_accessors = $app_stash->{'__NEED_ACCESSORS__'};
    my %accessors = map {($need_accessors->{$_}{'accessor'} => TRUE)} keys %$need_accessors;
    foreach my $pkg (keys %opts) {
        my $new_accessor = $opts{$pkg}->{'accessor'};

        throw gettext(
            "Model \"%s\" with accessors (\"%s\", \"%s\") is exists (class: \"%s\")",
            $pkg, $need_accessors->{$pkg}{'accessor'},
            $new_accessor, $package
        ) if exists($need_accessors->{$pkg});

        if (exists($accessors{$new_accessor})) {
            throw gettext("Model with accessor \"%s\" is exists (class: \"%s\")", $new_accessor, $package);
        } else {
            $accessors{$new_accessor} = TRUE;
            if (my $tags = $opts{$pkg}{tags}) {
                for my $tag (@$tags) {
                    push @{$app_stash->{'__TAGS_ACCESSORS__'}{$tag}}, $new_accessor;
                }
            }
        }
    }

    $app_stash->{'__NEED_ACCESSORS__'} = {%$need_accessors, %opts};
}

sub init_accessors {
    my ($self) = @_;

    my $package = ref($self);

    my $app_stash = package_stash(__PACKAGE__);

    my $accessors = $app_stash->{'__NEED_ACCESSORS__'} // {};

    my $self_stash = package_stash($package);

    $self_stash->{'__MODELS__'} //= {};

    my $action_log_model_template = qq[
package Application::Model::ActionLog::%s;

use base qw(Application::Model::ActionLog);

sub accessor {'%s'}
sub db_table_name {'%s'}

1;
];

    no strict 'refs';

    foreach my $class (keys(%$accessors)) {
        my $accessor_info  = $accessors->{$class};
        my $accessor       = $accessors->{$class}{'accessor'};
        my $has_action_log = $accessors->{$class}{'has_action_log'};

        next
          if exists($self_stash->{'__MODELS__'}{$accessor});

        throw gettext("Accessor cannot have name \"%s\", it is name of method", $accessor)
          if $self->can($accessor);

        $accessor_info->{'package'} = $package;
        $self_stash->{'__MODELS__'}{$accessor} = $accessor_info;

        my %import_args;
        $import_args{$_} = $accessor_info->{$_}
          foreach grep {$_ ne 'accessor' && $_ ne 'has_action_log' && $_ ne 'package'} keys(%$accessor_info);

        *{"${package}::${accessor}"} = sub {
            $_[0]->{$accessor} //= do {
                my $file_path = "$class.pm";
                $file_path =~ s/::/\//g;

                unless ($INC{$file_path}) {
                    eval {require $file_path};
                    throw sprintf('Failed require "%s": %s', $file_path, $@) if $@;
                }

                unless (exists $app_stash->{'ACCESSOR_INTO_CLASS'}{$class}{$accessor}) {
                    $app_stash->{'ACCESSOR_INTO_CLASS'}{$class}{$accessor} = 1;
                    $class->import('accessor', $accessor, %import_args);
                }
                $class->new(app => $_[0], accessor => $accessor);
            };
        };

        if ($has_action_log) {
            my $table_name = $has_action_log eq TRUE ? "${accessor}_action_log" : $has_action_log;
            my $package_name       = join('', map {ucfirst($_)} split('_', $accessor));
            my $full_package_name  = 'Application::Model::ActionLog::' . $package_name;
            my $action_log_package = sprintf($action_log_model_template, $package_name, $table_name, $table_name);

            $self_stash->{'__MODELS__'}{$table_name} = $accessor_info;

            *{"${package}::${table_name}"} = sub {
                $_[0]->{$table_name} //= do {
                    unless ($app_stash->{'__ACTION_LOG_IMPORT__'}{$full_package_name}) {
                        eval $action_log_package;
                        die Exception::SysDie->new($@) if $@;

                        $full_package_name->import(accessor => $table_name);
                        $app_stash->{'__ACTION_LOG_IMPORT__'}{$full_package_name} = TRUE;
                    }

                    $full_package_name->new(app => $_[0], accessor => $table_name);
                };
            };
        }
    }
}

=head2 set_cur_user_rights

  set rights for current user

B<Arguments:>

=over

=item

B<$rights> - array ref

=back

    $app->set_cur_user_rights([qw(RIGHT1 RIGHT2)]);

=cut

sub set_cur_user_rights {
    my ($self, $rights) = @_;

    $self->{'__CURRENT_USER_RIGHTS__'}{$_}++ foreach @$rights;
}

=head2 revoke_cur_user_rights

  revoke rights for current user

B<Arguments:>

=over

=item

B<$rights> - array ref

=back

    $app->revoke_cur_user_rights([qw(RIGHT1 RIGHT2)]);

=cut

sub revoke_cur_user_rights {
    my ($self, $rights) = @_;

    foreach (@$rights) {
        delete($self->{'__CURRENT_USER_RIGHTS__'}{$_})
          if exists($self->{'__CURRENT_USER_RIGHTS__'}{$_}) && --$self->{'__CURRENT_USER_RIGHTS__'}{$_} <= 0;
    }
}

=head2 set_option

Short method description

B<Arguments:>

=over

=item

B<$name> - type, description

=item

B<$value> - type, description

=back

B<Return value:> type, description

=cut

sub set_option {
    my ($self, $name, $value) = @_;

    $self->{'__OPTIONS__'}{$name} = $value;
}

=head2 set_tmp_app_locale

Short method description

B<Arguments:>

=over

=item

B<$locale_id> - type, description

=back

B<Return value:> type, description

=cut

sub set_tmp_app_locale {
    my ($self, $locale_id) = @_;

    my $old_locale_id = $self->get_option('locale');
    $self->set_app_locale($locale_id);

    return QBit::Application::_Utils::TmpLocale->new(app => $self, old_locale => $old_locale_id);
}

=head2 use_config

Short method description

B<Arguments:>

=over

=item

B<$filename> - type, description

=back

B<Return value:> type, description

=cut

sub read_config {
    my ($file_path) = @_;

    my $filename = basename($file_path);

    my $config = {};
    try {
        $config = from_json(read_file($file_path, binmode => ':utf8') || '{}');

        throw gettext('Expected hash') if ref($config) ne 'HASH';
    }
    catch {
        throw gettext('Read config file "%s" failed: %s', $filename, shift->message);
    };

    return $config;
}

sub use_config {
    my ($self, $filename) = @_;

    my $file_path = $self->{config_path} // lib::abs::path("../$filename");
    my $config = read_config($file_path);

    $self->_push_pkg_opts($filename => $config);
}

=head2 _push_pkg_opts

Short method description

B<Arguments:>

=over

=item

B<$filename> - type, description

=item

B<$config> - type, description

=back

B<Return value:> type, description

=cut

sub _push_pkg_opts {
    my ($self, $filename, $config) = @_;

    my $pkg_stash = package_stash(ref($self) || $self);

    $pkg_stash->{'__OPTIONS__'} = []
      unless exists($pkg_stash->{'__OPTIONS__'});

    push(
        @{$pkg_stash->{'__OPTIONS__'}},
        {
            filename => $filename,
            config   => $config,
        }
    );
}

TRUE;
