
=head1 Name

QBit::Application::Model::API::ADVQ - AdvQ's API realization.

=head1 Config opions

=over

=item

B<URLs> - hash, server list. Like

 URLs => {
     server_name => {
         search => [map {"http://test$_.advq.yandex.ru"} 1..6],
         chrono => [map {"http://test$_.advq.yandex.ru"} 1..6],
     }
 }

Server name gets by `hostname` (without parameters).

=back

=head1 Required models

=over

=item

B<memcached> => L<QBit::Application::Model::Memcached>.

=back

=cut

package QBit::Application::Model::API::ADVQ;

use qbit;

use base qw(QBit::Application::Model);

use PiConstants qw($DEFAULT_HTTP_TIMEOUT);

use LWP::UserAgent;
use YAML::XS;
use Net::INET6Glue::INET_is_INET6;

use Utils::Logger qw(ERRORF);

use Exception::API::ADVQ;
use Exception::API::ADVQ::HTTP;
use Exception::API::ADVQ::TAINTED;
use Exception::API::ADVQ::HostConfig;
use Exception::Validation::BadArguments;

__PACKAGE__->model_accessors(memcached => 'QBit::Application::Model::Memcached');

=head2 chrono

Get data from chrono DB.

B<Arguments as hash:>

=over

=item

B<words> - string or ref to array of string, requested phrases, required;

=item

B<type> - string, period name. One of C<monthly>, C<weekly>, C<daily>;

=item

B<regions> - string, region list ids, joined by ',';

=item

B<make_reg_gist> - ???.

=back

B<Return value:> ref to array of hashes. Structure depends on parameters.

=cut

sub chrono {
    my ($self, %opts) = @_;

    warn("Obsolete: use QBit::Application::Model::API::Yandex::ADVQ");

    $opts{'type'} //= 'monthly';

    my @param_pairs = (
        $self->_make_words_param_pairs($opts{'words'}),
        (map {"$_=" . uri_escape($opts{$_})} grep {defined($opts{$_})} qw(regions make_reg_gist))
    );

    return $self->_get_data(chrono => "/advq/$opts{'type'}_hist?" . join('&', @param_pairs));
}

sub init {
    my ($self) = @_;

    $self->SUPER::init();

    $self->{__LWP__} = LWP::UserAgent->new(timeout => $self->get_option('timeout', $DEFAULT_HTTP_TIMEOUT));
}

=head1 Methods

=head2 search

Get data from basic DB.

B<Arguments as hash>

=over

=item

B<words> - string or ref to array of string, requested phrases, required;

=item

B<regions> - string, region list ids, joined by ',';

=item

B<ph_page> - number, results page number, from zero;

=item

B<ph_page_size> - number, results page size;

=item

B<count_by_regions> - boolean, include into result additional statistic grouped by regions;

=item

B<dblist> - ???.

=back

B<Return value:> ref to array of hashes. Structure depends on parameters.

=cut

sub search {
    my ($self, %opts) = @_;

    warn("Obsolete: use QBit::Application::Model::API::Yandex::ADVQ");

    my @param_pairs = (
        $self->_make_words_param_pairs($opts{'words'}),
        (
            map  {"$_=" . uri_escape($opts{$_})}
            grep {defined($opts{$_})} qw(regions ph_page ph_page_size count_by_regions dblist)
        )
    );

    return $self->_get_data(search => '/advq/search?' . join('&', @param_pairs));
}

sub __fix_utf {
    my ($data) = @_;

    if (ref($data) eq '') {
        utf8::decode($data);
        return $data;
    } elsif (ref($data) eq 'ARRAY') {
        return [map {__fix_utf($_)} @$data];
    } elsif (ref($data) eq 'HASH') {
        return {map {$_ => __fix_utf($data->{$_})} keys(%$data)};
    } else {
        throw 'Unknown ref: ' . ref($data);
    }
}

sub _get_data {
    my ($self, $url_type, $path_params) = @_;

    my $cached_data = $self->memcached->get($url_type, $path_params);
    return $cached_data if $cached_data;

    my $url = $self->_get_url($url_type) . $path_params;

    my ($response, $data);

    for (1 .. 3) {
        $response = $self->{__LWP__}->get($url);

        if ($response->is_success()) {
            $data = $response->decoded_content('default_charset' => 'UTF-8');
            utf8::encode($data);    # turn into octets
            $data = Load($data)->{'requests'};
            $data = __fix_utf($data);
            last unless grep {$_->{'stat'}{'tainted'} || $_->{'tainted'}} @$data;
        } else {
            ERRORF("%s: %s. Response status: %s", $_, $url, $response->status_line());
            last if $response->message() =~ /timeout/i;
        }
    }

    throw Exception::API::ADVQ::HTTP gettext('Cannot get data from ADVQ: "%s"', $response->status_line)
      unless defined($data);

    throw Exception::API::ADVQ::TAINTED gettext('Server has returned tainted answer')
      if grep {$_->{'stat'}{'tainted'} || $_->{'tainted'}} @$data;

    $self->memcached->set($url_type, $path_params, $data, 60 * 60 * 24);

    return $data;
}

sub _get_url {
    my ($self, $type) = @_;

    my $urls = $self->get_option('URLs')->{$self->get_option('hostname')}{$type}
      // throw Exception::API::ADVQ::HostConfig gettext(
        'Type "%s" for hostname "%s" has not found in configuration file', $type, $self->get_option('hostname'));

    $self->{'current_url_index'}{$type} //= int(rand(@$urls));
    $self->{'current_url_index'}{$type} = 0 if ++$self->{'current_url_index'}{$type} >= @$urls;

    my $url = $urls->[$self->{'current_url_index'}{$type}];

    return $url;
}

sub _make_words_param_pairs {
    my ($self, $words) = @_;

    throw Exception::Validation::BadArguments gettext('Missed required argument "word"') unless defined($words);
    $words = [$words] unless ref($words);

    foreach (@$words) {
        $_ = lc($_);
        s/ +- / /g;
    }

    return map {'words=' . uri_escape($_)} @$words;
}

TRUE;
