package QBit::Application::Model::API::HTTP;

use qbit;

use base qw(QBit::Application::Model::API);

use LWP::UserAgent;
use HTTP::Request::Common qw();
use File::ReadBackwards;
use Time::HiRes qw(gettimeofday);

use PiConstants qw($TVM_HTTP_HEADER_NAME :HTTP);
use Utils::Logger qw(INFO  DEBUG  DEBUGF  ERROR);

use Exception::API::HTTP;

my %SPECIAL_FIELDS_NAMES = (
    map {$_ => TRUE}
      qw(
      :post
      :get
      :put
      :delete
      :patch
      :content_type
      :content_file
      :cached
      :expected_end_marker
      :return_fh
      :return_ref
      :headers
      :url
      :content
      :memcached
      :attempts
      :delay
      :timeout
      :timeout_retry
      )
);

sub get_structure_model_accessors {
    return {api_tvm => 'Application::Model::API::Yandex::TVM',};
}

sub last_response {
    my ($self, $response) = @_;
    if (@_ > 1) {
        $self->{response} = $response;
    }
    return $self->{response};
}

sub call {
    my ($self, $method, %params) = @_;

    $method //= '';

    my $content_file = $params{':content_file'};
    my $use_cache    = delete($params{':cached'});
    my $return_fh    = delete($params{':return_fh'});
    my $return_ref   = delete($params{':return_ref'});

    my $use_memcache = delete($params{':memcached'});

    throw Exception 'Param ":memcached" cannot be used with ":cached", ":content_file" and ":return_fh"'
      if $use_memcache and ($use_cache || $content_file || $return_fh);

    throw Exception 'Param ":content_file" cannot be used with ":return_fh" and ":return_ref"'
      if defined($content_file) && ($return_fh || $return_ref);

    throw Exception 'Param ":return_fh" cannot be used with ":return_ref"' if $return_fh && $return_ref;

    if ($use_cache) {
        throw Exception 'Param "method" cannot be empty with ":cached"' if length($method) == 0;
    } else {
        throw Exception 'Param ":return_fh" cannot be used without ":cached"' if $return_fh;
    }

    my $url = $params{':url'} || $self->get_option('url');
    throw Exception 'URL undef' unless defined $url;
    my $uri = $url . $method;

    my $start = gettimeofday();
    INFO("[$uri] start");

    my @request_fields = grep {!exists($SPECIAL_FIELDS_NAMES{$_})} keys(%params);
    my $url_params = {hash_transform(\%params, \@request_fields)};

    my ($cache_file_path, $is_cache_ok, $key_memcache, $result);
    if ($use_cache) {
        $cache_file_path = $content_file // $self->_get_cache_file_path($method, %$url_params);
        $is_cache_ok = $self->_check_cache_file($cache_file_path);
    } elsif ($use_memcache) {
        $key_memcache = join '|', $uri, map {$_ . '=' . $params{$_}} sort keys %$url_params;
        if ($result = $self->app->memcached->get($self->accessor, $key_memcache)) {
            # если данные битые, то считаем, что их и не было
            eval {
                $result      = from_json($result)->{data};
                $is_cache_ok = TRUE;
            };
        }
    }

    unless ($is_cache_ok) {
        # Если в конфиге установлен твм-алиас, добавляем специальный заголовок с твм-тикетом
        my $tvm_alias = $self->get_option('tvm_alias');
        if ($tvm_alias && $self->accessor() ne 'api_tvm') {
            $params{':headers'} //= {};
            unless ($params{':headers'}{$TVM_HTTP_HEADER_NAME}) {
                my $ticket = $self->api_tvm->get_service_ticket($tvm_alias);
                $params{':headers'}{$TVM_HTTP_HEADER_NAME} = $ticket;
            }
        }

        $result = $self->get($uri, %params, ($use_cache ? (':content_file' => $cache_file_path) : ()));
        if ($key_memcache) {
            $self->app->memcached->set($self->accessor, $key_memcache, to_json({data => $result}), $use_memcache);
        }
    }

    my $end = curdate(oformat => 'db_time');
    my $delta_seconds = gettimeofday() - $start;
    INFO("[$uri] end. elapsed $delta_seconds seconds." . ($is_cache_ok ? '(cached)' : ''));

    return '' if defined($content_file);

    if ($return_fh) {
        open(my $fh, '<', $cache_file_path) or die "Cannot open cache file $cache_file_path for reading";
        return $fh;
    }

    if ($use_cache) {
        my $ref = File::Slurp::read_file($cache_file_path, scalar_ref => 1);
        return $return_ref ? $ref : $$ref;
    } else {
        return $return_ref ? \$result : $result;
    }
}

sub get {
    my ($self, $uri, %params) = @_;

    my $keep_response = delete $params{'keep_response'};

    my @request_fields = grep {!exists($SPECIAL_FIELDS_NAMES{$_})} keys(%params);
    my $url_params = {hash_transform(\%params, \@request_fields)};

    my %headers = (
        ($params{':headers'} ? %{$params{':headers'}} : ()),
        $self->_get_headers_cfg, %{$self->_get_params_and_headers(%params)}
    );

    my $method = 'GET';
    $method = 'POST'   if $params{':post'};
    $method = 'PUT'    if $params{':put'};
    $method = 'DELETE' if $params{':delete'};
    $method = 'PATCH'  if $params{':patch'};

    throw "Thou shalt not use '' but ':content' instead" if exists $params{''};

    my $content = $self->request(
        $self->_prepare_request_params(
            uri                 => $uri,
            method              => $method,
            url_params          => $url_params,
            headers             => \%headers,
            content             => $params{':content'},
            content_file        => delete($headers{':content_file'}),
            attempts            => $params{':attempts'},
            delay               => $params{':delay'},
            timeout             => $params{':timeout'},
            timeout_retry       => $params{':timeout_retry'},
            expected_end_marker => $params{':expected_end_marker'},
            keep_response       => $keep_response
        )
    );

    return $content;
}

sub _prepare_request_params {
    my ($self, %params) = @_;

    # %params = (
    #    uri           => 'http://foo?bar=1',
    #    method        => 'GET', # 'POST, 'PUT'
    #    url_params    => { foo => 1, bar => 1 },
    #    headers       => {':content_type' => 'application/json', ..  },
    #    content       => "...", # "" || { foo =>1, bar => 1 } || ["foo", 1, "bar", 1 ]
    #    content_file  => '/tmp/...',
    #    attempts      => 3
    #    delay         => 1
    #    timeout_retry => FALSE,
    #    keep_response  => 1,
    #)

    my %req_params = (
        uri        => $params{'uri'},
        method     => $params{'method'} // 'GET',
        url_params => $params{'url_params'} // {},
        headers    => $params{'headers'} // {},

        attempts => $params{'attempts'} // $self->get_option('attempts', $DEFAULT_HTTP_ATTEMPTS),
        delay    => $params{'delay'}    // $self->get_option('delay',    $DEFAULT_HTTP_DELAY),
        timeout  => $params{'timeout'}  // $self->get_option('timeout',  $DEFAULT_HTTP_TIMEOUT),
        timeout_retry => $params{'timeout_retry'} // $self->get_option('timeout_retry'),

        expected_end_marker => $params{'expected_end_marker'} // '',
    );

    foreach my $key (qw( content  content_file  keep_response  )) {
        if (exists $params{$key}) {
            $req_params{$key} = $params{$key};
        }
    }

    throw Exception::API::HTTP 'Undefined uri' unless $req_params{'uri'};
    throw Exception::API::HTTP "Wrong method: $req_params{'method'}"
      unless $req_params{'method'} =~ /^(GET|POST|PUT|DELETE|PATCH)$/;

    return %req_params;
}

sub request {
    my ($self, %params) = @_;

    my %req_params = $self->_prepare_request_params(%params);

    my ($uri, $content, $content_file_path, $keep_response, $method, $url_params, $headers, $attempts, $delay, $timeout,
        $timeout_retry, $expected_end_marker)
      = @req_params{
        qw( uri  content  content_file   keep_response  method  url_params  headers  attempts  delay  timeout  timeout_retry  expected_end_marker)
      };

    $self->{'__LWP__'}->timeout($timeout);

    my $postdata = $content;
    if ($postdata && utf8::is_utf8($postdata)) {
        utf8::encode($postdata);
    }

    if ($method eq 'POST' && !$postdata && %$url_params) {
        $postdata   = $url_params;
        $url_params = {};
    }

    my $orig_uri = $uri;
    if (%$url_params) {
        my $uri_obj = URI->new($uri);
        $uri_obj->query_form(%$url_params);
        $uri = $uri_obj->as_string;
    }

    my $request;
    if ($method eq 'GET') {
        if (defined($postdata)) {
            $request = HTTP::Request::Common::GET($uri, %$headers, Content => $postdata);
        } else {
            $request = HTTP::Request::Common::GET($uri, %$headers);
        }
    }
    $request = HTTP::Request::Common::PUT($uri, %$headers, Content => $postdata) if $method eq 'PUT';
    $request = HTTP::Request::Common::POST($uri, %$headers, Content => $postdata) if $method eq 'POST';
    # $request creation should be unified, but we can not do this before https://st.yandex-team.ru/PI-22551
    # see comment https://github.yandex-team.ru/partner/partner2/pull/4828#issuecomment-2312603
    $request = HTTP::Request->new($method, $uri, HTTP::Headers->new(%$headers), $postdata) if $method eq 'DELETE';
    $request = HTTP::Request->new($method, $uri, HTTP::Headers->new(%$headers), $postdata) if $method eq 'PATCH';

    my $response         = undef;
    my $response_content = undef;
    my $retries          = 0;
    my $success          = FALSE;

    for my $retry (1 .. $attempts) {
        if ($retry - 1) {
            DEBUGF("RETRY #%s. SLEEP %s", $retries, $delay);
            sleep($delay);
        }

        DEBUGF('%s: %s', $method, $uri);

        $self->last_response(undef);

        $response = $self->{'__LWP__'}->request($request, $content_file_path);

        $self->last_response($response) if $keep_response;

        DEBUGF('RESPONSE CODE: %s', $response->code);
        if ($response->is_success()) {

            $response_content = $response->decoded_content() // $response->content;

            if ($content_file_path) {
                open(my $fh, '<', $content_file_path)
                  or throw Exception::API::HTTP "Failed to open $content_file_path: $!";
                my $n = read($fh, my $buf, 1024) // throw Exception::API::HTTP "Failed to read $content_file_path: $!";
                DEBUGF('RESPONSE BODY: %s...', $buf);
            } else {
                DEBUGF('RESPONSE BODY: %s...', substr($response_content, 0, 1024));
            }

            if ($self->_validate_response_content($response_content, $content_file_path, $expected_end_marker)) {
                $success = TRUE;
                last;
            }
        } else {
            my $msg = $response->status_line;
            $msg .= "\nURI: " . $orig_uri;

            $response_content = eval {$response->decoded_content()} // '';
            utf8::decode($response_content)
              unless utf8::is_utf8($response_content);

            $msg .= "\nResponse: " . substr($response_content, 0, 500) if $response_content;

            my $response_heders = $response->headers->as_string;
            $msg .= "\nHeaders: " . substr($response_heders, 0, 500) if $response_heders;

            INFO 'Got error response';
            INFO $msg;

            last
              if $response->code == 400
                  || $response->code == 404
                  || ($response->code == 408 && !$timeout_retry);
        }
    }

    my $log_params = {
        request  => $response->request->as_string,
        url      => $uri,
        status   => $response->code,
        headers  => $response->headers->as_string,
        response => $response_content,
        (defined($content) ? (content => $content) : (error => $response->status_line)),
    };

    $self->log($log_params) if $self->can('log');

    if ($ENV{FORCE_LOGGER_TO_SCREEN}) {
        local $log_params->{request} = substr($log_params->{request}, 0, 1_000);
        local $log_params->{response} = substr($log_params->{response} // '', 0, 1_000);
        local $log_params->{content}  = substr($log_params->{content}  // '', 0, 1_000);
        DEBUG($log_params);
    }

    throw Exception::API::HTTP $response unless $success;

    return $params{need_response} ? $response : $response_content;
}

sub init {
    my ($self) = @_;

    $self->SUPER::init();

    my $opts = $self->get_option('lwp_sock_opts');
    if ($opts) {
        require LWP::Protocol::http;
        push(@LWP::Protocol::http::EXTRA_SOCK_OPTS, @$opts);
    }
    $self->{'__LWP__'} = LWP::UserAgent->new(timeout => $self->get_option('timeout', $DEFAULT_HTTP_TIMEOUT));

    my %ssl_opts = %{$self->get_option('ssl_opts', {})};

    # Для того чтобы LWP работал с https ресурсами которые подписаны внутренним CA Яндекса.
    # (должен стоять deb пакет yandex-internal-root-ca)
    $ssl_opts{'SSL_ca_path'} //= '/etc/ssl/certs/';

    $self->{'__LWP__'}->ssl_opts(%ssl_opts);

    my $agent = $self->get_option('lwp_agent');
    $self->{'__LWP__'}->agent($agent) if $agent;
}

sub _validate_response_content {
    my ($self, $content, $content_file_path, $expected_end_marker) = @_;

    if ($expected_end_marker) {
        if (defined($content_file_path)) {
            if (-f $content_file_path && -s $content_file_path) {
                my $bw = File::ReadBackwards->new($content_file_path)
                  or throw Exception::API::HTTP "Failed to read $content_file_path: $!";
                my $last_line = $bw->readline();
                if ($last_line eq $expected_end_marker) {
                    return 1;
                } else {
                    INFO "NO END MARKER FOR $content_file_path";
                    return 0;
                }
            } else {
                return 0;
            }
        } elsif (defined($content)) {
            return $content =~ m/\Q$expected_end_marker\E$/;
        } else {
            return 0;
        }
    } else {
        return defined($content);
    }
}

sub _get_params_and_headers {
    my ($self, %params) = @_;

    my %names = (
        ':content_type' => 'Content-type',
        ':content_file' => ':content_file',
    );

    return {map {($names{$_} => $params{$_})} (grep {$names{$_}} keys %params)};
}

sub _get_headers_cfg {
    my ($self) = @_;

    my %headers_cfg = %{$self->get_option('headers', {})};
    my @allowed = grep exists($headers_cfg{$_}), qw/Host/;
    my %headers;
    @headers{@allowed} = delete @headers_cfg{@allowed};

    throw 'Bad config for headers: ' . join ', ', keys %headers_cfg if scalar keys %headers_cfg;

    return %headers;
}

TRUE;
