
=encoding UTF-8

=head1 Название

QBit::Application::Model::API::Yandex::Balance - API для работы с балансом

=head1 Описание

Реализут XMLRPC интерфейс с балансом.

О реализации интерфейса на стороне балансом можно посмотреть на вики
L</Balance/XmlRpc|https://wiki.yandex-team.ru/Balance/XmlRpc> и
L</Balance/BalanceService|http://wiki.yandex-team.ru/Balance/BalanceService>

Баланс доступен пользователем по адресу http://balance.yandex.ru

В балансе есть следующие сущности:

 * Клиент (client) - отдельная сущность, живущая именно в балансе.
   Обладает уникальным идентификатором client_id. Есть несколько параметров. В
   том числе: "Форма собственности" (может быть "Физическое лицо", "ООО", ...)
 * Представитель клиента (user) - пользователи Яндекса (с uid и логином),
   привязываются к клиенту. Уникальный индетификатор uid. У одного клиента
   может быть несколько представителей. Представитель может быть привязан
   только к одному клиенту (что совершенно логично: если было бы иначе,
   пользователю приходилось бы каждый раз выбирать для какого клиента он хочет
   выполнить действия)
 * Плательщики (person) - тоже отдельная сущность баланса. Как раз этой
   сущности Яндекс платит деньги, либо от которой получает деньги. Платильщики
   привязываются к Клиенту. У клиента может быть несколько плательщиков. Один
   и тот же плательщик может быть для разныех клиентов, но с какими-то
   оговорками. Есть 2 типа плательщиков: обычные (которые платят Яндексу) и
   партнерские (которым платит Яндекс). В ПИ мы работает только с
   плательщиками которым платит Яндекс. Плательщиков у клиентам может быть
   несколько, актуального плательщика можно выяснить из текущего договора. В
   договоре может быть только один плательщик.
 * Площадка - рекламная площадка в ПИ, привязывается к клиенту. На момент
   написания у баланса не было API посмотреть площадки, привязанные к
   клиенту, эта возможно сделать через веб интерфейс.
 * Договор - привязывается к client_id. У одного клиента может быть несколько
   договоров, но только один действующий в настоящий момент времени.
 * Оператор (operator)- пользователь Яндекса, который выполняет какие-то
   действия в балансе (например: 1. создает нового клиента, или 2. изменяет
   контактные данные представителя клиента). Это может быть либо сотрудник
   Яндекса (у которого есть необходимые права), либо сам пользователь.
 * площадка (place)

У баланса есть несколько инстансов (сред):

 * продакшн (боевая) среда - к ней привязан проадкш ПИ
 * тестовая среда - к ней привязаны беты ПИ
 * девелоперская среда - в ПИ не используется

В этом API идут обращения к следующим методам xml rpc баланса:

 * Balance.FindClient - по заданому фильтру показывает некоторые данные
   клиентов, которые соответствуют фильтру
 * Balance.GetPartnerContracts - получение договоров, доп. соглашений,
   информации о плательщике и информации о клиенте. Т.е. получение всей
   информации о клиенте, кроме данных о представителе.

 * Balance.GetClientPersons - получение данных о плательщиках клиента
 * Balance.GetClientUsers - получение данных о представителях клиента

 * Balance.RemoveUserClientAssociation - отвязывает предствителя от клиента
 * Balance.CreateOrUpdatePlace - создание или обновление площадки
 * Balance.CreateOrUpdatePartner - если не указан client_id, то создает
   клиента, представителя и плательщика. Если client_id указан, то обновляет
   данные.

=begin comment Проблема API баланса

К очень большому сожалению, в API баланса нет единого формата ключей
в хешах, которые отдает и принимсет API:

Метод Balance.GetClientPersons возвращает данные в виде (uppercase):

    'BANK_TYPE' => '2',
    'FNAME'     => "Иван",
    'CLIENT_ID' => '1010004',

А метод Balance.GetPartnerContracts в блоке о плательщике возвращает
(canonical):

    'bank_type' => '2',
    'fname'     => "Иван",
    'client_id' => '1010004',

При этом метод Balance.CreateOrUpdatePartner хочет получать данные в виде
(lcminus):

    'bank-type' => '2',
    'fname'     => "Иван",
    'client-id' => '1010004',

Внутри ПИ канонической считается считается perlish версия: 'some_thing'.

Для этих трех форматов ключей придуманы названия: uppercase, canonical,
lcminus и есть метод $self->_convert_hash_keys(), который преобразует
ключи к указанному виду.

=end comment

=cut

package QBit::Application::Model::API::Yandex::Balance;

use qbit;
use base qw(
  QBit::Application::Model::API::XMLRPC
  QBit::Application::Model::API::Yandex::Split::GetPagesTagsStat
  QBit::Application::Model::API::Yandex::Split::GetPagesStat
  QBit::Application::Model::API::Yandex::Split::GetDspStat
  QBit::Application::Model::API::Yandex::Split::GetInternalPagesStat
  QBit::Application::Model::API::Yandex::Split::GetInternalPagesTagsStat
  );

__PACKAGE__->model_accessors(
    kv_store        => 'QBit::Application::Model::KvStore',
    documents       => 'Application::Model::Documents',
    api_selfservice => 'Application::Model::API::Yandex::SelfService',
);

use XML::Twig;
use XML::Parser;
use Utils::TSV;
use File::Slurp;

use Exception::Balance;
use Exception::Balance::IncorrectAnswer;
use Exception::Balance::IncorrectAnswer::KnownErrors;
use Exception::Balance::SeveralResults;
use Exception::Balance::NotFound;
use Exception::API::XMLRPC;
use Exception::Validation::BadArguments;

use PiConstants qw(
  $FIRM_ID_YANDEX_LTD
  $AGGR_TUTBY
  );
use Utils::Logger qw( INFO  INFOF );
use QBit::Validator;

my %CLOSE_CONTRACT_COLLATERAL_TYPES = (
    GENERAL   => 90,
    SPENDABLE => 7050,
    PARTNERS  => 2090,
);

my @FIELDS_TO_COPY_CONTRACT = qw(
  client_id
  firm
  nds
  partner_pct
  payment_type
  person_id
  test_mode
  type
  );

my %FIELDS_TO_COPY_CONTRACT = (
    firm => 'firm_id',
    type => 'ctype',
);

=head2 hide_opts_set

B<Параметры:> 1) $self

Устанавливает признак необходимости скрывать полученные данные в логе ошибки

=cut

sub hide_opts_set {
    my $self = shift;

    $self->{_hide_opts} = TRUE;
}

=head2 hide_opts_get

B<Параметры:> 1) $self

Возвращает признак необходимости скрывать полученные данные в логе ошибки

=cut

sub hide_opts_get {
    my $self = shift;
    return delete $self->{_hide_opts};
}

=head2 call

B<Параметры:> 1) $self, 2) $func, 3) @opts

B<Возвращаемое значение:> 1) $result - ссылка на хеш с данными

Низкоуровневое обращение к XMLRPC.

В том случае если получиили от билинга ошибку пытаемся воспринять эту ошибку
как xml и отформатировать.

=cut

sub call {
    my ($self, $func, @opts) = @_;

    my $hide_opts = $self->hide_opts_get;

    # Для того чтобы эту штуку можно было тестировать
    return $self->{debug_result} if defined $self->{debug_result};

    my $return;

    try {
        $return = $self->SUPER::call($func, @opts);
    }
    catch Exception::API::XMLRPC with {
        my $error = $@->{text};
        my $bad_response;

        # Пытаюсь отформатировать xml
        my $xml = XML::Twig->new();
        eval {$xml->parse($error);};
        if ($@) {
            $xml = XML::Twig->new()->parse('<error/>');
            $xml->root->new(msg => $error)->paste(last_child => $xml->root);
            $xml->root->new(method => $func)->paste(last_child => $xml->root);
            $xml->root->new(contents => $error)->paste(last_child => $xml->root);
            $bad_response = TRUE;
        }

        my $dump_str = $hide_opts ? 'XXX' : substr(to_json(\@opts, pretty => TRUE), 0, 1000);
        $xml->root->new(opts => $dump_str)->paste(last_child => $xml->root);

        open my $fh, ">", \$error;
        $xml->set_pretty_print('indented');
        $xml->print($fh);

        throw Exception::Balance::IncorrectAnswer $error, undef, undef,
          sentry => {fingerprint => ['Balance', 'IncorrectAnswer', ($bad_response ? 'Bad response' : 'XMLRPC error')]};
    };

    return $return;
}

=head2 create_client

B<Параметры:> 1) $self 2) %opts

B<Возвращаемое значение:> 1) $client_id

Доступ к методу Balance.CreateClient - создание или обновление клиента

http://wiki.yandex-team.ru/Balance/XmlRpc#balance.createclient

Значения %opts:

=over

=item B<operator_uid> - число, UID пользователя, который создаёт/изменяет
площадку. (обязательный параметр)

=back

=cut

sub create_client {
    my ($self, %opts) = @_;

    throw gettext("Expected 'operator_uid'") unless defined $opts{operator_uid};

    my $operator_uid = delete $opts{operator_uid};

    my $result = $self->call('Balance.CreateClient', $operator_uid, \%opts,);

    throw Exception::Balance::IncorrectAnswer Dumper($result), undef, undef,
      sentry => {fingerprint => ['Balance', 'IncorrectAnswer', 'Not array']}
      if ref $result ne 'ARRAY';
    throw Exception::Balance::IncorrectAnswer Dumper($result) if $result->[0] ne '0';
    throw Exception::Balance::IncorrectAnswer Dumper($result) if $result->[1] ne 'SUCCESS';

    my $client_id = $result->[2];
    return $client_id;
}

=head2 create_or_update_contract_tag

B<Параметры:> 1) $self 2) %opts

B<Возвращаемое значение:> 1) TRUE

Доступ к методу Balance.CreateClient - создание или обновление коммерческих условий

https://wiki.yandex-team.ru/balance/xmlrpc#balance.createorupdatedistributiontag

Значения %opts:

=over

=item B<operator_uid> - число, UID пользователя, который создаёт/изменяет
площадку. (обязательный параметр)

=item B<id> - число, изменяемый/создаваемый идентификатор коммерческих условий.
(Обязательный параметр)

=item B<caption> - строка, изменяемое/создаваемое название коммерческих условий.
(Обязательный параметр)

=back

=cut

sub create_or_update_contract_tag {
    my ($self, %opts) = @_;

    my $contract_tag_2_balance = {
        id        => 'TagID',
        caption   => 'TagName',
        client_id => 'ClientID',
    };

    my $data = {hash_transform(\%opts, [], $contract_tag_2_balance)};

    my $result = $self->call('Balance.CreateOrUpdateDistributionTag', $opts{operator_uid}, $data);

    throw Exception::Balance::IncorrectAnswer Dumper($result), undef, undef,
      sentry => {fingerprint => ['Balance', 'IncorrectAnswer', 'Not array']}
      if (ref($result) ne 'ARRAY') || (ref($result->[0]) ne 'ARRAY');

    if (($result->[0]->[0] ne '0') or ($result->[0]->[1] ne 'SUCCESS')) {
        throw Exception::Balance::IncorrectAnswer::KnownErrors {
            error_id   => $result->[0]->[0],
            error_text => $result->[0]->[1],
        };
    }

    return TRUE;
}

=head2 create_or_update_partner

B<Параметры:> 1) $self 2) %opts

B<Возвращаемое значение:> 1) $client_id - это client_id
созданного/измененного клиента (возвращаетс только в случае успеха, в случае
каких-то проблем выбрасывается исключение)

Взаимодействие с методом Balance.CreateOrUpdatePartner - создание клиента,
представителя и плательщика или изменение данных.

Значения %opts:

=over

=item B<operator_uid> - число, UID пользователя, который создаёт/изменяет
площадку. (обязательный параметр)

=item B<mode> - режим отправки 'form' - это создание клиента, 'edit' - это
редактирование клиента.

=item B<...> - куча других параметров, которые вводит пользователь при
заполнении анкеты партнера

В случае ошибки метод Balance.CreateOrUpdatePartner вернет xml с описанием
ошибки.

=back

=cut

sub create_or_update_partner {
    my ($self, %opts) = @_;

    my $operator_uid = delete $opts{operator_uid};

    my $result = $self->call('Balance.CreateOrUpdatePartner', $operator_uid, \%opts,);

    throw Exception::Balance::IncorrectAnswer $result, undef, undef,
      sentry => {fingerprint => ['Balance', 'IncorrectAnswer', 'Not array']}
      if ref $result ne 'ARRAY';
    throw Exception::Balance::IncorrectAnswer $result if !$result->[0];

    my $client_id = $result->[0];

    return $client_id;
}

=head2 create_or_update_place

B<Параметры:> 1) $self 2) %opts

B<Возвращаемое значение:> 1) TRUE если удалось создать обновить площадку,
иначе будет выброшено исключение.

Доступ к методу Balance.CreateOrUpdatePlace - создание или обновление площадки
в балансе.

http://wiki.yandex-team.ru/Balance/XmlRpc#balance.createorupdateplace

Значения %opts:

=over

=item B<operator_uid> - число, UID пользователя, который создаёт/изменяет
площадку. (обязательный параметр)

=item B<client_id> - число, ID пользователя в балансе - владельца площадки.
Можно менять для ряда клиентов с client_id = (412775, 440062, 469655, 501212)
BALANCE-9853

=item B<page_id> - число, ID площадки в БК. (обязательный параметр)

=item B<domain> - строка, домен площадки.

=item B<campaign_type> - строка, тип площадки. (обязательный параметр)

=item B<viptype> - строка, внутренний тип площадки.

=item B<search_id> - число, ID площадки в ППС

=item B<is_payable> - нужно ли считать и перечислять деньги по влощадке

=item B<pay_type_id> - тип оплаты (для дистрибуции)

=item B<clid_type_id> - тип клида (для дистрибуции)

=back

=cut

sub create_or_update_place {
    my ($self, %opts) = @_;

    # Ходим в ручку 'Balance.CreateOrUpdatePlace' только в продакшене
    # Из-за особенностей налития тестовых сред ПИ2, БК, Баланса этот код на не проадкш средах
    # часто выдает ошибку '<msg>Invalid parameter for function: ClientID cannot be changed</msg>'
    # Смотри тикет st/INFRASTRUCTUREPI-2052
    if (($self->get_option('stage', 'unknown') eq 'production') || $ENV{TAP_VERSION}) {

        my $user2balance = {
            client_id       => 'ClientID',
            page_id         => 'ID',
            domain          => 'URL',
            campaign_type   => 'Type',
            viptype         => 'InternalType',
            search_id       => 'SearchID',
            is_payable      => 'IsPayable',
            pay_type_id     => 'PaymentTypeID',
            clid_type_id    => 'ClidType',
            contract_tag_id => 'TagID',
        };

        my $data = {hash_transform(\%opts, [], $user2balance)};

        my $result = $self->call('Balance.CreateOrUpdatePlace', $opts{operator_uid}, $data);

        throw Exception::Balance::IncorrectAnswer gettext('Could not interpret output from balance'), undef, undef,
          sentry => {fingerprint => ['Balance', 'IncorrectAnswer', 'Not array']}
          if ref $result ne 'ARRAY';
        throw Exception::Balance::IncorrectAnswer join(' ', @{$result}) if scalar $result->[0] != 0;

        if ($opts{is_tutby}) {
            $self->register_aggregator_pages(
                {
                    page_id  => $opts{page_id},
                    start_dt => $opts{create_date},
                }
            );
        }

    } else {
        INFO 'Not sending data to Balance.CreateOrUpdatePlace because it is not production stage.';
    }

    return TRUE;
}

=head2 create_person

Доступ к методу Balance.CreatePerson - создание или изменение плательщика.

B<Параметры:> 1) $self 2) %opts - хеш с данными плательщика которого нужно
создать. В обятательном порядке нужно передавать:

=over

=item B<client_id>

=item B<operator_uid> - uid пользователя кто проводит изменение (может быть
как пользользователь-партнер, так и системаный пользователь)

=item B<person_id> - id плательщкика для редактирования. В том случае если
передается -1, то плательщик будет создан

=back

B<Возвращаемое значение:> $person_id

    my $person_id = $self->api_balance->create_person(
        operator_uid => $user->{'id'},
        client_id => $user->{'client_id'},
        person_id => -1,  # -1 означает что нужно создать нового плательщика
        type => 'ur',
        %values4balance,
    );

=cut

sub create_person {
    my ($self, %opts) = @_;

    QBit::Validator->new(
        data     => \%opts,
        app      => $self->app,
        template => {
            type   => 'hash',
            fields => {
                operator_uid => {type => 'scalar',},
                client_id    => {type => 'scalar',},
                person_id    => {type => 'scalar',},
            },
            extra => 1,
        },
        throw => 1,
    );

    # баланс в xmlrpc протоколе хочет видеть person_id как число
    $opts{'person_id'} = SOAP::Data->type(int => $opts{'person_id'});

    # баланс в xmlrpc протоколе хочет видеть client_id как число
    $opts{'client_id'} = SOAP::Data->type(int => $opts{'client_id'});

    $self->hide_opts_set;
    my $result = $self->call('Balance2.CreatePerson', $opts{'operator_uid'}, \%opts);

    throw Exception::Balance::IncorrectAnswer Dumper($result), undef, undef,
      sentry => {fingerprint => ['Balance2.CreatePerson', 'IncorrectAnswer', 'Not array']}
      if ref $result ne 'ARRAY';
    throw Exception::Balance::IncorrectAnswer Dumper($result), undef, undef,
      sentry => {fingerprint => ['Balance2.CreatePerson', 'IncorrectAnswer', 'Bad array']}
      if scalar @$result != 1;

    my $person_id = $result->[0];

    return $person_id;
}

=head1 create_contract

 'client_id': <id клиента>,
 'currency': <Валюта, например 'USD'>,
 'firm_id': <id юрлица, для ООО Яндекс 1>,
 'manager_code': 42595,
 'nds': <айдишник НДС, например для ОСН в России: 18>,
 'person_id': <id плательщика>,
 'services': [1132],
 'start_dt': <дата начала действия оферты,>
 'is_offer': 1,
 'signed': <0/1 - не подписана/подписана>

=cut

sub create_contract {
    my ($self, %opts) = @_;

    QBit::Validator->new(
        data     => \%opts,
        app      => $self->app,
        template => {
            type   => 'hash',
            fields => {
                operator_uid => {type => 'scalar',},
                currency     => {type => 'scalar',},
                client_id    => {type => 'scalar',},
                person_id    => {type => 'scalar',},
                firm_id      => {type => 'scalar',},
                manager_code => {type => 'scalar',},
                nds          => {type => 'scalar',},
                services     => {type => 'array'},
                start_dt     => {type => 'scalar',},
                is_offer     => {type => 'scalar',},
                signed       => {type => 'scalar',},
                selfemployed => {type => 'scalar',},
            },
        },
        throw => 1,
    );

    # баланс в xmlrpc протоколе хочет видеть person_id как число
    $opts{'person_id'} = SOAP::Data->type(int => $opts{'person_id'});

    # баланс в xmlrpc протоколе хочет видеть client_id как число
    $opts{'client_id'} = SOAP::Data->type(int => $opts{'client_id'});

    $self->hide_opts_set;
    my $result = $self->call('Balance2.CreateCommonContract', $opts{'operator_uid'}, \%opts);

    throw Exception::Balance::IncorrectAnswer Dumper($result), undef, undef,
      sentry => {fingerprint => ['Balance2.CreateCommonContract', 'IncorrectAnswer', 'Not array']}
      if ref $result ne 'ARRAY';
    throw Exception::Balance::IncorrectAnswer Dumper($result), undef, undef,
      sentry => {fingerprint => ['Balance2.CreateCommonContract', 'IncorrectAnswer', 'Bad array']}
      if scalar @$result != 1;

    my $contract_id = $result->[0];

    return $contract_id;
}

=head2 create_offer

=cut

sub create_offer {
    my ($self, %opts) = @_;

    if (exists($opts{'person_id'})) {
        # баланс в xmlrpc протоколе хочет видеть person_id как число
        $opts{'person_id'} = SOAP::Data->type(int => $opts{'person_id'});
    }

    my $result = $self->call('Balance2.CreateOffer', $opts{'operator_uid'}, \%opts);

    local $Data::Dumper::Terse = 1;

    throw Exception::Balance::IncorrectAnswer Dumper($result), undef, undef,
      sentry => {fingerprint => ['Balance', 'IncorrectAnswer', 'Not array']}
      if ref $result ne 'ARRAY';
    throw Exception::Balance::IncorrectAnswer Dumper($result) if scalar @$result != 1;

    my $contract_id = $result->[0];

    return $contract_id;
}

sub create_collateral {
    my ($self, %opts) = @_;

    foreach (qw(operator_uid contract_id collateral_type)) {
        throw Exception::Balance gettext('Missed required fields: %s', $_) unless (defined $opts{$_});
    }

    my $operator = delete $opts{operator_uid};
    my $contract = delete $opts{contract_id};
    my $type     = delete $opts{collateral_type};

    $self->call('Balance.CreateCollateral', $operator, $contract, $type, \%opts);
}

sub close_contract {
    my ($self, %opts) = @_;

    my @param_list = qw(operator_uid contract_id END_DT);
    foreach ('contract_type', @param_list) {
        throw Exception::Balance gettext('Missed required fields: %s', $_) unless (defined $opts{$_});
    }
    my $type = delete $opts{contract_type};
    throw gettext("Incorrect %s = %s", 'contract_type', $type) unless ($CLOSE_CONTRACT_COLLATERAL_TYPES{$type});
    my %params = (
        collateral_type => $CLOSE_CONTRACT_COLLATERAL_TYPES{$type},
        END_REASON      => $opts{end_reason} // '1',
    );
    @params{@param_list} = @opts{@param_list};

    if ($opts{memo}) {
        @params{memo} = $opts{memo};
    }

    $self->create_collateral(%params);
}

sub create_collateral_selfemployed {
    my ($self, %opts) = @_;

    my @param_list = qw(operator_uid contract_id);
    foreach (@param_list) {
        throw Exception::Balance gettext('Missed required fields: %s', $_) unless (defined $opts{$_});
    }

    my %params = (
        collateral_type => 2160,
        selfemployed    => 1,
    );
    @params{@param_list} = @opts{@param_list};

    $self->create_collateral(%params);
}

sub create_collateral_game_offer {
    my ($self, %opts) = @_;

    my @param_list = qw(operator_uid contract_id memo);
    foreach (@param_list) {
        throw Exception::Balance gettext('Missed required fields: %s', $_) unless (defined $opts{$_});
    }

    my %params = (
        collateral_type => 2040,
        num             => "Ф-б/н",
    );
    @params{@param_list} = @opts{@param_list};

    $self->create_collateral(%params);
}

=head2 update_contract

L<https://wiki.yandex-team.ru/partner/w/external-services-api-balance-update-contract/>

=cut

sub update_contract {
    my ($self, %opts) = @_;

    throw "Expected 'operator_uid'" unless defined $opts{operator_uid};
    throw "Expected 'contract_id'"  unless defined $opts{contract_id};

    my $operator_uid = delete $opts{operator_uid};
    my $contract_id  = delete $opts{contract_id};

    my $result = $self->call('Balance2.UpdateContract', $operator_uid, $contract_id, \%opts);

    local $Data::Dumper::Terse = 1;

    throw Exception::Balance::IncorrectAnswer Dumper($result), undef, undef,
      sentry => {fingerprint => ['Balance', 'IncorrectAnswer', 'Not array']}
      if ref $result ne 'ARRAY';
    throw Exception::Balance::IncorrectAnswer Dumper($result) if scalar @$result != 1;

    my $contract = $result->[0];

    return $contract;
}

=head2 create_user_client_association

B<Параметры:> 1) $self 2) %opts

B<Возвращаемое значение:> -

Доступ к методу Balance.CreateUserClientAssociation - привязывание
представителя к клиенту.

Метод ничего не возвращает. В случае успеха метод молча отработает. В случае
какой-либо ошибки метод выбросит исключение.

Про исключения. Если случилось что-то совсем старшное и нерешаемое, то метод
выбрасывает исключение Exception::Balance::IncorrectAnswer. Если же произошла
ошибка про которую есть дополнительная информация, то будет прошено
искключение Exception::Balance::IncorrectAnswer::KnownErrors в котором есть 2
метода, которые позволяют получить дополнительную информацию об ошибке:
$exception->get_error_id() и $exception->get_error_text()

http://wiki.yandex-team.ru/Balance/XmlRpc#balance.createuserclientassociation

Значения %opts:

=over

=item B<operator_uid> - число, UID пользователя, который создает привязку
(обязательный параметр)

=item B<client_id> - число, Client ID к которому нужно привязать представителя
(обязательный параметр)

=item B<user_id> - число, uid пользователя, который будет являтеся
представителем. (обязательный параметр)

=back

Пример работы. В случае какой-то ошибки просто падаем с исключением:

    $app->api_balance->create_user_client_association(
        operator_uid => $operator_uid,
        client_id => $client_id,
        user_id => $user_id,
    );

Пример работы, где отрабатывется исключением:

    try {
        $app->api_balance->create_user_client_association(
            operator_uid => $operator_uid,
            client_id => $client_id,
            user_id => $user_id,
        );
    } catch Exception::Balance::IncorrectAnswer::KnownErrors with {
        my ($exception) = @_;
        p $exception->get_error_id();
        p $exception->get_error_text();
    };

=cut

sub create_user_client_association {
    my ($self, %opts) = @_;

    throw gettext("Expected 'operator_uid'") unless defined $opts{'operator_uid'};
    throw gettext("Expected 'client_id'")    unless defined $opts{'client_id'};
    throw gettext("Expected 'user_id'")      unless defined $opts{'user_id'};

    $self->_call_std('Balance2.CreateUserClientAssociation',
        $opts{'operator_uid'}, $opts{'client_id'}, $opts{'user_id'},);

    return TRUE;
}

=head2 find_client

B<Параметры:> 1) % хеш с запросом на поиск (описание ключей в доке
http://wiki.yandex-team.ru/Balance/XmlRpc#balance.findclient)

B<Возвращаемое значение:> 1) ссылка на массив со ссылками на хеш с ответом
баланса

Метод обращается к методу баланса Balance.FindClient. В случае каких-либо
ошибок возвращает исключение Exception::Balance::IncorrectAnswer. Возможна
ситуация что найден более чем один клиент - в этом случае массив будет
состоять более чем из одного элемента. В том случае если по запросу не найден
ни один клиент будет возвращена ссылка на пустой массив. Формат ключей -
canonical.

То данные которые выдает этот метод отличаются от данных в блоке Client метода
Balance.GetPartnerContracts.

    p $app->api_balance->find_client("Login" => "ivanbessarabov");
    {
        agency_id        0,
        city             "",
        client_id        1304746,
        client_type_id   0,
        email            "ivan@bessarabov.ru",
        fax              "",
        is_agency        0,
        name             "Бессарабов Иван",
        phone            "+7 (915) 480 2997",
        url              ""
    }

=cut

sub find_client {
    my ($self, %opts) = @_;

    my $data = $self->call('Balance.FindClient', \%opts,);

    my $result = [];

    throw Exception::Balance::IncorrectAnswer gettext('Could not interpret output from balance'), undef, undef,
      sentry => {fingerprint => ['Balance', 'IncorrectAnswer', 'Not array']}
      if ref $data ne 'ARRAY'
          or scalar $data->[0] != 0;

    foreach my $el (@{$data->[2]}) {
        push @{$result}, $self->_convert_hash_keys($el);
    }

    return $result;
}

=head2 get_client_id_by_uid

B<Параметры:> 1) $self 2) $uid

B<Возвращаемое значение:> 1) $client_id если указаный $uid является
представительем какоего-то клиента, иначе undef.

=cut

sub get_client_id_by_uid {
    my ($self, $uid) = @_;

    my $balance_answer = $self->find_client("PassportID" => $uid);

    if (ref $balance_answer eq 'ARRAY') {
        my $n = @{$balance_answer};

        if ($n == 0) {
            return undef;
        } elsif ($n == 1) {

            if ($balance_answer->[0]->{client_id}) {
                return $balance_answer->[0]->{client_id};
            } else {
                return undef;
            }

        } else {
            throw Exception::Balance::IncorrectAnswer
"For uid '$uid' got incorrect number of Balance.FindClient results: '$n'. It can containg only one result.";
        }

    } else {
        return undef;
    }
}

=head2 get_client_dsp

На входе $client_id.

На выходе ARRAYREF с хешами. Если нет плательщиков, то возвращается ARRAYREF
без элементов.

=cut

sub get_client_dsp {
    my ($self, $client_id) = @_;
    my $data = $self->call('Balance.GetClientDsp', $client_id);

    my %dsps;

    foreach (@{$data->[0]}) {
        $dsps{$_} = 1;
    }

    return \%dsps;
}

=head2 get_client_persons

На входе $client_id и $type.

На выходе ARRAYREF с хешами. Если нет плательщиков, то возвращается ARRAYREF
без элементов.

Используется метод баланса Balance.GetClientPersons. Возвращает из баланса
информацию о плательщиках, которые привязаны к указанному client_id.
Следует использовать этот метод, только когда у пользователя нет договора,
так как в договоре есть информация о текущем плательщике.

В случае если не найдены данные, то будет возвращена ссылка на пустой хеш.

Данные которые мы получаем через метод Balance.GetClientPersons несколько
отличаются от того чтоб мы получаем через метод Balance.GetPartnerContracts:

 * GetClientPersons поле dt в формате '2010-12-08 12:44:05', а в
   GetPartnerContracts поел dt в формате '2010-12-08'
 * У GetClientPersons есть значение у поля 'attributes', а у
   GetPartnerContracts нет (но значение никакой ценности не представляет).
 * У GetClientPersons есть значение у поля 'exports', а у
   GetPartnerContracts нет (но значение никакой ценности не представляет).

http://wiki.yandex-team.ru/Balance/XmlRpc#balance.getclientpersons

    # Число-констаната, означающая тип плательщика, которые нужно возвращать
    # 1 - тип плательщика 'partner' - Яндекс платит таким плательщикам
    # 0 - тип плательщика 'обычный' - такие плательщики платит Яндексу
    my $type = 1;

    p $app->api_balance->get_client_persons($client_id, $type);
    {
        account                  40817810004210005493,
        address_postcode         123056,
        authority_doc_details    "",
        ...
    }

=cut

sub get_client_persons {
    my ($self, $client_id, $type, $return_full_data) = @_;

    if (!defined($type) || !in_array($type, [0, 1])) {
        throw Exception::Validation::BadArguments gettext("Incorrect type");
    }

    my $data = $self->call('Balance.GetClientPersons', $client_id, $type);

    my @persons;

    foreach (@{$data->[0]}) {
        my $fixed_data = $self->_convert_hash_keys($_);
        $self->_delete_unused_person_data($fixed_data) unless $return_full_data;
        push @persons, $fixed_data;
    }

    return \@persons;
}

=head2 get_client_users

Используется метод баланса Balance.GetClientUsers. Возвращает из баланса
ссылку на массив с хешами информации о представителях клиента, которые
привязаны к указанному client_id.

В том случае, если у указанного Client ID нет представителей, то будет
возвращена ссылка на пустой массив.

В том случае, если в балансе нет клиента с указанным Client ID, то так же
будет возвращена ссылка на пустой массив.

Про метод GetClientUsers есть информация на странице
http://wiki.yandex-team.ru/Balance/BalanceService

    p $app->api_balance->get_client_users($client_id);
    [
        [0] {
            gecos         "Бессарабов Иван",
            login         "ivanbessarabov",
            passport_id   35309619
        }
    ]

У одного клиента может быть больше чем один представитель. Например, у
client_id = 943972 есть 2 представителя.

=cut

sub get_client_users {
    my ($self, $client_id) = @_;

    my $data = $self->call('Balance.GetClientUsers', $client_id);

    my $result = [];

    throw Exception::Balance::IncorrectAnswer gettext('Could not interpret output from balance'), undef, undef,
      sentry => {fingerprint => ['Balance', 'IncorrectAnswer', 'Not array']}
      if ref $data ne 'ARRAY';

    foreach my $el (@{$data->[0]}) {
        push @{$result}, $self->_convert_hash_keys($el);
    }

    return $result;
}

=head2 get_currency_rate

Access to method Balance.GetCurrencyRate - currency_rate getting.

B<Options:> 1) $self 2) %opts - hash with date and currency code which rate
is required.

=over

=item B<currency> - Currency code, e. g. USD, EUR.

=item B<date> - (Optional) Currency rate date. 'yyyy-mm-dd'

=back

B<Returning value:> $rate

    my $rate = $self->api_balance->get_currency_rate(
        currency => 'USD',
        date     => '2014-01-20'
    );

=cut

sub get_currency_rate {
    my ($self, %opts) = @_;

    my $currency = delete($opts{'currency'});
    my $date = delete($opts{'date'}) // curdate(oformat => 'db');

    throw gettext("Missed required field '%s'", 'currency') unless defined $currency;
    throw gettext("Incorrect %s = %s", 'date', $date) if defined $date && !trdate(db => sec => $date);

    return $self->_call_std('Balance2.GetCurrencyRate', $currency, $date)->{'rate'};
}

sub get_dsp_money_completition {
    my ($self, %params) = @_;

    return $self->_call_method_from_to(
        method => 'Balance2.GetDspMoneyCompletion',
        from   => $params{from},
        to     => $params{to},
    );
}

sub get_dsp_money_completition_with_page_id {
    my ($self, %params) = @_;

    return $self->_call_method_from_to(
        method => 'Balance2.GetDspMoneyCompletionWithPageId',
        from   => $params{from},
        to     => $params{to},
    );
}

=head2 get_dsp_stat

Работа с методом Balance2.GetDspStat

    my $data = $app->api_balance->get_dsp_stat(
        from => '2016-09-01',
        to => '2016-09-01',
    );

Опциональные параметры:

    my $data2 = $app->api_balance->get_dsp_stat(
        from => '2016-09-01',
        to => '2016-09-01',
        include_deals => 0, # может быть 0 или 1, по дефолту 0
        include_partner_stat_id => 0, # может быть 0 или 1, по дефолту 0
    );


L<https://wiki.yandex-team.ru/partner/w/get-dsp-stat/>

=cut

sub get_dsp_stat_orig {
    my ($self, %params) = @_;

    my $method = 'Balance2.GetDspStat';

    my $start = curdate(oformat => 'sec');

    my $cache_params = {
        'from' => $params{from},
        'to'   => $params{to},
        $params{include_deals} ? (include_deals => 1)
        : (),
        $params{include_partner_stat_id} ? (include_partner_stat_id => 1)
        : (),
    };

    INFOF('start [%s] (%s)', $method, join('; ', map {join '=', $_, $cache_params->{$_}} sort keys %$cache_params));

    my $cache_file_path = $self->_get_cache_file_path($method, %$cache_params);
    my $is_cache_ok = $self->_check_cache_file($cache_file_path);

    my $log_cached = '';
    if ($is_cache_ok) {
        $log_cached = " (cached)";
    } else {
        my $res = $self->call(
            $method, $params{from}, $params{to},
            ($params{include_deals}           // 0),
            ($params{include_partner_stat_id} // 0),
        );
        File::Slurp::write_file($cache_file_path, \$res->[0]);
    }

    my $end = curdate(oformat => 'sec');

    my $delta_seconds = $end - $start;

    INFOF(
        'end [%s] (%s). elapsed %s seconds%s',
        $method, join('; ', map {join '=', $_, $cache_params->{$_}} sort keys %$cache_params),
        $delta_seconds, $log_cached
    );

    if ($params{_get_tsv_fh}) {
        open(my $tsv_fh, '<', $cache_file_path) or die "Cannot open cache file $cache_file_path for reading";
        return $tsv_fh;
    } else {
        my $tsv_ref = File::Slurp::read_file($cache_file_path, scalar_ref => 1);

        if ($params{_get_tsv_ref}) {
            return $tsv_ref;
        } elsif ($params{_get_tsv}) {
            return $$tsv_ref;
        } else {
            return parse_tsv($$tsv_ref);
        }
    }
}

sub get_internal_pages_stat_orig {
    my ($self, %params) = @_;

    return $self->_call_method_from_to(
        method => 'Balance2.GetInternalPagesStat',
        from   => $params{from},
        to     => $params{to},
    );
}

sub get_internal_pages_tags_stat_orig {
    my ($self, %params) = @_;

    return $self->_call_method_from_to(
        method => 'Balance2.GetInternalPagesTagsStat',
        from   => $params{from},
        to     => $params{to},
    );
}

sub get_pages_stat_orig {
    my ($self, %params) = @_;

    return $self->_call_method_from_to(
        method => 'Balance.GetPagesStat',
        %params
    );
}

sub get_pages_tags_stat_orig {
    my ($self, %params) = @_;

    return $self->_call_method_from_to_curl_sort(
        method => 'Balance2.GetPagesTagsStat',
        %params
    );
}

# Balance.GetDistributionDownloads
# Возвращает структуру с данными
sub get_parsed_distribution_downloads {
    my ($self, %params) = @_;

    throw gettext("Don't use Balance.GetDistributionDownloads, use Balance.GetDistributionFixed.");

    return $self->_call_method_from_to(
        method => 'Balance.GetDistributionDownloads',
        from   => $params{from},
        to     => $params{to},
    );
}

# Balance.GetDistributionFixed
# Возвращает структуру с данными
sub get_parsed_distribution_fixed {
    my ($self, %params) = @_;

    return $self->_call_method_from_to(
        method => 'Balance.GetDistributionFixed',
        from   => $params{from},
        to     => $params{to},
    );
}

# Balance.GetDistributionRevenueShare
# Возвращает структуру с данными
sub get_parsed_distribution_revenue_share {
    my ($self, %params) = @_;

    return $self->_call_method_from_to(
        method => 'Balance.GetDistributionRevenueShare',
        from   => $params{from},
        to     => $params{to},
    );
}

# Balance.GetDistributionRevenueShareFull
# Возвращает структуру с данными
sub get_parsed_distribution_revenue_share_full {
    my ($self, %params) = @_;

    return $self->_call_method_from_to(
        method => 'Balance.GetDistributionRevenueShareFull',
        from   => $params{from},
        to     => $params{to},
    );
}

# Balance.GetDistributionSerpHits
# Возвращает структуру с данными
sub get_parsed_distribution_serp_hits {
    my ($self, %params) = @_;

    return $self->_call_method_from_to(
        method => 'Balance.GetDistributionSerpHits',
        from   => $params{from},
        to     => $params{to},
    );
}

=head2 get_partner_contracts

B<Параметры:> 1) % хеш с запросом на поиск:

=over

=item B<ClientID> - число, ID клиента в балансе.

=item B<ExternalID> - строка, номер договора.

=back

B<Возвращаемое значение:> 1) ссылка на массив хешей с ответом баланса

Метод обращается к методу баланса Balance.GetPartnerContracts. В случае
каких-либо ошибок выбрасывает исключение.

В данных плательщика удаляются ненужные поля с помощью метода
_delete_unused_person_data

Уведомления не подписываются.

    p $app->api_balance->get_partner_contracts("ClientID" => $client_id);
    {
        Client        { ... }, # клиент
        Collaterals   [ ... ], # дополнительные соглашения
        Contract      { ... }, # договор
        Person        { ... }, # плательщик
    }

=cut

sub get_partner_contracts {
    my ($self, %opts) = @_;

    my $data = $self->call('Balance.GetPartnerContracts', \%opts);

    $data = $data->[0];

    foreach (@$data) {
        $self->_delete_unused_person_data($_->{Person});
        $self->_fix_dates($_->{Contract}, @{$_->{Collaterals} // []});
    }

    return $data;
}

=head1 Методы

=head2 init

B<Параметры:> -

B<Возвращаемое значение:> -

Инициализация

TODO - добится от баланса принятия серелизованных строк не только типа string
и убрать хак.

=cut

sub init {
    my ($self) = @_;

    $self->SUPER::init();

    # Баланс не умеет воспринимать только "<value><string>...</", поэтому все
    # типы объектов мы серилизуем в string
    # Но из-за этого возникает warning:
    # Content-Length header value was wrong, fixed at /usr/share/perl5/LWP/Protocol/http.pm line 190.
    $self->{__RPC__}->{_serializer}->{_typelookup} = {
        dateTime => [1, sub {$_[0] =~ /^\d{8}T\d\d:\d\d:\d\d$/}, 'as_dateTime'],
        string => [2, sub {1}, 'as_string'],
    };

}

sub link_dsp_to_client {
    my ($self, %opts) = @_;

    throw Exception::Validation::BadArguments("Expected 'user_id'") unless defined $opts{user_id};

    my $user_id = delete $opts{user_id};

    my $result = $self->call('Balance2.LinkDspToClient', $user_id, \%opts)->[0];

    throw Exception::Balance::IncorrectAnswer gettext('Could not interpret output from balance'), undef, undef,
      sentry => {fingerprint => ['Balance', 'IncorrectAnswer', 'Not array']}
      if ref $result ne 'ARRAY';
    throw Exception::Balance::IncorrectAnswer join(' ', @{$result}) if scalar $result->[0] != 0;

    return TRUE;
}

sub link_dsp_to_client_with_confirm {
    my ($self, %opts) = @_;
    try {
        $self->link_dsp_to_client(%opts);
    }
    catch Exception::Balance::IncorrectAnswer with {
        my ($e) = @_;
        # возможно уже был создан link
        my %dsps = %{$self->app->api_balance->get_client_dsp($opts{CLIENT_ID})};
        unless (exists $dsps{$opts{DSP_ID}}) {
            throw $e;
        }
    };
}

=head2 query_catalog

B<Параметры:> 1) $self 2) $tables 3) $where 4) {} (опц.) - указание как привести поля

B<Возвращаемое значение:> 1) $data - структура с данными

http://wiki.yandex-team.ru/Balance/XmlRpc#balance.querycatalog

Например:

    $proj->billing->query_catalog(["T_BRAND"], '');

Это вернет структуру вида:

    $VAR1 = {
              'columns' => [
                           't_brand.bid',
                           't_brand.name',
                           't_brand.ename',
                           't_brand.note',
                           't_brand.status',
                           't_brand.file_id',
                           't_brand.hidden'
                         ],
              'result' => [
                          [
                            '706769.0000000000',
                            "ЭЛЕКТРОННЫЙ СПРАВОЧНИК КОНСТРУКТОРА",
                            'ELEKTRONNY SPRAVOCHNIK KONSTRUKTORA',
                            undef,
                            'A',
                            '2.0000000000',
                            undef
                          ],
                          ...
                          ]

            }

В случае если указан 4 параметр - {t_brand.name => 'name', t_brand.file_id => 'id'}, Вернет структуру вида:

    $VAR1 = [
        { id => '2.0000000000',  name => "ЭЛЕКТРОННЫЙ СПРАВОЧНИК КОНСТРУКТОРА" },
        ...
    ]


=cut

sub query_catalog {
    my ($self, $tables, $where, $map) = @_;

    my $result = $self->call('Balance.QueryCatalog', $tables, $where);

    if (ref $result eq 'ARRAY') {
        $result = $result->[0];
    } else {
        throw Exception::Balance::IncorrectAnswer gettext('Could not interpret output from balance'), undef, undef,
          sentry => {fingerprint => ['Balance', 'IncorrectAnswer', 'Not array']};
    }

    if ($map) {
        my ($list_columns, $list_data) = @{$result}{'columns', 'result'};
        # get fields names
        my @columns_out = map($map->{$_} || '', @$list_columns);
        # convert array into hashes with new f. names
        my $list_data_out = [
            map({
                    my $a = {};
                      @$a{@columns_out} = @$_;
                      delete($a->{''});
                      $a;
                } @$list_data)
        ];
        return $list_data_out;
    } else {
        return $result;
    }
}

=head2 remove_user_client_association

B<Параметры:> 1) $self 2) $uid оператора, который проводит изменение
3) $client_id 4) $uid представителя, которого нужно отвязать от клиента

B<Возвращаемое значение:> 1) true если представитель успешно отвязан

Доступ к методу Balance.RemoveUserClientAssociation - отвязывает предствителя
от клиента

http://wiki.yandex-team.ru/Balance/XmlRpc#balance.removeuserclientassociation

Метод выкидывает исключение, если не удалось отвязать представителя.

Метод умирает, если запустить его не на тестовой версиии баланса.

=cut

sub remove_user_client_association {
    my ($self, $operator_uid, $client_id, $user_uid) = @_;

    if ($self->get_option('url') ne 'http://greed-ts.paysys.yandex.ru:8002/xmlrpc') {
        throw gettext("Method RemoveUserClientAssociation should be run only with test balance instance");
    }

    my $data = $self->call('Balance.RemoveUserClientAssociation', $operator_uid, $client_id, $user_uid,);

    throw Exception::Balance::IncorrectAnswer gettext('Could not interpret output from balance'), undef, undef,
      sentry => {fingerprint => ['Balance', 'IncorrectAnswer', 'Not array']}
      if ref $data ne 'ARRAY';
    throw Exception::Balance::IncorrectAnswer join(' ', @{$data}) if scalar $data->[0] != 0;

    return TRUE;
}

=head2 get_bank

L<https://wiki.yandex-team.ru/Balance/XmlRpc/#balance.getbank>

    my $bik_data = $app->api_balance->get_bank(
        Bik => '044525225',
    );

После этого в $bik_data будет:

    [
        {
            bank_address => undef,
            bank_city => 'МОСКВА',
            bank_name => 'ПАО СБЕРБАНК',
            bik => '044525225',
            city => 'МОСКВА',
            cor_acc => 30101810400000000225,
            corr_account => 30101810400000000225,
            hidden => 0,
            id => 1103,
            info => undef,
            name => 'ПАО СБЕРБАНК',
            swift => 'SABRRUMMXXX',
            update_dt => '20180124T04:00:38',
        },
    ]

Так же можно вызвать с параметром Swift:

    my $swift_data = $app->api_balance->get_bank(
        Swift => 'INGBNL2A',
    );

После этого в $swift_data будет:

    [
        {
            address => 'FINANCIAL PLAZA, BIJLMERDREEF 109',
            bicint => 'INGBNL2AXXX',
            country => 'NETHERLANDS',
            name => 'ING BANK N.V.',
            place => 'AMSTERDAM',
            zipcode => '1102 BW AMSTERDAM',
        },
    ]

Если по указанным данным ничего не нашлось, то бросает исключение Exception::Balance::NotFound.

В случае других ошибок бросает другие исключения.

=cut

sub get_bank {
    my ($self, %opts) = @_;

    my $data;

    try {
        $data = $self->call('Balance2.GetBank', \%opts);
    }
    catch {
        my $exception = shift;

        my $error_message = $exception->message();

        if ($error_message =~ m'<code>CLASSBANK_NOT_FOUND</code>' || $error_message =~ m'<code>NOT_FOUND</code>') {
            throw Exception::Balance::NotFound;
        } else {
            throw $exception;
        }
    };

    return $data;
}

sub check_bank {
    my ($self, %params) = @_;

    my %opts;

    if (defined($params{bik}) && defined($params{swift})) {
        throw Exception::Validation::BadArguments "Can't use both bik and swift";
    } elsif (defined($params{bik})) {
        $opts{Bik} = $params{bik};
    } elsif (defined($params{swift})) {
        $opts{Swift} = $params{swift};
    } else {
        throw Exception::Validation::BadArguments "Must specify bik or swift";
    }

    my $data;

    try {
        $data = $self->get_bank(%opts);
    }
    catch Exception::Balance::NotFound with {};

    if (ref($data) eq 'ARRAY' && @$data == 1) {
        return {
            found => 1,
            info  => $data->[0],
        };
    } else {
        return {found => 0,};
    }
}

sub _call_method_from_to_curl_sort {
    my ($self, %params) = @_;

    throw gettext("Need 'method'") unless defined $params{method};
    throw gettext("Need 'from'")   unless defined $params{from};
    throw gettext("Need 'to'")     unless defined $params{to};

    my $method = delete($params{'method'});
    my $from   = delete($params{'from'});
    my $to     = delete($params{'to'});

    my $start = curdate(oformat => 'db_time');

    INFO("[$method] ($from - $to) start");

    my $cache_file_path = $self->_get_cache_file_path($method, 'from' => $from, 'to' => $to);
    my $is_cache_ok = $self->_check_cache_file($cache_file_path);

    my $log_cached = '';
    if ($is_cache_ok) {
        $log_cached = " (cached)";
    } else {
        my $balance_url = $self->get_option('api_balance', {})->{'url'};

        my $xml_request = qq{
            <?xml version="1.0" encoding="UTF-8"?>
            <methodCall>
                <methodName>$method</methodName>
                <params>
                    <param><value><string>$from</string></value></param>
                    <param><value><string>$to</string></value></param>
                </params>
            </methodCall>
        };

        # exit 1 if fault; leave the faultsting value in the output
        # remove tags; remove empty strings
        # the result should be a tsv that was inside this xml answer
        #
        my $sed_xml_parse = '/<name>faultString<\/name>/ {n; q 1}; /<.*>/ s/<.*>//; /^$/ d';

        # request using curl; parse with sed; sort by the 1st column (page_id)

        my $cmd = qq{
            set -o pipefail ;
            curl -s -X POST '$balance_url' -d '$xml_request' |
            sed '$sed_xml_parse' |
            sort -k 1 -n -s > $cache_file_path
        };

        my $error;

      TRY:
        for my $try (1 .. 3) {
            undef($error);

            system($cmd);

            if ($? == -1) {
                $error = "Failed to execute: $!";
            } elsif ($? & 127) {
                $error = sprintf("Child died with signal %d, %s coredump", ($? & 127), ($? & 128) ? 'with' : 'without');
            } elsif ($?) {
                $error = File::Slurp::read_file($cache_file_path);
                $error = sprintf("Child exited with value %d. XML faultString: %s", ($? >> 8), $error);
                system("rm $cache_file_path");
            } else {
                last TRY;
            }
            sleep(1);
        }

        throw Exception $error if defined($error);
    }

    my $end = curdate(oformat => 'db_time');
    my $delta_seconds = trdate('db_time' => 'sec', $end) - trdate('db_time' => 'sec', $start);

    INFO("[$method] ($from - $to) end. elapsed $delta_seconds seconds${log_cached}.");

    if ($params{_get_tsv_fh}) {
        open(my $tsv_fh, '<', $cache_file_path) or die "Cannot open cache file $cache_file_path for reading";
        return $tsv_fh;
    } else {
        my $tsv_ref = File::Slurp::read_file($cache_file_path, scalar_ref => 1);

        if ($params{_get_tsv_ref}) {
            return $tsv_ref;
        } elsif ($params{_get_tsv}) {
            return $$tsv_ref;
        } else {
            return parse_tsv($$tsv_ref);
        }
    }
}

=begin comment _call_method_from_to

=end comment

=cut

sub _call_method_from_to {
    my ($self, %params) = @_;

    throw gettext("Need 'method'") unless defined $params{method};
    throw gettext("Need 'from'")   unless defined $params{from};
    throw gettext("Need 'to'")     unless defined $params{to};

    my $method = delete($params{'method'});

    my $start = curdate(oformat => 'db_time');

    INFO("[$method] ($params{'from'} - $params{'to'}) start");

    my $cache_file_path = $self->_get_cache_file_path($method, 'from' => $params{from}, 'to' => $params{to});
    my $is_cache_ok = $self->_check_cache_file($cache_file_path);

    my $log_cached = '';
    if ($is_cache_ok) {
        $log_cached = " (cached)";
    } else {
        my $tsv = $self->call($method, $params{from}, $params{to})->[0];
        File::Slurp::write_file($cache_file_path, \$tsv);
    }

    my $end = curdate(oformat => 'db_time');
    my $delta_seconds = trdate('db_time' => 'sec', $end) - trdate('db_time' => 'sec', $start);

    INFO("[$method] ($params{'from'} - $params{'to'}) end. elapsed $delta_seconds seconds${log_cached}.");

    if ($params{_get_tsv_fh}) {
        open(my $tsv_fh, '<', $cache_file_path) or die "Cannot open cache file $cache_file_path for reading";
        return $tsv_fh;
    } else {
        my $tsv_ref = File::Slurp::read_file($cache_file_path, scalar_ref => 1);

        if ($params{_get_tsv_ref}) {
            return $tsv_ref;
        } elsif ($params{_get_tsv}) {
            return $$tsv_ref;
        } else {
            return parse_tsv($$tsv_ref);
        }
    }
}

=begin comment _call_std

=end comment

=cut

sub _call_std {
    my ($self, $method, @opts) = @_;

    my $result = $self->call($method, @opts);

    throw Exception::Balance::IncorrectAnswer Dumper($result), undef, undef,
      sentry => {fingerprint => ['Balance', 'IncorrectAnswer', 'Not array']}
      if (ref($result) ne 'ARRAY') || (ref($result->[0]) ne 'ARRAY');

    if (($result->[0]->[0] ne '0') or ($result->[0]->[1] ne 'SUCCESS')) {
        throw Exception::Balance::IncorrectAnswer::KnownErrors {
            error_id   => $result->[0][0],
            error_text => $result->[0][1],
        };
    }

    return $result->[0][2];
}

=begin comment _convert_hash_keys

Что это такое - читай в pod в разделе 'Описание'

=end comment

=cut

sub _convert_hash_keys {
    my ($self, $hash, $format) = @_;

    $format = 'canonical' if !defined($format);
    throw gettext("Incorrect format '%s'", $format) if !in_array($format, [qw(uppercase canonical lcminus)]);

    my $fixed_hash;

    if ($format eq 'uppercase') {

        $fixed_hash = {map {my $new_key = $_; $new_key =~ s/-/_/g; uc $new_key => $hash->{$_}} keys %$hash};

    } elsif ($format eq 'canonical') {

        $fixed_hash = {map {my $new_key = $_; $new_key =~ s/-/_/g; lc $new_key => $hash->{$_}} keys %$hash};

    } elsif ($format eq 'lcminus') {

        $fixed_hash = {map {my $new_key = $_; $new_key =~ s/_/-/g; lc $new_key => $hash->{$_}} keys %$hash};

    }

    return $fixed_hash;
}

=begin comment _delete_unused_person_data

B<Параметры:> 1) $self, 2) $person - ссылка на хеш с данными плательщика

B<Возвращаемое значение:> -

Метод предназанчеен для удаления некоторых полей из данных платильщика,
которые мы получаем в методах GetPartnerContracts и GetClientPersons.

Метод изменяет переданный параметр $person

Цель этого метода - это подготовить ПИ к ситуации, когда баланс отрежет
эти поля. Т.е. заранее отрезаем поля, убеждаемся что все работает, а потом
и баланс у себя убирает эти поля.

В этом методе есть массив полей, которые мы удаляем, по хорошему он должен
быть пуст - что означает что баланс у себя уже все удалил.

=end comment

=cut

sub _delete_unused_person_data {
    my ($self, $person) = @_;

    # В некоторых случаях Баланс в качетсве данных плательщика может вернуть
    # '0'. В этом случае не нужно править данные.
    return if ref($person) ne 'HASH';

    my @what_to_delete_from_person_data = qw(
      bankcity
      bank_data
      bank_inn
      corraccount

      address_town
      address_flat
      address_street
      address_construction
      address_district
      address_code
      address_home
      address_building
      address_region
      address_city
      address_gni
      );

    foreach my $k (keys %$person) {
        foreach my $el (@what_to_delete_from_person_data) {
            delete $person->{$k} if $k =~ /^$el$/i;
        }
    }
}

sub _fix_dates {
    my $self = shift;

    my @fields = qw(
      create_dt
      dt
      end_dt
      is_cancelled
      is_faxed
      is_signed
      service_start_dt
      start_dt
      update_dt
      );

    # Date can be in several formats (date, db_time, db_time_t)
    # It is need only date
    foreach my $data (@_) {
        foreach my $field (@fields) {
            if ($data->{$field}) {
                $data->{$field} = substr $data->{$field}, 0, 10;
            }
        }
    }
}

=head2
    $self->make_create_offer_params_adfox_paid_services(
        operator_uid => 666,
        client_id => 123,
        person_id => 111,
        contract_currency => '',
        adfox_login => 'account_adfox'
    );
=cut

sub make_create_offer_params_adfox_paid_services {
    my ($self, %opts) = @_;

    # 1120000000153798 = https://staff.yandex-team.ru/stanakov
    # see https://st.yandex-team.ru/PI-17263
    my $ADFOX_CONTRACT_MANAGER_ID = $self->get_option('adfox_contract_manager_id', '1120000000153798');

    return {
        'operator_uid' => $opts{operator_uid},
        'client_id'    => $opts{client_id},
        'currency'     => $opts{contract_currency},
        'firm_id'      => $FIRM_ID_YANDEX_LTD,
        'manager_uid'  => $ADFOX_CONTRACT_MANAGER_ID
        , # менеджер, которому будут отсылаться все уведомления по этому договору
        'payment_term'   => 20,                         # Срок оплаты счетов – 20 дней
        'payment_type'   => 3,                          # постоплата
        'person_id'      => $opts{person_id},
        'services'       => [102],                      # ADFox
        'start_dt'       => curdate(oformat => 'db'),
        'dmp_segments'   => 1,                          # Доступны сегменты DMP. 1 или 0
        'partner_credit' => 1,                          # PI-19825
    };
}

sub register_aggregator_pages {
    my ($self, $data) = @_;

    if (ref($data) eq 'HASH') {
        $data = [$data];
    }

    my $tm      = Time::HiRes::time;
    my $version = int($tm * 1_000);
    $self->api_selfservice->logbroker(
        topic => $self->get_option('aggregator_pages_topic'),
        data  => [
            map {
                +{
                    version => 0 + $version,
                    obj     => {
                        id       => 0 + $_->{page_id},
                        PageID   => 0 + $_->{page_id},
                        ClientID => 0 + ($_->{aggregator_client_id} // $AGGR_TUTBY->{client_id}),
                        StartDT  => $_->{start_dt},
                        EndDT    => $_->{end_dt},
                    },
                 }
              } @$data
        ],
    );
}

# Примеры вызова
#     $app->api_balance->update_person_for_payoneer(
#         client_id    =>  34079181,
#         operator_uid => 269268291,
#         payee_id     => 269268291,
#         person_id    =>  11717846,
#         type         => 'sw_ytph',
#     );
#
# ./rosetta_call --model=api_balance --method=update_person_for_payoneer --opts='{"client_id" : 34079181, "operator_uid" : 269268291, "payee_id" : 269268291, "person_id" : 11717846, "type" : "sw_ytph"}'
#
# curl -s 'http://greed-ts.paysys.yandex.ru:8002/xmlrpc' \
#     -X POST \
#     -H 'Accept: text/xml' \
#     -H 'Accept: multipart/*' \
#     -H 'Accept: application/soap' \
#     -H 'Content-Type: text/xml' \
#     -H 'Content-Length: 768' \
#     --data '<?xml version="1.0" encoding="UTF-8"?><methodCall><methodName>Balance2.CreatePerson</methodName><params><param><value><string>269268291</string></value></param><param><value><struct><member><name>payoneer_wallet</name><value><string>269268291</string></value></member><member><name>person_id</name><value><int>11717846</int></value></member><member><name>operator_uid</name><value><string>269268291</string></value></member><member><name>client_id</name><value><string>34079181</string></value></member><member><name>type</name><value><string>sw_ytph</string></value></member><member><name>bank_type</name><value><string>7</string></value></member><member><name>is_partner</name><value><string>1</string></value></member></struct></value></param></params></methodCall>'

sub update_person_for_payoneer {
    my ($self, %opts) = @_;

    my $payee_id = delete $opts{payee_id};
    throw Exception::Validation::BadArguments gettext('Missed required parameter "%s"', 'payee_id') unless $payee_id;

    $opts{bank_type}       = 7;
    $opts{is_partner}      = 1;
    $opts{payoneer_wallet} = $payee_id;

    return $self->create_person(%opts);
}

# Примеры вызова
#     $app->api_balance->update_contract_payment_type(
#         contract_id  =>   2593152,
#         operator_uid => 269268291,
#         pay_to       =>         7,
#     );
#
# ./rosetta_call --model=api_balance --method=update_contract_payment_type --opts='{"contract_id" : 2593152, "operator_uid" : 269268291, "pay_to" : 7}'
#
# curl -s 'http://greed-ts.paysys.yandex.ru:8002/xmlrpc' \
#     -X POST \
#     -H 'Accept: text/xml' \
#     -H 'Accept: multipart/*' \
#     -H 'Accept: application/soap' \
#     -H 'Content-Type: text/xml' \
#     -H 'Content-Length: 574' \
#     --data '<?xml version="1.0" encoding="UTF-8"?><methodCall><methodName>Balance.CreateCollateral</methodName><params><param><value><string>269268291</string></value></param><param><value><string>2593152</string></value></param><param><value><string>2110</string></value></param><param><value><struct><member><name>pay_to</name><value><string>7</string></value></member><member><name>dt</name><value><dateTime.iso8601>20210303T13:18:02</dateTime.iso8601></value></member><member><name>sign</name><value><string>1</string></value></member></struct></value></param></params></methodCall>'

sub update_contract_payment_type {
    my ($self, %opts) = @_;

    $opts{collateral_type} = 2110;
    $opts{DT}              = curdate(oformat => 'iso8601');
    $opts{sign}            = 1;

    return $self->create_collateral(%opts);
}

# Примеры вызова
#     $app->api_balance->update_person_for_payoneer(
#         contract     => '{ __OLD_CONTRACT_DATA__ }',
#         currency     =>        'USD',
#         dt           => '2021-03-03',
#         operator_uid =>    269268291,
#     );

sub create_new_contract_for_payoneer {
    my ($self, %opts) = @_;

    foreach (qw(contract dt currency operator_uid)) {
        throw Exception::Validation::BadArguments gettext('Missed required parameter "%s"', $_) unless $opts{$_};
    }

    my %new_contract = hash_transform($opts{contract}, \@FIELDS_TO_COPY_CONTRACT, \%FIELDS_TO_COPY_CONTRACT);

    $new_contract{currency}         = $opts{currency};
    $new_contract{operator_uid}     = $opts{operator_uid};
    $new_contract{pay_to}           = 7;
    $new_contract{service_start_dt} = $opts{dt};
    $new_contract{signed}           = 1;
    $new_contract{start_dt}         = $opts{dt};

    return $self->create_offer(%new_contract);
}

sub check_rubank_account {
    my ($self, %opts) = @_;

    my $data;
    try {
        $data = $self->call('Balance.CheckRUBankAccount', $opts{bik}, $opts{account} ? $opts{account} : ());
    }
    catch Exception::Balance::NotFound with {
        $data = [[1, 'NotFound']];
    };

    return {
        found => !$data->[0][0],
        info  => $data->[0][1],
    };
}

TRUE;
