
=head1 Name

QBit::Application::Model::API::Yandex::BlackBox

=head2 Настройки

 * Debug - $, boolean
 * URL - $, string. Урл интерфейса блэкбокса
 * MayOnlySocial - $, boolean. Возможна только социальная авторизация (без логина на Я).

=cut

package QBit::Application::Model::API::Yandex::BlackBox;

use qbit;

use base qw(QBit::Application::Model::API::HTTP);

use LWP::UserAgent;
use URI::Escape qw(uri_escape_utf8);
use HTTP::Request;
use XML::Simple;

use Utils::Logger qw(ERROR WARN);

use Exception::BlackBox;
use Exception::BlackBox::Net;
use Exception::BlackBox::InternalError;
use Exception::BlackBox::NeedResign;
use Exception::BlackBox::NeedAuth;
use Exception::BlackBox::NeedRealLogin;
use Exception::BlackBox::NotSecure;
use Exception::Validation::BadArguments;

sub accessor {'api_blackbox'}

__PACKAGE__->mk_ro_accessors(qw(ua));

sub get_structure_model_accessors {
    return {api_tvm => 'Application::Model::API::Yandex::TVM',};
}

our $DEBUG;
my $BLACKBOX_CACHE_TTL = 10;

sub init {
    my ($self) = @_;

    $self->SUPER::init();

    throw gettext('Need URL of BlackBox in config') unless $self->get_option('URL');

    $DEBUG = $self->get_option('Debug');
}

=head2 oauth

Получаем информацию о сессии из ящика. В случае ошибки 2 типа исключения:

    my $bb_res = $self->api_blackbox->oauth(
        token       => $self->request->http_header('Authorization'),
        remote_addr => $self->request->remote_addr(),
        userip      => $self->request->remote_addr(),
        fields      => [qw(
            accounts.login.uid
            account_info.fio.uid
        )],
    );

=cut

sub oauth {
    my ($self, %opts) = @_;

    my $user_ticket = delete $opts{user_ticket};

    my @missed_required_fields = grep {!defined($opts{$_})} qw(token userip);
    throw Exception::Validation::BadArguments ngettext(
        'Missed required argument %s',
        'Missed required arguments %s',
        scalar(@missed_required_fields),
        join(', ', @missed_required_fields)
    ) if @missed_required_fields;

    my @fields = @{$opts{'fields'} // []};

    my $res = $self->call(
        method      => 'oauth',
        oauth_token => $opts{'token'},
        userip      => $opts{'userip'},
        ($user_ticket ? (get_user_ticket => 1) : ()),
        (@fields ? (dbfields => join(',', @fields)) : ()),
    );

    WARN({blackbox => $res}) if $DEBUG;

    throw Exception::BlackBox::InternalError($res->{'error'}
          && $res->{'error'}->{'content'} ? $res->{'error'}->{'content'} : gettext('Unknown error'))
      unless $res->{'status'}
          && defined($res->{'status'}->{'content'});

    if ($res->{'status'}->{'content'} ne 'VALID') {
        throw Exception::BlackBox::NeedAuth 'Not authorized';
    }

    return $res;
}

=head2 sessionid

Получаем информацию о сессии из ящика. В случае ошибки 2 типа исключения:

Exception::BlackBox::NeedAuth - не авторизован (совсем или протухло)

Exception::BlackBox::NeedResign - требуется продление (по времени или запрос главного домена)

Exception::BlackBox::NotSecure - нет куки sessionid2 или она не валидна

    my $bb_res = $self->api_blackbox->sessionid(
        session_id      => $self->request->cookie('Session_id'),
        sessionid2      => $self->request->cookie('sessionid2'),        # optional
        remote_addr     => $self->request->remote_addr(),
        server_name     => $self->request->server_name(),
        fields          => [qw(
            accounts.login.uid
            account_info.fio.uid
            subscription.suid.85
        )],
        must_be_secure => 1, # optional. Если есть, то может быть выброшено исключение Exception::BlackBox::NotSecure
    );

Документация
 http://wiki.yandex-team.ru/AuthLib
 http://wiki.yandex-team.ru/passport/mda/intro

=cut

sub sessionid {
    my ($self, %opts) = @_;

    my $session_id  = delete $opts{'session_id'};
    my $sessionid2  = delete $opts{'sessionid2'};
    my $user_ticket = delete $opts{user_ticket};

    my $remote_addr = delete $opts{'remote_addr'};
    throw Exception::Validation::BadArguments gettext("Expected remote_addr") if !defined($remote_addr);

    my $server_name = delete $opts{'server_name'};
    throw Exception::Validation::BadArguments gettext("Expected server_name") if !defined($server_name);

    my @fields = @{delete $opts{'fields'}};

    my $must_be_secure = delete $opts{'must_be_secure'};

    throw Exception::Validation::BadArguments gettext("Got unknown parameters: %s", join(", ", keys(%opts))) if %opts;

    my $pass = $self->_get_addr_pass($server_name);
    my %pass_data = map(($_ => $pass->{$_}), qw(host_passport url_auth url_resign url_real_login));

    throw Exception::BlackBox::NeedAuth 'No session', %pass_data
      if (!$session_id);

    my $res = $self->call(
        method    => 'sessionid',
        sessionid => $session_id,
        (defined($sessionid2) ? (sslsessionid    => $sessionid2) : ()),
        ($user_ticket         ? (get_user_ticket => 1)           : ()),
        userip => $remote_addr,
        host   => $server_name,
        emails => 'getdefault',
        (@fields ? (dbfields => join(',', @fields)) : ()),
    );

    WARN({blackbox => $res}) if $DEBUG;

    throw Exception::BlackBox::InternalError($res->{'error'}
          && $res->{'error'}->{'content'} ? $res->{'error'}->{'content'} : gettext('Unknown error'))
      unless $res->{'status'}
          && defined($res->{'status'}->{'content'});

    if ($must_be_secure && !$res->{'auth'}->{'secure'}->{'content'} && $res->{'status'}->{'content'} ne 'NOAUTH') {
        throw Exception::BlackBox::NotSecure 'Not secure connection', %pass_data;
    }

    $res->{from} = \%pass_data;

    if ($res->{'status'}->{'content'} eq 'VALID') {
        # if social - go away and bring back YA.login
        throw Exception::BlackBox::NeedRealLogin 'Yandex login - must have. Only social login doesn\'t matter',
          %pass_data
          if ($res->{'auth'}{'social'} && !$res->{login}{content} && !$self->get_option('MayOnlySocial'));

        # All ok
        return $res;
    } elsif ($res->{'status'}{'id'} == 5) {
        throw Exception::BlackBox::NeedAuth 'Not authorized', %pass_data;
    }

    # TODO: think about JSON & POST
    if ($pass->{master_domain}) {
        if ($res->{'status'}->{'content'} eq 'NEED_RESET') {
            throw Exception::BlackBox::NeedResign 'Session wants to be fresh', %pass_data, res => $res;
        } else {
            throw Exception::BlackBox::NeedAuth 'Not authorized', %pass_data;
        }
    } else {
        if ($res->{'status'}->{'content'} eq 'NOAUTH') {
            throw Exception::BlackBox::NeedAuth 'Session too old', %pass_data;
        } else {
            throw Exception::BlackBox::NeedResign 'Session wants to be fresh or get from master domains', %pass_data,
              res     => $res,
              cookies => {Cookie_check => 'CheckCookieCheckCookie; domain=.yandex.' . $pass->{tld} . '; path=/;'};
        }
    }

    return $res;
}

=head2 userinfo

B<Параметры:> %opts

B<Возвращаемое значение:> $ со структорой ответа BlackBox

http://doc.yandex-team.ru/blackbox/reference/MethodUserInfo.xml

=cut

sub userinfo {
    my ($self, %opts) = @_;

    my @fields = @{delete $opts{fields}};
    my $multi  = delete $opts{multi};

    my @args = (
        method => 'userinfo',
        userip => ($opts{userip} || '127.0.0.1'),
        %opts,
        (@fields ? (dbfields => join(',', @fields)) : ()),
    );

    my $data = $multi ? $self->call_multi(@args) : $self->call(@args);

    return $data;
}

=head2 get_user_info

Методу нужно передать user_id, ручка сходит в паспрт в метод userinfo и вернет распарсеные данные.

    my $user_info = $app->api_blackbox->get_user_info( 607369863 );

Пример ответа:

    {
        canonical_login => 'bes-test-024',
        display_login => 'BeS.teST-024',
        email => 'BeS.teST-024@yandex.ru',
        id => 607369863,
        language => 'ru',
        lastname => 'Pupkin',
        midname => '',
        name => 'Vasily',
    }

=cut

sub get_user_info {
    my ($self, $user_id) = @_;

    throw Exception::Validation::BadArguments sprintf("user_id is not numeric: %s", $user_id)
      unless $user_id =~ /^[0-9]+\z/;

    my $userinfo = $self->userinfo(
        uid    => $user_id,
        fields => [qw(accounts.login.uid account_info.fio.uid userinfo.lang.uid)],
        emails => 'getdefault'
    );

    throw "There is no info about user_id $user_id in blackbox" unless $userinfo->{'uid'}{'content'};

    my $user = {
        id              => $userinfo->{'uid'}{'content'},
        display_login   => $userinfo->{'login'}{'content'},
        canonical_login => fix_login($userinfo->{'login'}{'content'}),
        email           => $userinfo->{'address-list'}->{'address'}->{content},
        language        => $userinfo->{'dbfield'}{'userinfo.lang.uid'},
    };
    ($user->{'lastname'}, $user->{'name'}, $user->{'midname'}) =
      (split(/\s+/, $userinfo->{'dbfield'}{'account_info.fio.uid'} || ''), '', '', '');

    return $user;
}

sub call {
    my ($self, %opts) = @_;

    my $content = unset_utf($self->SUPER::call('blackbox', ':memcached' => $BLACKBOX_CACHE_TTL, %opts));
    if ($content) {
        my $res = XML::Simple::XMLin($content, ForceContent => 1, ForceArray => ['dbfield']);
        throw Exception::BlackBox $res->{'error'}{'content'}
          if ( ($res->{'error'}{'content'} // '') ne 'OK'
            && !$res->{'login'}{'content'}
            && ($res->{'status'}{'id'} // 0) != 5);

        if ($res->{'dbfield'}) {
            $res->{'dbfield'}{$_} = $res->{'dbfield'}{$_}{'content'} foreach keys(%{$res->{'dbfield'}});
        }
        return $res;
    }
}

sub call_multi {
    my ($self, %opts) = @_;

    my $content = unset_utf($self->SUPER::call('blackbox', ':memcached' => $BLACKBOX_CACHE_TTL, %opts));
    if ($content) {
        my $res = XML::Simple::XMLin($content, ForceContent => 1, ForceArray => ['dbfield']);
        throw Exception::BlackBox $res->{'error'}{content}
          if $res->{error}{content} && $res->{error}{content} ne 'OK';

        throw Exception::BlackBox
          unless $res->{user};

        for my $uid (keys %{$res->{user}}) {
            my $ures = $res->{user}{$uid};
            if ($ures->{'dbfield'}) {
                $ures->{'dbfield'}{$_} = $ures->{'dbfield'}{$_}{'content'} for keys %{$ures->{'dbfield'}};
            }
        }
        return $res;
    }
}

=head2 _get_addr_pass

Получаем информацию о вхождении домена в главный или подчиненные домены

 tld - $, запрашиваемый top level domain
 master_domain - флаг, принадлежность к главному домену
 url_auth, url_resign, host_passport - $, соответствующие урлы для пасса и пасспорта (включают // и ?)

opts:

=cut

sub _get_addr_pass {
    my ($self, $server_name, %opts) = @_;

    my $master_domains = {
        'ru'     => 'passport.yandex.ru',
        'com'    => 'passport.yandex.com',
        'com.tr' => 'passport.yandex.com.tr',
    };

    my $tld = $server_name =~ /\.yandex\.(ru|com|com\.tr|ua|by|kz)$/i ? lc($1) : 'ru';
    my $is_master = !!$master_domains->{$tld};

    my $host_passport       = ($master_domains->{$tld} // $master_domains->{ru});
    my $url_pass_auth       = "https://$host_passport/passport?mode=auth";
    my $url_pass_resign     = "https://$host_passport/auth/update/?";
    my $url_pass_real_login = "https://$host_passport/passport?mode=postregistration&create_login=1";

    return {
        tld            => $tld,                    # our top level domain
        master_domain  => $is_master,              # is our domain is master ?
        host_passport  => $host_passport,
        url_resign     => $url_pass_resign,        # 'pass' url
        url_auth       => $url_pass_auth,          # 'passport' url
        url_real_login => $url_pass_real_login,    # 'passport' url
    };
}

sub get_uid_by_login {
    my ($self, $login) = @_;

    throw Exception::Validation::BadArguments "no login" unless $login;

    my $userinfo = $self->userinfo(
        login  => $login,
        fields => [],
    );

    throw "There is no info about login '$login' in blackbox" unless $userinfo->{'uid'}{'content'};

    return $userinfo->{'uid'}{'content'};
}

sub get_users_avatar_and_lang {
    my ($self, @uids) = @_;

    my $return = {};

    try {
        my $multi   = @uids > 1;
        my $bb_info = $self->userinfo(
            uid     => join(',', @uids),
            fields  => [qw(userinfo.lang.uid)],
            regname => "yes",
            multi   => $multi,
        );
        unless ($multi) {
            $return = {
                $uids[0] => {
                    avatar => $bb_info->{display_name}{avatar}{default}{content} // '0/0-0',
                    lang   => $bb_info->{dbfield}{'userinfo.lang.uid'}           // 'ru',
                },
            };
        } else {
            for my $uid (@uids) {
                $return->{$uid} = {
                    avatar => $bb_info->{user}{$uid}{display_name}{avatar}{default}{content} // '0/0-0',
                    lang   => $bb_info->{user}{$uid}{dbfield}{'userinfo.lang.uid'}           // 'ru',
                };
            }
        }
    }
    catch Exception::BlackBox;
    return $return;
}

sub check_tvm_service {
    my ($self) = @_;

    (my $bbc, $BLACKBOX_CACHE_TTL) = ($BLACKBOX_CACHE_TTL, 0);
    my $result;
    try {
        $self->get_uid_by_login('---');
    }
    catch {
        my ($e) = @_;
        $result = TRUE if $e->message =~ /failed to check service ticket/;
    };
    $BLACKBOX_CACHE_TTL = $bbc;

    return $result;
}

TRUE;
