package QBit::Application::Model::API::Yandex::Cabinet;

use qbit;

use base qw(QBit::Application::Model::API::HTTP);

use QBit::Validator;
use Utils::Logger qw(ERROR);

use Exception::Cabinet;

sub accessor {'api_cabinet'}

=encoding UTF-8

=head1 DESCRIPTION

Perl модель для работы с API кабинета разработчика.

https://wiki.yandex-team.ru/tech/developer/api

=head1 METHODS

=head2 check_key

    my $data = $app->api_cabinet->check_key(
        user_ip => '213.180.219.192',
        key => 'xxxxxxxx-xxxx-xxxx-xxxx-xxxxxxxxxxxx',
    );

После этого в $data будет содержаться:

    {
        key_info => {
            dt     => "2015-08-31T13:29:55.611000",
            hidden => JSON::XS::Boolean  {
                public methods (0)
                private methods (1) : __ANON__
                internals: 0
            },
            id     => "xxxxxxxx-xxxx-xxxx-xxxx-xxxxxxxxxxxx",
            name   => "Key #1",
            user   => {
                email => "yndx.bessarabov@yandex.ru",
                login => "yndx-bessarabov",
                name  => "Pupkin Vasily",
                uid   => 155209804,
                roles => ['user'],
                balance_client_id => undef,
            },
        },
        result   => "OK",
    }

В том случае если такого ключа нет или он не подключен к сервису, то
будет выброшено исключене.

https://wiki.yandex-team.ru/tech/developer/api#checkkey

=cut

sub check_key {
    my ($self, %opts) = @_;

    QBit::Validator->new(
        data     => \%opts,
        template => {
            type   => 'hash',
            fields => {
                user_ip => {type => 'scalar',},
                key     => {type => 'scalar',},
            }
        },
        throw => TRUE,
    );

    my $data;

    try {
        $data = $self->call('check_key', %opts, service_token => $self->get_option('service_token'));
    }
    catch {
        my ($exception) = @_;

        unless ($exception->isa('Exception::API::HTTP') && $exception->{'response'}->code == 404) {
            ERROR $exception->message;
        }

        throw Exception::Cabinet gettext('Key is not valid');
    };

    return from_json($data);
}

sub check_tvm_service {
    my ($self) = @_;

    my $result;
    try {
        $self->call(
            'check_key',
            user_ip       => '127.0.0.1',
            key           => 'xxxxxxxx-xxxx-xxxx-xxxx-xxxxxxxxxxxx',
            service_token => $self->get_option('service_token')
        );
    }
    catch {
        my ($e) = @_;
        if ($e->isa('Exception::API::HTTP') && $e->{response}->code == 401) {
            $result = TRUE;
        }
    };

    return $result;
}

TRUE;
