package QBit::Application::Model::API::Yandex::HTTPBK;

use qbit;

use base qw(QBit::Application::Model::API::HTTP);
use Utils::TSV;
use List::Compare qw();

use QBit::Validator;
use Utils::Logger qw(ERROR INFOF WARN);
use Utils::JSON qw(apply_json_patch);
use Yandex::Utils;

use Exception::BK;
use Exception::Validation::BadArguments;
use Exception::Validator::Fields;

use PiConstants qw(
  $DSP_MOBILE_TYPE_ID
  $DSP_FORMAT_TO_TYPE_MAP
  );

__PACKAGE__->model_accessors(api_selfservice => 'Application::Model::API::Yandex::SelfService',);

sub accessor {'api_http_bk'}

sub need_log_all {FALSE}

sub init {
    my ($self) = @_;

    $self->SUPER::init();

    throw Exception::BK 'At least one of allow_dsp_via_http or allow_dsp_via_logbroker should be set'
      unless $self->get_option('allow_dsp_via_http') || $self->get_option('allow_dsp_via_logbroker');

    return TRUE;
}

sub apply_dsp_patch {
    my ($self, $data, $patch_json) = @_;
    return apply_json_patch($data, $patch_json);
}

sub _prepare_dsp_object_for_logbroker {
    my ($self, $data) = @_;

    my %lb_data = hash_transform(
        $data,
        [],
        {
            dsp_id           => 'DSPID',
            id               => 'DSPID',
            tag              => 'DSPTag',
            dsp_types        => 'DSPType',
            show_probability => 'ShowProbability',
            short_caption    => 'Title',
            url              => 'Url',
        }
    );

    $lb_data{Options} = {
        map {
            my $opt_name = $_;
            $opt_name =~ s/_/-/g;
            $opt_name => TRUE
          } grep {
            $data->{$_}
          } qw(disabled postmoderated skipnoud unmoderated_rtb_auction use_pnocsy is_ssp_allowed works_on_all_platforms)
    };
    $self->apply_dsp_patch(\%lb_data, $data->{patch}) if $data->{patch};
    $lb_data{PublicTitle} = $data->{display_name} if $lb_data{Options}{'show-public-title'} && $data->{display_name};

    $lb_data{Options} = join(',', sort grep {$lb_data{Options}{$_}} keys %{$lb_data{Options}});
    $lb_data{$_} += 0 foreach (qw(DSPID DSPType ShowProbability));

    return {
        DSPID    => 0 + delete($lb_data{DSPID}),
        Data     => to_json(\%lb_data, canonical => TRUE),
        UnixTime => time,
    };
}

sub dsp_lb {
    my ($self, $opts) = @_;

    return FALSE unless $self->get_option('allow_dsp_via_logbroker');

    QBit::Validator->new(
        data     => $opts,
        template => {
            type   => 'hash',
            fields => {
                disabled      => {type    => 'boolean', optional => TRUE},
                dsp_types     => {type    => 'array',   optional => TRUE},
                formats       => {type    => 'array',   optional => TRUE},
                id            => {type    => 'int_un'},
                postmoderated => {type    => 'boolean', optional => TRUE},
                skipnoud      => {type    => 'boolean', optional => TRUE},
                short_caption => {len_max => 32},
                show_probability => {type    => 'int_un', max => 100, optional => TRUE},
                tag              => {len_max => 32},
                unmoderated_rtb_auction => {type    => 'boolean', optional => TRUE},
                url                     => {len_max => 255},
                use_pnocsy              => {type    => 'boolean'},
                display_name            => {len_max => 32,        optional => TRUE,},
                patch                   => {type    => 'scalar',  optional => TRUE,},
                is_ssp_allowed          => {type    => 'boolean', optional => TRUE,},
                works_on_all_platforms  => {type    => 'boolean', optional => TRUE,},
            },
        },
        throw => TRUE,
    );

    if (exists($opts->{dsp_types})) {
        my $bk_type;
        $bk_type |= 2**$_ foreach @{$opts->{dsp_types}};

        $opts->{dsp_types} = $bk_type;
    } else {
        $opts->{dsp_types} = 0;
    }

    if ($opts->{dsp_types} & (2**$DSP_MOBILE_TYPE_ID)) {
        for my $format (@{$opts->{formats} // []}) {
            $opts->{dsp_types} |= 2**$DSP_FORMAT_TO_TYPE_MAP->{$format};
        }
    }

    my $dsp_topic = $self->get_option('logbroker_topic');
    throw Exception::BK 'logbroker_topic should be set'
      unless $dsp_topic;

    $self->api_selfservice->logbroker(
        data        => [$self->_prepare_dsp_object_for_logbroker($opts)],
        topic       => $dsp_topic,
        skip_log_ok => !$self->need_log_all,
    );

    return TRUE;
}

=head2 add_dsp

=cut

sub add_dsp {
    my ($self, %opts) = @_;

    return FALSE unless $self->get_option('allow_dsp_via_http');

    QBit::Validator->new(
        data     => \%opts,
        template => {
            type   => 'hash',
            fields => {
                id               => {type    => 'int_un'},
                tag              => {len_max => 32},
                short_caption    => {len_max => 32},
                use_pnocsy       => {type    => 'boolean'},
                url              => {len_max => 255},
                disabled         => {type    => 'boolean', optional => TRUE},
                dsp_types        => {type    => 'array', optional => TRUE},
                skipnoud         => {in      => [qw(true false)], optional => TRUE},
                show_probability => {type    => 'int_un', max => 100, optional => TRUE},
                postmoderated           => {type    => 'boolean', optional => TRUE},
                unmoderated_rtb_auction => {type    => 'boolean', optional => TRUE},
                test_url                => {len_max => 255,       optional => TRUE},
                is_ssp_allowed          => {type    => 'boolean', optional => TRUE,},
                works_on_all_platforms  => {type    => 'boolean', optional => TRUE,},
            },
        },
        throw => TRUE,
    );

    if (exists($opts{'dsp_types'})) {
        my $bk_type;
        $bk_type |= 2**$_ foreach @{$opts{'dsp_types'}};

        $opts{'dsp_types'} = $bk_type;
    }

    foreach (qw(disabled postmoderated unmoderated_rtb_auction)) {
        $opts{$_} = $opts{$_} ? 1 : 0 if exists($opts{$_});
    }

    my $data = $self->call(
        'export/dsp-api.cgi',
        func => 'add_dsp',
        hash_transform(
            \%opts,
            ['url', 'disabled', 'skipnoud', 'show_probability', 'postmoderated'],
            {
                id                      => 'dsp_id',
                tag                     => 'dsp_tag',
                test_url                => 'testurl',
                dsp_types               => 'dsp_type',
                short_caption           => 'caption',
                use_pnocsy              => 'use-pnocsy',
                unmoderated_rtb_auction => 'unmoderated-rtb-auction'
            }
        ),
        'keep_response' => 1
    );

    my $result;
    try {
        $result = from_json($data);
    }
    catch {
        throw Exception::BK $self->last_response, 'export/dsp-api.cgi - got invalid JSON', $@->message,
          sentry => {fingerprint => ['HTTPBK', 'Invalid JSON']};
    };

    throw Exception::BK $self->last_response, $result->{'error'}, undef,
      sentry => {fingerprint => ['HTTPBK', 'Error', $result->{'code'}]}
      if $result->{'code'} != 0;

    return $result;
}

sub add_or_get_if_exists_dsp {
    my ($self, %opts) = @_;

    return FALSE unless $self->get_option('allow_dsp_via_http');

    my $result;
    try {
        $result = $self->add_dsp(%opts);
    }
    catch Exception::BK with {
        my ($e) = @_;
        try {
            my $info = $self->get_dsp_info($opts{'id'});
            $result = {data_key => $info->[0]->{DataKey}};
        }
        catch {
            # выкидываем исходную ошибку, которая возникла при добавлении
            throw $e;
        }
    };

    return $result;
}

=head1 all_pages_hits_stat_day

L<https://wiki.yandex-team.ru/users/mllnr/Подневнаястатистикапопейджам>

    ->all_pages_hits_stat_day('2014-07-16')

=cut

sub all_pages_hits_stat_day {
    my ($self, $date) = @_;

    my $bs_date = _convert_to($date, 'bs');

    my $data = $self->call('export/all-pages-hits-stat-day.cgi', POSTDATA => $bs_date,);

    my $result = from_json($data);

    throw Exception::BK $result->{'ErrorMessage'}, undef, undef,
      sentry => {fingerprint => ['HTTPBK', 'Error', $result->{'code'}]}
      if ref($result) eq 'HASH' && $result->{'Error'} == 1;

    return [
        map {
            {%$_, UpdateTime => _convert_to($_->{'UpdateTime'}, 'db')}
          } grep {
            $_->{'UpdateTime'} eq $bs_date
          } @{$result}
    ];
}

=head2 edit_dsp

=cut

sub edit_dsp {
    my ($self, $dsp_id, %opts) = @_;

    return FALSE unless $self->get_option('allow_dsp_via_http');

    throw Exception::Validation::BadArguments(gettext('Expected that DSP ID is defined')) unless defined($dsp_id);

    if (exists($opts{'dsp_types'})) {
        my $bk_type;
        $bk_type |= 2**$_ foreach @{$opts{'dsp_types'}};

        $opts{'dsp_types'} = $bk_type;
    }

    my %params = hash_transform(
        \%opts,
        [qw(dsp_id dsp_tag url disabled skipnoud show_probability postmoderated)],
        {
            test_url                  => 'testurl',
            dsp_types                 => 'dsp_type',
            'short_caption'           => 'caption',
            'unmoderated_rtb_auction' => 'unmoderated-rtb-auction'
        }
    );

    foreach (qw(skipnoud disabled postmoderated)) {
        $params{$_} = $params{$_} ? 1 : 0 if exists($params{$_});
    }

    my $data = $self->call(
        'export/dsp-api.cgi',
        func   => 'edit_dsp',
        dsp_id => $dsp_id,
        %params,
        'keep_response' => 1
    );

    my $result;
    try {
        $result = from_json($data);
    }
    catch {
        throw Exception::BK $self->last_response, 'export/dsp-api.cgi - got invalid JSON', $@->message,
          sentry => {fingerprint => ['HTTPBK', 'Invalid JSON']};
    };

    throw Exception::BK $self->last_response, $result->{'error'}, undef,
      sentry => {fingerprint => ['HTTPBK', 'Error', $result->{'code'}]}
      if $result->{'code'} != 0;

    return $result;
}

sub get_apfraudstat {
    my ($self, $date) = @_;

    my $bs_date = _convert_to($date, 'bs');

    my $data = $self->call('export/export_apfraudstat.cgi', date => $bs_date,);

    return unless defined($data);

    my @data = ();
    my @errors;

    push(
        @data,
        map {
            {
                PageID      => $_->[0],
                UpdateTime  => _convert_to($_->[1], 'db'),
                Shows       => $_->[2],
                FraudShows  => $_->[3],
                Clicks      => $_->[4],
                FraudClicks => $_->[5],
                Cost        => $_->[6],
                FraudCost   => $_->[7],
                AdShows     => $_->[8],
                AdClicks    => $_->[9],
            }
          }
          grep {
            @$_ == 10 or push(@errors, $_), FALSE
          }
          map {
            [split("\t", $_)]
          }
          grep {
            $_
          }
          split("\n", $data)
        );

    WARN("Invalid rows from 'export/export_apfraudstat.cgi($bs_date)': " . to_json(\@errors))
      if @errors;
    return \@data;
}

sub get_big_search_page_ids {
    my ($self) = @_;

    my $data = $self->call('export/export-tagstat-pages.cgi');

    return [split(/\s+/, $data)];
}

=head2 get_direct_rtb_stat

=cut

sub get_direct_rtb_stat {
    my ($self, %opts) = @_;

    foreach (qw(date_from date_to)) {
        throw sprintf('Expected date format YYYY-mm-dd. got "%s"', $opts{$_} // 'undef')
          unless check_date($opts{$_}, iformat => 'db');
        $opts{$_} =~ s/-//g;
    }

    throw sprintf('Expected ad_type. got "%s"', $opts{ad_type} // 'undef')
      unless $opts{ad_type};

    my $data = $self->call('export/export_direct_rtb_stat.cgi', %opts);

    return $data if ref($data);

    $data =~ s/^#//;

    throw Exception::BK 'export/export_direct_rtb_stat.cgi - no "#END" marker received', undef, undef,
      sentry => {fingerprint => ['HTTPBK', 'no "end" marker received']}
      unless $data =~ s/#END//;

    return parse_tsv($data);
}

=head2 get_dsp_info

=cut

sub get_dsp_info {
    my ($self, $dsp_id) = @_;

    my %params = (func => 'info');
    %params = (
        func   => 'get_dsp',
        dsp_id => $dsp_id
    ) if $dsp_id;

    my $data = $self->call('export/dsp-api.cgi', %params, 'keep_response' => 1);

    my $result;
    try {
        $result = from_json($data);
    }
    catch {
        throw Exception::BK $self->last_response, 'export/dsp-api.cgi - got invalid JSON', $@->message,
          sentry => {fingerprint => ['HTTPBK', 'Invalid JSON']};
    };

    throw Exception::BK $self->last_response, $result->{'error'}, undef,
      sentry => {fingerprint => ['HTTPBK', 'Error', $result->{'code'}]}
      if $result->{'code'} != 0;

    return $result->{'data'};
}

=head2 get_dsp_page_stat

=cut

sub get_dsp_page_stat {
    my ($self, %opts) = @_;

    foreach (qw(starttime stoptime)) {
        throw sprintf('Expected date format YYYY-mm-dd. got "%s"', $opts{$_} // 'undef')
          unless check_date($opts{$_}, iformat => 'db');
        $opts{$_} =~ s/-//g;
    }

    $opts{'max-position-count'} = 1;
    my $data = $self->call('export/export-dsppagestat.cgi', %opts);

    return $data if ref($data);

    my @result = ();

    while ($data =~ /([^\r\n]+)?\r?\n/imsg) {
        my @data = split(/\t/, $1);
        push(
            @result,
            {
                Date                 => $data[0],
                PageID               => $data[1],
                ImpID                => $data[2],
                DSPID                => $data[3],
                WinHits              => $data[4],
                WinPrice             => $data[5],
                WinPartnerPrice      => $data[6],
                Shows                => $data[7],
                AllHits              => $data[8],
                WinMaxPositionsCount => $data[9],
                AllRealPrice         => $data[10],
                (
                    $opts{'include-bad-stat'}
                    ? (
                        BadWinHits              => $data[11],
                        BadWinPrice             => $data[12],
                        BadWinPartnerPrice      => $data[13],
                        BadShows                => $data[14],
                        BadAllHits              => $data[15],
                        BadWinMaxPositionsCount => $data[16],
                        BadAllRealPrice         => $data[17]
                      )
                    : ()
                )
            }
        );
    }

    return \@result;
}

sub get_dsp_statistics_responses {
    my ($self, $dsp_id, $from_date, $to_date) = @_;

    my %opts = ();

    $opts{'fromdate'} = $from_date if defined($from_date);
    $opts{'todate'}   = $to_date   if defined($to_date);

    foreach (keys(%opts)) {
        throw sprintf('Expected date format YYYY-mm-dd. got "%s"', $opts{$_} // 'undef')
          unless check_date($opts{$_}, iformat => 'db');
        $opts{$_} =~ s/-//g;
    }

    $opts{'dspid'} = $dsp_id if defined($dsp_id);

    my $data = $self->call('export/export-dsppageerror.cgi', %opts);

    $data =~ m/^([^\r\n]+)?\r?\n(.*)?\r?\n?$/ms;

    if ($2) {
        $data = $2;
    } elsif ($1 && $1 =~ m/^DSPID/) {
        return [];
    } else {
        throw Exception::BK($1);
    }

    my @result = ();

    while ($data =~ /([^\r\n]+)?\r?\n/imsg) {
        my @data = split(/\t/, $1);
        $data[2] =~ s/([0-9]{4})([0-9]{2})([0-9]{2})/$1-$2-$3/;
        push(
            @result,
            {
                DSPID      => $data[0],
                PageID     => $data[1],
                UpdateDate => $data[2],
                ErrorType  => $data[3],
                ErrorsSum  => $data[4]
            }
        );
    }

    return \@result;
}

=head2 get_page_block_shows

    my $data = $app->api_http_bk->get_page_block_shows('2017-12-14');

Возвращает arrayref. Работает с ручкой БК export_pageblockshows.cgi
Дока на ручку - L<https://wiki.yandex-team.ru/partner/w/exportpageblockshows/>

    [
        {
            Flag => 0,
            PageAdShow => 1,
            PageID => 250073,
            PlaceID => 542,
            UpdateTime => '2017-12-14',
        },
        {
            Flag => 0,
            PageAdShow => 15,
            PageID => 258894,
            PlaceID => 542,
            UpdateTime => '2017-12-14',
        },
    ]

=cut

sub get_page_block_shows {
    my ($self, $date) = @_;

    my $bs_date = _convert_to($date, 'bs');

    my $method = 'export/export_pageblockshows.cgi';

    my $data = $self->call($method, date => $bs_date);

    throw Exception::BK "$method - no '#End' marker", undef, undef,
      sentry => {fingerprint => ['HTTPBK', 'no "end" marker received']}
      unless $data =~ s/#End\s*\z//;

    my @data = ();

    push(
        @data,
        map {
            {
                UpdateTime => _convert_to($_->[0], 'db'),
                PageID     => $_->[1],
                PlaceID    => $_->[2],
                Flag       => $_->[3],
                PageAdShow => $_->[4],
            }
          }
          map {
            [split("\t", $_)]
          }
          grep {
            $_
          }
          split("\n", $data)
        );

    INFOF "method %s number of line: %s", $method, scalar @data;

    return \@data;
}

=head1 get_partner_pages

    Cм. подробнее http://wiki.yandex-team.ru/partner/w/get-partner-pages

=cut

sub get_partner_pages {
    my ($self, $page_ids) = @_;

    my $data = $self->call('export/get-partner-pages.cgi', ($page_ids ? (pages => join(',', @$page_ids)) : ()));

    throw Exception::BK 'export/get-partner-pages.cgi - no "end" marker received', undef, undef,
      sentry => {fingerprint => ['HTTPBK', 'no "end" marker received']}
      unless $data =~ s/# end//;

    $data =~ s/# //;

    my $result = parse_tsv($data);

    return $result;
}

sub get_publisher_stat {
    my ($self, %args) = @_;

    my %opts = ();

    foreach (qw(startdate stopdate pageid)) {
        $opts{$_} = delete $args{$_} if exists($args{$_});
    }

    foreach (qw(startdate stopdate)) {
        throw sprintf('Expected date format YYYY-mm-dd. got "%s"', $opts{$_} // 'undef')
          unless check_date($opts{$_}, iformat => 'db');
        $opts{$_} =~ s/-//g;
    }

    $opts{'func'}       = 'get_publisher_stat';
    $opts{'end_marker'} = 1;                      # This options is for endpoint

    my $data = $self->call(
        'export/video_content.cgi',
        data                   => to_json(\%opts),
        ':expected_end_marker' => "#END\n",          # This option is for our internal validation/retry mechanism
        %args
    );

    # ??? зачем возвращать пустую строку тем более что она игнорится
    return $data if exists $args{':content_file'};

    return parse_tsv(
        $data,
        headers       => $self->get_publisher_stat_fields(),
        string_escape => TRUE,
    );
}

sub get_publisher_stat_fields {
    [
        qw(
          UpdateTime
          PageID
          DSPID
          PublisherID
          ContentID
          PublisherName
          ContentName
          Shows
          Wins
          Hits
          Price
          PartnerPrice
          CLID
          )
    ];
}

sub get_serp_tag_hits {
    my ($self, $starttime, $stoptime) = @_;

    my $bs_start = _convert_to($starttime, 'bs');
    my $bs_end   = _convert_to($stoptime,  'bs');

    my $data = $self->call(
        'export/export_serptaghits.cgi',
        starttime => $bs_start,
        stoptime  => $bs_end,
    );

    throw Exception::BK sprintf(
        'export/export_serptaghits.cgi -no "end" marker received. Starttime: "%s", endtime: "%s".',
        $bs_start, $bs_end
      ),
      undef, undef, sentry => {fingerprint => ['HTTPBK', 'no "end" marker received']}
      if $data !~ s/#\s*end\r?\n//si;

    my @result = ();

    while ($data =~ /([^\r\n]+)?\r?\n/imsg) {
        my @data = split(/\t/, $1);
        push(
            @result,
            {
                UpdateTime => _convert_to($data[0], 'db'),
                PageID     => $data[1],
                CLID       => $data[2],
                Shows      => $data[3]
            }
        );
    }

    return \@result;
}

sub import_cookie_match_settings {
    my ($self, %opts) = @_;

    my $data = [
        {
            CookieMatchTag => $opts{'tag'}      // '',
            DataKey        => $opts{'data_key'} // '',
            InitURL        => $opts{'init_url'} // '',
            Options        => {
                jsredir                => $opts{'js_redir'}             // '',
                skipdatakey            => $opts{'skip_data_key'}        // '',
                'trace-back-reference' => $opts{'trace_back_reference'} // ''
            }
        }
    ];

    my $json_result = $self->call(
        'export/import-cookie-match-settings.cgi',
        ':post'         => TRUE,
        ':content_type' => 'application/json',
        ':content'      => to_json($data),
        'keep_response' => 1
    );

    my $result;
    try {
        $result = from_json($json_result);
    }
    catch {
        throw Exception::BK $self->last_response, 'export/import-cookie-match-settings.cgi - got invalid JSON',
          $@->message, sentry => {fingerprint => ['HTTPBK', 'Invalid JSON']};
    };

    throw Exception::BK $self->last_response, $result->{'ErrorMessage'} if ref($result) eq 'HASH';
    throw Exception::BK $self->last_response, $result->[0]{'ErrorMessage'} if $result->[0]{'Error'};

    return $result->[0];
}

sub start_dsp {
    my ($self, $dsp_id) = @_;

    return FALSE unless $self->get_option('allow_dsp_via_http');

    throw Exception::Validation::BadArguments(gettext('Expected that DSP ID is defined')) unless defined($dsp_id);

    my $data = $self->call(
        'export/dsp-api.cgi',
        func            => 'start_dsp',
        dsp_id          => $dsp_id,
        'keep_response' => 1
    );

    my $result;
    try {
        $result = from_json($data);
    }
    catch {
        throw Exception::BK $self->last_response, 'export/dsp-api.cgi - got invalid JSON', $@->message,
          sentry => {fingerprint => ['HTTPBK', 'Invalid JSON']};
    };

    throw Exception::BK $self->last_response, $result->{'error'}, undef,
      sentry => {fingerprint => ['HTTPBK', 'Error', $result->{'code'}]}
      if $result->{'code'} != 0;

    return $result;
}

sub stop_dsp {
    my ($self, $dsp_id) = @_;

    return FALSE unless $self->get_option('allow_dsp_via_http');

    throw Exception::Validation::BadArguments(gettext('Expected that DSP ID is defined')) unless defined($dsp_id);

    my $data = $self->call(
        'export/dsp-api.cgi',
        func            => 'stop_dsp',
        dsp_id          => $dsp_id,
        'keep_response' => 1
    );

    my $result;
    try {
        $result = from_json($data);
    }
    catch {
        throw Exception::BK $self->last_response, 'export/dsp-api.cgi - got invalid JSON', $@->message,
          sentry => {fingerprint => ['HTTPBK', 'Invalid JSON']};
    };

    throw Exception::BK $self->last_response, $result->{'error'}, undef,
      sentry => {fingerprint => ['HTTPBK', 'Error', $result->{'code'}]}
      if $result->{'code'} != 0;

    return $result;
}

sub _convert_to {
    my ($date, $format) = @_;

    local $QBit::Date::TR_HS{'bs'} = {
        '>' => sub {sprintf('%04d%02d%02d000000', @{$_[0]}[0 .. 2])},
        '<' => sub {shift =~ /(\d{4})(\d{2})(\d{2})000000/ ? [$1, $2, $3, 0, 0, 0] : [0, 0, 0, 0, 0, 0]},
    };

    return $format eq 'bs' ? trdate('db', 'bs', $date) : trdate('bs', 'db', $date),;
}

TRUE;
