package QBit::Application::Model::API::Yandex::HTTPGeobase;

use qbit;
use base qw(QBit::Application::Model::API::HTTP);

use Exception;
use Exception::IncorrectParams;
use Exception::API::HTTP;
use Exception::GeoBase::BadRegion;
use Data::Validate::IP qw(is_ipv4 is_ipv6);
use PiConstants qw( $EARTH_REGION_ID );

sub accessor {'api_geobase'}

sub get_structure_model_accessors {
    my ($class) = @_;

    return {memcached => 'QBit::Application::Model::Memcached'};
}

=head2 get_region_by_ip

B<Arguments:>

=over

=item B<$ip> - строка, понимает IPv4 и IPv6.

=back

B<Return value:> указатель на хеш. Хеш как в методе L</to_hash>.

=cut

sub get_region_by_ip {
    my ($self, $ip) = @_;

    # Хак для того чтобы работали беты, собранные в креаторе.
    # Из-за настройки докера, беты видят что клиенты подключатся с ip адресов 172.17.0.*
    # А геобаза отдает ошибку при попытке запросить данные для этих ip адресов:
    #
    # $ curl 'http://geobase-test.qloud.yandex.ru/v1/region_by_ip?ip=172.17.0.52' ; echo ''
    # {"error":"(std::runtime_error) bad region idx 4294967295; total - 77982","code":400}

    if ($self->_is_docker_ip($ip) && ($self->get_option('stage', 'unknown') eq 'dev')) {
        return {
            capital_id           => 0,
            en_name              => 'Yandex',
            geo_parent_id        => 0,
            id                   => 9999,
            is_main              => 0,
            iso_name             => 'Yandex',
            latitiude_size       => '0.013937',
            latitude             => '55.734046',
            latitude_size        => '0.013937',
            longitude            => '37.588628',
            longitude_size       => '0.005388',
            name                 => "Яндекс (Москва)",
            native_name          => '',
            official_languages   => 'ru',
            parent_id            => 213,
            phone_code           => '',
            population           => 0,
            position             => 0,
            services             => [],
            short_en_name        => 'Yandex',
            synonyms             => '',
            type                 => -1,
            tzname               => 'Europe/Moscow',
            widespread_languages => 'ru',
            zip_code             => '',
            zoom                 => 10,
        };
    }

    my $reg;
    try {
        $reg = from_json($self->call('region_by_ip', ip => $ip));
    }
    catch Exception::API::HTTP with {
        my ($exception) = @_;
        my $response = $exception->{'response'};
        if ($response->code == 400 && $response->content) {
            my $result = from_json($response->decoded_content());
            if ($result->{'error'} && $result->{'error'} =~ m/bad region idx/) {
                throw Exception::GeoBase::BadRegion $result->{'error'};
            }
        }

        throw $exception;
    };

    return ($reg && $reg->{id}) ? $reg : undef;
}

=head2 get_region_parents_by_id

B<Arguments:>

=over

=item B<$id> - число, ID региона

=back

B<Return value:> указатель на массив чисел, ID родительских регионов.

=cut

sub get_region_parents_by_id {
    my ($self, $id) = @_;

    my $res;
    try {
        $res = from_json($self->call('parents', id => ($id || $EARTH_REGION_ID)));
    }
    catch Exception::API::HTTP with {
        my ($exception) = @_;
        my $response = $exception->{'response'};
        if ($response->code == 400 && $response->content) {
            my $result = from_json($response->decoded_content());
            if ($result->{'error'} && $result->{'error'} =~ m/unknown id/) {
                throw Exception::GeoBase::BadRegion $result->{'error'};
            }
        }

        throw $exception;
    };

    return $res;
}

=head2 get_region_parents_by_ip

B<Arguments:>

=over

=item B<$ip> - строка, понимает IPv4 и IPv6.

=back

B<Return value:> указатель на массив чисел, ID родительских регионов.

=cut

sub get_region_parents_by_ip {
    my ($self, $ip) = @_;

    my $region = $self->get_region_by_ip($ip);

    return $self->get_region_parents_by_id($region->{id});
}

sub get_cached_region_parents {
    my ($self, $id_or_ip, $value) = @_;

    throw Exception::IncorrectParams "$id_or_ip is not id or ip" unless $id_or_ip eq 'id' || $id_or_ip eq 'ip';

    my $call_func     = 'get_region_parents_by_' . $id_or_ip;
    my $cached_prefix = 'region_parents_by_' . $id_or_ip;

    my $regions = $self->memcached->get($cached_prefix => $value);

    unless ($regions) {
        try {
            $regions = $self->$call_func($value);
        }
        finally {
            my ($e) = @_;
            my $ex_name = blessed($e);
            if (!defined($ex_name) || $ex_name eq 'Exception::GeoBase::BadRegion') {
                $regions //= [];
                $self->memcached->set($cached_prefix => $value, $regions, 30 * 60);
            }
        };
    }

    return $regions;
}

sub _is_docker_ip {
    my ($self, $ip) = @_;

    $ip //= 'undef';

    return $ip =~ /^(?:\:\:ffff:)?172\.17\.0\./;
}

TRUE;
