package QBit::Application::Model::API::Yandex::HTTPMAAS;

use qbit;

use base qw(QBit::Application::Model::API::HTTP);

use Utils::Logger qw/INFO/;

use Utils::MonitoringUtils qw/send_to_graphite/;

use Exception::MAAS;
use Exception::MAAS::NotFound;

sub accessor {'api_http_maas'}

__PACKAGE__->model_accessors(memcached => 'QBit::Application::Model::Memcached');

=head1 Описание

Врапер над методами MAAS

Документация: https://wiki.yandex-team.ru/jandekspoisk/externalsource/MobileApplicationsAsAService/

=cut

=head2 find_app

Получаем данные о приложении.

Фильтруем лишние, трансформируем нужные.

Выход:
 {
       'source_store_id'    => 2,
       'source_title'       => 'Realmadrid App',
       'app_adult'          => '17+',
       'app_id'             => 'id327054744',
       'source_url'         => 'https://itunes.apple.com/ru/app/id327054744',
       'app_os'             => '8.0',
       'app_rating_count'   => '1',
       'app_name'           => 'Realmadrid App',
       'source_headline'    => 'Realmadrid App. http://www.realmadrid.com. com.realmadrid.mymadrid. Real Madrid C.F. Realmadrid App. Bugfixing, review issues with iPhone 4S and iPhone 5. Stay closer to the better club of the...',
       'app_rating'         => '1',
       'app_website'        => 'http://www.realmadrid.com'
 }

Сырой ответ (продакшен)

 curl 'http://extdata-mobile-apps.search.yandex.net/process/app?country=ru&content_type=app&store=itunes&app_id=1057245420'
 curl 'http://extdata-mobile-apps.search.yandex.net/process/app?country=ru&content_type=app&store=gplay&app_id=com.craftgames.plntcrft'

=cut

sub find_app {
    my ($self, $app_id, $os_id) = @_;

    my $data;

    our $cache_balance //= 0;

    my $store_by_os_id = {
        1 => 'gplay',
        2 => 'itunes',
    };

    my %data_params = (
        'app_id'       => $app_id,
        'content_type' => 'app',
        'country'      => 'ru',
        'store'        => $store_by_os_id->{$os_id},
    );

    $data = $self->memcached->get('maas', $app_id);
    unless ($data) {
        $cache_balance--;

        try {
            $data = $self->_get_app(%data_params);
        }
        catch {
            INFO $@->message;
        };

        send_to_graphite(value => 1, path => "API.HTTPMAAS.access_live") if $data && ref($data) eq 'HASH';

        $self->memcached->set('maas', $app_id, $data || {}, 60 * 60 * 24 * 3);
    } else {
        $cache_balance++;
    }

    send_to_graphite(value => $cache_balance, path => "API.HTTPMAAS.balance_memcached");

    # transform raw data

    my $data_doc;
    {
        no warnings FATAL => qw(uninitialized);
        $data_doc = $data->{response}->{results}->[0]->{groups}->[0]->{documents}->[0];
        throw Exception::MAAS::NotFound gettext('No data for app "%s"', $app_id)
          unless $data_doc && ref($data_doc) eq 'HASH';
    }

    my $result = {};

    $result->{source_store_id} = $data_params{store} eq 'itunes' ? 2 : 1;
    $result->{source_url} = $data_doc->{url};
    $result->{source_url} =~ s/(?<=itunes\.apple\.com\/ru\/app\/)(?=\d+)/id/i;
    $result->{source_title}     = $data_doc->{title}->[0]->{text};
    $result->{source_headline}  = $data_doc->{headline}->[0]->{text};
    $result->{app_website}      = $data_doc->{properties}->{website};
    $result->{app_id}           = $data_doc->{properties}->{app_id} || $data_doc->{properties}->{bundle};
    $result->{app_name}         = $data_doc->{properties}->{title_eng} || $data_doc->{properties}->{name};
    $result->{app_rating}       = $data_doc->{properties}->{rating};
    $result->{app_rating_count} = $data_doc->{properties}->{rating_count};
    $result->{app_adult}        = $data_doc->{properties}->{adult};
    $result->{app_os}           = $data_doc->{properties}->{os_version};
    $result->{app_os} =~ s/[^\d\.]+//g;

    foreach my $v (values(%$result)) {
        if ($v) {
            utf8::decode($v);
            utf8::upgrade($v);
        }
    }

    return $result;
}

=head2 _get_app

Врапер над методом process/app

Получаем данные о приложении

Параметры:

 app_id := STRING ("882268463")

 content_type := STRING ("app")

 country := STRING ("ru", "us")

 store := "itunes" | "gplay"

=cut

sub _get_app {
    my ($self, %opts) = @_;

    my $data = $self->call('process/app', %opts);
    # "call" also throws "404 Not found"

    my $result = from_json($data);

    throw Exception::MAAS $result->{'error'} if $result->{'error'};

    return $result;
}

sub get_apps_from_yt {
    my ($self, $missing_app_list) = @_;

    local *_select_then_parse = sub($$$$) {
        my ($api_yt, $replica_list, $path, $query) = @_;
        my $data_str = $api_yt->select_from_any_replica(
            replicas => $replica_list,
            path     => $path,
            yql      => $query,
            headers  => {'Accept' => 'application/json'},
            params   => {
                ':timeout'       => 300,
                ':attempts'      => 1,
                ':delay'         => 0,
                ':timeout_retry' => 0,
                input_row_limit  => 100_000_000_000,
            },
        );
        # somewhere there is a double encode, but I have not found it
        # it works this way
        utf8::decode($data_str);
        utf8::decode($data_str);
        return map {from_json($_)} split("\n", $data_str);
    };

    my @data_parsed = ();

    my %os_dict       = @Application::Model::Product::SSP::ImpressionLog::OS_DICT;
    my $OS_ID_ANDROID = $os_dict{'android'};
    my $OS_ID_IOS     = $os_dict{'ios'};

    my ($gplay_bundle_id_list, $itunes_app_id_list, $itunes_bundle_id_list) = (
        join(',', map {sprintf("'%s'", $_->{source_app})} @{$missing_app_list->{$OS_ID_ANDROID}}),
        join(',',
            map {sprintf("'id%d'", $_->{source_app})}
            grep {$_->{source_app} !~ /\D+/} @{$missing_app_list->{$OS_ID_IOS}}),
        join(',',
            map {sprintf("'%s'", $_->{source_app})} grep {$_->{source_app} =~ /\D+/} @{$missing_app_list->{$OS_ID_IOS}})
    );

    my $yt_replica_list = $self->app->get_option('api_http_maas')->{yt_replicas};
    my $yt_path_gplay   = '[//home/direct/extdata-mobile/gplay/latest]';
    my $yt_path_itunes  = '[//home/direct/extdata-mobile/itunes/latest]';

    my $yql_tmpl = q(
        bundle AS bundle,
        app_id AS app_id,
        website AS app_website,
        name AS app_name,
        '%s' AS app_os
    FROM %s
    WHERE
        lang = 'ru' AND
        %s IN (%s)
    );
    if ($gplay_bundle_id_list) {
        my $query = sprintf($yql_tmpl, $OS_ID_ANDROID, $yt_path_gplay, 'app_id', $gplay_bundle_id_list);
        push @data_parsed, _select_then_parse($self->app->api_yt, $yt_replica_list, $yt_path_gplay, $query);
    }
    if ($itunes_app_id_list) {
        my $query = sprintf($yql_tmpl, $OS_ID_IOS, $yt_path_itunes, 'app_id', $itunes_app_id_list);
        push @data_parsed, _select_then_parse($self->app->api_yt, $yt_replica_list, $yt_path_itunes, $query);
    }
    if ($itunes_bundle_id_list) {
        my $query = sprintf($yql_tmpl, $OS_ID_IOS, $yt_path_itunes, 'bundle', $itunes_bundle_id_list);
        push @data_parsed, _select_then_parse($self->app->api_yt, $yt_replica_list, $yt_path_itunes, $query);
    }
    return \@data_parsed;
}

TRUE;
