package QBit::Application::Model::API::Yandex::MediaStorageAvatars;

=encoding UTF-8

=cut

use qbit;

use base qw(QBit::Application::Model::API::HTTP);

use XML::Simple;

use Exception::API::MediaStorage;
use PiConstants qw($TVM_HTTP_HEADER_NAME);

=head
    see MDS Avatars response codes w/description
    https://wiki.yandex-team.ru/mds/avatars/#kodyvozvrata7
=cut

my $MDS_ERR_DESC = {
    'code_403_update_is_prohibited' => 'update is prohibited',
    'code_403_image_is_banned'      => 'image is banned',
    'code_400_imagemagick_failed'   => qr/imagemagick failed/,
};

sub accessor {'api_media_storage_avatars'}

sub get_structure_model_accessors {
    return {api_tvm => 'Application::Model::API::Yandex::TVM',};
}

=head1
    if we try to upload the same image twice MDS returns 403 Unauthorized,
    but image meta info can be in response->{attrs}
    $1 = '{
        "attrs": {
            "extra": {
                "svg": {
                    "path": "/get-partner/65861/99_cent.jpg/svg"
                }
            },
            "group-id": 65861,
            "imagename": "99_cent.jpg",
            "meta": {
                "crc64": "8E26F2EFC7269C63",
                "md5": "e38174c4a5118d7d4c64f4ab01d07ad6",
                "modification-time": 1549985590,
                "orig-animated": false,
                "orig-format": "JPEG",
                "orig-orientation": "",
                "orig-size": {
                    "x": 1178,
                    "y": 640
                },
                "orig-size-bytes": 349141,
                "processed_by_computer_vision": false,
                "processed_by_computer_vision_description": "computer vision is disabled",
                "processing": "finished"
            },
            "sizes": {
                "big": {
                    "height": 435,
                    "path": "/get-partner/65861/99_cent.jpg/big",
                    "width": 800
                },
                "card": {
                    "height": 150,
                    "path": "/get-partner/65861/99_cent.jpg/card",
                    "width": 210
                },
                "card_retina": {
                    "height": 300,
                    "path": "/get-partner/65861/99_cent.jpg/card_retina",
                    "width": 420
                },
                "list": {
                    "height": 50,
                    "path": "/get-partner/65861/99_cent.jpg/list",
                    "width": 70
                },
                "list_retina": {
                    "height": 100,
                    "path": "/get-partner/65861/99_cent.jpg/list_retina",
                    "width": 140
                },
                "orig": {
                    "height": 640,
                    "path": "/get-partner/65861/99_cent.jpg/orig",
                    "width": 1178
                }
            }
        },
        "description": "update is prohibited",
        "status": "error"
    }';

=cut

sub call {
    my ($self, $url, %params) = @_;

    throw Exception::Validation::BadArguments gettext('Expected "url"') unless $url;

    my $tvm_ticket = $self->api_tvm->get_service_ticket($self->get_option('tvm_alias'));
    throw Exception::API gettext('Can not get TVM ticket') unless $tvm_ticket;

    my $headers = HTTP::Headers->new();
    $headers->header($TVM_HTTP_HEADER_NAME => $tvm_ticket);

    $self->{'__LWP__'}->default_headers($headers) if $headers;

    my $response;

    try {
        my $r = $self->SUPER::call($url, %params);
        $response = $r ? from_json($r) : $r;
    }
    catch Exception::API::HTTP with {
        my ($exception) = @_;
        # if we try to upload the same image twice
        # MDS returns 403 Unauthorized, but image meta info can be in response
        if (403 == $exception->{response}->code && $url =~ m!/put\-!ig) {
            my $r = from_json($exception->{response}->content);
            if ($MDS_ERR_DESC->{'code_403_update_is_prohibited'} eq $r->{'description'}) {
                $response = $r->{'attrs'};
            } elsif ($MDS_ERR_DESC->{'code_403_image_is_banned'} eq $r->{'description'}) {
                throw Exception::Validation::BadArguments gettext('Can not save image: image is banned');
            }
        }
        # 404 file not found on delete is ok, we should delete it as well
        elsif (404 == $exception->{response}->code && $url =~ m!/delete!ig) {
            $response = '';
        }
        # 434 MDS can not download file by link
        elsif (434 == $exception->{response}->code && $url =~ m!/put\-!ig) {
            throw Exception::Validation::BadArguments gettext('Can not save image: link is unavailable');
        } elsif (400 == $exception->{response}->code && $url =~ m!/put\-!ig) {
            my $r = from_json($exception->{response}->content);
            if ($r->{'description'} =~ /$MDS_ERR_DESC->{'code_400_imagemagick_failed'}/ig) {
                throw Exception::Validation::BadArguments gettext('Can not save image: incorrect image format');
            }
        }
        # 409 image blacklisted | 415 image is too small | 415 face was found on the image
        elsif ((409 == $exception->{response}->code || 415 == $exception->{response}->code) && $url =~ m!/put\-!ig) {
            throw Exception::Validation::BadArguments gettext('Can not save image: this image is unacceptable');
        }
        # 429 too many rps
        elsif (429 == $exception->{response}->code) {
            throw Exception::Validation::BadArguments gettext(
                'Can not save image: servers are busy, please try again later');
        } else {
            throw Exception::API::MediaStorage $exception,
              sentry => {fingerprint => ['MediaStorageAvatars', 'Exception::API::HTTP', $exception->{response}->code]};
        }
    }
    catch {
        my ($exception) = @_;
        throw Exception::API::MediaStorage $exception,
          sentry => {fingerprint => ['MediaStorageAvatars', 'other exception']};
    };
    return $response;
}

=head1 get_link_with_upload_file

    my $link = $app->api_media_storage_avatars->get_link_with_upload_file('the_file.txt', "123");

И в $link будет записан публичный урл загруженного файла:

    "http://storage.mdst.yandex.net/get-partner/1184/the_file.txt"

=cut

sub get_link_with_upload_file {
    my ($self, $data, %opts) = @_;

    my $result = $self->upload_file($data, %opts);

    return $self->get_public_link($result);
}

=head1 get_link_with_upload_by_link

    my $link = $app->api_media_storage_avatars->get_link_with_upload_by_link('the_file.txt', 'http://100photos.time.com/photos/andreas-gursky-99-cent');

И в $link будет записан публичный урл загруженного файла:

    "http://storage.mdst.yandex.net/get-partner/1184/the_file.txt"

=cut

sub get_link_with_upload_by_link {
    my ($self, $link) = @_;

    my $result = $self->upload_link($link);

    return $self->get_public_link($result);
}

=head1 get_public_link
    returns public link of an image thumbnail with $size_name or 'orig' if not set
    https://avatars.mdst.yandex.net/get-partner/65861/99_cent.jpg/orig
=cut

sub get_public_link {
    my ($self, $uploaded_img_info, $size_name) = @_;

    throw Exception::Validation::BadArguments gettext('Expected "uploaded_img_info"') unless $uploaded_img_info;

    $size_name //= 'orig';

    return sprintf(
        '%s/get-%s/%s/%s/%s',
        $self->get_option('public_url'),
        $self->get_option('namespace'),
        $self->get_property_groupid($uploaded_img_info),
        $self->get_property_imagename($uploaded_img_info), $size_name
    );
}

sub _get_property {
    my ($uploaded_img_info, $property_name) = @_;

    throw Exception::Validation::BadArguments gettext('Expected "uploaded_img_info"') unless $uploaded_img_info;
    my $v;
    if (exists $uploaded_img_info->{$property_name}) {
        $v = $uploaded_img_info->{$property_name};
    } elsif (exists $uploaded_img_info->{'meta'}->{$property_name}) {
        $v = $uploaded_img_info->{'meta'}->{$property_name};
    }
    return $v;
}

=head1 get_property_groupid
    returns value of 'group-id' key from image info MDS returns after upload
    to use in making links to thumbnails
=cut

sub get_property_groupid {
    my ($self, $uploaded_img_info) = @_;
    return _get_property($uploaded_img_info, 'group-id');
}

=head1 get_property_groupid
    returns value of 'imagename' key from image info MDS returns after upload
    to use in making links to thumbnails
=cut

sub get_property_imagename {
    my ($self, $uploaded_img_info) = @_;
    return _get_property($uploaded_img_info, 'imagename');
}

=head1 get_property_md5
    returns value of 'md5' key from image info MDS returns after upload
=cut

sub get_property_md5 {
    my ($self, $uploaded_img_info) = @_;
    return _get_property($uploaded_img_info, 'md5');
}

=head1 get_property_sizes
    returns value of 'sizes' key from image info MDS returns after upload
=cut

sub get_property_sizes {
    my ($self, $uploaded_img_info) = @_;
    return _get_property($uploaded_img_info, 'sizes');
}

=head1 upload_file

    my $upload_result = $app->api_media_storage->upload_file('image.jpg', "file content");

После этого в $upload_result будет что-то вроде:

$1 = {
    'extra' => {
                 'svg' => {
                            'path' => '/get-partner/65861/99_cent.jpg/svg'
                          }
               },
    'group-id' => 65861,
    'imagename' => '99_cent.jpg',
    'meta' => {
                'crc64' => '8E26F2EFC7269C63',
                'md5' => 'e38174c4a5118d7d4c64f4ab01d07ad6',
                'modification-time' => 1549985590,
                'orig-animated' => bless( do{\(my $o = 0)}, 'JSON::XS::Boolean' ),
                'orig-format' => 'JPEG',
                'orig-orientation' => '',
                'orig-size' => {
                                 'x' => 1178,
                                 'y' => 640
                               },
                'orig-size-bytes' => 349141,
                'processed_by_computer_vision' => $1->{'meta'}{'orig-animated'},
                'processed_by_computer_vision_description' => 'computer vision is disabled',
                'processing' => 'finished'
              },
    'sizes' => {
                 'big' => {
                            'height' => 435,
                            'path' => '/get-partner/65861/99_cent.jpg/big',
                            'width' => 800
                          },
                 'card' => {
                             'height' => 150,
                             'path' => '/get-partner/65861/99_cent.jpg/card',
                             'width' => 210
                           },
                 'card_retina' => {
                                    'height' => 300,
                                    'path' => '/get-partner/65861/99_cent.jpg/card_retina',
                                    'width' => 420
                                  },
                 'list' => {
                             'height' => 50,
                             'path' => '/get-partner/65861/99_cent.jpg/list',
                             'width' => 70
                           },
                 'list_retina' => {
                                    'height' => 100,
                                    'path' => '/get-partner/65861/99_cent.jpg/list_retina',
                                    'width' => 140
                                  },
                 'orig' => {
                             'height' => 640,
                             'path' => '/get-partner/65861/99_cent.jpg/orig',
                             'width' => 1178
                           }
               }
  };

=cut

sub upload_file {
    my ($self, $data) = @_;

    throw Exception::Validation::BadArguments gettext('Expected "data"') unless length($data);

    return $self->call(
        $self->_get_mds_avatars_put_url(),
        ':post'    => TRUE,
        ':content' => {'file' => $data},
        ':headers' => {'Content-Type' => 'multipart/form-data'}
    );
}

=head1 upload_link

    my $upload_result = $app->api_media_storage_avatars->upload_link('99_cent.jpg', 'http://dujye7n3e5wjl.cloudfront.net/photographs/640-tall/time-100-influential-photos-andreas-gursky-99-cent-90.jpg');
    $upload_result see above
=cut

sub upload_link {
    my ($self, $link) = @_;

    throw Exception::Validation::BadArguments gettext('Expected "link"') unless $link;

    return $self->call($self->_get_mds_avatars_put_url(), ':get' => TRUE, 'url' => $link);
}

=head1 delete_file
    my $delete_result = $app->api_media_storage_avatars->delete_file($group_id, $filename);
=cut

sub delete_file {
    my ($self, $group_id, $filename) = @_;

    throw Exception::Validation::BadArguments gettext('Expected "filename"') unless $filename;
    throw Exception::Validation::BadArguments gettext('Expected "group_id"') unless $group_id;

    return $self->call($self->_get_mds_avatars_delete_url($group_id, $filename), ':get' => TRUE);
}

=head1 _get_mds_avatars_put_url

    my $url = $self->_get_mds_avatars_put_url();
    $url = string URL part after host to request MDS Avatars for upload file/link
=cut

sub _get_mds_avatars_put_url {
    my ($self) = @_;
    return sprintf(":%d/put-%s/", $self->get_option('put_port'), $self->get_option('namespace'));
}

=head1 _get_mds_avatars_delete_url

    my $url = $self->_get_mds_avatars_delete_url('group-id', 'filename.fml');
    $url = string URL part after host to request MDS Avatars for delete uploaded file
=cut

sub _get_mds_avatars_delete_url {
    my ($self, $group_id, $filename) = @_;
    return sprintf(
        ":%d/delete-%s/%s/%s",
        $self->get_option('put_port'),
        $self->get_option('namespace'),
        $group_id, $filename
    );
}

sub check_tvm_service {
    my ($self) = @_;

    my $result;
    try {
        $self->delete_file(1, 'null.exe');
    }
    catch {
        my ($e) = @_;
        if ($e->message =~ /401 Unauthorized/) {
            $result = TRUE;
        }
    };

    return $result;
}

TRUE;
