package QBit::Application::Model::API::Yandex::Metrika;

=encoding UTF-8

=head1 Название

QBit::Application::Model::API::Yandex::Metrika - API для работы с Метрикой

=head1 Описание

Реализут внешний API интерфейс с Метрикой.

см https://tech.yandex.ru/metrika/doc/api2/management/counters/addcounter-docpage/

=end comment

=cut

use qbit;
use base qw(QBit::Application::Model::API::HTTP);

use PiSecrets qw( get_secret );
use Utils::Logger qw( INFOF  ERROR);

use Exception::Metrika;
use Exception::API::Metrika;

my $TYPE = 'partner';

sub accessor {'api_metrika'}

# см https://tech.yandex.ru/metrika/doc/api2/management/counters/addcounter-docpage/
sub create_page_counter_for_login {
    my ($self, $domain, $page_id, $accessor, $owner_id) = @_;

    # NOTE! пара site-name должна быть уникальна, поэтому если хотим на домен
    #       несколько счетчиков(под каждый пэйдж), то в названии должен быть page_id!
    my $data_str = to_json(
        {
            counter => {
                site       => $domain,
                name       => _create_name($domain, $page_id),
                type       => $TYPE,
                partner_id => $page_id,
            }
        }
    );

    my $counter_id;
    try {

        # Пример (https://wiki.yandex-team.ru/partner/w/external-services-metrika/#primery):
        #   curl -s -XPOST -H 'Content-Type: application/json'  -H "Authorization: OAuth $TOKEN"  \
        #     https://api-metrika-test.metrika.yandex.net/management/v1/counters?ulogin=yndx-partner \
        #     -d '{"counter":{"site":"yandex.ru", "name":"", "type":"partner"}}'

        my $response_str = $self->call(
            '/management/v1/counters',
            ':post'    => 1,
            ':headers' => {'Content-Type' => 'application/json',},
            # query POST data
            ':content' => $data_str,
            # query GET params
            uid => $owner_id
        );

        my $response = from_json($response_str);
        $counter_id = $response->{counter}->{id};

        throw Exception::API::Metrika 'Cannot get counter_id for ' . $data_str unless $counter_id;
    }
    catch {
        my ($e) = @_;

        my $response_data = eval {from_json(fix_utf($e->{'response'}->decoded_content()))} // {};

        if ($response_data->{errors}) {
            $e->{text} = join "\n", map {$_->{message}} @{$response_data->{errors}};
        }

        throw Exception::API::Metrika $e, undef, undef,
          sentry => {fingerprint => ['Metrika', 'Error on process result']};
    };

    INFOF('Metrika counter=%s created (page_id=%s, model=%s)', $counter_id, $page_id, $accessor) if $counter_id;

    return $counter_id;
}

sub create_or_get_page_counter_for_login {
    my ($self, $domain, $page_id, $accessor, $owner_id) = @_;
    my $counter_id;
    try {
        $counter_id = $self->create_page_counter_for_login($domain, $page_id, $accessor, $owner_id);
    }
    catch Exception::API::Metrika with {
        my ($e) = @_;
        try {
            $counter_id = $self->get_counter_id($domain, $page_id, $accessor, $owner_id);
        }
        catch {
            # выкидываем исхожную ошибку создания
            throw $e;
        }
    };
    return $counter_id;
}

# см  https://tech.yandex.ru/metrika/doc/api2/api_v1/data-docpage/
sub get_stat {
    my ($self, $date_from, $date_to) = @_;

    my %opts;

    $opts{'date1'} = $date_from if defined($date_from);
    $opts{'date2'} = $date_to   if defined($date_to);

    foreach (keys(%opts)) {
        throw gettext('Expected date format YYYY-mm-dd') unless check_date($opts{$_}, iformat => 'db');
    }

    $opts{'ids'}        = $self->get_option('counter_id');
    $opts{'filters'}    = "ym:s:paramsLevel1=='" . $self->get_option('filter_name') . "'";
    $opts{'dimensions'} = 'ym:s:date,ym:s:paramsLevel2,ym:s:paramsLevel3,ym:s:paramsLevel4';
    $opts{'metrics'}    = 'ym:s:visits';
    $opts{'accuracy'}   = 'full';
    $opts{'limit'}      = '100000';

    my $json = $self->call('/stat/v1/data', %opts);

    my $data;
    try {
        $data = from_json($json);
    }
    catch {
        throw Exception::Metrika gettext('Incorrect answer'), undef, undef, sentry => {
            fingerprint => ['Metrika', 'Error on process result'],
            extra => {answer => $json}
        };
    };

    throw Exception::Metrika gettext('Answer from Metrika contains amount of rows over requested limit')
      if $data->{'total_rows'} > $opts{'limit'};

    my @result = map {
        {
            date          => $_->{dimensions}[0]->{name},
            page_id       => $_->{dimensions}[1]->{name},
            position_type => $_->{dimensions}[2]->{name},
            pixel_type    => $_->{dimensions}[3]->{name},
            total         => $_->{metrics}[0],
        }
    } @{$data->{data}};

    return \@result;
}

sub get_counter_id {
    my ($self, $domain, $page_id, $accessor, $owner_id) = @_;

    my $name = _create_name($domain, $page_id);

    my $response_str = $self->call(
        '/management/v1/counters',
        # query GET params
        search_string => $name,
        uid           => $owner_id
    );

    my $response = from_json($response_str);

    my $counters = $response->{counters} // [];

    for my $counter (@{$counters}) {
        # тк АПИ возвращает список, то пройдемся по нему и найдем нужный нам
        # хотя в большинстве случаях должен возвращаться один элемент
        if ($counter->{name} eq $name) {
            my $counter_id = $counter->{id};
            INFOF('Metrika counter=%s found (page_id=%s, model=%s)', $counter_id, $page_id, $accessor) if $counter_id;
            return $counter->{id};
        }
    }

    return undef;
}

sub _create_name {
    my ($domain, $page_id) = @_;
    return sprintf("Page %s (%s)", $page_id, $domain);
}

sub check_tvm_service {
    my ($self) = @_;

    my $result;
    try {
        my $response_str = $self->call(
            '/management/v1/counters',
            search_string => '---',
            uid           => 1
        );
    }
    catch {
        my ($e) = @_;
        if ($e->message =~ /403 Forbidden/) {
            $result = TRUE;
        }
    };

    return $result;
}

TRUE;
