
=head1 Name

QBit::Application::Model::API::Yandex::OEBS - Module for getting documents from OEBS.

=head1 Description

The OEBS module implements interface for getting documents infromation from OEBS Yandex system.

=cut

package QBit::Application::Model::API::Yandex::OEBS;

use qbit;
use base qw(QBit::Application::Model::API::HTTP);

use MIME::Base64;

use Exception::OEBS;
use Exception::Validation::BadArguments;

my $NOT_FOUND_ERROR_TEXT = q(Данные по запросу не найдены);

sub call {
    my ($self, $method, %opts) = @_;

    $opts{':headers'}{'Content-Type'} = 'application/json';

    my $response;
    try {
        my $r = $self->SUPER::call($method, %opts);
        $response = from_json($r, keep_utf => TRUE);
    }
    catch {
        my ($e) = @_;
        if ($e->message =~ /$NOT_FOUND_ERROR_TEXT/) {
            return [];
        } else {
            throw Exception::OEBS $e;
        }
    };
    if (ref $response eq 'HASH' && $response->{errorCode}) {
        throw Exception::OEBS $response->{errorMessage};
    }
    return $response;
}

sub check_tvm_service {
    my ($self) = @_;

    my $result;
    try {
        my $res = $self->call(
            'getPartnerActHeaders',
            ':post'    => TRUE,
            ':content' => to_json({ClientID => 1})
        );
    }
    catch {
        my ($e) = @_;
        if ($e->message =~ /403 Forbidden/) {
            $result = TRUE;
        }
    };

    return $result;
}

=head2 get_partner_act_contents

Gets contents of partner acts.

Example:
    my $partner_act_content = $self->api_oebs->get_partner_act_contents(1619548, 57288);
    See t/qbit-application-model-api-yandex-oebs.t for more examples.

B<Arguments:>

=over

=item

B<client_id> - integer, partner client id

=item

B<doc_id> - integer, partner act id

=back

B<Return value:> hashref with act contents.

=cut

sub get_partner_act_contents {
    my ($self, $client_id, $doc_id, %opts) = @_;

    throw Exception::Validation::BadArguments gettext("'%s': is missing", 'client_id') unless $client_id;
    throw Exception::Validation::BadArguments gettext("'%s': is missing", 'doc_id')    unless $doc_id;

    my $res = $self->call(
        'getPartnerActContents',
        ':post'    => TRUE,
        ':content' => to_json({ClientID => $client_id, DocID => $doc_id})
    ) // return undef;

    return undef unless $res->{'doc_body'};

    $res->{'doc_body'} = html_decode($res->{'doc_body'});

    return $res;
}

=head1 Methods

=head2 get_partner_act_headers

Gets headers of partner acts.

Example:
    my $partner_act_headers = $self->api_oebs->get_partner_act_headers(1619548);
    See t/qbit-application-model-api-yandex-oebs.t for more examples.

B<Arguments:>

=over

=item

B<client_id> - integer, partner client id

=back

B<Return value:> arrayref with hashref data entries.

=cut

sub get_partner_act_headers {
    my ($self, $client_id) = @_;

    throw Exception::Validation::BadArguments gettext("'%s': is missing", 'client_id') unless $client_id;

    my $res = $self->call(
        'getPartnerActHeaders',
        ':post'    => TRUE,
        ':content' => to_json({ClientID => $client_id})
    ) // return undef;

    return $res;
}

=head2 get_partner_document_contents

Gets contents of partner document.

Example:
    my $partner_document_contents = $self->api_oebs->get_partner_document_contents(1619548, 48345);
    See t/qbit-application-model-api-yandex-oebs.t for more examples.

B<Arguments:>

=over

=item

B<client_id> - integer, partner client id

=item

B<doc_id> - integer, partner act id

=back

B<Return value:> hashref with document contents.

=cut

sub get_partner_document_contents {
    my ($self, $client_id, $doc_id, %opts) = @_;

    throw Exception::Validation::BadArguments gettext("'%s': is missing", 'client_id') unless $client_id;
    throw Exception::Validation::BadArguments gettext("'%s': is missing", 'doc_id')    unless $doc_id;

    my $res = $self->call('GetPartnerDocumentContents', {ClientID => $client_id, DocID => $doc_id}) // return undef;

    return undef unless $res->[0]{'doc_body'};

    $res->[0]{'doc_body'} = html_decode($res->[0]{'doc_body'});
    $res->[0]{'doc_body'} = encode_base64($res->[0]{'doc_body'}) if $opts{'encode_base64'};

    return $res->[0];
}

=head2 get_partner_documents

Gets partner document properties.

Example:
    my $partner_documents = $self->api_oebs->get_partner_documents(1619548);
    See t/qbit-application-model-api-yandex-oebs.t for more examples.

B<Arguments:>

=over

=item

B<client_id> - integer, partner client id

=back

B<Return value:> arrayref with hashref data entries.

=cut

sub get_partner_documents {
    my ($self, $client_id) = @_;

    throw Exception::Validation::BadArguments gettext("'%s': is missing", 'client_id') unless $client_id;

    my $res = $self->call('GetPartnerDocuments', ClientID => $client_id) // return undef;

    return $res->[0];
}

TRUE;
