package QBit::Application::Model::API::Yandex::SelfemployedOeBS;

use qbit;
use base qw(QBit::Application::Model::API::HTTP);

use Exception::SelfemployedOeBS;
use Exception::SelfemployedOeBS::Unbound;
use Exception::SelfemployedOeBS::Unregistered;
use Exception::Validation::BadArguments;

sub call {
    my ($self, $method, %opts) = @_;

    $opts{':headers'}{'Content-Type'} = 'application/json';

    my $response;
    try {
        my $r = $self->SUPER::call($method, %opts);
        $response = from_json($r, keep_utf => TRUE);
    }
    catch {
        my ($e) = @_;
        throw Exception::SelfemployedOeBS $e;
    };
    if ($response->{errorCode}) {
        if ($response->{errorMessage} =~ /TAXPAYER_UNBOUND/) {
            throw Exception::SelfemployedOeBS::Unbound $response->{errorMessage},
              inn => $response->{errorMessage} =~ /.+\s+(\d{12})/ig;
        } elsif ($response->{errorMessage} =~ /TAXPAYER_UNREGISTERED/) {
            throw Exception::SelfemployedOeBS::Unregistered $response->{errorMessage},
              inn => $response->{errorMessage} =~ /.+\s+(\d{12})/ig;
        } else {
            throw Exception::SelfemployedOeBS $response->{errorMessage};
        }
    }
    return $response->{response};
}

# тест готовности сервиса
sub ping {
    my ($self) = @_;
    return $self->call('ping', ':get' => TRUE);
}

# для проверки работоспособности сервиса и авторизации
sub get_regions {
    my ($self) = @_;

    return $self->call(
        'get-regions',
        ':post'    => TRUE,
        ':content' => to_json({requestTime => curdate(oformat => "yt")}),
    );
}

# bind-by-inn
# Реализует вызов метода ПП НПД PostBindPartnerWithInnRequest
# Этот метод можно использовать для поиска налогоплательщика по ИНН и
# привязки в качестве партнера к нашей платформе (юр.лицу).
# Если налогоплательщик не зарегистрирован в системе "Мой налог" или зарегистрирован,
# но с другим ИНН, то запрос может на найти его. После того, как запрос отправлен в ПП НПД,
# от налогоплательщика требуется зайти в систему "Мой налог" и подтвердить привязку.

# inn - ИНН налогоплательщика
#{
#    "requestId": "Идентификатор заявки на привязку партнера"
#}
sub bind_by_inn {
    my ($self, $inn) = @_;

    return $self->call(
        'bind-by-inn',
        ':post'    => TRUE,
        ':content' => to_json(
            {
                inn => $inn,
                permissions =>
                  ['INCOME_REGISTRATION', 'INCOME_LIST', 'INCOME_SUMMARY', 'CANCEL_INCOME', 'PAYMENT_INFORMATION'],
            }
        ),
    );
}

# bind_status
# Получение статуса привязки налогоплательщика к нашей платформе
# Реализует вызов метода ПП НПД GetBindPartnerStatusRequest
# Этот метод можно использовать для проверки статуса заявки на привязку, была ли она подтверждена
# налогоплательщиком в системе "Мой налог".

# "requestId": "Идентификатор заявки на привязку" (получен из метода bind_by_inn)
#{
#    "resultCode": "Код статуса заявки",
#    "inn": "ИНН налогоплательщика",
#    "processingTime": "Дата рассмотрения заявки, заполняется только после принятия заявки",
#    "permissions": ["Список подтвержденных налогоплательщиком разрешений"]
#}
sub bind_status {
    my ($self, $requestId) = @_;

    return $self->call(
        'bind-status',
        ':post'    => TRUE,
        ':content' => to_json({requestId => $requestId,}),
    );
}

# get_status
# Получение информации и статуса налогоплательщика
# Реализует вызов метода ПП НПД GetTaxpayerStatusRequest
# Этот метод можно использовать для получения информации о привязанном налогоплательщике и
# проверке его статуса в качестве самозанятого. Если налогоплательщик не привязан к платформе,
# метод вернет ошибку.

# "inn": "ИНН налогоплательщика"
#{
#    "firstName": "Имя налогоплательщика",
#    "secondName": "Фамилия налогоплательщика",
#    "patronymic": "Отчество налогоплательщика",
#    "registrationTime": "Дата последней постановки на учёт",
#    "unregistrationTime": "Дата снятия с учёта. Если она заполнена и превышает registrationTime, то считается, что налогоплательщик потерял статус самозанятого",
#    "unregistrationReason": "Код причины снятия с учёта",
#    "activities": ["Виды деятельности"],
#    "region": "ОКТМО региона преимущественного ведения деятельности на текущий отчетный период",
#    "phone": "Телефон налогоплательщика в формате 7XXXXXXXXXX",
#    "email": "Адрес электронной почты налогоплательщика",
#    "accountNumber": "Номер счета для уплаты налога",
#    "lastUpdateTime": "Дата последнего обновления данных",
#    "registrationCertificateNumber": "Номер свидетельства о постановке на учёт"
#}
sub get_status {
    my ($self, $inn) = @_;

    return $self->call(
        'get-status',
        ':post'    => TRUE,
        ':content' => to_json({inn => $inn,}),
    );
}

# get_taxpayer_reg_status
# Получение информации о статусе налогоплательщика без привязки к платформе
# Реализует вызов метода ПП НПД GetTaxpayerRegistrationStatusRequest
# Этот метод можно использовать для получения статуса налогоплательщика, является ли он самозанятым,
# и истории его постановок и снятия с учёта.

# "inn": "ИНН налогоплательщика"
#{
#    "isRegistred": "Флаг, зарегистрирован ли в системе налогоплательщик с таким инн, true/false",
#    "regHistory": [
#      {
#        "registrationDate": "Дата постановки на учёт",
#        "unregistrationDate": "Дата снятия с учёта"
#      }
#    ]
#}
sub get_taxpayer_reg_status {
    my ($self, $inn) = @_;

    return $self->call(
        'get-taxpayer-reg-status',
        ':post'    => TRUE,
        ':content' => to_json({inn => $inn,}),
    );
}

# get-notifications
# Получение уведомлений из личного кабинета налогоплательщика
# Реализует вызов метода ПП НПД GetNotificationsRequest
# Этот метод можно использовать для получения списка уведомлений налогоплательщиков, как новых,
# так и прочитанных и заархивированных. За один раз разрешается запрашивать уведомления по 1000 налогоплательщиков

#     "inn": "Список ИНН налогоплательщиков",
#     "getAcknowleged": "Флаг, показывать ли прочитанные уведомления, true/false. По умолчанию false",
#     "getArchived": "Флаг, показывать ли заархивированные уведомления, true/false. По умолчанию false"

sub get_notifications {
    my ($self, $inn_list, $getAcknowleged, $getArchived) = @_;

    $getAcknowleged = $getAcknowleged ? JSON::XS::true : JSON::XS::false;
    $getArchived    = $getArchived    ? JSON::XS::true : JSON::XS::false;
    return $self->call(
        'get-notifications',
        ':post'    => TRUE,
        ':content' => to_json(
            {innList => [map {inn => $_, getAcknowleged => $getAcknowleged, getArchived => $getArchived}, @$inn_list]}
        ),
    );
}

# post-notif-read
# Отметка уведомлений как прочитанные в личном кабинете налогоплательщика
# Реализует вызов метода ПП НПД PostNotificationsAckRequest
# Этот метод можно использовать для отметки списка уведомлений по налогоплательщикам в качестве уже прочитанных.
# За один раз разрешается отмечать уведомления не более чем по 1000 налогоплательщиков.

# {
#     "innList": [
#     {
#         "inn": "ИНН налогоплательщика",
#         "messageId": ["Идентификаторы уведомлений"]
#     }
#    ]
# }

sub post_notif_read {
    my ($self, $inn_list) = @_;

    return $self->call(
        'post-notif-read',
        ':post'    => TRUE,
        ':content' => to_json({innList => $inn_list,}),
    );
}

# get_receipts
# Этот метод можно использовать для получения информации о доходах (чеках) и
# услугах, за которые были получены эти доходы. Запрос ограничен по датам. Так же
# предполагается пагинация доходов, то есть возврат доходов порциями фиксированного
# размера. На данный момент информации о максимальном размере страницы нет.

#{
#  "inn": "ИНН налогоплательщика",
#  "from": ["Дата, начиная с которой будут отражены доходы"],
#  "to": ["Дата, по которую будут отражены доходы"],
#  "limit": ["Количество доходов на странице, по умолчанию 100"],
#  "offset": ["Отступ от начала списка чеков (начиная с 0), по умолчанию 0"]
#}

#{
#  "errorMessage": null,
#  "errorXml": null,
#  "errorCode": null,
#  "response": {
#                "hasMore": "Флаг, есть ли ещё данные на следующей странице, true/false",
#                "pageTotalAmount": "Итоговая сумма всех чеков на текущей странице",
#                "pageTotalNotCancelledAmount": "Итоговая сумма всех действующих чеков на текущей странице",
#                "pageTotalCancelledAmount": "Итоговая сумма всех сторнированных чеков на текущей странице",
#                "receipts": [
#                              {
#                                "link": "Ссылка на чек",
#                                "totalAmount": "Сумма чека",
#                                "receiptId": "Идентификатор чека",
#                                "requestTime": "Дата формирования",
#                                "operationTime": "Дата расчёта",
#                                "partnerCode": "ID банка/платформы партнера",
#                                "cancelationTime": "Дата сторнирования",
#                                "services": [
#                                              {
#                                                "amount": "Стоимость единицы услуги",
#                                                "name": "Наименование услуги",
#                                                "quantity": "Количество"
#                                              }
#                                            ]
#                              }
#                            ]
#              }
#}
sub get_receipts {
    my ($self, $inn, $from, $to, $limit, $offset) = @_;

    return $self->call(
        'get-receipts',
        ':post'    => TRUE,
        ':content' => to_json(
            {
                inn    => $inn,
                from   => $from,
                to     => $to,
                limit  => $limit // 1000,
                offset => $offset // 0,
            }
        ),
    );
}

sub check_tvm_service {
    my ($self) = @_;

    my $result;
    try {
        $self->get_status(1);
    }
    catch {
        my ($e) = @_;
        if ($e->message =~ /403 Forbidden/) {
            $result = TRUE;
        }
    };

    return $result;
}

TRUE;
