package QBit::Application::Model::API::Yandex::Sender;

=encoding UTF-8

=head1 Название

QBit::Application::Model::API::Yandex::Sender - API для работы с Рассылятором

=head1 Описание

Реализуeт внешний API интерфейс с Рассылятором.

# https://github.yandex-team.ru/sendr/sendr/blob/master/docs/transaction-api.md
# Документация Вики https://wiki.yandex-team.ru/sender/manual/

=end comment

=cut

use qbit;
use base qw(QBit::Application::Model::API::HTTP);
use MIME::Base64;
use File::Slurp qw(read_file);
use PiSecrets qw(get_secret);
use PiConstants qw($PARTNERS_TEST_MAIL $IGNORABLE_MAIL_LISTS_KEY);
use Utils::Logger qw(INFOF);
use CHI;

use Exception::Validation::BadArguments;

my $SENDER_TOKEN;
our $CHI;

__PACKAGE__->model_accessors(
    kv_store   => 'QBit::Application::Model::KvStore',
    partner_db => 'Application::Model::PartnerDB',
);

sub init {
    my ($self) = @_;

    $self->SUPER::init();

    $self->{headers} = {
        'Content-Type'            => 'application/json',
        'X-Sender-I-Am-Sender'    => JSON::XS::true,
        'X-Sender-Prefer-Account' => $self->get_option('account', ''),
    };

    # альтернативный способ авторизации
    # $self->_set_new_auth_token();
    # $self->{headers}{Authorization} = 'Basic ' . encode_base64($self->get_option('token', ''))
    #   unless $self->get_option('tvm_alias');

    return $self;
}

=head2 send

    Отправить почтовое уведомление через Рассылятор.

    to_email     - email получателя (to_yandex_puid, to)
    template     - ключ щаблона и языка
    args         - переменные шаблона
    async        - ассинхронный запрос
    cc           - копия
    bcc          - скрытая копия
    is_real_user - 1|0 default - 0
    remote_addr  - ip address, обязателен, если флаг is_real_user=1

    Пример:
        $sender->send(
            # required, to_email в приоритете
            to_email       => ilia-aksenov@yandex-team.ru                           or
            to             => [{"email" => "alice@example.com", "name" => "Alice"}] or
            to_yandex_puid => id

            template => WMFW1P44-M012,              # required
            args     => {key1 => 'value1'},         # optional
            async    => 1,                          # optional, default value is true
            сс => [
                {
                    "email" => "email" # required
                    "name"  => "name"  # optional
                }
            ],
            bcc => [],
            attachment => [
                {
                "filename" : "a.txt",
                "mime_type" : "application/text",
                "data" : "<binary data>"
                }
            ],
        );

    Ответ:
        "params" : {
            "control" : {
                "async" : true,
                "countdown" : null,
                "expires" : 86400,
                "for_testing" : false
            },
            "source" : {
                "to_email" : "vkoakrev@yandex-team.ru",
                "args" : "{\"var1\":\"value1\"}"
            }
        },
        "result":{
            "status":"OK",
            "message_id":"<20200819200830.826528.82d3ac43c67f40449ac5dbfb6da0d3f0@3f9714fb21f6>",
            "task_id":"793964fc-cc76-4bcd-a739-eeb415a653b8"
        }

        status - OK или ERROR
        message - в случае, если статус ERROR ключ message будет содержать текст ошибки

=cut

sub send {
    my ($self, %opts) = @_;

    map {$opts{lc($_)} = delete $opts{$_}} grep {$_ =~ /[A-Z]/} keys %opts;

    throw Exception::Validation::BadArguments gettext("'%s': is missing", 'to_email')
      unless grep {$_ =~ /^(to|to_email|to_yandex_puid)$/ && $opts{$_}} keys %opts;

    $self->_check_required(\%opts, [qw/template/]);

    if ($self->_is_campaign_disabled($opts{template})) {
        INFOF "Ignoring disabled mail notification %s", $opts{template};
        return {result => {status => 'DISABLED'}};
    }

    $opts{async} = $opts{async} ? JSON::XS::true : JSON::XS::false if defined $opts{async};

    my %deleted_flags;
    unless ($self->get_option('real_send', '')) {
        $deleted_flags{$_} = delete $opts{$_} for (grep {defined $opts{$_}} qw(to cc bcc));
        $opts{to_email} = $PARTNERS_TEST_MAIL;
    }

    if (delete $opts{is_real_user}) {
        $self->_check_required(\%opts, ['remote_addr']);

        $opts{has_ugc}                            = JSON::XS::true;
        $self->{headers}{'X-Sender-Real-User-IP'} = $opts{remote_addr};
        $self->{headers}{'X-Sender-I-Am-Sender'}  = JSON::XS::false;
    }

    $opts{response} = $self->call(
        sprintf("/%s/transactional/%s/send", $self->get_option('account', ''), $opts{template}),
        ':post'    => 1,
        ':headers' => $self->{headers},
        ':content' => to_json \%opts,
    );

    %opts = (%opts, %deleted_flags) if %deleted_flags;
    $self->_log(\%opts);
    return from_json($opts{response}, keep_utf => TRUE);
}

=head2 status

    Провeрка статуса отправки

    message_id - id сообщения

        my $response = $sender->send(...);

        $sender->status(
            message_id => $response->{result}->{message_id},
            template   => 'template_name'
        );

    Ответ:

        200-299 - OK
        500 - Internal error
        404 - запись с таким идентификатором отсутствует (скорее всего истех срок хранения)
        104 - письмо находится в обработке

        {
            "msg" : "SMTPSenderRefused: Invalid sender sndr.bnc.test@yandex.ru",
            "code" : 550,
            "message_id" : "<20200819200819.670208.52e313c5f1b4437ea0de509fcd9a563c@3f9714fb21f6>",
            "retry" : false
        }

=cut

sub status {
    my ($self, %opts) = @_;

    $self->_check_required(\%opts, ['template', 'message_id']);

    $opts{status} = $self->call(
        sprintf(
            "/%s/transactional/%s/status?message_id=%s",
            $self->get_option('account', ''),
            $opts{template}, $opts{message_id}
        ),
        ':get'     => 1,
        ':headers' => $self->{headers},
    );

    $self->_log(\%opts);
    return from_json($opts{status}, keep_utf => TRUE);
}

=head2 campaign

    Информация о кампании


    $sender->campaign(
        template   => 'KTG9UR54-UDH'
    );

    Ответ:

    {
        "letters":[
            {
                "created_at":"2021-09-07T10:13:05.208Z",
                "code":"A",
                "id":"566116"
            }
        ],
        "clone_of":"8IRE2M54-SG11",
        "scheduled_for":null,
        "scheduling_status":null,
        "sending_state_modified_at":null,
        "ignore_unsubscribe":false,
        "unsubscribe_redirect":null,
        "id":"560154",
        "submitted_by":"ilia-aksenov",
        "title":"[EMAIL-143] ✉ Welcome-письма для партнеров РСЯ 0.2",
        "created_by":"sanchezzzzz",
        "state":"active",
        "type":"transact",
        "valid_until":null,
        "description":"",
        "tags":[
            "email-143",
            "welcome",
            "welcome-yan",
            "yan"
        ],
        "deleted":false,
        "ab_winner_code":null,
        "lists":[],
        "sending_state":null,
        "sending_control":null,
        "account":"rspartner",
        "magic_list_id":"sendr-560154",
        "slug":"KTG9UR54-UDH",
        "moderation_state":null,
        "submitted_at":"2021-09-13T17:15:00.467Z",
        "created_at":"2021-09-07T10:13:05.191Z",
        "service_label":null,
        "modified_at":"2021-09-13T17:15:00.468Z",
        "project":null,
        "event_hooks":[],
        "single_use_maillist":false
    }

=cut

sub campaign {
    my ($self, %opts) = @_;

    $self->_check_required(\%opts, ['template']);

    return from_json $self->call(
        sprintf("/%s/campaign/%s/", $self->get_option('account', ''), $opts{template}),
        ':get'     => 1,
        ':headers' => $self->{headers},
      ),
      keep_utf => TRUE;
}

sub _is_campaign_disabled {
    my ($self, $template) = @_;

    $CHI //= CHI->new(driver => 'RawMemory', datastore => {});
    my $result = $CHI->get($template);
    if (!defined $result) {
        $result = 0;
        my $kv = from_json($self->kv_store->get($IGNORABLE_MAIL_LISTS_KEY) // '{}')->{$template} // 0;
        if ($kv) {
            my $status = $self->campaign(template => $template);
            $result = $status->{sending_control} // '' eq 'cancel';
        }
        $CHI->set($template, $result, '5 min');
    }
    return $result;
}

sub _set_new_auth_token {
    my ($self) = @_;

    unless ($SENDER_TOKEN = $self->get_option('token')) {
        $SENDER_TOKEN = get_secret('sender-token');
        $self->set_option('token', $SENDER_TOKEN);
        return TRUE;
    }
    return FALSE;
}

sub _log {
    my ($self, $obj) = @_;

    if ('send' eq (split '::', (caller(1))[3])[-1]) {
        for my $recipient (qw/to_email to cc bcc/) {
            $obj->{recipient}->{$recipient} = delete $obj->{$recipient}
              if defined $obj->{$recipient};
        }

        $obj->{message_id} = from_json($obj->{response})->{result}{message_id};

        for my $opt (grep {$_ !~ /^(recipient|template|response|status|message_id)$/} keys %$obj) {
            $obj->{opts}->{$opt} = delete $obj->{$opt};
        }

        map {$obj->{$_} = to_json $obj->{$_}} grep {ref($obj->{$_}) && $_ ne 'response'} keys %$obj;

        $self->app->partner_db->sender_log->add($obj);
    } else {
        $self->_check_required($obj, ['message_id', 'template', 'status']);

        my $id = $self->partner_db->sender_log->get_all(
            fields => ['id'],
            filter => [AND => [[message_id => '=' => \$obj->{message_id}], [template => '=' => \$obj->{template}]]]
        )->[0]->{'id'};

        throw Exception::Validation::BadArguments gettext('sender_log_error', $obj->{template}, $obj->{message_id})
          unless $id;

        $self->app->partner_db->sender_log->edit($id, {status => $obj->{status}});
    }

    return TRUE;
}

sub _check_required {
    my ($self, $obj, $list) = @_;

    for my $required (@$list) {
        throw Exception::Validation::BadArguments gettext("'%s': is missing", $required) unless $obj->{$required};
    }
    return TRUE;
}

sub check_tvm_service {
    my ($self) = @_;

    my $result;
    try {
        $self->status(
            template   => 'test',
            message_id => 1,
        );
    }
    catch {
        my ($e) = @_;
        if ($e->message =~ / 40[13] /) {
            $result = TRUE;
        }
    };

    return $result;
}

TRUE;
