package QBit::Application::Model::API::Yandex::YQL;

=encoding UTF-8

см YQL HTTP API - https://yql.yandex-team.ru/docs/yt/interfaces/http#primer-zapuska-zaprosa
Swagger https://yql.yandex-team.ru/docs/http/reference/#/operations

=cut

use qbit;

use base qw(QBit::Application::Model::API::HTTP);

use PiSecrets qw( get_secret );
use Utils::Logger qw(INFOF ERRORF);

use Exception;

sub accessor {'api_yql'}

my $YQL_TOKEN;
my $YQL_WAIT_SECONDS_DEFAULT  = 600;
my $YQL_SLEEP_SECONDS_DEFAULT = 5;

sub yql_start_operation_and_get_result {
    my ($self, %opts) = @_;

    my ($clusters, $start_params, $get_params) = @opts{qw(clusters start_params get_params)};

    my $yql_operation_result;
    my $error;
    for my $cluster (@$clusters) {
        try {
            if (my $operation_id = $self->_yql_start_operation(cluster => $cluster, %$start_params)) {
                $yql_operation_result = $self->_yql_get_operation_result($operation_id, %$get_params);
            }
        }
        catch {
            my ($e) = @_;
            $error = $e;
            INFOF 'Cluster "%s" return error: %s', $cluster, substr($e->message(), 0, 1024);
        };
        last if defined($yql_operation_result);
    }

    unless (defined($yql_operation_result)) {
        throw $error if $error;
        throw Exception 'Cannot get yql result';
    }

    return $yql_operation_result;
}

=head2
 starts operation with given query content $opts{params}->{content}
 returns started operation_id
=cut

sub _yql_start_operation {
    my ($self, %opts) = @_;

    $opts{params}{cluster} = delete $opts{cluster};

    my $operation_id = $self->_execute('operations', %opts)->{id};

    return $operation_id;
}

=head2
    NOTE! result is always TRUNCATED!!
    tries to get operation $opts{operation_id} result within $opts{wait_seconds}
    #curl -X GET -H "Authorization: OAuth $YQL_TOKEN" https://yql.yandex.net/api/v2/operations/5cebad2261e58e307c8bb9e3/results?filters=DATA

    returns yql response | undef if operation was not comleted within wait_seconds
=cut

sub _yql_get_operation_result {
    my ($self, $operation_id, %opts) = @_;

    my $wait_seconds             = $opts{wait_seconds};
    my $sleep_before_first_check = $opts{sleep_before_first_check};
    my $want_result              = $opts{want_result} // 1;

    my $yql_op_completed = $self->_yql_wait_operation(
        operation_id             => $operation_id,
        wait_seconds             => $wait_seconds,
        sleep_seconds            => $opts{sleep_seconds},
        sleep_before_first_check => $sleep_before_first_check,
    );

    if ($want_result) {
        my %params = (
            format      => $opts{format}      // 'tsv',
            write_index => $opts{write_index} // 0,
            limit       => $opts{limit}       // 1000,
            answer_raw  => 1,
            filters     => 'DATA',
        );

        my $response = from_json $self->_execute("operations/$operation_id/results", %params);
        my $Write = $response->{data}[0]{Write};
        # если есть Ref, значит в ответе только sample data
        # нужно сделать запрос к tmp таблице с полным ответом
        if (my $Ref = $Write->[0]{Ref}[0]) {
            my $host   = $Ref->{Reference}[1];
            my $table  = $Ref->{Reference}[2];
            my $fields = join ', ', @{$Ref->{Columns}};

            $response = $self->app->api_yt->read_table(
                host    => $host,
                yql     => "$fields FROM [//$table]",
                path    => "//$table",
                headers => {'X-YT-Parameters' => '{output_format=<encode_utf8=%false>json}',},
                params  => {
                    ':timeout'  => 60,
                    ':attempts' => 3,
                    ':delay'    => 0,
                }
            );
            return $response;
        }
        my @fields = map {$_->[0]} @{$Write->[0]{Type}[1][1]};

        my $result = '';
        for my $row (@{$Write->[0]{Data}}) {
            my $data = {};
            for (my $i = 0; $i < scalar(@fields); $i++) {
                my $val = ref($row->[$i]) eq 'ARRAY' ? $row->[$i]->[0] : $row->[$i];
                $data->{$fields[$i]} = $val;
            }
            $result .= sprintf("%s\n", to_json($data, canonical => TRUE));
        }
        return $result;
    } else {
        return $yql_op_completed;
    }
}

=head2
 pings operation $opts{operation_id}
 for $opts{wait_seconds}
 every $opts{sleep_seconds}

 returns yql response
=cut

sub _yql_wait_operation {
    my ($self, %opts) = @_;

    my $is_completed = FALSE;

    my ($operation_id, $wait_seconds, $sleep_seconds, $sleep_before_first_check) =
      @opts{qw( operation_id wait_seconds sleep_seconds sleep_before_first_check)};
    $wait_seconds  //= $YQL_WAIT_SECONDS_DEFAULT;
    $sleep_seconds //= $YQL_SLEEP_SECONDS_DEFAULT;

    my $start_time = time();

    sleep($sleep_before_first_check) if defined($sleep_before_first_check) && $sleep_before_first_check < $wait_seconds;

    while (time() < $start_time + $wait_seconds) {

        my $yql_response = $self->_execute("operations/$operation_id");

        if (grep {$yql_response->{status} eq $_} qw(PENDING RUNNING)) {
            INFOF('yql operation[%s] in progress before %s seconds: status=[%s]',
                $operation_id, $wait_seconds, $yql_response->{status});
            sleep($sleep_seconds);
        } elsif ($yql_response->{status} eq 'COMPLETED') {
            INFOF('yql operation[%s] completed before %s seconds: status=[%s]',
                $operation_id, $wait_seconds, $yql_response->{status});
            $is_completed = TRUE;
            last;
        } else {
            # 'IDLE', 'ERROR', 'ABORTING', 'ABORTED', 'UNKNOWN'
            my $yql_status = $self->_execute("operations/$operation_id/results");

            if ($yql_status->{status} eq 'ERROR') {
                throw Exception::API($yql_status->{errors} ? $yql_status->{errors}->[-1]->{message} : 'unknown');
            } else {
                throw Exception::API sprintf(
                    'yql operation[%s] not completed after %s seconds: status=[%s]; err_message=[%s]',
                    $operation_id,
                    $wait_seconds,
                    $yql_status->{status},
                    $yql_status->{errors}
                    ? (
                        ref $yql_status->{errors} eq 'ARRAY'
                        ? (@{$yql_status->{errors}} ? $yql_status->{errors}->[-1]->{message} : 'unknown')
                        : $yql_status->{errors}
                      )
                    : 'unknown'
                );
            }
            # full results can be found at https://yql.yandex-team.ru/Operations/$operation_id
            last;
        }
    }

    return $is_completed;
}

=head2
 makes a request to yql http api //yql.yandex-team.ru/docs/http/reference
 ie: curl -X GET -H "Authorization: OAuth $YQL_TOKEN" https://yql.yandex.net/api/v2/operations/5cebad2261e58e307c8bb9e3/results?filters=DATA

 $method = url part after api endpoint
 $opts{host} = will add 'USE host;' before yql query
 $opts{headers} = additional http headers
 $opts{params}->{content} = query content
 $opts{params}->{action} = query action ["UNKNOWN", "SAVE", "PARSE", "COMPILE", "VALIDATE", "OPTIMIZE", "RUN", "EXTRACT_PARAMS_META"]
 $opts{params}->{type} = query type ['UNKNOWN', 'SQL', 'YQL', 'MKQL', 'CLICKHOUSE', 'UDF_META', 'SQLv1']

 returns from_json(api response)
=cut

sub _execute {
    my ($self, $method, %opts) = @_;

    $YQL_TOKEN //= get_secret('yql-token');

    my $answer_raw = $opts{answer_raw};
    my $headers = $opts{headers} // {};

    my $params  = $opts{params} // {};
    my $cluster = $params->{cluster};
    my $content = $params->{content};
    my $action  = $params->{action} // 'RUN';
    my $type    = $params->{type} // 'SQLv1';

    my %request_params = (
        (
            $content
            ? (
                ":post"    => 1,
                ":content" => to_json(
                    {
                        content => _prefix_with_use_cluster($cluster, $content),
                        action  => $action,
                        type    => $type,
                    }
                )
              )
            : (":get" => 1,)
        ),
        ":headers" => {
            "Content-Type"  => "application/json",
            "Authorization" => "OAuth $YQL_TOKEN",
            %$headers
        },
        %opts
    );

    my $data = $self->call($method, %request_params);

    return $answer_raw ? $data : from_json($data);
}

sub _prefix_with_use_cluster {
    my ($cluster, $query) = @_;
    return "USE $cluster; $query";
}

1;
