package QBit::Application::Model::API::Yandex::YT;

=encoding UTF-8

=cut

use qbit;

use base qw( QBit::Application::Model::API::Yandex::YT::Base );

use Utils::Logger qw(FATAL INFO ERRORF);

use Exception::API::YT;
use Exception::IncorrectParams;

sub accessor {'api_yt'}

=head2 list_tables_from_any_replica

Функция пытается получить список таблиц/потомков узла (path) идя по списку реплик и возращает первый успешный результат

$app->api_yt->list_tables_from_any_replica(
    replicas => [...],   # список хостов ДЦ, например ['bannah', 'hahn']
    path     => '//...', # путь к узлу
);

При неудаче на всех хостах падает.

=cut

sub list_tables_from_any_replica {
    my ($self, %opts) = @_;

    my ($replicas, $host, $path, $params, $headers) = @opts{qw(replicas host path params headers)};

    $params //= {};

    throw Exception::IncorrectParams 'host arg is depreceted' if $host;

    my $data  = undef;
    my $error = undef;
    foreach $host (@$replicas) {
        my $exception;
        try {
            $data = $self->list(
                'host'    => $host,
                'path'    => $path,
                'headers' => $headers,
                'params'  => {
                    ':timeout'       => 20,
                    ':attempts'      => 1,
                    ':delay'         => 0,
                    ':timeout_retry' => 0,
                    %$params,
                },
            );
        }
        catch {
            ($exception) = @_;
            $error //= $exception;
        };
        last if (!$exception && defined($data));
    }
    throw Exception::API::YT $error unless (defined $data);

    return $data;
}

=head2 read_table_from_any_replica

Функция читает таблицу целиком идя по списку реплик и возращает первый успешный результат

$app->api_yt->read_table_from_any_replica(
    replicas => [...],   # список хостов ДЦ, например ['bannah', 'hahn']
    path     => '//...', # путь к таблице
);

При неудаче на всех хостах падает.

=cut

sub read_table_from_any_replica {
    my ($self, %opts) = @_;

    my ($replicas, $host, $path, $params, $headers) = @opts{qw( replicas host path params headers params)};

    $params //= {};

    throw Exception::IncorrectParams 'host arg is depreceted' if $host;

    my $data  = undef;
    my $error = undef;
    foreach $host (@$replicas) {
        my $exception;
        try {
            $data = $self->read_table(
                'host'    => $host,
                'path'    => $path,
                'headers' => $headers,
                'params'  => $params,
            );
        }
        catch {
            ($exception) = @_;
            $error //= $exception;
        };
        last if (!$exception && defined($data));
    }
    throw Exception::API::YT $error unless (defined $data);

    return $data;
}

=head2 select_from_any_replica

Функция делает YQL select запрос в YT таблицу идя по списку реплик и возращает первый успешный результат

$app->api_yt->select_from_any_replica(
    replicas => [...],   # список хостов ДЦ, например ['seneca-sas', 'bannah', 'markov']
    path     => '//...', # путь к таблице
    yql      => '...',   # YQL запрос типа  "PageID from [//yabs/BadDomainsStat] limit 1"
);

=cut

sub select_from_any_replica {
    my ($self, %opts) = @_;

    my ($replicas, $path, $columns, $yql, $params, $headers) =
      @opts{qw( replicas  path  columns  yql  params  headers )};

    $params //= {};

    throw Exception::IncorrectParams 'Wrong type $replicas' unless $replicas && ref($replicas) eq 'ARRAY';
    throw Exception::IncorrectParams 'Wrong $yql' unless $yql;

    my $data  = undef;
    my $error = undef;
    foreach my $host (@$replicas) {
        $error = undef;
        try {
            $data = $self->select_rows(
                host    => $host,
                path    => $path,
                yql     => $yql,
                headers => $headers,
                params  => {
                    %$params,
                    ':attempts'      => 1,
                    ':timeout_retry' => 0,
                },
            );
        }
        catch {
            ($error) = @_;
        };

        last if !$error && defined $data;
    }

    unless (defined $data) {
        throw Exception::API::YT $error, undef, undef,
          sentry => {fingerprint => ['YT', 'select_from_any_replica', $path]};
    }

    return $data;
}

=head2 rewrite_table_on_cluster

Функция перезаписывает таблицу в YT целиком, одим запросом
на всех или явно указаных кластерах

При удаче ничего не возвращает, иначе throw
NOTE! Падает только не смогли записать ни на один их кластеров (неудача на одном кластере не является ошибкой)

$app->api_yt->rewrite_table(path=>$path, data=>$data)

=cut

sub rewrite_table_on_cluster {
    my ($self, %opts) = @_;

    my ($path, $data, $replicas, $column_names, $params) = @opts{qw( path  data  replicas  column_names params)};

    throw Exception::IncorrectParams '"path" not specified' unless $path;
    throw Exception::IncorrectParams '"data" must be an ARRAY' if ref($data) ne 'ARRAY';
    throw Exception::IncorrectParams '"data" must be an ARRAY of HASHes'
      if !$column_names && ref($data->[0]) ne 'HASH';
    throw Exception::IncorrectParams '"replicas" must be an ARRAY' if $replicas && ref($replicas) ne 'ARRAY';
    throw Exception::IncorrectParams '"column_names" must be an ARRAY'
      if $column_names && ref($column_names) ne 'ARRAY';

    $replicas //= $self->app->get_option('yt')->{'replicas'};
    $column_names //= [sort keys %{$data->[0]}];

    my $replicas_count = scalar(@$replicas);

    my $error;
    foreach my $replica_host (@$replicas) {
        INFO "YT replica $replica_host";
        try {
            my $host = $self->app->api_yt->get_heavy_host($replica_host);

            $self->write_table(
                host         => $host,
                path         => $path,
                column_names => $column_names,
                data         => $data,
                order_by     => $opts{'order_by'},
                params       => $params
            );
        }
        catch {
            ($error) = @_;

            unless (--$replicas_count) {
                throw Exception::API::YT $error;
            } else {
                INFO $error->message;
            }
        }
    }

    return 1;
}

1;
