package QBit::Application::Model::API::Yandex::YT::Base;

use qbit;

use base qw( QBit::Application::Model::API::HTTP);    # Exporter

use Encode;

use PiSecrets qw( get_secret );
use Utils::Logger qw( INFO  INFOF );
use Utils::Order qw(get_sort_function);

use Exception::API::YT;
use Exception::IncorrectParams;

=head2

Функция возвращает список таблиц по пути (список потомков узла)

При удаче возвращает JSON-список потомков, иначе throw>

$app->api_yt->list_tables (host=>$host, path=>$path)

=cut

sub list {
    my ($self, %opts) = @_;

    my ($host, $path, $headers, $params) = @opts{qw( host path headers params )};

    my $data = $self->call(
        host    => sprintf('%s.yt.yandex.net', $host),
        method  => 'list',
        path    => $path,
        params  => $params,
        headers => $headers,
    );

    return $data;
}

=head2

Функция делает YQL select запрос в динамическую YT таблицу

При удаче ничего не возвращает, иначе throw>

$app->api_yt->select_rows(host=>$host, path=>$path, yql=>$yql)

=cut

sub select_rows {
    my ($self, %opts) = @_;

    my ($host, $path, $yql, $headers, $params) = @opts{qw( host  path  yql  headers  params )};

    my $data = $self->call(
        host    => sprintf('%s.yt.yandex.net', $host),
        method  => 'select_rows',
        path    => $path,
        params  => $params,
        headers => {
            'X-YT-Parameters' => sprintf('{"query"="%s"}', $yql),
            %$headers,
        },
    );

    return $data;
}

=head2

Функция читает статическую YT таблицу целиком

При удаче ничего не возвращает, иначе throw>

$app->api_yt->read_table(host=>$host, path=>$path )

=cut

sub read_table {
    my ($self, %opts) = @_;

    my ($host, $path, $headers, $params) = @opts{qw( host  path  headers  params )};

    my $data = $self->call(
        host    => sprintf('%s.yt.yandex.net', $host),
        method  => 'read_table',
        path    => $path,
        params  => $params,
        headers => $headers,
    );

    return $data;
}

=head2 $app->api_yt->get_heavy_host($cluster)

Функция получает какой-либо из хостов кластера, на которые можно отправлять тяжелые операции

=cut

sub get_heavy_host {
    my ($self, $cluster) = @_;

    my $res = $self->SUPER::call(
        '/hosts',
        ':url'     => 'http://' . $cluster . '.yt.yandex.net',
        ':headers' => {'accept' => 'text/plain'},
        params     => {
            ':timeout'  => 10,
            ':attempts' => 3,
            ':delay'    => 0,
        }
    );
    my @hosts = split /\n/, $res;
    throw Exception::API::YT "can't find YT cluster $cluster heavy hosts" unless @hosts;
    return $hosts[0];
}

=head2

Функция перезаписывает таблицу в YT целиком, одим запросом
При удаче ничего не возвращает, иначе throw>

NOTE! Данные должны быть заранее ОТСОРТИРОВАНЫ в соответсвиии с настройками ключей таблицы

$app->api_yt->write_table(host=>$host, path=>$path, data=>$data)

=cut

sub write_table {
    my ($self, %opts) = @_;

    my ($host, $path, $column_names, $data, $params) = @opts{qw( host  path  column_names  data  params)};

    throw Exception::IncorrectParams 'path not specified' unless $path;
    throw Exception::IncorrectParams '"column_names" has wrong format'
      if ref($column_names) ne 'ARRAY' || !@$column_names;
    throw Exception::IncorrectParams '"data" has wrong format'
      if ref($data) ne 'ARRAY' || (defined($data->[0]) && !ref($data->[0]));

    my $row_is_hash = ref($data->[0]) eq 'HASH';

    my $order_by;
    if (defined($opts{'order_by'})) {
        $order_by = get_sort_function($opts{'order_by'}, row_is_hash => $row_is_hash, column_names => $column_names);
    }

    my $data_str = '';
    foreach my $row ($order_by ? sort {$order_by->($a, $b)} @$data : @$data) {
        my $values = $row;
        $values = [@$row{@$column_names}] if $row_is_hash;
        # NOTE! в конце последней строки также должен быть "\n"
        $data_str .= join("\t", map {s|\t|\\t|g; encode('utf8', $_)} @$values) . "\n";
    }

    my $yt_params = sprintf '{"input_format"=<"enable_type_conversion"=%%true;"columns"=["%s"];>"schemaful_dsv"}',
      join('";"', @$column_names);

    $self->call(
        host    => $host,
        method  => 'write_table',
        path    => $path,
        content => \$data_str,
        headers => {
            'accept'          => 'application/json',    # ???
            'X-YT-Parameters' => $yt_params,
        },
        params => $params
    );

    INFOF '%d rows sent to "%s" to "%s" YT cluster', scalar(@$data), $path, $host;

    return 1;
}

=head2

Функция удаляет записи в YT таблице
Отправляет data в формате JSONEachRow (потому что только он позволяет изменять сразу нескольо записей одним запросом)

При удаче ничего не возвращает, иначе throw>

$app->api_yt->insert_rows(host=>$host, path=>$path, data=>$data)

=cut

sub delete_rows {
    my ($self, %opts) = @_;

    $self->insert_rows(%opts, is_delete => 1);

    return 1;
}

=head2

Функция делает upsert в YT таблицу (обновляет или добавляет) или удаляет записи
Отправляет data в формате JSONEachRow (потому что только он позволяет изменять сразу нескольо записей одним запросом)

При удаче ничего не возвращает, иначе throw>

$app->api_yt->insert_rows(host=>$host, path=>$path, data=>$data)

=cut

sub insert_rows {
    my ($self, %opts) = @_;

    my ($path, $data, $require_sync_replica, $host, $is_delete, $params) =
      @opts{qw( path  data  require_sync_replica  host  is_delete  params)};
    my $method = $is_delete ? 'delete_rows' : 'insert_rows';

    throw Exception::IncorrectParams 'path not specified' unless $path;
    throw Exception::IncorrectParams 'data must be an ARRAY of HASHes'
      if ref($data) ne 'ARRAY' || !@$data || ref($data->[0]) ne 'HASH';

    # JSON each row
    my $data_str = '';
    foreach my $row (@$data) {
        $data_str .= to_json($row) . "\n";
    }

    $host //= $self->get_heavy_host($self->app->get_option('yt')->{'meta_cluster'});

    $self->call(
        host    => $host,
        method  => $method,
        path    => $path,
        content => \$data_str,
        headers => {
            'Content-Type'    => 'application/json',
            'accept'          => 'application/json',
            'X-YT-Parameters' => '{"input_format"=<"encode_utf8"=%false;>"json"}',
        },
        params => {
            %$params,
            defined $require_sync_replica
            ? ('require_sync_replica' => $require_sync_replica ? 'true' : 'false')
            : ()
        },
    );

    INFOF '%d rows sent to "%s" to "%s" YT cluster', scalar(@$data), $path, $host;

    return 1;
}

sub call {
    my ($self, %opts) = @_;

    my ($host, $path, $method, $content, $headers, $params) = @opts{qw( host  path  method  content  headers  params )};
    $headers //= {};
    $params  //= {};

    my $yt_methods_to_http = {
        delete_rows => ':put',
        insert_rows => ':put',
        read_table  => ':get',
        select_rows => ':get',
        write_table => ':put',
        list        => ':get',
        remove      => ':post',
    };

    my $http_method = $yt_methods_to_http->{$method // ''};

    throw Exception::IncorrectParams 'host not specified' unless $host;
    throw Exception::IncorrectParams sprintf('unknown method "%s"', $method // 'undef') unless $http_method;
    throw Exception::IncorrectParams 'path not specified' unless $path;
    throw Exception::IncorrectParams 'data must be ref to string' if $content && ref($content) ne 'SCALAR';

    my $token = get_secret('yt-token');

    my %params = (
        $http_method => 1,
        ':url'       => 'http://' . $host,
        ':headers'   => {
            'X-YT-Header-Format' => '<format=text>yson',
            'Authorization'      => 'OAuth ' . $token,
            %$headers
        },
        'path' => $path,
        ($content ? (':content' => $$content) : ()),
        %$params
    );

    my $url = sprintf '/api/v3/%s', $method;

    my $data = undef;
    try {
        $data = $self->SUPER::call($url, %params);
    }
    catch {
        my ($error) = @_;
        throw Exception::API::YT $error;
    };

    return $data;
}

1;
