package QBit::Application::Model::DB;

use qbit;

use base qw(QBit::Application::Model);

use DBI;

use Exception::DB;
use Exception::DB::DuplicateEntry;

use Utils::Logger qw/INFO WARN/;

our $DEBUG = $ENV{'FORCE_LOGGER_TO_SCREEN'} // FALSE;

our $DBH_KEY = \undef;

__PACKAGE__->abstract_methods(qw(query filter _get_table_object _create_sql_db _connect _is_connection_error));

sub begin {
    my ($self) = @_;

    $self->{'__SAVEPOINTS__'} == 0
      ? $self->_do('BEGIN')
      : $self->_do('SAVEPOINT SP' . $self->{'__SAVEPOINTS__'});

    ++$self->{'__SAVEPOINTS__'};
}

sub close_coroutine_dbhs {
    my ($self) = @_;

    my $cur_dbh_key = $self->get_key();

    map {delete($self->{'__DBH__'}{$_})} grep {$_ ne $cur_dbh_key} keys(%{$self->{'__DBH__'}});
}

sub close_dbh {
    my ($self) = @_;

    delete($self->{'__DBH__'}{$self->get_key()});
}

sub commit {
    my ($self) = @_;

    --$self->{'__SAVEPOINTS__'}
      ? $self->_do('RELEASE SAVEPOINT SP' . $self->{'__SAVEPOINTS__'})
      : $self->_do('COMMIT');
}

sub create_sql {
    my ($self, %opts) = @_;

    $self->_connect();

    my $SQL = '';

    if ($opts{tables}) {

        $SQL .= join("\n\n", map {"drop table if exists \`$_\`;\n"} @{$opts{tables}}) if $opts{with_drop};

        $SQL .= join("\n\n", map {$self->$_->create_sql()} $self->_sorted_tables(@{$opts{tables}}));

    } else {

        my $meta = $self->get_all_meta();

        $SQL .= join("\n\n", map {$self->$_->create_sql()} $self->_sorted_tables(keys(%{$meta->{'tables'}})))
          if exists($meta->{'tables'});

    }

    return "$SQL\n";
}

sub finish {
    my ($self) = @_;

    if ($self->{'__SAVEPOINTS__'}) {
        $self->{'__SAVEPOINTS__'} = 1;
        $self->rollback();

        throw gettext("Unclosed transaction");
    }
}

sub get_all_meta {
    my ($self, $package) = @_;

    $package = (ref($self) || $self) unless defined($package);
    my $meta = {};

    foreach my $pkg (eval("\@${package}::ISA")) {
        next unless $pkg->isa(__PACKAGE__);
        $self->_add_meta($meta, $pkg->get_all_meta($pkg));
    }

    $self->_add_meta($meta, package_stash($package)->{'__META__'} || {});

    return $meta;
}

sub get_dbh {
    my ($self) = @_;

    return $self->{'__DBH__'}{$self->get_key()};
}

sub get_key {return $$ . (defined($$DBH_KEY) ? $$DBH_KEY : '')}

sub init {
    my ($self) = @_;

    $self->SUPER::init();

    unless (package_stash(ref($self))->{'__METHODS_CREATED__'}) {
        my $meta = $self->get_all_meta();

        $self->make_tables($meta->{'tables'} // {});

        package_stash(ref($self))->{'__METHODS_CREATED__'} = TRUE;
    }

    $self->{'__SAVEPOINTS__'} = 0;
}

=head2 make_tables

B<Arguments:>

=over

=item

B<$tables> - reference of a hash (required).

  {
      table_name => <string (table name)>,
      table_name => <object (QBit::Application::Model::DB::Table)>,
      table_name => <hash (definition of a table)>,
  }

=back

B<Example:>

  $app->db->make_tables({clients => 'users'});
  # or
  $app->db->make_tables({clients => $app->db->users});
  # or
  $app->db->make_tables({
      clients => {
         fields       => ...,
         primary_key  => ...,
         indexes      => ...,
         foreign_keys => ...,
         collate      => ...,
         engine       => ...,
      }
  });

  # after
  $app->db->clients->create();
  $app->db->clients->add_multi(...);
  $app->db->clients->drop();

=cut

sub make_tables {
    my ($self, $tables_meta) = @_;

    my $class     = ref($self);
    my $pkg_stash = package_stash($class);

    my ($meta, %tables);
    foreach my $table_name (keys(%$tables_meta)) {
        my $table_meta = $tables_meta->{$table_name};

        my %table = ();
        if (ref($table_meta) eq 'HASH') {
            %table = %$table_meta;
        } else {
            my $table_name_template = blessed($table_meta) ? $table_meta->name : $table_meta;

            $meta //= $self->get_all_meta();

            throw gettext('Table "%s" not found', $table_name_template)
              unless exists($meta->{'tables'}{$table_name_template});

            %table = %{$meta->{'tables'}{$table_name_template}};
        }

        if ($pkg_stash->{'__METHODS_CREATED__'}) {
            $pkg_stash->{'__META__'}{'tables'}{$table_name} = {%table};
        }

        $table{'class'} = $self->_get_table_class(type => $table{'type'});
        $table{'fields'}       = [$table{'class'}->default_fields(%table),       @{$table{'fields'}       || []}];
        $table{'indexes'}      = [$table{'class'}->default_indexes(%table),      @{$table{'indexes'}      || []}];
        $table{'foreign_keys'} = [$table{'class'}->default_foreign_keys(%table), @{$table{'foreign_keys'} || []}];
        $table{'primary_key'}  = $table{'class'}->default_primary_key(%table)
          unless exists($table{'primary_key'});

        $tables{$table_name} = \%table;
    }

    $self->{'__TABLE_TREE_LEVEL__'}{$_} = $self->_table_tree_level(\%tables, $_, 0) foreach keys(%tables);
    $self->{'__TABLES__'} //= {};

    my $preload_accessors = $self->app->get_option('preload_accessors');

    foreach my $table_name ($self->_sorted_tables(keys(%tables))) {
        throw gettext('Cannot create table object, "%s" is reserved', $table_name)
          if $self->can($table_name);
        {
            no strict 'refs';

            *{$class . '::' . $table_name} = sub {
                $_[0]->{'__TABLES__'}{$table_name} //= do {
                    throw gettext('No package for table "%s"', $table_name)
                      unless defined($tables{$table_name}->{'class'});

                    $tables{$table_name}->{'class'}->new(
                        %{$tables{$table_name}},
                        name => $table_name,
                        db   => $_[0],
                    );
                };
            };
        };

        $self->$table_name if $preload_accessors;
    }
}

sub init_db {
    my ($self, $tables_filter) = @_;

    if ($tables_filter && ref($tables_filter) eq 'ARRAY') {
        $tables_filter = {map {$_ => 1} @$tables_filter};
    }

    $self->_connect();

    my $meta = $self->get_all_meta();

    if (exists($meta->{'tables'})) {
        my @sorted_tables = $self->_sorted_tables(
            grep {!$tables_filter || $tables_filter->{$_}}
              keys(%{$meta->{'tables'}})
        );

        foreach my $table_name (@sorted_tables) {
            $self->_do($self->$table_name->create_sql());
        }
    }
}

sub meta {
    my ($package, %meta) = @_;

    throw gettext("First argument must be package name, QBit::Application::Model::DB descendant")
      if !$package
          || ref($package)
          || !$package->isa('QBit::Application::Model::DB');

    my $pkg_stash = package_stash(ref($package) || $package);
    $pkg_stash->{'__META__'} = \%meta;
}

sub quote {
    my ($self, $name) = @_;

    my ($res) = $self->_sub_with_connected_dbh(
        sub {
            my ($self, $name) = @_;
            return $self->get_dbh()->quote($name);
        },
        [$self, $name]
    );

    return $res;
}

sub quote_identifier {
    my ($self, $name) = @_;

    my ($res) = $self->_sub_with_connected_dbh(
        sub {
            my ($self, $name) = @_;
            return $self->get_dbh()->quote_identifier($name);
        },
        [$self, $name]
    );

    return $res;
}

sub rollback {
    my ($self) = @_;

    --$self->{'__SAVEPOINTS__'}
      ? $self->_do('ROLLBACK TO SAVEPOINT SP' . $self->{'__SAVEPOINTS__'})
      : $self->_do('ROLLBACK');
}

sub transaction {
    my ($self, $sub) = @_;

    $self->begin();
    try {
        $sub->();
    }
    catch {
        my $ex = shift;

        if (!$ex->isa('Exception::DB') || $ex->isa('Exception::DB::DuplicateEntry') || $self->{'__SAVEPOINTS__'} == 1) {
            $self->rollback();
        } else {
            --$self->{'__SAVEPOINTS__'};
        }

        throw $ex;
    };

    $self->commit();
}

sub _add_meta {
    my ($self, $res, $meta) = @_;

    foreach my $obj_type (keys %{$meta}) {
        foreach my $obj (keys %{$meta->{$obj_type}}) {
            warn gettext('Object "%s" (%s) overrided', $obj, $obj_type)
              if exists($res->{$obj_type}{$obj});
            $res->{$obj_type}{$obj} = $meta->{$obj_type}{$obj};
        }
    }
}

sub _do {
    my ($self, $sql, @params) = @_;

    $sql = '/* ' . to_json({system => $ENV{SYSTEM} // '', login => $ENV{LOGIN} // ''}, canonical => 1) . ' */ ' . $sql;

    $self->_debug_sql($sql, TRUE) if $ENV{'DEBUG_SQL'};

    my ($res) = $self->_sub_with_connected_dbh(
        sub {
            my ($self, $sql, @params) = @_;

            my $err_code;
            my $time = time();
            return $self->get_dbh()->do($sql, undef, @params)
              || ($err_code = $self->get_dbh()->err())
              && (($time = time() - $time) || 1)
              && throw Exception::DB $self->get_dbh()->errstr()
              . " ($err_code, $time sec)\n"
              . $self->_log_sql($sql, \@params),
              errorcode => $err_code,
              sentry    => {fingerprint => ['db', $err_code]};
        },
        [$self, $sql, @params],
    );

    return $res;
}

sub _get_all {
    my ($self, $sql, @params) = @_;

    $sql = '/* ' . to_json({system => $ENV{SYSTEM} // '', login => $ENV{LOGIN} // ''}, canonical => 1) . ' */ ' . $sql;

    $self->_debug_sql($sql) if $ENV{'DEBUG_SQL'};

    my ($data) = $self->_sub_with_connected_dbh(
        sub {
            my ($self, $sql, @params) = @_;

            my $err_code;
            my $sth = $self->get_dbh()->prepare($sql)
              || ($err_code = $self->get_dbh()->err())
              && throw Exception::DB $self->get_dbh()->errstr() . " ($err_code)\n" . $self->_log_sql($sql, \@params),
              errorcode => $err_code,
              sentry    => {fingerprint => ['db', $err_code]};

            my $time = time();
            $sth->execute(@params)
              || ($err_code = $self->get_dbh()->err())
              && (($time = time() - $time) || 1)
              && throw Exception::DB $sth->errstr() . " ($err_code, $time sec)\n" . $self->_log_sql($sql, \@params),
              errorcode => $err_code,
              sentry    => {fingerprint => ['db', $err_code]};

            my $data = $sth->fetchall_arrayref({})
              || ($err_code = $self->get_dbh()->err())
              && throw Exception::DB $sth->errstr() . " ($err_code)\n" . $self->_log_sql($sql, \@params),
              errorcode => $err_code,
              sentry    => {fingerprint => ['db', $err_code]};

            $sth->finish()
              || ($err_code = $self->get_dbh()->err())
              && throw Exception::DB $sth->errstr() . " ($err_code)\n" . $self->_log_sql($sql, \@params),
              errorcode => $err_code,
              sentry    => {fingerprint => ['db', $err_code]};

            return $data;
        },
        [$self, $sql, @params],
    );

    return $data;
}

sub _debug_sql {
    my ($self, $sql, $is_do) = @_;

    if ($ENV{'DEBUG_SQL'}) {
        my $prefix;
        if ($self->accessor() eq 'partner_db' && $ENV{'DEBUG_SQL'} ne 'clickhouse') {
            $prefix = 'MYSQL';
        } elsif ($self->accessor() eq 'clickhouse_db' && $ENV{'DEBUG_SQL'} ne 'mysql') {
            $prefix = 'CLICKHOUSE';
        }

        if ($prefix) {
            if ($is_do) {
                WARN("$prefix: " . $sql);
            } else {
                INFO("$prefix: " . $sql);
            }
        }
    }
}

sub _log_sql {
    my ($self, $sql, $params) = @_;

    if ($params && @$params) {
        my $i = 0;
        $sql =~ s|\?|$self->quote( $params->[$i++] )|ge;
    }

    if ($DEBUG) {
        if (-t STDOUT) {
            INFO $sql;
        } else {
            wtf($sql);
        }
    }

    return $sql;
}

sub _sorted_tables {
    my ($self, @table_names) = @_;

    return
      sort {($self->{'__TABLE_TREE_LEVEL__'}{$a} || 0) <=> ($self->{'__TABLE_TREE_LEVEL__'}{$b} || 0) || $a cmp $b}
      @table_names;
}

sub _sub_with_connected_dbh {
    my ($self, $sub, $params, $try) = @_;

    $try ||= 1;
    my @res;
    my $inside_transaction = $self->{'__SAVEPOINTS__'};

    try {
        # Don't (re)connect inside transactions
        $self->_connect() unless $inside_transaction;
        @res = $sub->(@{$params || []});
    }
    catch {
        my $exception = shift;

        if (!defined($self->get_dbh())
            || $self->_is_connection_error($exception->{'errorcode'} || $self->get_dbh()->err()))
        {
            # Connection-related errors
            $self->close_dbh() if defined($self->get_dbh());

            # Don't try to reconnect inside transactions
            if (!$inside_transaction && $try < 3) {
                # It looks like we cannot start the next iteration from within catch() sub so we use recursion
                @res = $self->_sub_with_connected_dbh($sub, $params, $try + 1);
            } else {
                throw $exception;
            }
        } else {
            # Other errors
            throw $exception;
        }
    };

    return @res;
}

sub _table_tree_level {
    my ($self, $tables, $table_name, $level) = @_;

    return $self->{'__TABLE_TREE_LEVEL__'}{$table_name} + $level
      if exists($self->{'__TABLE_TREE_LEVEL__'}{$table_name});

    my @foreign_tables = (
        (map {$_->[1]} @{$tables->{$table_name}{'foreign_keys'}}),
        @{$tables->{$table_name}{'inherits'} || []},
        (exists($tables->{$table_name}{'view_of'}) ? @{$tables->{$table_name}{'view_of'}->($self->app)} : ())
    );

    return @foreign_tables
      ? array_max(map {$self->_table_tree_level($tables, $_, $level + 1)} @foreign_tables)
      : $level;
}

TRUE;
