package QBit::Application::Model::DB::Table;

use qbit;

use base qw(QBit::Application::Model::DB::Class);

__PACKAGE__->mk_ro_accessors(
    qw(
      name
      inherits
      primary_key
      indexes
      foreign_keys
      collate
      engine
      is_view
      view_of
      view_sql
      partition_by
      order_by
      sample_by
      settings
      )
);

__PACKAGE__->abstract_methods(
    qw(
      create_sql
      add_multi
      add
      edit
      delete
      _get_field_object
      _convert_fk_auto_type
      )
);

sub default_fields { }

sub default_foreign_keys { }

sub default_indexes { }

sub default_primary_key { }

sub field_names {
    my ($self) = @_;

    return map {$_->{'name'}} @{$self->fields};
}

sub fields {
    my ($self) = @_;

    return [(map {@{$self->db->$_->fields()}} @{$self->inherits || []}), @{$self->{'fields'}}];
}

sub get {
    my ($self, $id, %opts) = @_;

    throw gettext("No primary key") unless @{$self->primary_key};

    if (ref($id) eq 'ARRAY') {
        $id = {map {$self->primary_key->[$_] => $id->[$_]} 0 .. @$id - 1};
    } elsif (!ref($id)) {
        $id = {$self->primary_key->[0] => $id};
    }

    throw gettext(
        "Bad fields in id (table: '%s' pk: '%s' id: '%s')",
        $self->name,
        join(", ", @{$self->primary_key}),
        join(", ", sort keys(%$id))
    ) if grep {!exists($id->{$_})} @{$self->primary_key};

    return $self->get_all(%opts, filter => {map {$_ => $id->{$_}} @{$self->primary_key}})->[0];
}

sub get_all {
    my ($self, %opts) = @_;

    my $query = $self->db->query->select(
        table => $self,
        hash_transform(\%opts, [qw(fields filter)]),
    );

    $query->group_by(@{$opts{'group_by'}}) if $opts{'group_by'};

    $query->having($opts{'having'}) if $opts{'having'};

    $query->order_by(@{$opts{'order_by'}}) if $opts{'order_by'};

    $query->limit($opts{'offset'} // 0, $opts{'limit'}) if $opts{'limit'};

    $query->distinct() if $opts{'distinct'};

    $query->for_update() if $opts{'for_update'};

    $query->all_langs(TRUE) if $opts{'all_langs'};

    return $query->get_all();
}

sub have_fields {
    my ($self, $fields) = @_;

    $fields = [$fields] if ref($fields) ne 'ARRAY';

    my %field_names_hs = map {$_ => TRUE} $self->field_names;

    return @$fields == grep {$field_names_hs{$_}} @$fields;
}

sub init {
    my ($self) = @_;

    $self->SUPER::init();

    throw gettext('Required opt "fields"')
      unless $self->{'fields'};

    foreach my $field (@{$self->{'fields'}}) {    # Если нет типа, ищем тип в foreign_keys
        unless (exists($field->{'type'})) {
          FT: foreach my $fk (@{$self->{'foreign_keys'} || []}) {
                for (0 .. @{$fk->[0]} - 1) {
                    if ($field->{'name'} eq $fk->[0][$_]) {
                        my $fk_table_name = $fk->[1];
                        $self->_convert_fk_auto_type($field,
                            $self->db->$fk_table_name->{'__FIELDS_HS__'}{$fk->[2][$_]});
                        last FT;
                    }
                }
            }
        }
        $field = $self->_get_field_object(%$field, db => $self->db, table => $self);
        $self->{'__FIELDS_HS__'}{$field->{'name'}} = $field;
    }
}

sub alter {
    my ($self, %opts) = @_;

    my $sql = "ALTER TABLE\n    " . $self->quote_identifier($self->name) . "\n";

    if ($opts{'disable_keys'}) {
        $sql .= "DISABLE KEYS\n";
    } elsif ($opts{'enable_keys'}) {
        $sql .= "ENABLE KEYS\n";
    }

    $self->db->_do($sql);
}

sub create {
    my ($self, %opts) = @_;

    $self->db->_do($self->create_sql(%opts));
}

sub truncate {
    my ($self) = @_;

    $self->db->_do('TRUNCATE TABLE ' . $self->quote_identifier($self->name));
}

sub drop {
    my ($self, %opts) = @_;

    throw 'You cannot drop table in transaction' if $self->db->{'__SAVEPOINTS__'};

    $self->db->_do('DROP TABLE ' . ($opts{'if_exists'} ? 'IF EXISTS ' : '') . $self->quote_identifier($self->name));
}

=head2 swap

B<Arguments:>

=over

=item *

B<$name_or_table> - table name or object

=back

B<Example:>

  $app->db->users->swap('clients');

  # RENAME TABLE `users` TO `users_<pid>_<time>`, `clients` TO `users`, `users_<pid>_<time>` TO `clients`;

  $app->db->users->swap($app->db->clients); # same

=cut

sub swap {
    my ($self, $name_or_table) = @_;

    my $table_name_to_swap = blessed($name_or_table)
      && $name_or_table->isa('QBit::Application::Model::DB::Table') ? $name_or_table->name : $name_or_table;

    my $tmp_table_name = sprintf('%s_%d_%d', $self->name, $$, time);

    $self->db->_do('RENAME TABLE '
          . $self->quote_identifier($self->name) . ' TO '
          . $self->quote_identifier($tmp_table_name)
          . ",\n    "
          . $self->quote_identifier($table_name_to_swap) . ' TO '
          . $self->quote_identifier($self->name)
          . ",\n    "
          . $self->quote_identifier($tmp_table_name) . ' TO '
          . $self->quote_identifier($table_name_to_swap));
}

sub _fields_hs {
    my ($self) = @_;

    return {map {$_->{'name'} => $_} @{$self->fields}};
}

sub _pkeys_or_filter_to_filter {
    my ($self, $pkeys_or_filter) = @_;

    unless (blessed($pkeys_or_filter) && $pkeys_or_filter->isa('QBit::Application::Model::DB::Filter')) {
        if (ref($pkeys_or_filter) eq 'ARRAY') {
            $pkeys_or_filter = [$pkeys_or_filter] if !ref($pkeys_or_filter->[0]) && @{$self->primary_key} > 1;
        } else {
            $pkeys_or_filter = [$pkeys_or_filter];
        }

        my $filter = $self->db->filter();
        foreach my $pk (@$pkeys_or_filter) {
            if (!ref($pk) && @{$self->primary_key} == 1) {
                $pk = {$self->primary_key->[0] => $pk};
            } elsif (ref($pk) eq 'ARRAY') {
                $pk = {map {$self->primary_key->[$_] => $pk->[$_]} 0 .. @{$self->primary_key} - 1};
            }

            if (ref($pk) ne 'HASH' || grep {!defined($pk->{$_})} @{$self->primary_key}) {
                throw gettext('Bad primary key "%s" for table "%s"', $pk, $self->name());
            }
            $filter->or({map {$_ => $pk->{$_}} @{$self->primary_key}});
        }
        $pkeys_or_filter = $filter;
    }

    return $pkeys_or_filter;
}

TRUE;
