package QBit::Application::Model::DB::clickhouse::Table;

use qbit;

use base qw(QBit::Application::Model::DB::Table);

use QBit::Application::Model::DB::clickhouse::Field;

use PiConstants qw($CLICKHOUSE_CHUNK);

BEGIN {
    no strict 'refs';

    foreach my $method (qw(edit delete replace)) {
        *{__PACKAGE__ . "::$method"} = sub {throw gettext('Method "%s" not supported', $method)}
    }
}

sub init {
    my ($self) = @_;

    $self->QBit::Application::Model::DB::Class::init();

    throw gettext('Required opt "fields"')
      unless $self->{'fields'};

    foreach my $field (@{$self->{'fields'}}) {
        $field = $self->_get_field_object(%$field, db => $self->db, table => $self);
        $self->{'__FIELDS_HS__'}{$field->{'name'}} = $field;
    }
}

sub add {
    my ($self, $data, %opts) = @_;

    return $self->add_multi([$data], %opts);
}

sub add_multi {
    my ($self, $data, %opts) = @_;

    my $fields = $self->_fields_hs();

    my $field_names;
    if ($opts{'fields_order'}) {
        my @unknown_fields = grep {!exists($fields->{$_})} @{$opts{'fields_order'}};

        throw gettext('In table %s not found follows fields: %s', $self->name(), join(', ', @unknown_fields))
          if @unknown_fields;

        $field_names = $opts{'fields_order'};
    } elsif (blessed($data->[0]) && $data->[0]->isa('QBit::Application::Model::DB::Query')) {
        $field_names = [sort map {$fields->{$_}->name} keys %$fields];
    } else {
        my $data_fields;
        if ($opts{'identical_rows'}) {
            $data_fields = [sort keys(%{$data->[0] // {}})];
        } else {
            $data_fields = array_uniq(map {keys(%$_)} @$data);
        }

        if ($opts{'ignore_extra_fields'}) {
            $field_names = arrays_intersection([map {$fields->{$_}->name} keys %$fields], $data_fields);
        } else {
            my @unknown_fields = grep {!exists($fields->{$_})} @$data_fields;

            throw gettext('In table %s not found follows fields: %s', $self->name(), join(', ', @unknown_fields))
              if @unknown_fields;

            $field_names = $data_fields;
        }

        $field_names = [sort @$field_names];
    }

    throw Exception::Validation::BadArguments gettext('Expected fields') unless $field_names;

    my @locales = sort keys(%{$self->db->get_option('locales', {})});
    @locales = (undef) unless @locales;

    my $add_rows = 0;

    my $sql_header = 'INSERT INTO ' . $self->quote_identifier($self->name) . ' (';

    my @real_field_names;
    foreach my $name (@$field_names) {
        if ($fields->{$name}{'i18n'}) {
            push(@real_field_names, defined($_) ? "${name}_${_}" : $name) foreach @locales;
        } else {
            push(@real_field_names, $name);
        }
    }
    $sql_header .= join(', ', map {$self->quote_identifier($_)} @real_field_names) . ") ";

    my $db = $self->db();

    if (blessed($data->[0]) && $data->[0]->isa('QBit::Application::Model::DB::Query')) {
        # NOTE! При вставке нельзя передавать 'FORMAT JSONCompact' (#PI-12223)
        $data->[0]->format('NOFormat');

        my ($sql, @params) = $data->[0]->get_sql_with_data();

        return $db->_do($sql_header . $sql, @params);
    }

    $sql_header .= "VALUES\n";

    while (my @add_data = splice(@$data, 0, $CLICKHOUSE_CHUNK)) {
        my ($delimiter, $values) = ('', '');

        foreach my $row (@add_data) {
            my @params;
            foreach my $name (@$field_names) {
                if ($fields->{$name}{'i18n'}) {
                    if (ref($row->{$name}) eq 'HASH') {
                        my @missed_langs = grep {!exists($row->{$name}{$_})} @locales;
                        throw Exception::Validation::BadArguments gettext('Undefined languages "%s" for field "%s"',
                            join(', ', @missed_langs), $name)
                          if @missed_langs;
                        push(@params, $fields->{$name}->quote($row->{$name}{$_})) foreach @locales;
                    } elsif (!ref($row->{$name})) {
                        push(@params, $fields->{$name}->quote($row->{$name})) foreach @locales;
                    } else {
                        throw Exception::Validation::BadArguments gettext('Invalid value in table->add');
                    }
                } else {
                    push(@params, $fields->{$name}->quote($row->{$name}));
                }
            }

            $values .= "$delimiter(" . join(', ', @params) . ')';

            $delimiter ||= ",\n";
        }

        $db->_do($sql_header . $values);

        $add_rows += @add_data;
    }

    return $add_rows;
}

sub create_sql {
    my ($self, %opts) = @_;

    throw gettext('Inherites does not realize') if $self->inherits;

    my $engine = $self->engine() or throw gettext('Expected "engine" for table "%s"', $self->name);

    my ($engine_to_sql) =
      $self->db->query(without_table_alias => TRUE)->_field_to_sql(undef, $engine, {table => $self});

    my $database = $opts{'with_database'} ? $self->db->get_option('database') . '.' : '';

    my $extra = '';
    for my $p (qw(partition_by order_by sample_by)) {
        if (my $val = $self->$p) {
            my $str = uc($p);
            $str =~ s/_/ /;
            my ($sql) = $self->db->query(without_table_alias => TRUE)->_field_to_sql(undef, $val, {table => $self});
            $extra .= "\n$str $sql";
        }
    }

    return
        "CREATE TABLE IF NOT EXISTS $database"
      . $self->quote_identifier($self->name)
      . " (\n    "
      . join(",\n    ", (map {$_->create_sql()} @{$self->fields}),) . "\n"
      . ") ENGINE = $engine_to_sql$extra;\n";
}

sub _get_field_object {
    my ($self, %opts) = @_;

    throw gettext('Required option "name"') unless defined($opts{'name'});
    throw gettext('Required option "type" for field "%s"', $opts{'name'}) unless defined($opts{'type'});

    return QBit::Application::Model::DB::clickhouse::Field->new(%opts);
}

sub optimize {
    my ($self) = @_;

    $self->db->_do('OPTIMIZE TABLE ' . $self->quote_identifier($self->name));
}

TRUE;
