package QBit::Application::Model::DB::mysql;

use qbit;

use base qw(QBit::Application::Model::DB);

use QBit::Application::Model::DB::mysql::Table;
use QBit::Application::Model::DB::mysql::Query;
use QBit::Application::Model::DB::Filter;

use Coro::Mysql;

use Exception::DB;
use Exception::DB::DuplicateEntry;

eval {require Exception::DB::DuplicateEntry};

sub query {
    my ($self) = @_;

    return QBit::Application::Model::DB::mysql::Query->new(db => $self);
}

sub filter {
    my ($self, $filter, %opts) = @_;

    return QBit::Application::Model::DB::Filter->new($filter, %opts, db => $self);
}

sub lock {
    my ($self, %opts) = @_;

    my $tables = $opts{tables};

    my $sql = 'LOCK TABLES ' . join ',',
      map {$self->quote_identifier($_) . ' ' . (lc($tables->{$_}) eq 'read' ? 'READ' : 'WRITE')} keys %$tables;

    return $self->_do($sql);
}

sub unlock {
    my ($self) = @_;

    my $sql = 'UNLOCK TABLES';
    return $self->_do($sql);
}

sub _create_sql_db {
    my ($self) = @_;

    return
        'CREATE DATABASE '
      . $self->get_dbh()->quote_identifier($self->get_option('database'))
      . "\nDEFAULT CHARACTER SET UTF8;\n" . 'USE '
      . $self->get_dbh()->quote_identifier($self->get_option('database')) . ";\n\n";
}

sub _do {
    my ($self, $sql, @params) = @_;

    my $res;
    try {
        $res = $self->SUPER::_do($sql, @params);
    }
    catch Exception::DB with {
        my $e = shift;
        $e->{'text'} =~ /^Duplicate entry/
          ? throw Exception::DB::DuplicateEntry $e
          : throw $e;
    };

    return $res;
}

sub _get_table_class {
    my ($self, %opts) = @_;

    my $table_class;
    if (defined($opts{'type'})) {
        my $try_class = "QBit::Application::Model::DB::mysql::Table::$opts{'type'}";
        $table_class = $try_class if eval("require $try_class");

        throw gettext('Unknown table class "%s"', $opts{'type'}) unless defined($table_class);
    } else {
        $table_class = 'QBit::Application::Model::DB::mysql::Table';
    }

    return $table_class;
}

my $COUNT = 0;

sub _connect {
    my ($self, %opts) = @_;

    if (!defined($self->get_dbh()) || $opts{'use_coro_mysql'}) {
        foreach (qw(database host port)) {
            $opts{$_} //= $self->get_option($_);
        }

        my $dsn = 'DBI:mysql:'
          . join(
            ';', map {$_ . '=' . $opts{$_}}
              grep {defined($opts{$_})} qw(database host port)
          );

        my $dbh = DBI->connect(
            $dsn,
            $self->get_option('user',     ''),
            $self->get_option('password', ''),
            {
                PrintError => 0,
                RaiseError => 0,
                AutoCommit => 1,
            },
        ) || throw Exception::DB DBI::errstr(), sentry => {fingerprint => ['mysql', DBI::err()]};

        $dbh->{'mysql_auto_reconnect'} = FALSE;
        $dbh->do('SET NAMES utf8');
        $dbh->{'mysql_enable_utf8'} = TRUE;

        if ($opts{'use_coro_mysql'}) {
            $dbh = Coro::Mysql::unblock($dbh);
        }

        $self->{'__DBH__'}{$self->get_key} = $dbh;
    }
}

sub _is_connection_error {
    my ($self, $code) = @_;

    return in_array($code || 0, [2006]);
}

sub set_isolation_level_read_committed {
    my ($self) = @_;

    $self->_do('SET SESSION TRANSACTION ISOLATION LEVEL READ COMMITTED;');
}

TRUE;
