package QBit::Application::Model::DB::mysql::Table;

use qbit;

use base qw(QBit::Application::Model::DB::Table);

use QBit::Application::Model::DB::mysql::Field;

use Exception::Validation::BadArguments;

use PiConstants qw($MYSQL_CHUNK);

sub add {
    my ($self, $data, %opts) = @_;

    $data = {$self->primary_key->[0] => $data} if !ref($data) && @{$self->primary_key || []} == 1;

    $self->add_multi([$data], %opts);

    my $fields_hs = $self->_fields_hs();
    my @res       = map {
        !defined($data->{$_})
          && $fields_hs->{$_}{'autoincrement'}
          ? $self->db->_get_all('SELECT LAST_INSERT_ID() AS `id`')->[0]{'id'}
          : $data->{$_}
    } @{$self->primary_key || []};

    return @res == 1 ? $res[0] : \@res;
}

sub edit {
    my ($self, $pkeys_or_filter, $data, %opts) = @_;

    my $query = $self->db->query()->select(table => $self, fields => {});

    my @fields = sort keys(%$data);

    my $sql = 'UPDATE ' . $self->quote_identifier($self->name) . "\n" . 'SET ';

    my $fields = $self->_fields_hs();

    my @locales = sort keys(%{$self->db->get_option('locales', {})});

    my @sql_fields = ();
    foreach my $name (@fields) {
        if ($fields->{$name}{'i18n'} && @locales) {
            my $value = $data->{$name};

            if (ref($value) eq 'HASH') {
                throw Exception gettext('You can not use expression for i18n fields in update')
                  unless grep {exists($value->{$_})} @locales;
            } elsif (ref($value)) {
                throw Exception gettext('You can not use expression for i18n fields in update');
            }

            foreach my $locale (@locales) {
                push(
                    @sql_fields,
                    {
                        field_name => "${name}_${locale}",
                        data       => [ref($value) eq 'HASH' ? $value->{$locale} : $value],
                    }
                );
            }
        } else {
            my ($sql_field, @field_value) = $query->_field_to_sql(
                undef,
                ref($data->{$name}) ? $data->{$name} : \$data->{$name},
                $query->_get_table($self)
            );

            push(
                @sql_fields,
                {
                    field_name => $name,
                    sql        => $sql_field,
                    data       => \@field_value,
                }
            );
        }
    }

    my @field_data = ();
    $sql .= join(
        ",\n    ",
        map {
            push(@field_data, @{$_->{'data'}});
            $self->quote_identifier($_->{'field_name'}) . ' = ' . ($_->{'sql'} ? $_->{'sql'} : '?')
          } @sql_fields
    ) . "\n";

    my $filter_expr = $query->filter($self->_pkeys_or_filter_to_filter($pkeys_or_filter))->expression();
    my ($filter_sql, @filter_data) = $query->_field_to_sql(undef, $filter_expr, $query->_get_table($self));
    $sql .= 'WHERE ' . $filter_sql;

    if (exists($opts{limit})) {
        $sql .= "\nLIMIT $opts{limit}";
    }

    return $self->db->_do($sql, @field_data, @filter_data);
}

sub delete {
    my ($self, $pkeys_or_filter, %opts) = @_;

    my $query = $self->db->query()->select(table => $self, fields => {});
    my $filter_expr = $query->filter($self->_pkeys_or_filter_to_filter($pkeys_or_filter))->expression();
    my ($filter_sql, @filter_data) = $query->_field_to_sql(undef, $filter_expr, $query->_get_table($self));

    my $sql_tail = '';
    if (exists $opts{limit}) {
        $sql_tail .= "\nLIMIT $opts{limit}";
    }

    $self->db->_do('DELETE FROM ' . $self->quote_identifier($self->name) . "\nWHERE $filter_sql $sql_tail",
        @filter_data);
}

=head2 add_multi

Функция делает вставку/замену в MySQL таблицу пачки строк

opts:
 - без опций            INSERT  INTO <table> (<first row fields>) VALUES (<values 1>), ..., (<values N>)
 - ignore_extra_fields  не падает если в данных есть неизвестные столбцы
 - replace              REPLACE INTO ...
 - ignore               ... IGNORE INTO
 - duplicate_update     ... INTO ... ON DUPLICATE KEY UPDATE <non PK field 1>=VALUES(<non PK field 1>), ...,

=cut

sub add_multi {
    my ($self, $data, %opts) = @_;

    my $fields = $self->_fields_hs();

    my $field_names;
    if ($opts{'fields'}) {
        my @unknown_fields = grep {!exists($fields->{$_})} @{$opts{'fields'}};

        throw gettext('In table %s not found follows fields: %s', $self->name(), join(', ', @unknown_fields))
          if @unknown_fields;

        $field_names = $opts{'fields'};
    } elsif (blessed($data->[0]) && $data->[0]->isa('QBit::Application::Model::DB::Query')) {
        $field_names = [sort map {$fields->{$_}->name} keys %$fields];
    } else {
        my $data_fields = [sort keys(%{$data->[0] // {}})];

        if ($opts{'ignore_extra_fields'}) {
            $field_names = arrays_intersection([map {$fields->{$_}->name} keys %$fields], $data_fields);
        } else {
            my @unknown_fields = grep {!exists($fields->{$_})} @$data_fields;

            throw gettext('In table %s not found follows fields: %s', $self->name(), join(', ', @unknown_fields))
              if @unknown_fields;

            $field_names = $data_fields;
        }

        $field_names = [sort @$field_names];
    }

    throw Exception::Validation::BadArguments gettext('Expected fields') unless $field_names;

    my @locales = sort keys(%{$self->db->get_option('locales', {})});
    @locales = (undef) unless @locales;

    my $sql_header =
        ($opts{'replace'} ? 'REPLACE'  : 'INSERT')
      . ($opts{'ignore'}  ? ' IGNORE ' : '')
      . ' INTO '
      . $self->quote_identifier($self->name) . ' (';
    my @real_field_names;
    foreach my $name (@$field_names) {
        if ($fields->{$name}{'i18n'}) {
            push(@real_field_names, defined($_) ? "${name}_${_}" : $name) foreach @locales;
        } else {
            push(@real_field_names, $name);
        }
    }
    $sql_header .= join(', ', map {$self->quote_identifier($_)} @real_field_names) . ") ";

    my $sql_footer = '';
    if ($opts{'duplicate_update'}) {
        my $primary_keys = {map {$_ => 1} @{$self->primary_key // []}};
        my @update_fields = grep {!$primary_keys->{$_}} @real_field_names;
        if (@update_fields) {
            $sql_footer = ' ON DUPLICATE KEY UPDATE '
              . join(', ',
                map {$self->db->quote_identifier($_) . "=VALUES(" . $self->db->quote_identifier($_) . ")"}
                  @update_fields);
        }
    }

    if (blessed($data->[0]) && $data->[0]->isa('QBit::Application::Model::DB::Query')) {
        my ($sql, @params) = $data->[0]->get_sql_with_data();

        return $self->db->_do($sql_header . $sql . $sql_footer, @params);
    }

    $sql_header .= "VALUES\n";

    my $add_rows = 0;

    my $need_transact = @$data > $MYSQL_CHUNK;
    $self->db->begin() if $need_transact;

    try {
        $self->db->_sub_with_connected_dbh(
            sub {
                my ($db, @data) = @_;

                my ($sql, $values);
                my $sth;
                my $err_code;

                while (my @add_data = splice(@data, 0, $MYSQL_CHUNK)) {
                    if (@add_data != $MYSQL_CHUNK) {
                        $sql    = undef;
                        $values = undef;
                    }

                    my @params = ();
                    foreach my $row (@add_data) {
                        unless ($values) {
                            $values = '(?' . ', ?' x (@real_field_names - 1) . ')';
                            $values = $values . (",\n" . $values) x (@add_data - 1);
                        }

                        foreach my $name (@$field_names) {
                            if ($fields->{$name}{'i18n'}) {
                                if (ref($row->{$name}) eq 'HASH') {
                                    my @missed_langs = grep {!exists($row->{$name}{$_})} @locales;
                                    throw Exception::Validation::BadArguments gettext(
                                        'Undefined languages "%s" for field "%s"',
                                        join(', ', @missed_langs), $name)
                                      if @missed_langs;
                                    push(@params, $row->{$name}{$_}) foreach @locales;
                                } elsif (!ref($row->{$name})) {
                                    push(@params, $row->{$name}) foreach @locales;
                                } else {
                                    throw Exception::Validation::BadArguments gettext('Invalid value in table->add');
                                }
                            } else {
                                push(@params, $row->{$name});
                            }
                        }
                    }

                    unless ($sql) {
                        $sql = $sql_header . $values . $sql_footer;
                        $sth = $db->get_dbh()->prepare($sql)
                          || ($err_code = $db->get_dbh()->err())
                          && throw Exception::DB $db->get_dbh()->errstr()
                          . " ($err_code)\n"
                          . $db->_log_sql($sql, \@params),
                          errorcode => $err_code;
                    }

                    $add_rows += $sth->execute(@params)
                      || ($err_code = $db->get_dbh()->err())
                      && throw Exception::DB $sth->errstr() . " ($err_code)\n" . $db->_log_sql($sql, \@params),
                      errorcode => $err_code;

                    $db->_log_sql($sql, \@params) if $QBit::Application::Model::DB::DEBUG;

                }
            },
            [$self->db, @$data]
        );
    }
    catch Exception::DB with {
        my $e = shift;
        $e->{'text'} =~ /^Duplicate entry/
          ? throw Exception::DB::DuplicateEntry $e
          : throw $e;
    };

    $self->db->commit() if $need_transact;

    return $add_rows;
}

sub create_sql {
    my ($self, %opts) = @_;
    $opts{if_not_exists} //= 1;
    throw gettext('Inherites does not realize') if $self->inherits;

    my $collate = defined($self->collate()) ? (" COLLATE '" . $self->collate() . "'") : '';
    my $engine = defined($self->engine()) ? ($self->engine()) : 'InnoDB';

    return $self->view_sql->($self->db->app) if $self->is_view;
    return
        'CREATE TABLE '
      . ($opts{if_not_exists} ? 'IF NOT EXISTS ' : '')
      . $self->quote_identifier($self->name)
      . " (\n    "
      . join(
        ",\n    ",
        (map {$_->create_sql()} @{$self->fields}),
        (
            $self->primary_key
            ? 'PRIMARY KEY (' . join(', ', map {$self->quote_identifier($_)} @{$self->primary_key}) . ')'
            : ()
        ),
        (map {$self->_create_sql_index($_)} @{$self->indexes            || []}),
        (map {$self->_create_sql_foreign_key($_)} @{$self->foreign_keys || []}),
      )
      . "\n"
      . ") ENGINE='$engine' DEFAULT CHARACTER SET 'UTF8'$collate;";
}

sub replace {
    my ($self, $data, %opts) = @_;

    $self->add($data, %opts, replace => 1);
}

sub _convert_fk_auto_type {
    my ($self, $field, $fk_field) = @_;

    $field->{$_} = $fk_field->{$_}
      foreach grep {exists($fk_field->{$_}) && !exists($field->{$_})} qw(type unsigned not_null length);
}

sub _create_sql_foreign_key {
    my ($self, $key) = @_;

    return 'FOREIGN KEY '
      . $self->quote_identifier(
        substr(join('_', 'fk', $self->name, '', @{$key->[0]}, '_', $key->[1], '', @{$key->[2]}), 0, 63))
      . ' ('
      . join(', ', map {$self->quote_identifier($_)} @{$key->[0]}) . ")\n"
      . '        REFERENCES '
      . $self->quote_identifier($key->[1]) . ' ('
      . join(', ', map {$self->quote_identifier($_)} @{$key->[2]}) . ")\n"
      . "            ON UPDATE RESTRICT\n"
      . "            ON DELETE RESTRICT";
}

sub _create_sql_index {
    my ($self, $index) = @_;

    my @fields = map {ref($_) ? $_ : {name => $_}} @{$index->{'fields'}};

    return
        ($index->{'unique'} ? 'UNIQUE ' : '') 
      . 'INDEX '
      . $self->quote_identifier(
        substr(join('_', ($index->{'unique'} ? 'uniq' : ()), $self->name, '', map {$_->{'name'}} @fields), 0, 64))
      . ' ('
      . join(', ', map {$self->quote_identifier($_->{'name'}) . ($_->{'length'} ? "($_->{'length'})" : '')} @fields)
      . ')';
}

sub _get_field_object {
    my ($self, %opts) = @_;

    return QBit::Application::Model::DB::mysql::Field->new(%opts);
}

TRUE;
