package QBit::Application::Model::DBManager;

use qbit;

use base qw(QBit::Application::Model);

use QBit::Application::Model::DBManager::_Utils::Fields;
use QBit::Application::Model::DBManager::Filter;

use File::Find;
use File::Spec;

use Parse::Eyapp;

use Exception::DBManager::Grammar;
use Exception::Validation::BadArguments;

__PACKAGE__->abstract_methods(qw(query add db_table_name));

my $FILTER_PATH = 'QBit::Application::Model::DBManager::Filter';

# TODO: сделать общий метод
#sub init {
#    my ($self) = @_;
#
#    $self->SUPER::init();
#
#    $self->model_fields($self->get_structure_model_fields());
#    $self->model_filter($self->get_structure_model_filter());
#}

sub model_fields {
    my ($self, @fields) = @_;

    # When this sub is called for an object store in the object.
    # When this sub is called for a package store in that package stash.
    #
    my $storage = defined(blessed($self)) ? $self : package_stash($self);

    my $fields_class = 'QBit::Application::Model::DBManager::_Utils::Fields';
    my $fields = ref($fields[0]) eq 'HASH' ? $fields[0] : {@fields};

    $storage->{'__MODEL_FIELDS__'} = $fields;

    # my $inited_fields = $fields_class->init_fields($fields);
    my $inited_fields = $fields_class->init_fields($fields, (ref($self) || $self));

    $fields_class->init_check_rights($inited_fields);

    $storage->{'__MODEL_FIELDS_INITIALIZED__'} = $inited_fields;

    $storage->{'__MODEL_FIELDS_SORT_ORDERS__'} = $fields_class->init_field_sort($inited_fields);
}

sub get_structure_model_fields {return {}}

sub import {
    my ($package, %opts) = @_;

    $package->SUPER::import(%opts);

    my $dir = File::Spec->catdir(split(/::/, $FILTER_PATH));

    my @dirs = map {File::Spec->catdir($_, $dir)} @INC;

    my $prune   = 0;
    my $basedir = undef;

    my @results = ();
    foreach $basedir (@dirs) {
        next unless -d $basedir;

        find(
            {
                wanted => sub {
                    my $name = File::Spec->abs2rel($_, $basedir);
                    return unless $name && $name ne File::Spec->curdir();

                    if (-d && $prune) {
                        $prune = 1;
                        return;
                    }

                    return unless /\.pm$/ && -r;

                    $name =~ s/\.pm$//;
                    $name = join('::', File::Spec->splitdir($name));

                    push(@results, $name);
                },
                no_chdir => 1,
                follow   => 1
            },
            $basedir
        );
    }

    # filter duplicate modules
    my %seen = ();
    @results = grep {not $seen{$_}++} @results;

    @results = map "$FILTER_PATH\::$_", @results;

    @results = map {($_ =~ m{^(\w+(?:(?:::|')\w+)*)$})[0] || die "$_ does not look like a package name"} @results;

    foreach my $m (@results) {
        eval " require $m; import $m; ";
        die $@ if $@;
    }
}

__PACKAGE__->mk_ro_self_or_stash_accessors(
    {
        get_model_fields             => '__MODEL_FIELDS__',
        get_model_fields_initialized => '__MODEL_FIELDS_INITIALIZED__',
        get_model_fields_sort_orders => '__MODEL_FIELDS_SORT_ORDERS__',
    }
);

sub model_filter {
    my ($self, @filter) = @_;

    # When this sub is called for an object store in the object.
    # When this sub is called for a package store in that package stash.
    #
    my $storage = defined(blessed($self)) ? $self : package_stash($self);

    my $filter = ref($filter[0]) eq 'HASH' ? $filter[0] : {@filter};

    $storage->{'__DB_FILTER_DBACCESSOR__'} = $filter->{'db_accessor'} || 'db';

    $storage->{'__DB_FILTER_FIELDS__'} = $filter->{'fields'};
}

__PACKAGE__->mk_ro_self_or_stash_accessors(
    {
        get_model_filter_fields     => '__DB_FILTER_FIELDS__',
        get_model_filter_dbaccessor => '__DB_FILTER_DBACCESSOR__',
    }
);

sub get_structure_model_filter {return {}}

sub get_db_filter_simple_fields {
    my ($self, %opts) = @_;

    $opts{'fields'} = $self->get_db_filter_fields() unless exists($opts{'fields'});

    my @res;
    for my $name (sort keys %{$opts{'fields'}}) {
        my $value = $opts{'fields'}{$name};
        push(@res, {name => $name, label => $value->{'label'}})
          if $self->{'__DB_FILTER__'}{$name}->is_simple;
    }

    return \@res;
}

sub pre_process_fields { }

sub found_rows {
    my ($self) = @_;

    return $self->{'__FOUND_ROWS__'};
}

sub get {
    my ($self, $pk, %opts) = @_;

    return undef unless defined($pk);

    my $pk_fields = $self->get_pk_fields();

    $pk = {$pk_fields->[0] => $pk} if ref($pk) ne 'HASH';

    my @missed_fields = grep {!exists($pk->{$_})} @$pk_fields;
    throw Exception::Validation::BadArguments sprintf("Invalid primary key fields (package: %s, expected fields: %s)",
        ref($self), join(', ', map "\"$_\"", @missed_fields))
      if @missed_fields;

    return $self->get_all(%opts, filter => [AND => [map {[$_ => '=' => $pk->{$_}]} @$pk_fields]])->[0];
}

sub get_all {
    my ($self, %opts) = @_;

    my $fields = $self->_get_fields_obj($opts{'fields'}, $opts{'all_locales'});

    my %last_fields = %{$fields->get_fields()};    # shallow copy

    # Hide unavailable fields from last_fields
    foreach (@{$fields->get_field_need_delete_names()}) {
        delete($last_fields{$_});
    }

    my $query = $self->query(
        fields  => $fields,
        filter  => $self->get_db_filter($opts{'filter'}),
        options => \%opts,
    )->all_langs($opts{'all_locales'});

    $query->distinct   if $opts{'distinct'};
    $query->for_update if $opts{'for_update'};

    if ($opts{'order_by'}) {
        my $all_fields = $self->_get_fields_obj([keys(%{$self->get_model_fields()})]);

        my %db_fields = map {$_ => TRUE} keys(%{$all_fields->get_db_fields()});

        my @order_by = map {[ref($_) ? ($_->[0], $_->[1]) : ($_, 0)]}
          grep {exists($db_fields{ref($_) ? $_->[0] : $_})} @{$opts{'order_by'}};

        $query->order_by(@order_by) if @order_by;
    }

    $query->limit($opts{'offset'}, $opts{'limit'}) if $opts{'limit'};

    $query->calc_rows(1) if $opts{'calc_rows'};

    my $result = $query->get_all();

    $self->{'__FOUND_ROWS__'} = $query->found_rows() if $opts{'calc_rows'};

    if (@$result) {
        $self->pre_process_fields($fields, $result);
        $result = $fields->process_data($result);
    }

    $self->{'__LAST_FIELDS__'} = \%last_fields;

    return $result;
}

#Аксессоры моделей должны совпадать с названиями таблиц в базе.
sub get_all_with_join {
    my ($self, %opts) = @_;

    my $model = $self->{'accessor'};

    my $got_filter = delete($opts{'filter'}) // ['AND', []];

    my @join_models  = ();
    my $join_options = {};

    foreach my $filter (@{$got_filter->[1]}) {
        $self->_parse_filter(\@join_models, $join_options, $model, $filter,
            $self->get_db_filter_fields(private => TRUE));
    }

    my $pk = $self->get_pk_fields();

    my $query = $self->query(
        fields => $self->_get_fields_obj($pk),
        (
            exists($join_options->{$model})
            ? (filter => $self->get_db_filter(['AND', $join_options->{$model}{'filter'}]))
            : ()
        )
    )->group_by(@$pk)->calc_rows(1);

    $query->limit(delete($opts{'offset'}), delete($opts{'limit'})) if $opts{'limit'};

    foreach my $j_model (@join_models) {
        $query->join(
            table  => $self->partner_db->$j_model,
            alias  => 'alias_' . $j_model,
            fields => [],
            (
                exists($join_options->{$j_model}{'filter'})
                ? (filter => $self->$j_model->get_db_filter(['AND', $join_options->{$j_model}{'filter'}]))
                : ()
            ),
            join_on => $join_options->{$j_model}{'join_on'}
        );
    }

    my $data = $query->get_all();

    my $meta = $self->get_all_with_meta(%opts, filter => {id => [map {$_->{'id'}} @$data]});

    $meta->{'meta'}{'found_rows'} = $query->found_rows();

    return $meta;
}

sub get_all_with_meta {
    my ($self, %opts) = @_;

    my %meta_opts = map {$_ => TRUE} @{delete($opts{'meta'}) || []};
    $opts{'calc_rows'} = TRUE if $meta_opts{'found_rows'};

    my $data = $self->get_all(%opts);

    my %meta;
    $meta{'last_fields'} = [sort keys %{$self->last_fields()}] if $meta_opts{'last_fields'};
    $meta{'found_rows'} = $self->found_rows() if $meta_opts{'found_rows'};

    return {
        data => $data,
        meta => \%meta,
    };
}

sub get_db_filter {
    my ($self, $data, %opts) = @_;

    if (!defined($data) || blessed($data)) {
        return $data;
    }

    return ref($data) ? $self->_get_db_filter_from_data($data, %opts) : $self->_get_db_filter_from_text($data, %opts);
}

sub get_db_filter_fields {
    my ($self, %opts) = @_;

    $opts{'cache'} ||= {};

    my $filter_fields = $self->get_model_filter_fields();

    if (exists($opts{fields})) {
        foreach my $field (@{$opts{fields}}) {
            throw Exception::Validation::BadArguments gettext('Filter by unknown field "%s" in model %s', $field,
                ref($self))
              unless exists($filter_fields->{$field});
        }
    }
    my @fields = exists($opts{fields}) ? (@{delete($opts{fields})}) : (keys %$filter_fields);

    foreach my $field (@fields) {
        if (exists($self->{'__DB_FILTER__'}{$field})) {
            next;
        }

        my $fdata = $filter_fields->{$field};

        throw Exception::Validation::BadArguments gettext('Missed filter type (package: "%s", filter: "%s")',
            ref($self), $field)
          unless defined($fdata->{'type'});

        my $filter_class = $FILTER_PATH . '::' . $fdata->{'type'};

        $self->{'__DB_FILTER__'}{$field} = $filter_class->new(%$fdata, field_name => $field, db_manager => $self);
    }

    my %fields = %{clone($self->get_model_filter_fields()) || {}};

    my %requested_fields = map {$_ => TRUE} @fields;
    foreach my $field (keys(%fields)) {
        unless ($requested_fields{$field}) {
            delete($fields{$field});
            next;
        }

        my $save = TRUE;

        $save = $self->{'__DB_FILTER__'}{$field}->pre_process($fields{$field}, $field, %opts)
          if !$opts{'without_pre_process'} && $self->{'__DB_FILTER__'}{$field}->can('pre_process');

        unless ($save) {
            delete($fields{$field});
            next;
        }

        $fields{$field}->{'label'} = $fields{$field}->{'label'}()
          if exists($fields{$field}->{'label'}) && ref($fields{$field}->{'label'}) eq 'CODE';
    }

    unless ($opts{'private'}) {
        _clean_filter_fields($self, \%fields);
    }

    return \%fields;
}

sub _clean_filter_fields {
    my ($accessor, $filter_fields) = @_;

    foreach my $field_name (keys(%$filter_fields)) {
        my $field = $filter_fields->{$field_name};

        if (exists($field->{'value_type'})) {
            $field->{'type'} = $field->{'value_type'};
        }

        my $model_accessor = $field->{'model_accessor'};
        if (defined($model_accessor) && exists($field->{'subfields'})) {
            _clean_filter_fields($accessor->$model_accessor, $field->{'subfields'});
        }

        $filter_fields->{$field_name} =
          {hash_transform($field, [qw(type label), @{$accessor->{'__DB_FILTER__'}{$field_name}->public_keys || []}])};
    }
}

sub get_depends_for_field {
    my ($self, $field_name) = @_;

    my $model_fields = $self->get_model_fields();

    my $field = $model_fields->{$field_name} or throw Exception gettext('Unknown field: %s', $field_name);

    return [@{$field->{'depends_on'} // []}, @{$field->{'forced_depends_on'} // []}];
}

sub get_pk_fields {
    my ($self) = @_;

    my $fields = $self->get_model_fields();

    return [sort {$a cmp $b} grep {$fields->{$_}{'pk'}} keys(%$fields)];
}

sub last_fields {
    my ($self) = @_;

    return $self->{'__LAST_FIELDS__'};
}

sub _db {
    my ($self) = @_;

    my $accessor_name = $self->get_model_filter_dbaccessor();
    return $self->$accessor_name;
}

sub _get_db_filter_from_data {
    my ($self, $data, %opts) = @_;

    return undef unless $data;

    return $data if blessed($data) && $data->isa('QBit::Application::Model::DB::Filter');

    return [AND => [\undef]] if ref($data) && ref($data) eq 'ARRAY' && @$data == 1 && !defined($data->[0]);

    return $self->_get_db_filter_from_data([AND => [map {[$_ => '=' => $data->{$_}]} keys(%$data)]], %opts)
      if ref($data) eq 'HASH';

    if (ref($data) eq 'ARRAY' && @$data == 2 && ref($data->[1]) eq 'ARRAY') {
        throw Exception::Validation::BadArguments gettext('Unknow operation "%s"', uc($data->[0]))
          unless in_array(uc($data->[0]), [qw(OR AND)]);

        return ($opts{'type'} || '') eq 'text'
          ? '(' . join(' ' . uc($data->[0]) . ' ', map {$self->_get_db_filter_from_data($_, %opts)} @{$data->[1]}) . ')'
          : $self->_db()
          ->filter([uc($data->[0]) => [map {$self->_get_db_filter_from_data($_, %opts)->expression()} @{$data->[1]}]]);
    } elsif (ref($data) eq 'ARRAY' && @$data == 3) {
        my $field = $data->[0];
        $opts{'model_fields'}{$field} ||= $self->get_db_filter_fields(private => TRUE, fields => [$field])->{$field};
        my $model_fields = $opts{'model_fields'};

        throw Exception::Validation::BadArguments gettext('Unknown field "%s"', $field)
          unless defined($model_fields->{$field});

        $self->{'__DB_FILTER__'}{$field}->check($data, $model_fields->{$field})
          if $self->{'__DB_FILTER__'}{$field}->can('check');

        return ($opts{'type'} || '') eq 'text'
          ? $self->{'__DB_FILTER__'}{$field}->as_text($data, $model_fields->{$field}, %opts)
          : return $self->_db()->filter(
              $model_fields->{$field}{'db_filter'}
            ? $model_fields->{$field}{'db_filter'}($self, $data, $model_fields->{$field}, %opts)
            : $self->{'__DB_FILTER__'}{$field}->as_filter($data, $model_fields->{$field}, %opts)
          );

    } else {
        throw Exception::Validation::BadArguments gettext('Bad filter data');
    }
}

sub _get_db_filter_from_text {
    my ($self, $data, %opts) = @_;

    my $db_accessor = $self->get_model_filter_dbaccessor();
    my $model_fields = $opts{'model_fields'} ||= $self->get_db_filter_fields(private => TRUE);

    my $grammar = <<EOF;
%{
use qbit;
no warnings 'redefine';
%}

%whites = /([ \\t\\r\\n]*)/
EOF

    my %tokens = %{$self->_grammar_tokens(%opts, model_fields => $model_fields)};
    $tokens{$_} = QBit::Application::Model::DBManager::Filter::tokens($_) foreach (qw(AND OR));

    $grammar .= "\n%token $_ = {\n    $tokens{$_}->{'re'};\n}\n"
      foreach sort {$tokens{$b}->{'priority'} <=> $tokens{$a}->{'priority'}} keys(%tokens);

    $grammar .= <<EOF;

%left OR
%left AND

%tree
#%strict

%%
start:      expr { \$_[1] }
        ;
EOF

    my @expr = %{$self->_grammar_expr(%opts, model_fields => $model_fields)};
    $grammar .= "\n$expr[0]: $expr[1]";

    my $nonterminals = $self->_grammar_nonterminals(%opts, model_fields => $model_fields);
    $grammar .= "\n\n$_: $nonterminals->{$_}" foreach keys(%$nonterminals);

    $grammar .= "\n%%";

    my $grammar_class_name = ref($self) . '::Grammar';

    my $p = Parse::Eyapp->new_grammar(
        input     => $grammar,
        classname => $grammar_class_name,
    );
    throw $p->Warnings if $p->Warnings;

    my $parser = $grammar_class_name->new();
    $parser->{'__DB__'}    = $self->$db_accessor;
    $parser->{'__MODEL__'} = $self;
    $parser->input(\$data);

    my $filter = $parser->YYParse(
        yyerror => sub {
            my $token = $_[0]->YYCurval();

            my $text = gettext(
                'Syntax error near "%s". Expected one of these tokens: %s',
                $token ? $token : gettext('end of input'),
                join(', ', $_[0]->YYExpect())
            );
            throw Exception::DBManager::Grammar $text;
        }
    );

    return $filter if ($opts{'type'} || '') eq 'json_data';

    return $self->_get_db_filter_from_data(
        $filter, %opts,
        model_fields => $model_fields,
        db_accessor  => $db_accessor
    );
}

sub _get_fields_obj {
    my ($self, $fields, $all_langs) = @_;

    return QBit::Application::Model::DBManager::_Utils::Fields->new(
        $self->get_model_fields_initialized(),
        $self->get_model_fields_sort_orders(),
        $fields, $self, all_langs => $all_langs,
    );
}

sub _grammar_expr {
    my ($self, %opts) = @_;

    $opts{'gns'} ||= '';

    my $res =
"$opts{'gns'}expr AND $opts{'gns'}expr { QBit::Application::Model::DBManager::Filter::__merge_expr(\$_[1], \$_[3], 'AND') }
        |   $opts{'gns'}expr OR $opts{'gns'}expr  { QBit::Application::Model::DBManager::Filter::__merge_expr(\$_[1], \$_[3], 'OR') }
        |    '(' $opts{'gns'}expr ')' { \$_[2] }\n";

    foreach my $field_name (keys(%{$opts{'model_fields'}})) {
        $res .= "        |   " . $_ . "\n"
          foreach
          @{$self->{'__DB_FILTER__'}{$field_name}->expressions($field_name, $opts{'model_fields'}->{$field_name}, %opts)
              || []};
    }

    $res .= "        ;";

    return {"$opts{'gns'}expr" => $res};
}

sub _grammar_nonterminals {
    my ($self, %opts) = @_;

    my %nonterminals;

    foreach my $field_name (keys(%{$opts{'model_fields'}})) {
        push_hs(%nonterminals,
            $self->{'__DB_FILTER__'}{$field_name}
              ->nonterminals($field_name, $opts{'model_fields'}->{$field_name}, %opts))
          if $self->{'__DB_FILTER__'}{$field_name}->can('nonterminals');
    }

    return \%nonterminals;
}

sub _grammar_tokens {
    my ($self, %opts) = @_;

    my %tokens;

    foreach my $field_name (keys(%{$opts{'model_fields'}})) {
        $tokens{uc($field_name)} = {
            re       => "/\\G(" . uc($field_name) . ")/igc and return (" . uc($field_name) . " => \$1)",
            priority => length($field_name)
        };

        foreach my $token (@{$self->{'__DB_FILTER__'}{$field_name}->need_tokens || []}) {
            $tokens{$token} = QBit::Application::Model::DBManager::Filter::tokens($token);
        }

        push_hs(%tokens,
            $self->{'__DB_FILTER__'}{$field_name}->tokens($field_name, $opts{'model_fields'}->{$field_name}, %opts))
          if $self->{'__DB_FILTER__'}{$field_name}->can('tokens');
    }

    return \%tokens;
}

sub _parse_filter {
    my ($self, $join_models, $join_options, $model, $filter, $model_filters) = @_;

    if (exists($model_filters->{$filter->[0]}) && $model_filters->{$filter->[0]}{'type'} ne 'subfilter') {
        push(@{$join_options->{$model}{'filter'}}, $filter);
    } elsif (exists($model_filters->{$filter->[0]}) && $model_filters->{$filter->[0]}{'type'} eq 'subfilter') {
        my $field    = $model_filters->{$filter->[0]};
        my $accessor = $field->{'model_accessor'};

        push(@$join_models, $accessor) unless in_array($accessor, $join_models);

        $join_options->{$accessor}{'join_on'} =
          [$field->{'fk_field'} => '=' => {$field->{'field'} => $self->partner_db->$model}]
          unless exists($join_options->{$accessor});

        $self->_parse_filter($join_models, $join_options, $accessor, $filter->[2],
            $self->$accessor->get_db_filter_fields(private => TRUE));
    } else {
        throw gettext('Unknown filter');
    }
}

TRUE;
