package QBit::Application::Model::DBManager::Filter::subfilter;

use qbit;

use base qw(QBit::Application::Model::DBManager::Filter);

use Exception::Validation::BadArguments;

=head2

$data - сам фильтр.  Пример: ["blocks", "MATCH", ["block_public_id", "IN", ["65055-98" ] ] ] ]

$field:
    type            - тип фильтра - subfilter
    model_accessor  - аксессор сабмодели
    field           - поле для связи в своей модели
    fk_field        - поле для связи в сабмодели
    model_filter    - Если нужно докинуть специфичный фильтр в свою модель
    submodel_filter - Если нужно докинуть специфичный фильтр в сабмодель

%opts
    { 'model_fields' => { <field_name> => <field_data>, .. } }

=cut

sub as_filter {
    my ($self, $data, $field, %opts) = @_;

    my $accessor_name = $field->{'model_accessor'};

    my $tmp_rights;
    $tmp_rights = $self->{'db_manager'}->app->add_tmp_rights($field->{'add_tmp_rights'})
      if $field->{'add_tmp_rights'};

    my $submodel_filter = $field->{'submodel_filter'} ? ['AND', [$data->[2], $field->{'submodel_filter'}]] : $data->[2];
    my $filter = [
        ($field->{'field'} || $data->[0]) => ($data->[1] =~ /NOT/ ? '<> ALL' : '= ANY') =>
          $self->{'db_manager'}->$accessor_name->query(
            fields => $self->{'db_manager'}->$accessor_name->_get_fields_obj(
                exists($field->{'fk_field'})
                ? [$field->{'fk_field'}]
                : $self->{'db_manager'}->$accessor_name->get_pk_fields()
            ),
            filter => $self->{'db_manager'}->$accessor_name->_get_db_filter_from_data(
                $submodel_filter, %opts,
                model_fields => $opts{'model_fields'}->{$data->[0]}{'subfields'},
                type         => 'filter'
            )
          )
    ];

    return $field->{'model_filter'} ? ['AND', [$field->{'model_filter'}, $filter]] : $filter;
}

sub as_text {
    my ($self, $data, $field, %opts) = @_;

    my $accessor_name = $field->{'model_accessor'};
    return "$_[1]->[0] $_[1]->[1] {"
      . $self->{'db_manager'}->$accessor_name->_get_db_filter_from_data(
        $data->[2], %opts,
        model_fields => $opts{'model_fields'}->{$data->[0]}{'subfields'},
        type         => 'text'
      ) . '}';
}

sub check {
    throw Exception::Validation::BadArguments gettext('Bad operator "%s" for "%s" (type: %s) in filter', $_[1]->[1],
        $_[1]->[0], $_[0]->type())
      unless $_[1]->[1] eq 'MATCH' || $_[1]->[1] eq 'NOT MATCH';

    throw Exception::Validation::BadArguments gettext('Incorrect data for filter by field "%s" (type: %s)', $_[1]->[0],
        $_[0]->type)
      if ref($_[1]->[2]) ne 'ARRAY';
}

sub expressions {
    my ($self, $field_name, $field) = @_;

    my $uc_field_name = uc($field_name);

    return [
        $uc_field_name
          . " MATCH     '{' "
          . ($field->{'ns'} || '')
          . '___expr'
          . " '}' { [$field_name => 'MATCH'     => \$_[4]] }",
        $uc_field_name
          . " NOT MATCH '{' "
          . ($field->{'ns'} || '')
          . '___expr'
          . " '}' { [$field_name => 'NOT MATCH' => \$_[5]] }"
    ];
}

sub is_simple {return FALSE;}

sub need_tokens {return [qw(NOT MATCH)];}

sub nonterminals {
    my ($self, $field_name, $field) = @_;

    return $field->{'nonterminals'};
}

sub pre_process {
    my ($self, $field, $field_name, %opts) = @_;

    my $accessor_name = $field->{'model_accessor'};

    $opts{'subfilters'} ||= {};
    return FALSE if $opts{'subfilters'}->{$self->{'db_manager'}->$accessor_name};
    local ($opts{'subfilters'}->{$self->{'db_manager'}}) = TRUE;

    $opts{'ns'}           ||= [];
    $opts{'nonterminals'} ||= {};
    $opts{'tokens'}       ||= {};

    if (!exists($opts{'cache'}->{$self->{'db_manager'}->$accessor_name})) {
        $opts{'cache'}->{$self->{'db_manager'}->$accessor_name} =
          $self->{'db_manager'}->$accessor_name->get_db_filter_fields(
            %opts,
            ns      => [@{$opts{'ns'}}, $field_name],
            private => TRUE
          );
    }

    my $subfilter_fields = $opts{'cache'}->{$self->{'db_manager'}->$accessor_name};

    $field->{'subfields'} = clone($subfilter_fields);

    $field->{'ns'} = lc(join('___', @{$opts{'ns'}}, $field_name));

    my @expr = %{
        $self->{'db_manager'}->$accessor_name->_grammar_expr(
            %opts,
            ns           => [@{$opts{'ns'}}, $field_name],
            gns          => $field->{'ns'} . '___',
            model_fields => $subfilter_fields
        )
      };

    $opts{'nonterminals'}->{$expr[0]} = $expr[1];
    push_hs(
        $opts{'nonterminals'},
        $self->{'db_manager'}->$accessor_name->_grammar_nonterminals(
            %opts,
            ns           => [@{$opts{'ns'}}, $field_name],
            model_fields => $subfilter_fields
        )
    );
    $field->{'nonterminals'} = $opts{'nonterminals'};

    push_hs(
        $opts{'tokens'},
        $self->{'db_manager'}->$accessor_name->_grammar_tokens(
            %opts,
            ns           => [@{$opts{'ns'}}, $field_name],
            model_fields => $subfilter_fields
        )
    );
    $field->{'tokens'} = $opts{'tokens'};

    return TRUE;
}

sub public_keys {return [qw(subfields)];}

sub tokens {
    my ($self, $field_name, $field) = @_;

    return $field->{'tokens'};
}

TRUE;
